c***********************************************************************
      program xyz2poscar
c***********************************************************************
c-----------------------------------------------------------------------
c     //   variables
c-----------------------------------------------------------------------

c     //   initialize
      implicit none

c     //   unit conversion
      real(8) :: bohr2ang = 0.529177246d0

c     //   cartesian coordinates
      real(8), dimension(:), allocatable :: x
      real(8), dimension(:), allocatable :: y
      real(8), dimension(:), allocatable :: z

c     //   box matrix
      real(8), dimension(3,3) :: box

c     //   atomic species
      character(len=6), dimension(:), allocatable :: species

c     //   atomic species
      character(len=6), dimension(:), allocatable :: spec

c     //   number of atoms per species
      integer, dimension(:), allocatable :: natom_spec

c     //   number of atoms
      integer :: natom

c     //   number of species
      integer :: nspec

c     //   file unit
      integer :: iounit = 10

c     //   file names
      character(len=80) :: inpfile
      character(len=80) :: xyzfile
      character(len=80) :: posfile

c     //   units
      character(len=10) :: boxunit
      character(len=10) :: xyzunit

c     //   integers
      integer :: i, j, k, ierr, ispec, iargc

c     //   characters
      character(len=80) :: charline

c-----------------------------------------------------------------------
c     //   initial message
c-----------------------------------------------------------------------

      if ( iargc() .ne. 3 ) then

         write( 6, '(a)' )
         write( 6, '(a)' ) 'Program xyz2poscar'
         write( 6, '(a)' )
         write( 6, '(a)' ) 'Usage: xyz2poscar.x $1 $2 $3'
         write( 6, '(a)' )
         write( 6, '(a)' ) '$1: PIMD input file: (input.dat)'
         write( 6, '(a)' ) '$2: xyz file:        (structure.dat)'
         write( 6, '(a)' ) '$3: poscar file      (POSCAR)'
         write( 6, '(a)' )
         write( 6, '(a)' ) 'Example:'
         write( 6, '(a)' ) 'xyz2poscar.x input.dat structure.dat' //
     &                     ' POSCAR'
         write( 6, '(a)' )

         stop

      else

         write( 6, '(a)' )
         write( 6, '(a)' ) 'Program xyz2poscar'
         write( 6, '(a)' )

      end if

c-----------------------------------------------------------------------
c     //   files
c-----------------------------------------------------------------------

      call getarg( 1, inpfile )
      call getarg( 2, xyzfile )
      call getarg( 3, posfile )

c-----------------------------------------------------------------------
c     //   read box from inpfile
c-----------------------------------------------------------------------

c     //   read box
      open ( iounit, file = trim(inpfile) )
      do
         read ( iounit, *, iostat=ierr ) charline
         if ( ierr .ne. 0 ) exit
         if ( charline(1:11) .eq. '<iboundary>' ) exit
      end do
      read ( iounit, *, iostat=ierr ) boxunit
      read ( iounit, *, iostat=ierr ) box(1,1:3)
      read ( iounit, *, iostat=ierr ) box(2,1:3)
      read ( iounit, *, iostat=ierr ) box(3,1:3)
      close( iounit )

c     //   error termination
      if ( ierr .ne. 0 ) then
         write( 6, '(a)' ) 'Error: ' // trim(inpfile) // '.'
         write( 6, '(a)' ) 'keyword iboundary read incorrectly.'
         write( 6, '(a)' )
         stop
      end if

c     //   unit conversion
      if ( ( boxunit(1:1) .eq. 'B' ) .or.
     &     ( boxunit(1:1) .eq. '1' ) .or.
     &     ( boxunit(1:1) .eq. '2' ) ) then
         box(:,:) = box(:,:) * bohr2ang
      else if ( boxunit(1:1) .eq. '0' ) then
         ierr = 1
         write( 6, '(a)' ) 'Error: ' // trim(inpfile) // '.'
         write( 6, '(a)' ) 'keyword iboundary read incorrectly.'
         write( 6, '(a)' )
         stop
      end if

c     //   message
      if ( boxunit(1:1) .eq. 'B' ) then
         write( 6, '(a,10x,a)' )
     &      'Read ' // trim(inpfile) // ':', 'box in bohr.'
      else
         write( 6, '(a,10x,a)' )
     &      'Read ' // trim(inpfile) // ':', 'box in angstrom.'
      end if

c-----------------------------------------------------------------------
c     //   read structure from xyzfile
c-----------------------------------------------------------------------

c     //   read number of atoms
      open ( iounit, file = trim(xyzfile) )
      read ( iounit, * ) natom
      close( iounit )

c     //   memory allocation
      allocate( species(natom) )
      allocate( x(natom) )
      allocate( y(natom) )
      allocate( z(natom) )

c     //   memory allocation
      allocate( spec(natom) )
      allocate( natom_spec(natom) )

c     //   structure
      open ( iounit, file = xyzfile )
      read ( iounit, *, iostat=ierr )
      read ( iounit, *, iostat=ierr ) xyzunit
      do i = 1, natom
         read( iounit, *, iostat=ierr ) species(i), x(i), y(i), z(i)
      end do
      close( iounit )

c     //   error termination
      if ( ierr .ne. 0 ) then
         write( 6, '(a)' ) 'Error: ' // trim(xyzfile) // '.'
         write( 6, '(a)' )
         stop
      end if

c     //   unit conversion
      if ( xyzunit(1:1) .eq. 'B' ) then
         x(:) = x(:) * bohr2ang
         y(:) = y(:) * bohr2ang
         z(:) = z(:) * bohr2ang
      end if

c     //   message
      if ( xyzunit(1:1) .eq. 'B' ) then
         write( 6, '(a,3x,a)' )
     &      'Read ' // trim(xyzfile) // ':', 'atoms in bohr.'
      else
         write( 6, '(a,6x,a)' )
     &      'Read ' // trim(xyzfile) // ':', 'atoms in angstrom.'
      end if

c-----------------------------------------------------------------------
c     //   number of species
c-----------------------------------------------------------------------

c     //   1st species
      ispec = 1
      spec(1) = species(1)

c     //   remaining species
      do i = 2, natom
         k = 0
         do j = 1, ispec
            if ( species(i)(1:2) .eq. spec(j)(1:2) ) k = 1
         end do
         if ( k .eq. 0 ) then
            ispec = ispec + 1
            spec(ispec) = species(i)
         end if
      end do

c     //   number of species
      nspec = ispec

c     //   number of atoms per species
      do j = 1, nspec
         natom_spec(j) = 0
         do i = 1, natom
            if ( species(i)(1:2) .ne. spec(j)(1:2) ) cycle
            natom_spec(j) = natom_spec(j) + 1
         end do
      end do

      i = 0
      do j = 1, nspec
         i = i + natom_spec(j)
      end do

c     //   message
      write( 6, '(a,3x,i5)' )
     &   'Number of species:', nspec
      write( 6, '(a,5x,i5)' )
     &   'Number of atoms:  ', natom
      do j = 1, nspec
         write( 6, '(a,i5)' ) spec(j), natom_spec(j)
      end do
      write( 6, '(a,i5)' )
     &   'Number of atoms (sum): ', i
      write( 6, '(a)' )

c-----------------------------------------------------------------------
c     //   poscar file
c-----------------------------------------------------------------------

c     //   open file
      open ( iounit, file = trim(posfile) )

c     //   comment line
      write( iounit, '(a)' ) 'Created by PIMD.'

c     //   scaling factor
      write( iounit, '(f16.8)' )  1.d0

c     //   lattice vectors
      write( iounit, '(3f16.8)' ) box(1:3,1)
      write( iounit, '(3f16.8)' ) box(1:3,2)
      write( iounit, '(3f16.8)' ) box(1:3,3)

c     //   species
      do j = 1, nspec-1
         write( iounit, '(a)', advance='no' ) spec(j)(1:6)
      end do
      write( iounit, '(a)' ) spec(nspec)(1:6)

c     //   species
      do j = 1, nspec-1
         write( iounit, '(i6)', advance='no' ) natom_spec(j)
      end do
      write( iounit, '(i6)' ) natom_spec(nspec)

c     //   cartesian
      write( iounit, '(a)' ) 'Cartesian'

c     //   write in the order of species
      do j = 1, nspec
      do i = 1, natom
         if ( species(i)(1:2) .ne. spec(j)(1:2) ) cycle
         write( iounit, '(3f16.8,a,i6)' )
     &     x(i), y(i), z(i), ' ! ', i
      end do
      end do

c     //   close file
      close( iounit )

c     //   message
      write( 6, '(a)' ) 'Written to ' // trim(posfile) // '.'
      write( 6, '(a)' ) 
      write( 6, '(a)' ) 'Normal termination.'
      write( 6, '(a)' ) 

c-----------------------------------------------------------------------
c     //   end of code
c-----------------------------------------------------------------------

      stop
      end

