c///////////////////////////////////////////////////////////////////////
c
c      Author:          M. Shiga
c      Last updated:    Aug 21, 2024 by M. Shiga
c      Description:     print xsf file
c
c///////////////////////////////////////////////////////////////////////
c***********************************************************************
      program xsf2npf
c***********************************************************************

c     //   local variables
      implicit none

c     //   atomic coordinates
      real(8), dimension(:,:), allocatable :: x, y, z

c     //   potential energies
      real(8), dimension(:), allocatable :: pot

c     //   atomic forces
      real(8), dimension(:,:), allocatable :: fx, fy, fz

c     //   atomic forces
      real(8), dimension(3,3) :: box

c     //   total charge
      real(8) :: charge

c     //   atomic symbols
      character(len=4), dimension(:), allocatable :: species

c     //   input and output files
      character(len=120) :: xsf, npf

c     //   characters
      character(len=120) :: char

c     //   constants
      real(8), parameter:: au_energy  = 4.3597482d-18
      real(8), parameter:: au_charge  = 1.60217646d-19
      real(8), parameter:: au_length  = 0.529177249d-10

c     //   integers
      integer :: i, j, nstep, natom, nbead, iargc, ierr

c     //   real numbers
      real(8) :: const_1, const_2, const_3

c-----------------------------------------------------------------------
c     //   initial message
c-----------------------------------------------------------------------

      write( 6, '(a)' )
      write( 6, '(a)' ) 'Program xsf2npf_pi'
      write( 6, '(a)' )

      if ( iargc() .ne. 5 ) then

         write( 6, '(a)' ) 'Usage: xsf2npf_pi.x $1 $2 $3 $4 $5'
         write( 6, '(a)' )
         write( 6, '(a)' ) '$1: number of atoms'
         write( 6, '(a)' ) '$2: number of beads'
         write( 6, '(a)' ) '$3: total charge'
         write( 6, '(a)' ) '$4: input xsf file (trj.xsf)'
         write( 6, '(a)' ) '$5: output n2p2 file (trj.npf)'
         write( 6, '(a)' )

         stop

      end if

c-----------------------------------------------------------------------
c     //   read values
c-----------------------------------------------------------------------

c     //   number of atoms
      call getarg( 1, char )
      read( char, * ) natom

c     //   number of atoms
      call getarg( 2, char )
      read( char, * ) nbead

c     //   charge
      call getarg( 3, char )
      read( char, * ) charge

c     //   directory
      call getarg( 4, xsf )

c     //   output
      call getarg( 5, npf )

c-----------------------------------------------------------------------
c     //    constant
c-----------------------------------------------------------------------

c     /*   conversion factor   */
      const_1 = au_length * 1.d+10

c     /*   conversion factor   */
      const_2 =  au_charge / au_energy

c     /*   conversion factor   */
      const_3 =  1.d0 / const_2 / const_1

c-----------------------------------------------------------------------
c     //    memory allocation
c-----------------------------------------------------------------------

c     //   atomic symbols
      allocate( species(natom) )

c     //   atomic coordinates
      allocate( x(natom,nbead) )
      allocate( y(natom,nbead) )
      allocate( z(natom,nbead) )

c     //   potential energies
      allocate( pot(nbead) )

c     //   atomic forces
      allocate( fx(natom,nbead) )
      allocate( fy(natom,nbead) )
      allocate( fz(natom,nbead) )

c-----------------------------------------------------------------------
c     //   list of xsf files
c-----------------------------------------------------------------------

c     //   step
      nstep = 0

c     //   open xsf file
      open ( 10, file = trim(xsf) )

c     //   open npf file
      open ( 11, file = trim(npf) )

c     //   read a line
      read ( 10, *, iostat=ierr ) char

c     //   loop of files
      do

c        /*   loop of beads   */
         do j = 1, nbead

c           //   potential energy in eV
            read ( 10, *, iostat=ierr ) char, char, char, char, pot(j)

c        /*   loop of beads   */
         end do

c        //   look for error
         if ( ierr .ne. 0 ) exit

c        //   eV to hartree
         pot(:) = pot(:) * const_2

c        //   read a line
         read ( 10, *, iostat=ierr ) char

c        //   boundary: periodic
         if ( char(1:7) .eq. 'CRYSTAL' ) then

c           //   read a line
            read ( 10, *, iostat=ierr ) char

c           /*   write three lines   */
            read( 10, * ) box(1,1), box(2,1), box(3,1)
            read( 10, * ) box(1,2), box(2,2), box(3,2)
            read( 10, * ) box(1,3), box(2,3), box(3,3)

c           //   look for error
            if ( ierr .ne. 0 ) exit

c           //   read two lines
            read ( 10, *, iostat=ierr ) char
            read ( 10, *, iostat=ierr ) char

c           //   look for error
            if ( ierr .ne. 0 ) exit

c        //   boundary: free
         else

c           //   read a line
            read ( 10, *, iostat=ierr ) char

c           //   look for error
            if ( ierr .ne. 0 ) exit

c        //   boundary
         end if

c        /*   loop of beads   */
         do j = 1, nbead

c           /*   loop of atoms   */
            do i = 1, natom

c              /*   read one line   */
               read( 10, *, iostat=ierr )
     &            species(i),
     &            x(i,j), y(i,j), z(i,j),
     &            fx(i,j), fy(i,j), fz(i,j)

c           /*   loop of atoms   */
            end do

c        /*   loop of beads   */
         end do

c        //   look for error
         if ( ierr .ne. 0 ) exit

c        //   step
         nstep = nstep + 1

c        //   unit conversion
         box(:,:) = box(:,:) / const_1

c        /*   geometry in bohr   */
         x(:,:) = x(:,:) / const_1
         y(:,:) = y(:,:) / const_1
         z(:,:) = z(:,:) / const_1

c        /*   force in hartree per bohr   */
         fx(:,:) = fx(:,:) / const_3
         fy(:,:) = fy(:,:) / const_3
         fz(:,:) = fz(:,:) / const_3

c        /*   loop of beads   */
         do j = 1, nbead

c           /*   write one line   */
            write( 11, '(a)' ) 'begin'
            write( 11, '(a)' ) 'comment'
            write( 11, '(a,3e16.8)' ) 'lattice', box(1:3,1)
            write( 11, '(a,3e16.8)' ) 'lattice', box(1:3,2)
            write( 11, '(a,3e16.8)' ) 'lattice', box(1:3,3)

c           /*   loop of atoms   */
            do i = 1, natom

               write( 11, '(a,3e16.8,2x,a,5e16.8)' )
     &            'atom   ',
     &            x(i,j), y(i,j), z(i,j),
     &            species(i),
     &            0.d0, 0.d0,
     &            fx(i,j), fy(i,j), fz(i,j)

c           /*   loop of atoms   */
            end do

            write( 11, '(a,e24.16)' ) 'energy ', pot(j)
            write( 11, '(a,e24.16)' ) 'charge ', charge
            write( 11, '(a)' ) 'end'

c        /*   loop of beads   */
         end do

c     //   loop of files
      end do

c     //   close xsf file
      close( 10 )

c     //   close npf file
      close( 11 )

c-----------------------------------------------------------------------
c     //   final message
c-----------------------------------------------------------------------

      write( 6, '(a,i8)' )
     &   'Number of structures:', nstep
      write( 6, '(a)' )

      write( 6, '(a)' )
     &   'Normal termination of xsf2npf_pi.'
      write( 6, '(a)' )

      stop
      end
