!***********************************************************************
      program waterbox
!***********************************************************************
!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

!     //   initialize
      implicit none

!     //   circular constant
      real(8) :: pi

!     //   box
      real(8) :: box(3,3)

!     //   cubic box
      real(8) :: boxl

!     //   number of atoms
      integer :: natom

!     //   number of fixed atoms
      integer :: natom0 = 0

!     //   number of molecules
      integer :: nmol

!     //   integers
      integer :: maxtry = 100000

!     //   integers
      integer :: i, j, k, l, m, itry, iflag, ierr

!     //   real numbers
      real(8) :: psi, theta, phi, r, rx, ry, rz, r1, r2, xg, yg, zg, d0

!     //   real numbers
      real(8) :: s(3), a(3), b(3), c(3), d(3)

!     //   atomic coordinates
      real(8), dimension(:), allocatable :: x, y, z

!     //   fixed atomic coordinates
      real(8), dimension(:), allocatable :: x0, y0, z0

!     //   atomic coordinates of water
      real(8), dimension(3) :: xmol, ymol, zmol

!     //   characters
      character(len=80) :: char, xyzfile, inpfile, xyz0file

!     //   characters
      character(len=8) ::  option, unit

!     //   characters
      character(len=8), dimension(:), allocatable ::  spec0

!     //   unit conversion factor
      real(8) :: bohr2ang = 0.529177246d0

!-----------------------------------------------------------------------
!     //   initial message
!-----------------------------------------------------------------------

      if ( ( iargc() .lt. 4 ) .or. ( ( iargc() .gt. 5 ) ) ) then

         write( 6, '(a)' )
         write( 6, '(a)' ) 'Program waterbox'
         write( 6, '(a)' )
         write( 6, '(a)' ) 'Usage: waterbox $1 $2 $3 $4 $5'
         write( 6, '(a)' )
         write( 6, '(a)' ) '$1:  option: CUBIC/PPHEX'
         write( 6, '(a)' )
         write( 6, '(a)' ) 'When $1 is CUBIC:'
         write( 6, '(a)' ) '$2:  number of water molecules'
         write( 6, '(a)' ) '$3:  box side length [angstrom]'
         write( 6, '(a)' ) '$4:  output xyz file'
         write( 6, '(a)' ) '$5:  (blank)'
         write( 6, '(a)' )
         write( 6, '(a)' ) 'When $1 is PPHEX:'
         write( 6, '(a)' ) '$2:  number of water molecules'
         write( 6, '(a)' ) '$3:  input box file (input.dat)'
         write( 6, '(a)' ) '$4:  output xyz file'
         write( 6, '(a)' ) '$5:  (blank)'
         write( 6, '(a)' )
         write( 6, '(a)' ) 'When $1 is SLAB:'
         write( 6, '(a)' ) '$2:  number of water molecules'
         write( 6, '(a)' ) '$3:  input box file (input.dat)'
         write( 6, '(a)' ) '$4:  output xyz file'
         write( 6, '(a)' ) '$5:  input frozen atom file in xyz format'
         write( 6, '(a)' )
         write( 6, '(a)' ) 'Atomic radius for water:'
         write( 6, '(a)' ) 'O: 1.25 [angstrom],  H: 0.65 [angstrom]'
         write( 6, '(a)' )
         write( 6, '(a)' ) 'Atomic radius for slab:'
         write( 6, '(a)' ) 'M: 3.00 [angstrom]'
         write( 6, '(a)' )
         write( 6, '(a)' ) 'Examples:'
         write( 6, '(a)' ) &
     &      './waterbox CUBIC 256 19.7 water.xyz'
         write( 6, '(a)' ) &
     &      './waterbox PPHEX 256 input.dat water.xyz'
         write( 6, '(a)' ) &
     &      './waterbox SLAB 256 input.dat water.xyz frozen.xyz'
         write( 6, '(a)' )

         stop

      else

         write( 6, '(a)' )
         write( 6, '(a)' ) 'Program waterbox'
         write( 6, '(a)' )

      end if

!-----------------------------------------------------------------------
!     //   circular constant
!-----------------------------------------------------------------------

      pi = acos(-1.d0)

!-----------------------------------------------------------------------
!     //   atomic radius parameters
!-----------------------------------------------------------------------

      d(1) = 1.25d0
      d(2) = 0.65d0
      d(3) = 0.65d0

!-----------------------------------------------------------------------
!     //   atomic radius parameters for slab
!-----------------------------------------------------------------------

      d0   = 2.00d0

!-----------------------------------------------------------------------
!     //   read number of molecules
!-----------------------------------------------------------------------

      call getarg( 1, char )
      read( char, * ) option

!-----------------------------------------------------------------------
!     //   option
!-----------------------------------------------------------------------

!     //   option = CUBIC
      if ( option(1:5) .eq. 'CUBIC' ) then

         call getarg( 2, char )
         read( char, * ) nmol

         call getarg( 3, char )
         read( char, * ) boxl

         call getarg( 4, char )
         read( char, * ) xyzfile

!        //   cubic box
         box(1,1) = boxl
         box(1,2) = 0.d0
         box(1,3) = 0.d0
         box(2,1) = 0.d0
         box(2,2) = boxl
         box(2,3) = 0.d0
         box(3,1) = 0.d0
         box(3,2) = 0.d0
         box(3,3) = boxl

!     //   option = CUBIC
      else if ( option(1:5) .eq. 'PPHEX' ) then

         call getarg( 2, char )
         read( char, * ) nmol

         call getarg( 3, char )
         read( char, * ) inpfile

         open ( 10, file = trim(inpfile) )
         do
            read ( 10, *, iostat=ierr ) char
            if ( ierr .ne. 0 ) exit
            if ( char(1:11) .eq. '<iboundary>' ) then
               read ( 10, *, iostat=ierr ) unit
               read ( 10, *, iostat=ierr ) box(1,1:3)
               read ( 10, *, iostat=ierr ) box(2,1:3)
               read ( 10, *, iostat=ierr ) box(3,1:3)
               exit
            end if
         end do
         close( 10 )

!        //   error message
         if ( ierr .ne. 0 ) then
            write( 6, '(a)' ) &
     &         'Error: Reading box from ' // trim(inpfile) // '.'
            write( 6, '(a)' )
            stop
         end if

!        //   unit conversion: bohr -> angstrom
         if ( unit(1:1) .eq. '1' ) box(:,:) = box(:,:) * bohr2ang
         if ( unit(1:1) .eq. '2' ) box(:,:) = box(:,:) * bohr2ang
         if ( unit(1:1) .eq. 'B' ) box(:,:) = box(:,:) * bohr2ang

         call getarg( 4, char )
         read( char, * ) xyzfile

!     //   option = SLAB
      else if ( option(1:4) .eq. 'SLAB' ) then

         call getarg( 2, char )
         read( char, * ) nmol

         call getarg( 3, char )
         read( char, * ) inpfile

         open ( 10, file = trim(inpfile) )
         do
            read ( 10, *, iostat=ierr ) char
            if ( ierr .ne. 0 ) exit
            if ( char(1:11) .eq. '<iboundary>' ) then
               read ( 10, *, iostat=ierr ) unit
               read ( 10, *, iostat=ierr ) box(1,1:3)
               read ( 10, *, iostat=ierr ) box(2,1:3)
               read ( 10, *, iostat=ierr ) box(3,1:3)
               exit
            end if
         end do
         close( 10 )

!        //   error message
         if ( ierr .ne. 0 ) then
            write( 6, '(a)' ) &
     &         'Error: Reading box from ' // trim(inpfile) // '.'
            write( 6, '(a)' )
            stop
         end if

!        //   unit conversion: bohr -> angstrom
         if ( unit(1:1) .eq. '1' ) box(:,:) = box(:,:) * bohr2ang
         if ( unit(1:1) .eq. '2' ) box(:,:) = box(:,:) * bohr2ang
         if ( unit(1:1) .eq. 'B' ) box(:,:) = box(:,:) * bohr2ang

         call getarg( 4, char )
         read( char, * ) xyzfile

         call getarg( 5, char )
         read( char, * ) xyz0file

         open ( 10, file = trim(xyz0file) )
         read ( 10, *, iostat=ierr ) natom0
         close( 10 )

!        //   error message
         if ( ierr .ne. 0 ) then
            write( 6, '(a)' ) &
     &         'Error: Reading atoms from ' // trim(xyz0file) // '.'
            write( 6, '(a)' )
            stop
         end if

         allocate( x0(natom0) )
         allocate( y0(natom0) )
         allocate( z0(natom0) )
         allocate( spec0(natom0) )

         open ( 10, file = trim(xyz0file) )
         read ( 10, *, iostat=ierr )
         read ( 10, *, iostat=ierr ) unit
         do i = 1, natom0
            read ( 10, *, iostat=ierr ) spec0(i), x0(i), y0(i), z0(i)
         end do
         close( 10 )

!        //   error message
         if ( ierr .ne. 0 ) then
            write( 6, '(a)' ) &
     &         'Error: Reading atoms from ' // trim(xyz0file) // '.'
            write( 6, '(a)' )
            stop
         end if

!        //   unit conversion: bohr -> angstrom
         if ( unit(1:1) .eq. 'B' ) then
            x0(:) = x0(:) * bohr2ang
            y0(:) = y0(:) * bohr2ang
            z0(:) = z0(:) * bohr2ang
         end if

!     //   option
      else

!        //   error message
         if ( ierr .ne. 0 ) then
            write( 6, '(a,i8)' ) 'Error: Reading option.'
            write( 6, '(a)' )
            stop
         end if

!     //   option
      end if

!-----------------------------------------------------------------------
!     //   number of atoms
!-----------------------------------------------------------------------

      natom = 3 * nmol

!-----------------------------------------------------------------------
!     //   memory allocation
!-----------------------------------------------------------------------

      allocate( x(natom) )
      allocate( y(natom) )
      allocate( z(natom) )

!-----------------------------------------------------------------------
!     //   water molecule
!-----------------------------------------------------------------------

!     //   O atom
      xmol(1) =  0.0000
      ymol(1) =  0.0000
      zmol(1) =  0.0000

!     //   H atom
      xmol(2) =  0.7880
      ymol(2) =  0.7880
      zmol(2) =  0.0000

!     //   H atom
      xmol(3) = -0.7880
      ymol(3) =  0.7880
      zmol(3) =  0.0000

!-----------------------------------------------------------------------
!     //   loop of molecules
!-----------------------------------------------------------------------

      do k = 0, natom-3, 3

!-----------------------------------------------------------------------
!        //   a trial geometry of new molecule
!-----------------------------------------------------------------------

         do itry = 1, maxtry

!-----------------------------------------------------------------------
!           //   random translation
!-----------------------------------------------------------------------

            call random_number(r)
            s(1)  = r

            call random_number(r)
            s(2)  = r

            call random_number(r)
            s(3)  = r

            xg = box(1,1)*s(1) + box(1,2)*s(2) + box(1,3)*s(3)
            yg = box(2,1)*s(1) + box(2,2)*s(2) + box(2,3)*s(3)
            zg = box(3,1)*s(1) + box(3,2)*s(2) + box(3,3)*s(3)

!-----------------------------------------------------------------------
!           //   random rotation
!-----------------------------------------------------------------------

            call random_number(r)
            phi   = r * 2.d0*pi

            call random_number(r)
            theta = r * 2.d0*pi

            call random_number(r)
            psi   = r * 2.d0*pi

!-----------------------------------------------------------------------
!           //   water molecule
!-----------------------------------------------------------------------

            x(k+1:k+3) = xmol(1:3)
            y(k+1:k+3) = ymol(1:3)
            z(k+1:k+3) = zmol(1:3)

!-----------------------------------------------------------------------
!           //   random rotation around z axis
!-----------------------------------------------------------------------

            a(1:3) =   cos(phi)*x(k+1:k+3) + sin(phi)*y(k+1:k+3)
            b(1:3) = - sin(phi)*x(k+1:k+3) + cos(phi)*y(k+1:k+3)

            x(k+1:k+3) = a(1:3)
            y(k+1:k+3) = b(1:3)

!-----------------------------------------------------------------------
!           //   random rotation around x axis
!-----------------------------------------------------------------------

            b(1:3) =   cos(theta)*y(k+1:k+3) + sin(theta)*z(k+1:k+3)
            c(1:3) = - sin(theta)*y(k+1:k+3) + cos(theta)*z(k+1:k+3)

            y(k+1:k+3) = b(1:3)
            z(k+1:k+3) = c(1:3)

!-----------------------------------------------------------------------
!           //   random rotation around z axis
!-----------------------------------------------------------------------

            a(1:3) =   cos(phi)*x(k+1:k+3) + sin(phi)*y(k+1:k+3)
            b(1:3) = - sin(phi)*x(k+1:k+3) + cos(phi)*y(k+1:k+3)

            x(k+1:k+3) = a(1:3)
            y(k+1:k+3) = b(1:3)

!-----------------------------------------------------------------------
!           //   random rotation around z axis
!-----------------------------------------------------------------------

            do j = 1, 3
               x(k+j) = x(k+j) + xg
               y(k+j) = y(k+j) + yg
               z(k+j) = z(k+j) + zg
            end do

!-----------------------------------------------------------------------
!           //   check interatomic distances
!-----------------------------------------------------------------------

!           //   flag
            iflag = 0

!           //   loop of atom pairs
            do l = 1, 3

!              //   loop of atom pairs
               do j = 0, k-3, 3

!                 //   loop of atom pairs
                  do m = 1, 3

!                    //   distance
                     rx = x(k+l) - x(j+m)
                     ry = y(k+l) - y(j+m)
                     rz = z(k+l) - z(j+m)

!                    //   apply periodic boundary condition
                     call pbc_atom_box( rx, ry, rz, box )

!                    //   distance squared
                     r2 = rx*rx + ry*ry + rz*rz

!                    //   distance squared
                     r1 = sqrt(r2)

!                    //   reject if too close
                     if ( r1 .lt. (d(l)+d(m)) ) then
                        iflag = 1
                        go to 100
                     end if

!                 //   loop of atom pairs
                  end do

!              //   loop of atom pairs
               end do

!           //   loop of atom pairs
            end do

!           //   loop of atom pairs
            do l = 1, 3

!              //   loop of atom pairs
               do m = 1, natom0

!                 //   distance
                  rx = x(k+l) - x0(m)
                  ry = y(k+l) - y0(m)
                  rz = z(k+l) - z0(m)

!                 //   apply periodic boundary condition
                  call pbc_atom_box( rx, ry, rz, box )

!                 //   distance squared
                  r2 = rx*rx + ry*ry + rz*rz

!                 //   distance squared
                  r1 = sqrt(r2)

!                 //   reject if too close
                  if ( r1 .lt. (d(l)+d0) ) then
                     iflag = 1
                     go to 100
                  end if

!              //   loop of atom pairs
               end do

!           //   loop of atom pairs
            end do

!           //   flag
  100       if ( iflag .eq. 0 ) exit

!-----------------------------------------------------------------------
!        //   loop of trial geometry
!-----------------------------------------------------------------------

         end do

!-----------------------------------------------------------------------
!        //   error termination
!-----------------------------------------------------------------------

         if ( iflag .eq. 1 ) then
            write( 6, '(a,i8)' ) &
     &         'Error: Could not place water molecule.', (k+2)/3
            write( 6, '(a)' ) 
            stop
         end if

!-----------------------------------------------------------------------
!     //  loop of molecules
!-----------------------------------------------------------------------

      end do

!-----------------------------------------------------------------------
!     //   loop of molecules
!-----------------------------------------------------------------------

      open ( 10, file = trim(xyzfile) )

      write( 10, '(i8)' ) natom+natom0
      write( 10, '(a)' ) 'ANGSTROM'

      do i = 1, natom0
         write( 10, '(a4,3f12.6)' ) spec0(i), x0(i), y0(i), z0(i)
      end do

      do i = 1, 3

         if ( i .eq. 1 ) char = 'O'
         if ( i .eq. 2 ) char = 'H'
         if ( i .eq. 3 ) char = 'H'

         do k = i, natom, 3
            write( 10, '(a4,3f12.6)' ) char(1:4), x(k), y(k), z(k)
         end do

      end do

      close( 10 )

!-----------------------------------------------------------------------
!     //   message
!-----------------------------------------------------------------------

      write( 6, '(a)' ) 'Normal termination of waterbox.'
      write( 6, '(a)' ) 

      stop
      end





!***********************************************************************
      subroutine pbc_atom_box ( xi, yi, zi, box )
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      real(8) :: ai, bi, ci, xi, yi, zi, box(3,3), boxinv(3,3)

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      call inv3( box, boxinv )

      ai = boxinv(1,1)*xi + boxinv(1,2)*yi + boxinv(1,3)*zi
      bi = boxinv(2,1)*xi + boxinv(2,2)*yi + boxinv(2,3)*zi
      ci = boxinv(3,1)*xi + boxinv(3,2)*yi + boxinv(3,3)*zi

      ai = ai - dnint(ai)
      bi = bi - dnint(bi)
      ci = ci - dnint(ci)

      xi = box(1,1)*ai + box(1,2)*bi + box(1,3)*ci
      yi = box(2,1)*ai + box(2,2)*bi + box(2,3)*ci
      zi = box(3,1)*ai + box(3,2)*bi + box(3,3)*ci

      return
      end





!***********************************************************************
      subroutine inv3 ( a, ainv )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j

      real(8) :: a(3,3), ainv(3,3), det3, deta

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      ainv(1,1) = + a(2,2)*a(3,3) - a(2,3)*a(3,2)
      ainv(1,2) = + a(3,2)*a(1,3) - a(1,2)*a(3,3)
      ainv(1,3) = + a(1,2)*a(2,3) - a(2,2)*a(1,3)

      ainv(2,1) = + a(2,3)*a(3,1) - a(3,3)*a(2,1)
      ainv(2,2) = + a(3,3)*a(1,1) - a(3,1)*a(1,3)
      ainv(2,3) = + a(1,3)*a(2,1) - a(2,3)*a(1,1)

      ainv(3,1) = + a(2,1)*a(3,2) - a(3,1)*a(2,2)
      ainv(3,2) = + a(3,1)*a(1,2) - a(1,1)*a(3,2)
      ainv(3,3) = + a(1,1)*a(2,2) - a(1,2)*a(2,1)

      deta = det3 ( a )

      do j = 1, 3
      do i = 1, 3
         ainv(i,j) = ainv(i,j)/deta
      end do
      end do

      return
      end





!***********************************************************************
      real(8) function det3 ( a )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      real(8) :: a(3,3)

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      det3 = + a(1,1)*a(2,2)*a(3,3) - a(1,1)*a(2,3)*a(3,2) &
     &       + a(2,1)*a(3,2)*a(1,3) - a(2,1)*a(1,2)*a(3,3) &
     &       + a(3,1)*a(1,2)*a(2,3) - a(3,1)*a(2,2)*a(1,3)

      return
      end
