c***********************************************************************
      program velacf
c***********************************************************************
c-----------------------------------------------------------------------
c     //   variables
c-----------------------------------------------------------------------

c     //   local variables
      implicit none

c     //   atomic velocity
      real(8), dimension(:,:), allocatable :: vx, vy, vz

c     //   centroid velocity
      real(8), dimension(:,:), allocatable :: vxc, vyc, vzc

c     //   autocorrelation function
      real(8), dimension(:), allocatable :: cxx, cyy, czz, csum

c     //   power spectra
      real(8), dimension(:), allocatable :: pxx, pyy, pzz, psum

c     //   step counter
      real(8), dimension(:), allocatable :: steps

c     //   number of atoms
      integer :: natom

c     //   number of atoms for ikind
      integer :: natom_kind

c     //   atomic kind for velocity acf
      integer :: ikind

c     //   number of beads
      integer :: nbead

c     //   number of steps
      integer :: nstep

c     //   correlation steps
      integer :: mstep

c     //   step size in au, femtoseconds
      real(8) :: dt, dt_fs

c     //   time in au, femtoseconds
      real(8) :: t, t_fs

c     //   angular frequency in au, cminv
      real(8) :: omega, omega_cminv

c     //   angular frequency increment
      real(8) :: domega, domega_cminv

c     //   time frame in femtoseconds
      real(8) :: tacf, tacf_fs

c     //   circular constant
      real(8) :: pi

c     //   unit conversion factors
      real(8), parameter :: au_time   = 0.024188843d-15
      real(8), parameter :: au_length = 0.529177249d-10

c     /*   speed of light in SI unit   */
      real(8), parameter :: speedlight_SI = 2.99892458d+8

c     //   filenames
      character(len=80) :: trjfile, strfile, acffile, powfile

c     //   atomic kind (integer)
      integer, dimension(:), allocatable :: kind

c     //   integers
      integer :: i, j, k, istep, jstep, kstep, ierr, nomega

c     //   characters
      character(len=80) :: char

c     //   file unit
      integer :: iounit = 10

c     //   real number
      real(8) :: d, factor_cminv, omega_max_cminv

c-----------------------------------------------------------------------
c     //   initial message
c-----------------------------------------------------------------------

      if ( iargc() .ne. 9 ) then

         write( 6, '(a)' )
         write( 6, '(a)' ) 'Program velacf'
         write( 6, '(a)' )
         write( 6, '(a)' ) 'Usage: velacf.x $1 $2 $3 $4 $5'
         write( 6, '(a)' )
         write( 6, '(a)' ) '$1: trajectory file (trj.out)'
         write( 6, '(a)' ) '$2: structure file (structure.dat)'
         write( 6, '(a)' ) '$3: output file for velocity acf'
         write( 6, '(a)' ) '$4: output file for power spectra'
         write( 6, '(a)' ) '$5: number of beads'
         write( 6, '(a)' ) '$6: step size (fs)'
         write( 6, '(a)' ) '$7: correlation steps'
         write( 6, '(a)' ) '$8: atomic kind for velocity acf (integer)'
         write( 6, '(a)' ) '$9: highest frequency [cm**-1]'
         write( 6, '(a)' )
         write( 6, '(a)' ) 'atomic kind: 5th column of structure.dat'
         write( 6, '(a)' )

         stop

      else

         write( 6, '(a)' )
         write( 6, '(a)' ) 'Program velacf'
         write( 6, '(a)' )

      end if

c-----------------------------------------------------------------------
c     //   constants
c-----------------------------------------------------------------------

      pi = acos(-1.d0)

c     //   constant
      factor_cminv = 1.d0 / (2.d0*pi*speedlight_SI*au_time*100.d0)

c-----------------------------------------------------------------------
c     //   read values
c-----------------------------------------------------------------------

c     //   input filename
      call getarg( 1, trjfile )

c     //   input filename
      call getarg( 2, strfile )

c     //   output filename
      call getarg( 3, acffile )

c     //   output filename
      call getarg( 4, powfile )

c     //   number of beads
      call getarg( 5, char )
      read( char, * ) nbead

c     //   step size
      call getarg( 6, char )
      read( char, * ) dt_fs

c     //   correlation steps
      call getarg( 7, char )
      read( char, * ) mstep

c     //   atomic kind
      call getarg( 8, char )
      read( char, * ) ikind

c     //   highest frequency
      call getarg( 9, char )
      read( char, * ) omega_max_cminv

c-----------------------------------------------------------------------
c     //   print information
c-----------------------------------------------------------------------

      dt =  dt_fs / ( au_time * 1.d+15 )

      tacf_fs = dble(mstep) * dt_fs

      tacf =  tacf_fs / ( au_time * 1.d+15 )

      domega = 2.d0 * pi / tacf

      domega_cminv = domega * factor_cminv

      nomega = int( omega_max_cminv / domega_cminv )

      write( 6, '(a,f10.2)' ) 'Time length (fs):       ', tacf_fs
      write( 6, '(a,f10.2)' ) 'Frequency mesh (/cm):   ', domega_cminv
      write( 6, '(a,f10.2)' ) 'dt (au):                ', dt

c-----------------------------------------------------------------------
c     //   read structure: number of atoms
c-----------------------------------------------------------------------

      open( iounit, file = strfile )

      read( iounit, *, iostat=ierr ) natom

      close( iounit )

c-----------------------------------------------------------------------
c     //   print information
c-----------------------------------------------------------------------

      write( 6, '(a,i8)' ) 'Number of atoms:          ', natom
      write( 6, '(a,i8)' ) 'Number of beads:          ', nbead

c-----------------------------------------------------------------------
c     //   memory allocation: atomic kind
c-----------------------------------------------------------------------

      allocate( vx(natom,nbead) )
      allocate( vy(natom,nbead) )
      allocate( vz(natom,nbead) )

      allocate( kind(natom) )

c-----------------------------------------------------------------------
c     //   read structure: atomic kind
c-----------------------------------------------------------------------

      kind(:) = 0

      open( iounit, file = strfile )

      read( iounit, *, iostat=ierr )
      read( iounit, *, iostat=ierr )

      k = 0

      do i = 1, natom

         read( iounit, *, iostat=ierr ) char, d, d, d, j

         if ( ierr .ne. 0 ) j = 1

         if ( j .eq. ikind ) then
            k = k + 1
            kind(k) = i
         end if

      end do

      close( iounit )

      natom_kind = k

c-----------------------------------------------------------------------
c     //   print information
c-----------------------------------------------------------------------

      write( 6, '(a,i8)' ) 'Number of atoms of kind:  ', natom_kind

c-----------------------------------------------------------------------
c     //   read trajectory: number of steps
c-----------------------------------------------------------------------

      open( iounit, file = trjfile )

      k = 0

      do

         do j = 1, nbead
         do i = 1, natom
            read( iounit, *, iostat=ierr ) char, d, d, d, d, d, d
         end do
         end do

         if ( ierr .ne. 0 ) exit

         k = k + 1

      end do

      close( iounit )

      nstep = k

c-----------------------------------------------------------------------
c     //   print information
c-----------------------------------------------------------------------

      write( 6, '(a,i8)' ) 'Trajectory length:        ', nstep

c-----------------------------------------------------------------------
c     //   memory allocation
c-----------------------------------------------------------------------

      allocate( vxc(natom_kind,nstep) )
      allocate( vyc(natom_kind,nstep) )
      allocate( vzc(natom_kind,nstep) )

c-----------------------------------------------------------------------
c     //   print information
c-----------------------------------------------------------------------

      i = ( 8 * 3 * natom_kind * nstep ) / ( 1024 * 1024 ) + 1

      write( 6, '(a,i8)' ) 'MB allocated for acf:     ', i

c-----------------------------------------------------------------------
c     //   initialize
c-----------------------------------------------------------------------

      vxc(:,:) = 0.d0
      vyc(:,:) = 0.d0
      vzc(:,:) = 0.d0

c-----------------------------------------------------------------------
c     //   read trajectory
c-----------------------------------------------------------------------

      open( iounit, file = trjfile )

      do istep = 1, nstep

         do j = 1, nbead
         do i = 1, natom
            read( iounit, *, iostat=ierr )
     &         k, d, d, d, vx(i,j), vy(i,j), vz(i,j)
         end do
         end do

         do j = 1, nbead
         do k = 1, natom_kind
            i = kind(k)
            vxc(k,istep) = vxc(k,istep) + vx(i,j) / dble(nbead)
            vyc(k,istep) = vyc(k,istep) + vy(i,j) / dble(nbead)
            vzc(k,istep) = vyc(k,istep) + vz(i,j) / dble(nbead)
         end do
         end do

      end do

      close( iounit )

c-----------------------------------------------------------------------
c     //   print information
c-----------------------------------------------------------------------

      write( 6, '(a)' ) 'DONE: read velocity trajectory.'

c-----------------------------------------------------------------------
c     //   memory allocation
c-----------------------------------------------------------------------

      allocate( cxx(0:mstep) )
      allocate( cyy(0:mstep) )
      allocate( czz(0:mstep) )
      allocate( csum(0:mstep) )

      allocate( steps(0:mstep) )

c-----------------------------------------------------------------------
c     //   memory allocation
c-----------------------------------------------------------------------

      cxx(:) = 0.d0
      cyy(:) = 0.d0
      czz(:) = 0.d0

      steps(:) = 0.d0

c-----------------------------------------------------------------------
c     //   compute autocorrelation function
c-----------------------------------------------------------------------

      do istep = 1, nstep
      do jstep = istep, min(istep+mstep,nstep)

         kstep = jstep - istep

         steps(kstep) = steps(kstep) + 1.d0

         do k = 1, natom_kind
            cxx(kstep) = cxx(kstep) + vxc(k,istep) * vxc(k,jstep)
            cyy(kstep) = cyy(kstep) + vyc(k,istep) * vyc(k,jstep)
            czz(kstep) = czz(kstep) + vzc(k,istep) * vzc(k,jstep)
         end do

      end do
      end do

      do kstep = 0, mstep

         if ( nint(steps(kstep)) .eq. 0 ) cycle

         cxx(kstep) = cxx(kstep) / steps(kstep) / dble(natom_kind)
         cyy(kstep) = cyy(kstep) / steps(kstep) / dble(natom_kind)
         czz(kstep) = czz(kstep) / steps(kstep) / dble(natom_kind)

         csum(kstep) = ( cxx(kstep) + cyy(kstep) + czz(kstep) ) / 3.d0

      end do

c-----------------------------------------------------------------------
c     //   print information
c-----------------------------------------------------------------------

      write( 6, '(a)' ) 'DONE: velocity autocorrelation function.'

c-----------------------------------------------------------------------
c     //   print autocorrelation function
c-----------------------------------------------------------------------

      open( iounit, file = acffile )

      do kstep = 0, mstep

         t_fs = dble(kstep) * dt_fs

         write( iounit, '(f16.4,4e16.8)' )
     &      t_fs, csum(kstep), cxx(kstep), cyy(kstep), czz(kstep)

      end do

      close( iounit )

c-----------------------------------------------------------------------
c     //   memory allocation
c-----------------------------------------------------------------------

      allocate( pxx(0:mstep) )
      allocate( pyy(0:mstep) )
      allocate( pzz(0:mstep) )
      allocate( psum(0:mstep) )

c-----------------------------------------------------------------------
c     //   power spectra
c-----------------------------------------------------------------------

      do j = 0, nomega

         omega = dble(j) * domega

         do i = 0, mstep

            t = dble(i) * dt

            pxx(j)  = pxx(j)  + cxx(i)  * cos(omega*t) * dt
            pyy(j)  = pyy(j)  + cxx(i)  * cos(omega*t) * dt
            pzz(j)  = pzz(j)  + cxx(i)  * cos(omega*t) * dt
            psum(j) = psum(j) + csum(i) * cos(omega*t) * dt

         end do

      end do

c-----------------------------------------------------------------------
c     //   print information
c-----------------------------------------------------------------------

      write( 6, '(a)' ) 'DONE: power spectra.'

c-----------------------------------------------------------------------
c     //   print power spectra
c-----------------------------------------------------------------------

      open( iounit, file = powfile )

      do j = 0, nomega

         omega_cminv = dble(j) * domega_cminv

         write( iounit, '(f8.1,4e16.8)' )
     &      omega_cminv, psum(j), pxx(j), pyy(j), pzz(j)

      end do

      close( iounit )

c-----------------------------------------------------------------------
c     //   end of program
c-----------------------------------------------------------------------

      write( 6, '(a)' )
      write( 6, '(a)' ) 'Normal termination of velacf.'
      write( 6, '(a)' )

      stop
      end

