!***********************************************************************
      program vdos_group
!***********************************************************************
!-----------------------------------------------------------------------
!     //   variables
!-----------------------------------------------------------------------

!     //   local variables
      implicit none

!     //   unit conversion factors
      real(8), parameter :: au_time   = 0.024188843d-15
      real(8), parameter :: amu_mass  = 1.6605402d-27
      real(8), parameter :: au_mass   = 9.1093897d-31

!     /*   speed of light in SI unit   */
      real(8), parameter :: speedlight_SI = 2.99892458d+8

!     //   Boltzmann constant
      real(8), parameter :: boltz = 0.316682968d-5

!     /*   circular constant   */
      real(8) :: pi

!     /*   au to wavenumber   */
      real(8) :: factor_cminv

!     /*   wavelength   */
      real(8) :: dnu_cminv

!     /*   number of MD steps   */
      integer :: nstep

!     /*   MD step number   */
      integer :: istep

!     /*   number of atoms   */
      integer :: natom

!     /*   number of beads   */
      integer :: nbead

!     /*   input and output file   */
      integer :: iounit = 10

!     /*   maximum frequency   */
      integer :: nomega

!     /*   number of atomic symbols   */
      integer :: nsymbol

!     /*   atomic mass   */
      real(8), dimension(:), allocatable :: physmass

!     /*   atomic species   */
      character(len=8), dimension(:), allocatable :: species

!     /*   atomic symbols   */
      character(len=8), dimension(:), allocatable :: symbol

!     /*   mass of atomic symbols   */
      real(8), dimension(:), allocatable :: physmass_symbol

!     /*   atomic velocity   */
      real(8), dimension(:,:), allocatable :: vx, vy, vz

!     /*   atomic centroid velocity   */
      real(8), dimension(:), allocatable :: vxc, vyc, vzc

!     /*   step size   */
      real(8) :: dt_fs, dt

!     /*   time   */
      real(8) :: t

!     /*   total time   */
      real(8) :: t_total

!     /*   frequency   */
      real(8) :: omega

!     /*   temperature   */
      real(8) :: temperature

!     /*   normal mode eigenvalues   */
      real(8), dimension(:), allocatable :: e

!     /*   normal mode eigenvectors   */
      real(8), dimension(:,:), allocatable :: u

!     /*   total vibrational density of states   */
      real(8) :: vdos

!     /*   sum of total vibrational density of states   */
      real(8) :: vdos_sum

!     /*   partial vibrational density of states   */
      real(8), dimension(:), allocatable :: pvdos

!     //   filename
      character(len=80) :: trjfile, strfile, inpfile, eigfile, grpfile

!     /*   real numbers   */
      real(8) :: tmp, coswt, sinwt, sqrtm, fg, factor

!     /*   integer   */
      integer :: i, j, k, l, ierr, iargc

!     /*   character   */
      character(len=80) :: char

!     /*   functions   */
      real(8), dimension(:,:), allocatable :: a, b
      real(8), dimension(:,:), allocatable :: f, g

!     /*   number of mode groups   */
      integer :: ngroup

!     /*   group index   */
      integer, dimension(:), allocatable :: igroup

!     /*   group index   */
      integer, dimension(:), allocatable :: jgroup

!-----------------------------------------------------------------------
!     //   initial message
!-----------------------------------------------------------------------

!     //   comments
      if ( iargc() .ne. 10 ) then

         write( 6, '(a)' )
         write( 6, '(a)' ) 'Program vdos_group'
         write( 6, '(a)' )
         write( 6, '(a)' ) 'This code decomposes vibrational ' // &
     &                     'density of states by its contributions'
         write( 6, '(a)' ) 'from harmonic normal modes.'
         write( 6, '(a)' )
         write( 6, '(a)' ) '- Requires velocity trajectory from MD'
         write( 6, '(a)' ) '  and eigenvector matrix from NMA.'
         write( 6, '(a)' )
         write( 6, '(a)' ) '- Requires mode grouping, see below.'
         write( 6, '(a)' )
         write( 6, '(a)' ) 'Usage: vdos_group.x $1 $2 $3 $4 $5 $6 $7' &
     &                     // ' $8 $9 $10'
         write( 6, '(a)' )
         write( 6, '(a)' ) '$1:  trj.out (velocity trajectory)'
         write( 6, '(a)' ) '$2:  structure.dat (atomic masses)'
         write( 6, '(a)' ) '$3:  input_default.dat (atomic kinds)'
         write( 6, '(a)' ) '$4:  eigen.out (eigenvector matrix)'
         write( 6, '(a)' ) '$5:  group.dat (normal mode groups)'
         write( 6, '(a)' ) '$6:  number of beads'
         write( 6, '(a)' ) '$7:  number of steps'
         write( 6, '(a)' ) '$8:  step interval of velocity [fs]'
         write( 6, '(a)' ) '$9:  temperature [K]'
         write( 6, '(a)' ) '$10: highest frequency [cm**-1]'
         write( 6, '(a)' )
         write( 6, '(a)' ) 'Example:'
         write( 6, '(a)' ) 'vdos_group.x trj.out structure.dat' &
     &                     // ' input_default.dat eigen.out' &
     &                     // ' group.dat 1 10000 0.5 300.0 5000.0'
         write( 6, '(a)' )
         write( 6, '(a)' ) 'Format of group.dat:'
         write( 6, '(a)' )
         write( 6, '(a)' ) 'line 1: number of mode groups ($n)'
         write( 6, '(a)' ) '  [if $n=0, each mode is set as a group]'
         write( 6, '(a)' )
         write( 6, '(a)' ) 'line 2: $1, $2, ..., $n'
         write( 6, '(a)' ) '   $i is the first mode of i-th group'
         write( 6, '(a)' )

         stop

!     //   comments
      else

         write( 6, '(a)' )
         write( 6, '(a)' ) 'Program vdos_group'
         write( 6, '(a)' )

!     //   comments
      end if

!-----------------------------------------------------------------------
!     //   read command line
!-----------------------------------------------------------------------

!     //   trajectory filename
      call getarg( 1, trjfile )

!     //   structure filename
      call getarg( 2, strfile )

!     //   default input filename
      call getarg( 3, inpfile )

!     //   eigenvector filename
      call getarg( 4, eigfile )

!     //   group filename
      call getarg( 5, grpfile )

!     //   number of beads
      call getarg( 6, char )
      read( char, * ) nbead

!     //   number of nstep
      call getarg( 7, char )
      read( char, * ) nstep

!     //   step size in femtoseconds
      call getarg( 8, char )
      read( char, * ) dt_fs

!     //   temperature in kelvin
      call getarg( 9, char )
      read( char, * ) temperature

!     //   highest frequency
      call getarg( 10, char )
      read( char, * ) omega
      nomega = nint(omega)

!-----------------------------------------------------------------------
!     //   constants
!-----------------------------------------------------------------------

!     //   circular constant
      pi = acos( -1.d0 )

!     //   step size in au
      dt = dt_fs / au_time * 1.d-15

!     //   constant
      factor_cminv = 1.d0 / (2.d0*pi*speedlight_SI*au_time*100.d0)

!     //   unit frequency with cminv
      dnu_cminv = 1.d0 / (2.d0*pi*factor_cminv)

!-----------------------------------------------------------------------
!     //   read number of atoms
!-----------------------------------------------------------------------

!     //   open file
      open( iounit, file = trim(strfile) )

!     //   skip lines
      read( iounit, *, iostat=ierr ) natom

!     //   close file
      close( iounit )

!     //   check point
      if ( ierr .ne. 0 ) then
         write( 6, '(a)' ) 'ERROR in: ' // trim(strfile) // '.'
         stop
      else
         write( 6, '(a)' ) 'READ: ' // trim(strfile) // '.'
      end if

!-----------------------------------------------------------------------
!     //   set groups
!-----------------------------------------------------------------------

!     //   read number of groups
      open( iounit, file = trim(grpfile) )
      read( iounit, *, iostat=ierr ) ngroup
      close( iounit )

!     //   default: all modes are individual group
      if ( ngroup .le. 0 ) ngroup = 3*natom

!     //   memory allocation: group index
      allocate( igroup(3*natom) )
      allocate( jgroup(ngroup) )

!     //   default grouping
      if ( ngroup .eq. (3*natom) ) then

!        //   set group index
         do i = 1, 3*natom
            jgroup(i) = i
         end do

!     //   manual grouping
      else

!        //   read group index
         open( iounit, file = trim(grpfile) )
         read( iounit, *, iostat=ierr )
         read( iounit, *, iostat=ierr ) jgroup(1:ngroup)
         close( iounit )

!     //   default of manual grouping
      end if

!-----------------------------------------------------------------------
!     //   group
!-----------------------------------------------------------------------

!     //   initialize
      igroup(1:3*natom) = 0

!     //   loop of groups
      do i = 1, ngroup-1

!        //   loop of modes in group i
         do l = jgroup(i), jgroup(i+1)-1

!           //  mode in group i
            igroup(l) = i

!        //   loop of modes in group i
         end do

!     //   loop of groups
      end do

!     //   loop of modes in last group
      do l = jgroup(ngroup), 3*natom

!        //  mode in group i
         igroup(l) = i

!     //   loop of modes in last group
      end do

!-----------------------------------------------------------------------
!     //   memory allocation
!-----------------------------------------------------------------------

!     //   atomic mass
      allocate( physmass(natom) )

!     //   atomic species
      allocate( species(natom) )

!     //   atomic velocity
      allocate( vx(natom,nbead) )
      allocate( vy(natom,nbead) )
      allocate( vz(natom,nbead) )

!     //   atomic centroid velocity
      allocate( vxc(natom) )
      allocate( vyc(natom) )
      allocate( vzc(natom) )

!     /*   normal mode eigenvalues   */
      allocate( e(3*natom) )

!     /*   normal mode eigenvectors   */
      allocate( u(3*natom,3*natom) )

!     /*   partial vibrational density of states   */
      allocate( pvdos(3*natom) )

!     /*   functions   */
      allocate( a(3*natom,nomega) )
      allocate( b(3*natom,nomega) )
      allocate( f(3*natom,nomega) )
      allocate( g(3*natom,nomega) )

!-----------------------------------------------------------------------
!     //   read atomic symbols from input_default.dat
!-----------------------------------------------------------------------

!     //   open file
      open( iounit, file = trim(inpfile) )

!     //   read loop
      do

!        //   read line
         read( iounit, *, iostat=ierr ) char

!        //   check read
         if ( ierr .ne. 0 ) exit

!        //   if keyword found
         if ( char(1:9) .eq. '<nsymbol>' ) then

!           //   read line
            read( iounit, *, iostat=ierr ) nsymbol

!           /*   memory allocation: atomic symbols   */
            allocate( symbol(nsymbol) )

!           /*   memory allocation: atomic masses   */
            allocate( physmass_symbol(nsymbol) )

!           //   number of symbols
            do i = 1, nsymbol

!              //   read symbol and atomic mass
               read( iounit, *, iostat=ierr ) &
     &            symbol(i), j, physmass_symbol(i)

!           //   number of symbols
            end do

!           //   read finished
            exit

!        //   if keyword found
         end if

!     //   read loop
      end do

!     //   close file
      close( iounit )

!     //   check point
      if ( ierr .ne. 0 ) then
         write( 6, '(a)' ) 'ERROR in: ' // trim(inpfile) // '.'
         stop
      else
         write( 6, '(a)' ) 'READ: ' // trim(inpfile) // '.'
      end if

!-----------------------------------------------------------------------
!     //   read atomic species from structure.dat, set atomic mass
!-----------------------------------------------------------------------

!     //   open file
      open( iounit, file = trim(strfile) )

!     //   skip lines
      read( iounit, *, iostat=ierr )
      read( iounit, *, iostat=ierr )

!     //   loop of atoms
      do i = 1, natom

!        //   read species
         read( iounit, *, iostat=ierr ) species(i)

!        //   check read
         if ( ierr .ne. 0 ) exit

!        //   initialize mass
         physmass(i) = 0.d0

!        //   loop of symbols
         do j = 1, nsymbol

!           //   substitute mass if symbols match species
            if ( species(i)(1:8) .eq. symbol(j)(1:8) ) then
               physmass(i) = physmass_symbol(j)
            end if

!        //   loop of symbols
         end do

!        //   mass unit converted from amu to au
         physmass(i) = physmass(i) * amu_mass / au_mass

!     //   loop of atoms
      end do

!     //   close file
      close( iounit )

!     //   check point
      if ( ierr .ne. 0 ) then
         write( 6, '(a)' ) 'ERROR in: ' // trim(strfile) // '.'
         stop
      else
         write( 6, '(a)' ) 'READ: ' // trim(strfile) // '.'
      end if

!-----------------------------------------------------------------------
!     //   read normal modes
!-----------------------------------------------------------------------

!     //   open file
      open( iounit, file = 'eigen.out' )

!     //   loop of normal modes
      do i = 1, 3*natom

!        //   read eigenvalues
         read( iounit, *, iostat=ierr ) e(i)

!        //   read eigenvectors
         do j = 1, 3*natom, 3
            read( iounit, *, iostat=ierr ) u(j:j+2,i)
         end do

!        //   check read
         if ( ierr .ne. 0 ) exit

!     //   loop of normal modes
      end do

!     //   close file
      close( iounit )

!     //   check point
      if ( ierr .ne. 0 ) then
         write( 6, '(a)' ) 'ERROR in: ' // trim(eigfile) // '.'
         stop
      else
         write( 6, '(a)' ) 'READ: ' // trim(eigfile) // '.'
      end if

!-----------------------------------------------------------------------
!     //   reset functions
!-----------------------------------------------------------------------

      a(:,:) = 0.d0
      b(:,:) = 0.d0

      f(:,:) = 0.d0
      g(:,:) = 0.d0

!     //   total time
      t_total = dble(nstep) * dt

!     //   constant factor
      factor = 2.d0 / ( boltz * temperature ) / t_total

!-----------------------------------------------------------------------
!     //   start main loop
!-----------------------------------------------------------------------

!     //   open trajectory file
      open( iounit, file = 'trj.out' )

!     //   loop of MD steps
      do istep = 1, nstep

!        //   time
         t = dble(istep) * dt

!        //   read velocity
         do j = 1, nbead
         do i = 1, natom
            read( iounit, *, iostat=ierr ) &
     &         k, tmp, tmp, tmp, vx(i,j), vy(i,j), vz(i,j)
         end do
         end do

!        //   check read
         if ( ierr .ne. 0 ) exit

!        //   loop of atoms
         do i = 1, natom

!           //   centroid velocity
            vxc(i) = 0.d0
            vyc(i) = 0.d0
            vzc(i) = 0.d0

!           //   average over beads
            do j = 1, nbead
               vxc(i) = vxc(i) + vx(i,j)
               vyc(i) = vyc(i) + vy(i,j)
               vzc(i) = vzc(i) + vz(i,j)
            end do

!           //   centroid velocity
            vxc(i) = vxc(i) / dble(nbead)
            vyc(i) = vyc(i) / dble(nbead)
            vzc(i) = vzc(i) / dble(nbead)

!        //   loop of atoms
         end do

!        //   loop of frequencies
         do l = 1, nomega

!           //   frequency in au
            omega = dble(l) / factor_cminv

!           //   cosine and sine functions
            coswt = cos( omega * t )
            sinwt = sin( omega * t )

!           //   loop of atoms
            do i = 1, natom

!              //   square root of mass
               sqrtm = sqrt( physmass(i) )

!              //   counter
               j = 3*(i-1) + 1

!              //   make a and b vectors
               a(j,l) = a(j,l) + sqrtm * vxc(i) * coswt * dt
               b(j,l) = b(j,l) + sqrtm * vxc(i) * sinwt * dt

!              //   counter
               j = 3*(i-1) + 2

!              //   make a and b vectors
               a(j,l) = a(j,l) + sqrtm * vyc(i) * coswt * dt
               b(j,l) = b(j,l) + sqrtm * vyc(i) * sinwt * dt

!              //   counter
               j = 3*(i-1) + 3

!              //   make a and b vectors
               a(j,l) = a(j,l) + sqrtm * vzc(i) * coswt * dt
               b(j,l) = b(j,l) + sqrtm * vzc(i) * sinwt * dt

!           //   loop of atoms
            end do

!        //   loop of frequencies
         end do

!     //   loop of MD steps
      end do

!     //   close trajectory file
      close( iounit )

!     //   check point
      write( 6, '(a)' ) 'READ: ' // trim(trjfile) // '.'

!-----------------------------------------------------------------------
!     //   make f and g functions
!-----------------------------------------------------------------------

!     //   loop of frequencies
      do l = 1, nomega

!        //   loop of atoms
         do k = 1, 3*natom
         do i = 1, 3*natom

!           //   normal mode transform
            f(k,l) = f(k,l) + u(i,k) * a(i,l)
            g(k,l) = g(k,l) + u(i,k) * b(i,l)

!        //   loop of atoms
         end do
         end do

!     //   loop of frequencies
      end do

!-----------------------------------------------------------------------
!     //   print output
!-----------------------------------------------------------------------

!     //   open output file
      open( iounit, file = 'vdos_group.out' )

!     //   print header
      write( iounit, '(a)', advance='no' ) "----------------------"

!     //   print header
      do i = 1, ngroup-1
         write( iounit, '(a)', advance='no' ) "----------------"
      end do

!     //   print header
      write( iounit, '(a)' ) "----------------"

!     //   print header
      write( iounit, '(a)', advance='no' ) "  freq       all-modes"

!     //   print header
      do i = 1, ngroup-1
         write( iounit, '(a,i4)', advance='no' ) "        mode", i
      end do

!     //   print header
      write( iounit, '(a,i4)' ) "        mode", i

!     //   print header
      write( iounit, '(a)', advance='no' ) "----------------------"

!     //   print header
      do i = 1, ngroup-1
         write( iounit, '(a)', advance='no' ) "----------------"
      end do

!     //   print header
      write( iounit, '(a)' ) "----------------"

!     //   sum of total density of states
      vdos_sum = 0.d0

!     //   loop of frequencies
      do l = 1, nomega

!        //   total density of states
         vdos = 0.d0

!        //   partial density of states
         do i = 1, ngroup
            pvdos(i) = 0.d0
         end do

!        //   loop of atoms
         do k = 1, 3*natom

!           //   group
            i = igroup(k)

!           //   contribution
            fg = ( f(k,l) * f(k,l) + g(k,l) * g(k,l) ) * factor

!           //   partial density of states
            if ( i .gt. 0 ) pvdos(i) = pvdos(i) + fg

!           //   total density of states
            vdos = vdos + fg

!        //   loop of atoms
         end do

!        //   normalize by the sum rule
         vdos = vdos * dnu_cminv

!        //   sum of total density of states
         vdos_sum = vdos_sum + vdos

!        //   normalize by the sum rule
         pvdos(1:ngroup) = pvdos(1:ngroup) * dnu_cminv

!        //   print result
         write( iounit, '(i6,e16.8)', advance='no' ) l, vdos

!        //   print result
         do i = 1, ngroup-1
            write( iounit, '(e16.8)', advance='no' ) pvdos(i)
         end do

!        //   print result
         write( iounit, '(e16.8)' ) pvdos(ngroup)

!     //   loop of frequencies
      end do

!     //   close output file
      close( iounit )

!-----------------------------------------------------------------------
!     //   end of program
!-----------------------------------------------------------------------

      write( 6, '(a)' )
      write( 6, '(a,f10.2)' ) &
    &    'Sum of total density of states:', vdos_sum
      write( 6, '(a,f10.2)' ) &
    &    'Vibrational degrees of freedom:', dble(3*natom-3)
      write( 6, '(a)' )
      write( 6, '(a)' ) 'Normal termination.'
      write( 6, '(a)' )

      stop
      end
