c=======================================================================
c
c     References:
c
c        Lin, Blanco, Goddard III, J. Chem. Phys. 119, 11792 (2003).
c
c=======================================================================
c***********************************************************************
      program vdos_MPI
c***********************************************************************
c-----------------------------------------------------------------------

c     //   variables
      implicit none

c     //   file unit
      integer :: iounit = 10

c     //   number of processes
      integer :: nprocs

c     //   process rank
      integer :: myrank

c     //   step number
      integer :: istep

c     //   number of steps
      integer :: nstep

c     //   number of beads
      integer :: nbead

c     //   number of frequencies
      integer :: nomega

c     //   number of kinds
      integer :: nkind

c     //   atomic kind
      integer, dimension(:), allocatable :: ikind

c     //   centroid velocity
      real(8), dimension(:), allocatable :: vx, vy, vz

c     //   centroid velocity
      real(8), dimension(:,:), allocatable :: wx, wy, wz

c     //   real numbers
      real(8), dimension(:), allocatable :: vxcos, vycos, vzcos
      real(8), dimension(:), allocatable :: vxsin, vysin, vzsin

c     //   step size
      real(8) :: dt_fs, dt

c     //   total time
      real(8) :: t_total

c     //   time
      real(8) :: t

c     //   circular constant
      real(8) :: pi

c     //   frequency
      real(8) :: omega

c     //   kubo transformed spectra I_kubo(omega)
      real(8), dimension(:,:), allocatable :: f_kubo

c     //   I_kubo(omega) smeared
      real(8), dimension(:,:), allocatable :: f_smear

c     //   sum of I_kubo(omega) smeared
      real(8), dimension(:), allocatable :: f_smear_sum

c     //   kubo correlation function
      real(8), dimension(:,:), allocatable :: c_kubo

c     //   quantum corrected correlation function
      real(8), dimension(:,:), allocatable :: c

c     //   integral of f_kubo
      real(8) :: f_kubo_int

c     //   temperature
      real(8) :: temperature

c     //   characters
      character(len=80) :: char

c     //   filename
      character(len=80) :: trjfile, strfile, inpfile

c     //   unit conversion factors
      real(8), parameter :: au_time   = 0.024188843d-15
      real(8), parameter :: au_length = 0.529177249d-10
      real(8), parameter :: amu_mass  = 1.6605402d-27
      real(8), parameter :: au_mass   = 9.1093897d-31

c     //   Boltzmann constant
      real(8), parameter :: boltz = 0.316682968d-5

c     //   Planck constant divided by 2 pi
      real(8), parameter :: hbar = 1.d0

c     /*   speed of light in SI unit   */
      real(8), parameter :: speedlight_SI = 2.99892458d+8

c     //   speed of light
      real(8) :: speedlight

c     //   smearing parameter
      real(8) :: domega

c     //   number of atoms
      integer :: natom

c     //   atomic species
      character(len=8), dimension(:), allocatable :: species

c     //   atomic mass
      real(8), dimension(:), allocatable :: physmass

c     //   atomic mass
      real(8), dimension(:), allocatable :: physmass_kind

c     //   integers
      integer :: i, j, k, l, ierr, iargc

c     /*   number of atomic symbols   */
      integer :: nsymbol

c     /*   atomic symbols   */
      character(len=8), dimension(:), allocatable :: symbol

c     /*   atomic masses   */
      real(8), dimension(:), allocatable :: physmass_symbol

c     //   real numbers
      real(8) :: beta, factor_cminv, dnu_cminv

c     //   real numbers
      real(8) :: bho, d, coswt, sinwt, dx, dy, dz, qc, factor

c     //   mpi
      include 'mpif.h'

c-----------------------------------------------------------------------
c     //   start MPI
c-----------------------------------------------------------------------

      call MPI_INIT( ierr )
      call MPI_COMM_SIZE( MPI_COMM_WORLD, nprocs, ierr )
      call MPI_COMM_RANK( MPI_COMM_WORLD, myrank, ierr )

c-----------------------------------------------------------------------
c     //   initial message
c-----------------------------------------------------------------------

c     //   master rank
      if ( myrank .eq. 0 ) then

c        //   comments
         if ( iargc() .ne. 9 ) then

            write( 6, '(a)' )
            write( 6, '(a)' ) 'Program vdos_MPI'
            write( 6, '(a)' )
            write( 6, '(a)' ) 'Usage: vdos_MPI.x $1 $2 $3 $4 $5 $6 $7'
     &                        // ' $8 $9'
            write( 6, '(a)' )
            write( 6, '(a)' ) '$1: trj.out (velocity trajectory)'
            write( 6, '(a)' ) '$2: structure.dat (atomic masses)'
            write( 6, '(a)' ) '$3: input_default.dat (atomic kinds)'
            write( 6, '(a)' ) '$4: number of beads'
            write( 6, '(a)' ) '$5: number of steps (set 0 to check)'
            write( 6, '(a)' ) '$6: step interval of velocity [fs]'
            write( 6, '(a)' ) '$7: temperature [K]'
            write( 6, '(a)' ) '$8: highest frequency [cm**-1]'
            write( 6, '(a)' ) '$9: smearing parameter [cm**-1]'
            write( 6, '(a)' )
            write( 6, '(a)' ) 'Example:'
            write( 6, '(a)' ) 'vdos_MPI.x trj.out structure.dat'
     &                        // ' input_default.dat 16 0 0.5 300.0'
     &                        // ' 5000.0 10.0'
            write( 6, '(a)' )
            call my_mpi_finalize
            stop

c        //   comments
         else

            write( 6, '(a)' )
            write( 6, '(a)' ) 'Program vdos_MPI'
            write( 6, '(a)' )

c        //   comments
         end if

c     //   master rank
      end if

c-----------------------------------------------------------------------
c     //   read values
c-----------------------------------------------------------------------

c     //   trajectory filename
      call getarg( 1, trjfile )

c     //   structure filename
      call getarg( 2, strfile )

c     //   structure filename
      call getarg( 3, inpfile )

c     //   number of beads
      call getarg( 4, char )
      read( char, * ) nbead

c     //   number of nstep
      call getarg( 5, char )
      read( char, * ) nstep

c     //   step size in femtoseconds
      call getarg( 6, char )
      read( char, * ) dt_fs

c     //   temperature in kelvin
      call getarg( 7, char )
      read( char, * ) temperature

c     //   highest frequency
      call getarg( 8, char )
      read( char, * ) omega
      nomega = nint(omega)

c     //   smearing frequency
      call getarg( 9, char )
      read( char, * ) domega

c     //   domega must be greater than 0.00001
      domega = max(domega,1.d0)

c-----------------------------------------------------------------------
c     //   constants
c-----------------------------------------------------------------------

c     //   circular constant
      pi = acos( -1.d0 )

c     //   step size in au
      dt = dt_fs / au_time * 1.d-15

c     //   inverse temperature
      beta = 1.d0 / ( boltz * temperature )

c     //   constant
      factor_cminv = 1.d0 / (2.d0*pi*speedlight_SI*au_time*100.d0)

c     //   speed of light
      speedlight = speedlight_SI / ( au_length / au_time )

c     //   unit frequency with cminv
      dnu_cminv = 1.d0 / (2.d0*pi*factor_cminv)

c-----------------------------------------------------------------------
c     //   number of atoms
c-----------------------------------------------------------------------

c     //   master rank
      if ( myrank .eq. 0 ) then

c        //   open file
         open ( iounit, file = trim(strfile) )

c        //   number of atoms
         read ( iounit, *, iostat=ierr ) natom

c        //   close file
         close( iounit )

c     //   master rank
      end if

c     //   communication
      call my_mpi_bcast_int_0_world( ierr )

c     //   check error
      call error_handling_MPI( ierr, 'Number of atoms', 15 )

c     //   communication
      call my_mpi_bcast_int_0_world( natom )

c-----------------------------------------------------------------------
c     //   print information
c-----------------------------------------------------------------------

c     //   number of atoms
      if ( myrank .eq. 0 ) then
         write( 6, '(a,i6)' ) 'Number of atoms: ', natom
      end if

c-----------------------------------------------------------------------
c     //   memory allocation
c-----------------------------------------------------------------------

c     //   atomic kind
      allocate( ikind(natom) )

c     //   velocity
      allocate( vx(natom) )
      allocate( vy(natom) )
      allocate( vz(natom) )

c     //   Fourier transform
      allocate( vxcos(natom) )
      allocate( vycos(natom) )
      allocate( vzcos(natom) )
      allocate( vxsin(natom) )
      allocate( vysin(natom) )
      allocate( vzsin(natom) )

c     //   atomic mass
      allocate( physmass(natom) )

c-----------------------------------------------------------------------
c     /*   read atomic symbols                                        */
c-----------------------------------------------------------------------

c     //   master rank
      if ( myrank .eq. 0 ) then

c        //   atomic species
         allocate( species(natom) )

c        //   open file
         open ( iounit, file = trim(strfile) )

c        //   number of atoms
         read ( iounit, *, iostat=ierr )

c        //   comment line
         read ( iounit, *, iostat=ierr )

c        //   number of atomic kinds
         nkind = 1

c        //   loop of atoms
         do i = 1, natom

c           //   atomic symbol
            read( iounit, *, iostat=ierr ) species(i), d, d, d, ikind(i)

c           //   atomic symbol
            if ( ierr .ne. 0 ) then
               read( iounit, *, iostat=ierr ) species(i)
               ikind(i) = 1
            end if

c           //   number of atomic kinds
            nkind = max( ikind(i), nkind )

c        //   loop of atoms
         end do

c        //   close file
         close( iounit )

c        //   error check
         if ( ierr .ne. 0 ) go to 100

c        //   atomic masses
         physmass(:) = 0.d0

c        /*   file open   */
         open ( iounit, file = 'input_default.dat' )

c        /*   search for tag    */
         call search_tag ( '<nsymbol>', 9, iounit, ierr )

c        /*   number of atomic symbols   */
         read ( iounit, *, iostat=ierr ) nsymbol

c        /*   file close   */
         close( iounit )

c        /*   memory allocation: atomic symbols   */
         allocate( symbol(nsymbol) )

c        /*   memory allocation: atomic masses   */
         allocate( physmass_symbol(nsymbol) )

c        /*   file open   */
         open ( iounit, file = 'input_default.dat' )

c        /*   search for tag    */
         call search_tag ( '<nsymbol>', 9, iounit, ierr )

c        /*   number of atomic symbols   */
         read ( iounit, *, iostat=ierr )

c        /*   loop of symbols   */
         do i = 1, nsymbol

c           /*   read symbol, atomic number, atomic mass   */
            read ( iounit, *, iostat=ierr )
     &         symbol(i), j, physmass_symbol(i)

c        /*   loop of symbols   */
         end do

c        /*   file close   */
         close( iounit )

c        //   atomic masses
         physmass(:) = 0.d0

c        /*   loop of atoms   */
         do i = 1, natom

c           //   flag
            ierr = 1

c           /*   loop of symbols   */
            do l = 1, nsymbol

c              /*   if symbol matched   */
               if ( species(i)(1:8) .eq. symbol(l)(1:8) ) then

c                 /*   substitute mass   */
                  physmass(i) = physmass_symbol(l)*amu_mass/au_mass

c                 //   flag
                  ierr = 0

c                 /*   go to next loop   */
                  exit

c              /*   if symbol matched   */
               end if

c           /*   loop of symbols   */
            end do

c           /*   go to next loop   */
            if ( ierr .ne. 0 ) exit

c        /*   loop of atoms   */
         end do

c        /*   memory allocation: atomic symbols   */
         deallocate( symbol )

c        /*   memory allocation: atomic masses   */
         deallocate( physmass_symbol )

c        //   atomic species
  100    deallocate( species )

c     //   master rank
      end if

c     //   communication
      call my_mpi_bcast_int_0_world( ierr )

c     //   master rank
      if ( ierr .ne. 0 ) then

         if ( myrank .eq. 0 ) then
            write( 6, '(a)' )
            write( 6, '(a)' ) 'Error in atomic mass.'
            write( 6, '(a)' )
         end if
         call my_mpi_finalize
         stop

      end if

c-----------------------------------------------------------------------
c     //   communication
c-----------------------------------------------------------------------

c     //   atomic mass
      call my_mpi_bcast_real_1_world( physmass, natom )

c     //   number of kinds
      call my_mpi_bcast_int_0_world( nkind )

c     //   kind of atom
      call my_mpi_bcast_int_1_world( ikind, natom )

c-----------------------------------------------------------------------
c     //   read velocity trajectory
c-----------------------------------------------------------------------

c     //   kubo-transformed spectrum
      allocate( f_kubo(nomega,nkind) )

c     //   smeared spectrum
      allocate( f_smear(nomega,nkind) )

c     //   sum of spectrum
      allocate( f_smear_sum(nomega) )

c     //   atomic mass
      allocate( physmass_kind(nkind) )

c-----------------------------------------------------------------------
c     //   atomic kind
c-----------------------------------------------------------------------

      do i = 1, natom
         k = ikind(i)
         physmass_kind(k) = physmass(i)
      end do

c-----------------------------------------------------------------------
c     //   read velocity trajectory
c-----------------------------------------------------------------------

c     //   master rank
      if ( myrank .eq. 0 ) then

c        //   check number of steps
         if ( nstep .le. 0 ) then

c           //   open file
            open ( iounit, file = trim(trjfile) )

c           //   step counter
            istep = 0

c           //   error flag
            ierr = 0

c           //   loop of steps
            do

c              //   centroid
               vx(:) = 0.d0
               vy(:) = 0.d0
               vz(:) = 0.d0

c              //   read centroid
               do j = 1, nbead
               do k = 1, natom
                  read( iounit, *, iostat=ierr ) i, d, d, d, dx, dy, dz
                  vx(k) = vx(k) + dx / dble(nbead)
                  vy(k) = vy(k) + dy / dble(nbead)
                  vz(k) = vz(k) + dz / dble(nbead)
               end do
               end do

c              //   detect end of file
               if ( ierr .ne. 0 ) exit

c              //   step counter
               istep = istep + 1

c           //   loop of steps
            end do

c           //   last step
            nstep = istep

c           //   close file
            close( iounit )

c        //   check number of steps
         end if

c     //   master rank
      end if

c     //   communication
      call my_mpi_bcast_int_0_world( nstep )

c-----------------------------------------------------------------------
c     //   print information
c-----------------------------------------------------------------------

c     //   number of atoms
      if ( myrank .eq. 0 ) then
         write( 6, '(a,i6)' ) 'Number of steps: ', nstep
      end if

c-----------------------------------------------------------------------
c     //   read velocity trajectory, compute spectral density
c-----------------------------------------------------------------------

c     //   print
      if ( myrank .eq. 0 ) then
         write( 6, '(a)' )
      end if

c     //   total time
      t_total = dble(nstep) * dt

c     //   spectral density of kubo-transformed autocorrelation
      f_kubo(:,:) = 0.d0

c     //   with quantum correction factor, smeared
      f_smear(:,:) = 0.d0

c     //   with quantum correction factor, smeared
      f_smear_sum(:) = 0.d0

c-----------------------------------------------------------------------
c        //   compute spectra
c-----------------------------------------------------------------------

c     //   velocity
      allocate( wx(natom,nstep) )
      allocate( wy(natom,nstep) )
      allocate( wz(natom,nstep) )

c     //   centroid
      wx(:,:) = 0.d0
      wy(:,:) = 0.d0
      wz(:,:) = 0.d0

c     //   master
      if ( myrank .eq. 0 ) then

c        //   open file
         open ( iounit, file = trim(trjfile) )

c        //   loop of steps
         do istep = 1, nstep

c           //   read centroid
            do j = 1, nbead
            do k = 1, natom
               read( iounit, *, iostat=ierr ) i, d, d, d, dx, dy, dz
               wx(k,istep) = wx(k,istep) + dx / dble(nbead)
               wy(k,istep) = wy(k,istep) + dy / dble(nbead)
               wz(k,istep) = wz(k,istep) + dz / dble(nbead)
            end do
            end do

c        //   loop of steps
         end do

c        //   close file
         close( iounit )

c     //   master
      end if

c     //   communication
      call my_mpi_bcast_real_2_world( wx, natom, nstep )
      call my_mpi_bcast_real_2_world( wy, natom, nstep )
      call my_mpi_bcast_real_2_world( wz, natom, nstep )

c     //   loop of frequencies
      do l = myrank+1, nomega, nprocs

c        //   frequency in au
         omega = dble(l) / factor_cminv

c        //   Fourier transform of velocity
         vxcos(:) = 0.d0
         vycos(:) = 0.d0
         vzcos(:) = 0.d0
         vxsin(:) = 0.d0
         vysin(:) = 0.d0
         vzsin(:) = 0.d0

c        //   loop of steps
         do istep = 1, nstep

c           //   time
            t = dble(istep) * dt

c           //   exponential factor
            coswt = cos( omega * t )
            sinwt = sin( omega * t )

c           //   Fourier transform of velocity
            do k = 1, natom
               vxcos(k) = vxcos(k) + wx(k,istep) * coswt * dt
               vycos(k) = vycos(k) + wy(k,istep) * coswt * dt
               vzcos(k) = vzcos(k) + wz(k,istep) * coswt * dt
               vxsin(k) = vxsin(k) + wx(k,istep) * sinwt * dt
               vysin(k) = vysin(k) + wy(k,istep) * sinwt * dt
               vzsin(k) = vzsin(k) + wz(k,istep) * sinwt * dt
            end do

c        //   loop of steps
         end do

c        //   loop of atoms
         do k = 1, natom

c           //   atom
            j = ikind(k)

c           //   norm squared weighted by atomic masses
            f_kubo(l,j) = f_kubo(l,j)
     &         + ( vxcos(k)*vxcos(k) + vxsin(k)*vxsin(k)
     &           + vycos(k)*vycos(k) + vysin(k)*vysin(k)
     &           + vzcos(k)*vzcos(k) + vzsin(k)*vzsin(k) )
     &         * physmass(k)

c        //   loop of atoms
         end do

c        //   close file
         close( iounit )

c        //   loop of atomic kinds
         do j = 1, nkind

c           //   divided by total time
            f_kubo(l,j) = f_kubo(l,j) / t_total * ( 2.d0 * beta )

c        //   loop of atomic kinds
         end do

c        //   periodic output
         if ( myrank .eq. 0 ) then
            if ( mod(l-1,100) .eq. 0 ) then
               write( 6, '(a,f6.1)' )
     &            'DONE (%):', dble(l*100)/dble(nomega)
            end if
         end if

c     //   loop of frequencies
      end do

c     //   print
      if ( myrank .eq. 0 ) then
         write( 6, '(a,f6.1)' ) 'DONE (%):', dble(100)
      end if

c-----------------------------------------------------------------------
c        //   communications
c-----------------------------------------------------------------------

c     //   communication
      call my_mpi_allreduce_real_2_world( f_kubo, nomega, nkind )

c-----------------------------------------------------------------------
c     //   integral of spectrum
c-----------------------------------------------------------------------

c     //   integral of spectrum
      f_kubo_int = 0.d0

c     //   loop of frequencies
      do l = 1, nomega

c        //   integral of spectrum
         do j = 1, nkind
            f_kubo_int = f_kubo_int + f_kubo(l,j) * dnu_cminv
         end do

c     //   loop of frequencies
      end do

c     //   master rank
      if ( myrank .eq. 0 ) then
          write( 6, '(a)' )
          write( 6, '(a,f12.4)' ) 'Integral: ', f_kubo_int
          write( 6, '(a)' )
      end if

c-----------------------------------------------------------------------
c     //   smearing
c-----------------------------------------------------------------------

c     //   factor
      factor = 1.d0 / sqrt( pi * ( domega * domega ) / 0.5d0 )

c     //   loop of frequencies
      do k = 1, nomega

c        //   loop of frequencies
         do l = 1, nomega

c           //   frequency difference
            d = dble(k-l)

c           //   exponent
            d = 0.5d0 * d * d / ( domega * domega )

c           //   loop of kinds
            do j = 1, nkind

c              //   loop of kinds
               f_smear(k,j) = f_smear(k,j)
     &            + f_kubo(l,j) * factor * exp( - d )

c           //   loop of kinds
            end do

c        //   loop of frequencies
         end do

c        //   loop of kinds
         do j = 1, nkind

c           //   sum of all kinds
            f_smear_sum(k) = f_smear_sum(k) + f_smear(k,j)

c        //   loop of kinds
         end do

c     //   loop of frequencies
      end do

c-----------------------------------------------------------------------
c     //   print
c-----------------------------------------------------------------------

c     //   master rank
      if ( myrank .eq. 0 ) then

c        //   open file
         open ( iounit, file = 'vdos.out' )

c        //   header
         write( iounit, '(a)' )
     &      '--------------------------------------' //
     &      '--------------------------------------------'
         write( iounit, '(a)' )
     &      ' freq [cm-1]     I_kubo(sum)          I(sum)' //
     &      '  kind    I_kubo(kind)         I(kind)'
         write( iounit, '(a)' )
     &      '--------------------------------------' //
     &      '--------------------------------------------'

c        //   loop of frequencies
         do k = 1, nomega

c           //   frequency in au
            omega = dble(k) / factor_cminv

c           //   factor
            bho = beta * hbar * omega

c           //   quantum correction
            qc = bho / ( 1.d0 - exp(-bho) )

c           //   print data
            do j = 1, nkind
               write( iounit, '(f12.1,2e16.8,i6,2e16.8)' )
     &            dble(k),
     &            f_smear_sum(k)*dnu_cminv,
     &            qc*f_smear_sum(k)*dnu_cminv,
     &            j,
     &            f_smear(k,j)*dnu_cminv,
     &            qc*f_smear(k,j)*dnu_cminv
            end do

c        //   loop of frequencies
         end do

c        //   close file
         close( iounit )

c     //   master rank
      end if

c-----------------------------------------------------------------------
c     //   quantum corrected velocity correlation function
c-----------------------------------------------------------------------

c     //   quantum corrected correlation function
      allocate( c(0:nstep,nkind) )

c     //   kubo correlation function
      allocate( c_kubo(0:nstep,nkind) )

c     //   initialize
      c(:,:) = 0.d0

c     //   initialize
      c_kubo(:,:) = 0.d0

c     //   loop of steps
      do istep = 0, nstep

c        //   time
         t = dble(istep) * dt

c        //   loop of frequencies
         do k = 1, nomega

c           //   frequency in au
            omega = dble(k) / factor_cminv

c           //   factor
            bho = beta * hbar * omega

c           //   quantum correction
            qc = bho / ( 1.d0 - exp(-bho) )

c           //   loop of kinds
            do j = 1, nkind

c              //   factor
               factor = 2.d0 / physmass_kind(j) / ( 2.d0 * beta )

c              //   quantum corrected correlation function
               c(istep,j) = c(istep,j)
     &           + qc * f_smear(k,j) * cos(omega*t) * dnu_cminv * factor

c              //   kubo transformed correlation function
               c_kubo(istep,j) = c_kubo(istep,j)
     &           + f_smear(k,j) * cos(omega*t) * dnu_cminv * factor

c           //   loop of kinds
            end do

c        //   loop of frequencies
         end do

c     //   loop of steps
      end do

c-----------------------------------------------------------------------
c     //   print
c-----------------------------------------------------------------------

c     //   master rank
      if ( myrank .eq. 0 ) then

c        //   open file
         open ( iounit, file = 'vacf.out' )

c        //   header
         write( iounit, '(a)' )
     &      '--------------------------------------------'
         write( iounit, '(a)' )
     &      '   time [fs]     c_kubo [au]          c [au]'
         write( iounit, '(a)' )
     &      '--------------------------------------------'

c        //   loop of steps
         do istep = 0, nstep

c           //   loop of kinds
            do j = 1, nkind

               write( iounit, '(f12.3,2e16.8)' )
     &            dble(istep)*dt_fs, c_kubo(istep,j), c(istep,j)

c           //   loop of kinds
            end do

c        //   loop of steps
         end do

c        //   close file
         close( iounit )

c     //   master rank
      end if

c-----------------------------------------------------------------------
c     //   finalize mpi
c-----------------------------------------------------------------------

      call my_mpi_finalize

c-----------------------------------------------------------------------
c     //   end of code
c-----------------------------------------------------------------------

      stop
      end





c***********************************************************************
      subroutine search_tag ( char_tag, length_tag, iounit, ierr )
c***********************************************************************

c-----------------------------------------------------------------------
c     /*   local variables                                            */
c-----------------------------------------------------------------------

      implicit none

      integer:: k, length_tag, iounit, ierr

      character(len=80) :: char_line

      character(len=length_tag) :: char_tag

c-----------------------------------------------------------------------
c     /*   do loop start                                              */
c-----------------------------------------------------------------------

      do

c-----------------------------------------------------------------------
c        /*   read a line                                             */
c-----------------------------------------------------------------------

c         read (iounit,'(a80)',iostat=ierr) char_line
         read (iounit,*,iostat=ierr) char_line

c-----------------------------------------------------------------------
c        /*   return if error is found                                */
c-----------------------------------------------------------------------

         if ( ierr .ne. 0 ) return

c-----------------------------------------------------------------------
c        /*   search for the tag                                      */
c-----------------------------------------------------------------------

         k = index(char_line(1:length_tag),char_tag(1:length_tag))

c-----------------------------------------------------------------------
c        /*   return as soon as we find the tag                       */
c-----------------------------------------------------------------------

         if ( k .ge. 1 ) return

c-----------------------------------------------------------------------
c     /*   do loop end                                                */
c-----------------------------------------------------------------------

      end do

      return
      end





c***********************************************************************
      subroutine my_mpi_barrier_world
c***********************************************************************

      implicit none

      integer :: ierr

      include 'mpif.h'

      call MPI_BARRIER ( MPI_COMM_WORLD, ierr )

      return
      end





c***********************************************************************
      subroutine my_mpi_finalize
c***********************************************************************

      implicit none

      integer :: ierr

      include 'mpif.h'

c     /*   finalize MPI prallelization   */
      call MPI_FINALIZE ( ierr )

      return
      end





c***********************************************************************
      subroutine my_mpi_bcast_int_0_world ( i )
c***********************************************************************

      implicit none

      integer               ::  i, ierr
      integer, dimension(1) ::  j

      include 'mpif.h'

      j(1) = i

      call MPI_BCAST ( j, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr )

      i = j(1)

      return
      end





c***********************************************************************
      subroutine my_mpi_bcast_int_1_world ( i, n )
c***********************************************************************

      implicit none

      integer :: ierr, n
      integer, dimension(n) :: i

      include 'mpif.h'

      call MPI_BCAST ( i, n, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr )

      return
      end





c***********************************************************************
      subroutine my_mpi_bcast_real_1_world ( a, n )
c***********************************************************************

      implicit none

      integer :: ierr, n
      real(8) :: a(n)

      include 'mpif.h'

      call MPI_BCAST ( a, n, MPI_DOUBLE_PRECISION,
     &                 0, MPI_COMM_WORLD, ierr )

      return
      end





c***********************************************************************
      subroutine my_mpi_bcast_real_2_world ( a, n1, n2 )
c***********************************************************************

      implicit none

      integer :: i, j, k, n, n1, n2, ierr
      real(8) :: a(n1,n2), b(n1*n2)

      include 'mpif.h'

      k = 0
      do j = 1, n2
      do i = 1, n1
         k = k + 1
         b(k) = a(i,j)
      end do
      end do

      n = n1*n2

      call MPI_BCAST ( b, n, MPI_DOUBLE_PRECISION,
     &                 0, MPI_COMM_WORLD, ierr )

      k = 0
      do j = 1, n2
      do i = 1, n1
         k = k + 1
         a(i,j) = b(k)
      end do
      end do

      return
      end





c***********************************************************************
      subroutine my_mpi_allreduce_real_1_world ( a, n )
c***********************************************************************

      implicit none

      integer :: i, n, ierr
      real(8) :: a(n), b1(n), b2(n)

      include 'mpif.h'

      do i = 1, n
         b1(i) = a(i)
      end do

      call MPI_ALLREDUCE ( b1, b2, n, MPI_DOUBLE_PRECISION,
     &                     MPI_SUM, MPI_COMM_WORLD, ierr )

      do i = 1, n
         a(i) = b2(i)
      end do

      return
      end





c***********************************************************************
      subroutine my_mpi_allreduce_real_2_world ( a, n1, n2 )
c***********************************************************************

      implicit none

      integer :: i, j, k, n, n1, n2, ierr
      real(8) :: a(n1,n2), b1(n1*n2), b2(n1*n2)

      include 'mpif.h'

      k = 0
      do j = 1, n2
      do i = 1, n1
         k = k + 1
         b1(k) = a(i,j)
      end do
      end do

      n = n1*n2

      call MPI_ALLREDUCE ( b1, b2, n, MPI_DOUBLE_PRECISION,
     &                     MPI_SUM, MPI_COMM_WORLD, ierr )

      k = 0
      do j = 1, n2
      do i = 1, n1
         k = k + 1
         a(i,j) = b2(k)
      end do
      end do

      return
      end





c***********************************************************************
      subroutine error_handling_MPI( ierr, char_tag, length_tag )
c***********************************************************************
c-----------------------------------------------------------------------
c     /*   local variables                                            */
c-----------------------------------------------------------------------

      implicit none

      integer:: ierr, length_tag

      character(len=length_tag) :: char_tag

c-----------------------------------------------------------------------
c     /*   start                                                      */
c-----------------------------------------------------------------------

      if ( ierr .ne. 0 ) then

         write( 6, '(a)' ) 'Error termination at: ' // char_tag // '.'

         call my_mpi_finalize

         stop

      end if

      return
      end

