c***********************************************************************
      program vdos2vfe
c***********************************************************************
c-----------------------------------------------------------------------
c     //   local variables
c-----------------------------------------------------------------------

c     //   initialize
      implicit none

c     //   file number
      integer :: iounit = 10

c     //   number of frequencies
      integer :: nomega

c     //   circular constant
      real(8) :: pi

c     //   planck constant divided by 2 pi
      real(8), parameter :: hbar = 1.d0

c     //   boltzmann constant
      real(8), parameter :: boltz = 0.316682968d-5

c     //   speed of light in SI units
      real(8), parameter :: speedlight_SI = 2.99892458d+8

c     //   time in atomic units
      real(8), parameter :: au_time = 0.024188843d-15

c     //   cut off frequency
      real(8) :: omega_cutoff = 0.5d0

c     //   frequencies
      real(8), dimension(:), allocatable :: omega

c     //   density of states
      real(8), dimension(:), allocatable :: vdos

c     //   temperature
      real(8) :: temperature

c     //   internal energy correction
      real(8) :: uqtc

c     //   free energy correction
      real(8) :: fqtc

c     //   characters
      character(len=80) :: char, boundary

c     //   input file
      character(len=80) :: infile

c     //   number of atoms
      integer :: natom

c     //   number of atomic kinds
      integer :: nkind

c     //   number of degrees of freedom
      integer :: ndof

c     //   real numbers
      real(8) :: ho, bho, ho_half, bho_half, beta, factor_norm, vsum

c     //   real numbers
      real(8) :: factor_cminv, omega_au

cc     //   real numbers
c      real(8) :: dnu_cminv

c     //   integers
      integer :: i, j, ierr

c-----------------------------------------------------------------------
c     //   initial message
c-----------------------------------------------------------------------

      if ( iargc() .ne. 5 ) then

         write( 6, '(a)' )
         write( 6, '(a)' ) 'Program vdos2vfe'
         write( 6, '(a)' )
         write( 6, '(a)' ) 'Usage: vdos2vfe.x $1 $2 $3 $4 $5'
         write( 6, '(a)' )
         write( 6, '(a)' ) '$1: input filename for vdos'
         write( 6, '(a)' ) '$2: atomic kinds in vdos'
         write( 6, '(a)' ) '$3: temperature [K]'
         write( 6, '(a)' ) '$4: number of atoms'
         write( 6, '(a)' ) '$5: boundary - free (F) or periodic (P)'
         write( 6, '(a)' )
         write( 6, '(a)' ) 'Example: vdos2vfe.x vdos.out 2 300.0 109 P'
         write( 6, '(a)' )

         stop

      else

         write( 6, '(a)' )
         write( 6, '(a)' ) 'Program vdos2vfe'
         write( 6, '(a)' )

      end if

c-----------------------------------------------------------------------
c     //   read data
c-----------------------------------------------------------------------

c     //   input file
      call getarg( 1, infile )

c     //   number of atomic kinds
      call getarg( 2, char )
      read( char, * ) nkind

c     //   temperature
      call getarg( 3, char )
      read( char, * ) temperature

c     //   number of atoms
      call getarg( 4, char )
      read( char, * ) natom

c     //   boundary condition
      call getarg( 5, boundary )

c-----------------------------------------------------------------------
c     //   constants
c-----------------------------------------------------------------------

c     //   circular constant
      pi = acos(-1.d0)

c     //   inverse of temperature
      beta = 1.d0 / ( boltz * temperature )

c     //   conversion from cminv to hartree
      factor_cminv = 1.d0 / (2*pi*speedlight_SI*au_time*100)

cc     //   unit frequency with cminv
c      dnu_cminv = 1.d0 / (2.d0*pi*factor_cminv)

c-----------------------------------------------------------------------
c     //   number of vibrational degrees of freedom
c-----------------------------------------------------------------------

c     //   internal energy correction
      uqtc = 0.d0

c     //   free energy correction
      fqtc = 0.d0

c     //   error flag
      ierr = 0

c     //   number of degrees of freedom
      ndof = 0

c     //   periodic boundary
      if ( boundary(1:1) .eq. 'P' ) then

         ndof = 3*natom - 3
         uqtc = 1.5d0/beta
         fqtc = 1.5d0/beta

      else if ( boundary(1:1) .eq. 'F' ) then

         if ( natom .eq. 1 ) then
            ndof = 0
            uqtc = 1.5d0/beta
            fqtc = 1.5d0/beta
         else if ( natom .eq. 2 ) then
            ndof = 1
            uqtc = 2.5d0/beta
            fqtc = 2.5d0/beta
         else
            ndof = 3*natom - 6
            uqtc = 3.d0/beta
            fqtc = 3.d0/beta
         end if

      else

         ierr = 1

      end if

      if ( ndof .eq. 0 ) ierr = 1

      if ( ierr .ne. 0 ) then
         write( 6, '(a,2i6)' ) 'Error!', natom, ndof
         write( 6, '(a)' )
         stop
      end if

c-----------------------------------------------------------------------
c     //   read number of lines
c-----------------------------------------------------------------------

c     //   open file
      open ( iounit, file = trim(infile) )

c     //   read header lines
      read( iounit, *, iostat=ierr )
      read( iounit, *, iostat=ierr )
      read( iounit, *, iostat=ierr )

c     //   reset counter
      i = 0

c     //   loop of lines
      do

c        //   line by line
         read( iounit, *, iostat=ierr )

c        //   exit loop on error
         if ( ierr .ne. 0 ) exit

c        //   update counter
         i = i + 1

c     //   loop of lines
      end do

c     //   number of lines
      nomega = i / nkind

c     //   close file
      close( iounit )

c-----------------------------------------------------------------------
c     //   memory allocation
c-----------------------------------------------------------------------

c     //   frequencies
      allocate( omega(nomega) )

c     //   vibrational density of states
      allocate( vdos(nomega) )

c-----------------------------------------------------------------------
c     //   read number of lines
c-----------------------------------------------------------------------

c     //   open file
      open ( iounit, file = trim(infile) )

c     //   read header lines
      read( iounit, *, iostat=ierr )
      read( iounit, *, iostat=ierr )
      read( iounit, *, iostat=ierr )

c     //   loop of frequencies
      do i = 1, nomega

c        //   read frequency and vibrational density of states
         read( iounit, *, iostat=ierr ) omega(i), vdos(i)

c        //   skip lines
         do j = 1, nkind-1
            read( iounit, *, iostat=ierr )
         end do

c     //   loop of frequencies
      end do

c     //   close file
      close( iounit )

c-----------------------------------------------------------------------
c     //   normalize vdos
c-----------------------------------------------------------------------

c     //   sum over frequencies
      vsum = 0.d0

c     //   loop of frequencies
      do i = 1, nomega

c        //   cutoff
         if ( omega(i) .le. omega_cutoff ) cycle

c        //   sum over frequencies
         vsum = vsum + vdos(i)

c     //   loop of frequencies
      end do

c     //   normalization factor
      factor_norm = dble(ndof) / vsum

c     //   loop of frequencies
      do i = 1, nomega

c        //   sum over frequencies
         vdos(i) = vdos(i) * factor_norm

c     //   loop of frequencies
      end do

c-----------------------------------------------------------------------
c     //   count number of lines
c-----------------------------------------------------------------------

c     //   loop of frequencies
      do i = 1, nomega

c        //   cutoff
         if ( omega(i) .le. omega_cutoff ) cycle

c        //   in hartrees
         omega_au = omega(i) / factor_cminv

c        //   ho: vibrational energy
         ho = hbar * omega_au

c        //   half of ho
         ho_half = ho / 2.d0

c        //   bho: ho times beta
         bho = beta * ho

c        //   half of bho
         bho_half = bho / 2.d0

c        //   internal energy correction
         uqtc = uqtc + vdos(i)
     &        * ( ho_half + ho * exp(-bho) / (1.d0 - exp(-bho)) )

c        //   free energy correction
         fqtc = fqtc + vdos(i)
     &        * ( ho_half + 1.d0/beta * log(1.d0 - exp(-bho)) )

c     //   loop of frequencies
      end do

c-----------------------------------------------------------------------
c     //   print results
c-----------------------------------------------------------------------

      write( 6, '(a,5x,i8)' ) 'Degrees of freedom:', ndof

      write( 6, '(a,i8)' ) 'Highest frequency [/cm]:', nomega

      write( 6, '(a,5x,f12.6)' ) 'Scaling factor:', factor_norm

      write( 6, '(a)' )

      write( 6, '(a)' ) 'Helmholtz free energy (A):'
      write( 6, '(a)' )
     &   '------------------------------------------------'
      write( 6, '(a)' )
     &   '          U [au]        -TS [au]          A [au]'
      write( 6, '(a)' )
     &   '------------------------------------------------'

      write( 6, '(3f16.8)' )
     &   uqtc, fqtc-uqtc, fqtc

      write( 6, '(a)' ) 
      write( 6, '(a)' ) 'Normal termination of vdos2vfe.'
      write( 6, '(a)' ) 

c-----------------------------------------------------------------------
c     //   end of program
c-----------------------------------------------------------------------

      stop
      end

