c***********************************************************************
      module uc_variables
c***********************************************************************

c     /*   number of atoms   */
      integer :: natom

c     /*   atomic species   */
      character(len=8), dimension(:), allocatable :: species

c     /*   atomic positions   */
      real(8), dimension(:), allocatable :: x
      real(8), dimension(:), allocatable :: y
      real(8), dimension(:), allocatable :: z

c     /*   box matrix   */
      real(8), dimension(3,3) :: box

c     /*   bohr to angstrom   */
      real(8), parameter :: bohr2ang = 0.529177249d+00

c     /*   number of boxes   */
      integer :: na
      integer :: nb
      integer :: nc

c     /*   number of atoms in unit cell   */
      integer :: natom_uc

c     /*   number of atomic kinds   */
      integer :: nkind_uc

c     /*   atomic positions in unit cell   */
      real(8), dimension(:), allocatable :: x_uc
      real(8), dimension(:), allocatable :: y_uc
      real(8), dimension(:), allocatable :: z_uc

c     /*   atomic kinds   */
      integer, dimension(:), allocatable :: ikind

c     /*   atomic kinds   */
      integer, dimension(:), allocatable :: ikind_uc

c     /*   atomic species   */
      character(len=8), dimension(:), allocatable :: spec_uc

c     /*   atomic species   */
      character(len=8), dimension(:), allocatable :: species_uc

c     /*   unit cell matrix   */
      real(8), dimension(3,3) :: box_uc

c     /*   arguments   */
      character(len=16), dimension(10) :: char_arg

c     /*   length unit   */
      character(len=10) :: unit_uc

c     /*   lattice type   */
      character(len=10) :: lattice

c     /*   lattice parameters   */
      real(8) :: a_uc
      real(8) :: b_uc
      real(8) :: c_uc

c     /*   lattice parameters   */
      real(8) :: alpha_uc
      real(8) :: beta_uc
      real(8) :: gamma_uc

c***********************************************************************
      end module uc_variables
c***********************************************************************





c***********************************************************************
      program uc2sc
c***********************************************************************
c-----------------------------------------------------------------------
c     /*   shared variables                                           */
c-----------------------------------------------------------------------

      use uc_variables, only :
     &   x, y, z, x_uc, y_uc, z_uc, box, box_uc, bohr2ang,
     &   natom, ikind, na, nb, nc, natom_uc, ikind_uc,
     &   species, species_uc, lattice

c-----------------------------------------------------------------------
c     /*   local variables                                            */
c-----------------------------------------------------------------------

c     /*   reset  */
      implicit none

c     /*   integers   */
      integer :: i, j, ia, ib, ic

c     /*   real numbers   */
      real(8) :: da, db, dc, dx, dy, dz

c-----------------------------------------------------------------------
c     /*   usage                                                      */
c-----------------------------------------------------------------------

c     //   arguments
      if ( iargc() .lt. 1 ) then

c        //   print message
         write( 6, * ) 
         write( 6, * ) 'CODE:        uc2sc.x'
         write( 6, * ) 
         write( 6, * ) '             creates supercell for.'
         write( 6, * ) '             FCC, BCC, FCC1, BCC1, B1, B2.'
         write( 6, * ) 
         write( 6, * ) 'USAGE:       uc2sc.x $1 $2 $3 $4 $5 $6 $7'
         write( 6, * ) '$1:          lattice type (FCC).'
         write( 6, * ) '$2:          atomic species (Ni).'
         write( 6, * ) '$3:          lattice constant (3.52).'
         write( 6, * ) '$4:          length unit (ANGSTROM).'
         write( 6, * ) '$5:          boxes in a direction (4).'
         write( 6, * ) '$6:          boxes in b direction (4).'
         write( 6, * ) '$7:          boxes in c direction (4).'
         write( 6, * ) 
         write( 6, * ) 'USAGE:       uc2sc.x $1 $2 $3 $4 $5 $6 $7 $8'
         write( 6, * ) '$1:          lattice type (B2).'
         write( 6, * ) '$2:          atomic species (Cs).'
         write( 6, * ) '$2:          atomic species (Cl).'
         write( 6, * ) '$3:          lattice constant (4.12).'
         write( 6, * ) '$4:          length unit (ANGSTROM).'
         write( 6, * ) '$5:          boxes in a direction (4).'
         write( 6, * ) '$6:          boxes in b direction (4).'
         write( 6, * ) '$7:          boxes in c direction (4).'
         write( 6, * ) 

c        //   error termination
         stop

c     //   arguments
      end if

c-----------------------------------------------------------------------
c     /*   first argument: lattice                                    */
c-----------------------------------------------------------------------

c     /*   read argument  */
      call getarg( 1, lattice )

c-----------------------------------------------------------------------
c     /*   lattice parameters                                         */
c-----------------------------------------------------------------------

      if      ( lattice(1:4) .eq. "FCC " ) then
         call uc_fcc
      else if ( lattice(1:4) .eq. "FCC1" ) then
         call uc_fcc1
      else if ( lattice(1:4) .eq. "BCC " ) then
         call uc_bcc
      else if ( lattice(1:4) .eq. "BCC1" ) then
         call uc_bcc1
      else if ( lattice(1:4) .eq. "HCP " ) then
         call uc_hcp
      else if ( lattice(1:4) .eq. "B1  " ) then
         call uc_b1
      else if ( lattice(1:4) .eq. "B2  " ) then
         call uc_b2
      else
         write( 6, '(a)' ) 'Error.'
         stop
      end if

c-----------------------------------------------------------------------
c     /*   system setup                                               */
c-----------------------------------------------------------------------

c     /*   number of atoms   */
      natom = na * nb * nc * natom_uc

c     /*   memory allocation   */
      allocate( x(natom) )
      allocate( y(natom) )
      allocate( z(natom) )

c     /*   memory allocation   */
      allocate( species(natom) )
      allocate( ikind(natom) )

c     /*   box matrix   */
      box(1,1) = dble(na) * box_uc(1,1)
      box(2,1) = dble(na) * box_uc(2,1)
      box(3,1) = dble(na) * box_uc(3,1)
      box(1,2) = dble(nb) * box_uc(1,2)
      box(2,2) = dble(nb) * box_uc(2,2)
      box(3,2) = dble(nb) * box_uc(3,2)
      box(1,3) = dble(nc) * box_uc(1,3)
      box(2,3) = dble(nc) * box_uc(2,3)
      box(3,3) = dble(nc) * box_uc(3,3)

c-----------------------------------------------------------------------
c     /*   main loop                                                  */
c-----------------------------------------------------------------------

c     /*   atomic position   */
      x(:) = 0.d0
      y(:) = 0.d0
      z(:) = 0.d0

c     /*   counter   */
      i = 0

c     /*   loop of boxes   */
      do ia = 0, na-1
      do ib = 0, nb-1
      do ic = 0, nc-1

c        /*   cell index   */
         da = dble(ia)
         db = dble(ib)
         dc = dble(ic)

c        /*   vector   */
         dx = da*box_uc(1,1) + db*box_uc(1,2) + dc*box_uc(1,3)
         dy = da*box_uc(2,1) + db*box_uc(2,2) + dc*box_uc(2,3)
         dz = da*box_uc(3,1) + db*box_uc(3,2) + dc*box_uc(3,3)

c        /*   loop of atoms in unit cell   */
         do j = 1, natom_uc

c          /*   counter   */
           i = i + 1

c          /*   atomic species   */
           species(i) = species_uc(j)

c          /*   atomic position   */
           x(i) = x(i) + dx + x_uc(j)
           y(i) = y(i) + dy + y_uc(j)
           z(i) = z(i) + dz + z_uc(j)

c          /*   atomi kind   */
           ikind(i) = ikind_uc(j)

c        /*   loop of atoms in unit cell   */
         end do

c     /*   loop of boxes   */
      end do
      end do
      end do

c-----------------------------------------------------------------------
c     /*   print box matrix                                           */
c-----------------------------------------------------------------------
c     /*   blank   */
      write( 6, '(a)' )

c     /*   blank   */
      write( 6, '(a)' )
     &   '--------------------------------------------------------'
      write( 6, '(a)' )
     &   '                        box matrix                      '
      write( 6, '(a)' )
     &   '--------------------------------------------------------'

c     /*   blank   */
      write( 6, '(a)' )

cc     /*   unit   */
c      write( 6, '(a)' ) 'BOHR'

cc     /*   box matrix   */
c      write( 6, '(4x,3f16.8)' ) box(1,1), box(1,2), box(1,3)
c      write( 6, '(4x,3f16.8)' ) box(2,1), box(2,2), box(2,3)
c      write( 6, '(4x,3f16.8)' ) box(3,1), box(3,2), box(3,3)

c     /*   unit   */
      write( 6, '(a)' ) 'ANGSTROM'

c     /*   box matrix   */
      write( 6, '(4x,3f16.8)' )
     &   box(1,1)*bohr2ang, box(1,2)*bohr2ang, box(1,3)*bohr2ang
      write( 6, '(4x,3f16.8)' )
     &   box(2,1)*bohr2ang, box(2,2)*bohr2ang, box(2,3)*bohr2ang
      write( 6, '(4x,3f16.8)' )
     &   box(3,1)*bohr2ang, box(3,2)*bohr2ang, box(3,3)*bohr2ang

c-----------------------------------------------------------------------
c     /*   print atomic position                                      */
c-----------------------------------------------------------------------

c     /*   blank   */
      write( 6, '(a)' )

c     /*   blank   */
      write( 6, '(a)' )
     &   '--------------------------------------------------------'
      write( 6, '(a)' )
     &   '                    atomic positions                    '
      write( 6, '(a)' )
     &   '--------------------------------------------------------'

c     /*   blank   */
      write( 6, '(a)' )

c     /*   number of atoms   */
      write( 6, '(i8)' ) natom

cc     /*   unit   */
c      write( 6, '(a)' ) 'BOHR'

c     /*   unit   */
      write( 6, '(a)' ) 'ANGSTROM'

c     /*   loop of atoms   */
      do i = 1, natom

c        /*   atomic species, positions, kinds   */
c         write( 6, '(a4,3f16.8,i4)' )
c     &      species(i), x(i), y(i), z(i), ikind(i)

c        /*   atomic species, positions, kinds   */
         write( 6, '(a4,3f16.8,i4)' )
     &      species(i), x(i)*bohr2ang, y(i)*bohr2ang, z(i)*bohr2ang,
     &      ikind(i)

c     /*   loop of atoms   */
      end do

c-----------------------------------------------------------------------

      stop
      end





c***********************************************************************
      subroutine uc_fcc
c***********************************************************************
c-----------------------------------------------------------------------
c     /*   shared variables                                           */
c-----------------------------------------------------------------------

      use uc_variables, only :
     &   bohr2ang, a_uc, box_uc, x_uc, y_uc, z_uc,
     &   ikind_uc, natom_uc, nkind_uc, na, nb, nc,
     &   unit_uc, species_uc, spec_uc, char_arg

c-----------------------------------------------------------------------
c     /*   local variables                                            */
c-----------------------------------------------------------------------

c     /*   initialize   */
      implicit none

c     /*   integers   */
      integer :: i, ierr

c     /*   real numbers   */
      real(8) :: factor = 1.d0

c-----------------------------------------------------------------------
c     /*   unit cell                                                  */
c-----------------------------------------------------------------------

c     /*   number of atoms in a unit cell   */
      natom_uc = 4

c     /*   number of atomic kinds   */
      nkind_uc = 1

c-----------------------------------------------------------------------
c     /*   memory allocation                                          */
c-----------------------------------------------------------------------

c     /*   atomic positions in unit cell    */
      allocate( x_uc(natom_uc) )
      allocate( y_uc(natom_uc) )
      allocate( z_uc(natom_uc) )

c     /*   atomic species    */
      allocate( species_uc(natom_uc) )

c     /*   atomic kinds   */
      allocate( ikind_uc(natom_uc) )

c     /*   number of atomic kinds   */
      allocate( spec_uc(nkind_uc) )

c-----------------------------------------------------------------------
c     /*   read arguments                                             */
c-----------------------------------------------------------------------

      do i = 2, 7
         call getarg( i, char_arg(i) )
      end do

c-----------------------------------------------------------------------
c     /*   example:  BCC Ni 3.52 ANGSTROM 4 4 4                       */
c-----------------------------------------------------------------------

c     /*   argument 2:  atomic species   */
      spec_uc(1) = char_arg(2)

c     /*   argument 3:  lattice constant   */
      read ( char_arg(3), *, iostat=ierr ) a_uc

c     /*   argument 4:  unit of lattice constant   */
      unit_uc = char_arg(4)

c     /*   argument 5:  number of lattices in a, b, c directions   */
      read ( char_arg(5), *, iostat=ierr ) na
      read ( char_arg(6), *, iostat=ierr ) nb
      read ( char_arg(7), *, iostat=ierr ) nc

c     /*   unit conversion factor   */
      if ( unit_uc(1:1) .eq. 'A' ) factor = 1.d0 / bohr2ang

c-----------------------------------------------------------------------
c     /*   settings of FCC                                            */
c-----------------------------------------------------------------------

c     /*   atomic species   */
      species_uc(1) = spec_uc(1)
      species_uc(2) = spec_uc(1)
      species_uc(3) = spec_uc(1)
      species_uc(4) = spec_uc(1)

c     /*   atomic kinds   */
      ikind_uc(1)   = 1
      ikind_uc(2)   = 1
      ikind_uc(3)   = 1
      ikind_uc(4)   = 1

c     /*   unit cell matrix   */
      box_uc(1,1) = a_uc * factor
      box_uc(1,2) = 0.d0
      box_uc(1,3) = 0.d0
      box_uc(2,1) = 0.d0
      box_uc(2,2) = a_uc * factor
      box_uc(2,3) = 0.d0
      box_uc(3,1) = 0.d0
      box_uc(3,2) = 0.d0
      box_uc(3,3) = a_uc * factor

c     /*   atomic positions   */
      x_uc(1) =  0.d0
      y_uc(1) =  0.d0
      z_uc(1) =  0.d0
      x_uc(2) =  0.d0
      y_uc(2) =  a_uc / 2.d0 * factor
      z_uc(2) =  a_uc / 2.d0 * factor
      x_uc(3) =  a_uc / 2.d0 * factor
      y_uc(3) =  0.d0
      z_uc(3) =  a_uc / 2.d0 * factor
      x_uc(4) =  a_uc / 2.d0 * factor
      y_uc(4) =  a_uc / 2.d0 * factor
      z_uc(4) =  0.d0

      return
      end





c***********************************************************************
      subroutine uc_fcc1
c***********************************************************************
c-----------------------------------------------------------------------
c     /*   shared variables                                           */
c-----------------------------------------------------------------------

      use uc_variables, only :
     &   bohr2ang, a_uc, box_uc, x_uc, y_uc, z_uc,
     &   ikind_uc, natom_uc, nkind_uc, na, nb, nc,
     &   unit_uc, species_uc, spec_uc, char_arg

c-----------------------------------------------------------------------
c     /*   local variables                                            */
c-----------------------------------------------------------------------

c     /*   initialize   */
      implicit none

c     /*   integers   */
      integer :: i, ierr

c     /*   real numbers   */
      real(8) :: factor = 1.d0

c-----------------------------------------------------------------------
c     /*   unit cell                                                  */
c-----------------------------------------------------------------------

c     /*   number of atoms in a unit cell   */
      natom_uc = 1

c     /*   number of atomic kinds   */
      nkind_uc = 1

c-----------------------------------------------------------------------
c     /*   memory allocation                                          */
c-----------------------------------------------------------------------

c     /*   atomic positions in unit cell    */
      allocate( x_uc(natom_uc) )
      allocate( y_uc(natom_uc) )
      allocate( z_uc(natom_uc) )

c     /*   atomic species    */
      allocate( species_uc(natom_uc) )

c     /*   atomic kinds   */
      allocate( ikind_uc(natom_uc) )

c     /*   number of atomic kinds   */
      allocate( spec_uc(nkind_uc) )

c-----------------------------------------------------------------------
c     /*   read arguments                                             */
c-----------------------------------------------------------------------

      do i = 2, 7
         call getarg( i, char_arg(i) )
      end do

c-----------------------------------------------------------------------
c     /*   example:  BCC Ni 3.52 ANGSTROM 4 4 4                       */
c-----------------------------------------------------------------------

c     /*   argument 2:  atomic species   */
      spec_uc(1) = char_arg(2)

c     /*   argument 3:  lattice constant   */
      read ( char_arg(3), *, iostat=ierr ) a_uc

c     /*   argument 4:  unit of lattice constant   */
      unit_uc = char_arg(4)

c     /*   argument 5:  number of lattices in a, b, c directions   */
      read ( char_arg(5), *, iostat=ierr ) na
      read ( char_arg(6), *, iostat=ierr ) nb
      read ( char_arg(7), *, iostat=ierr ) nc

c     /*   unit conversion factor   */
      if ( unit_uc(1:1) .eq. 'A' ) factor = 1.d0 / bohr2ang

c-----------------------------------------------------------------------
c     /*   settings of FCC                                            */
c-----------------------------------------------------------------------

c     /*   atomic species   */
      species_uc(1) = spec_uc(1)

c     /*   atomic kinds   */
      ikind_uc(1)   = 1

cc     //   original axis lattice vector in [0,1,1]/sqrt(2)
c      box_uc(1,1) =   0.0d0
c      box_uc(2,1) =   a_uc * factor / 2.d0
c      box_uc(3,1) =   a_uc * factor / 2.d0
c
cc     //   original axis lottice vector in [1,0,-1]/sqrt(2)
c      box_uc(1,2) = - a_uc * factor / 2.d0
c      box_uc(2,2) =   0.0d0
c      box_uc(3,2) =   a_uc * factor / 2.d0
c
cc     //   original axis lattice vector in [1,-1,0]/sqrt(2)
c      box_uc(1,3) = - a_uc * factor / 2.d0
c      box_uc(2,3) =   a_uc * factor / 2.d0
c      box_uc(3,3) =   0.0d0
c
c      //   new axis
c      //   axis x = [0,1,1]/sqrt(2)
c      //   axis y = [-2,-1,1]/sqrt(6)
c      //   axis z = [-1,1,-1]/sqrt(3)

c     //   new axis lattice vector in 011 direction
      box_uc(1,1) = a_uc * factor / sqrt(2.d0)
      box_uc(2,1) = 0.0d0
      box_uc(3,1) = 0.0d0

c     //   new axis lottice vector in 101 direction
      box_uc(1,2) = a_uc * factor / sqrt(8.d0)
      box_uc(2,2) = a_uc * factor * sqrt(3.d0) / sqrt(8.d0)
      box_uc(3,2) = 0.0d0

c     //   new axis lattice vector in 110 direction
      box_uc(1,3) = a_uc * factor / sqrt(8.d0)
      box_uc(2,3) = a_uc * factor / sqrt(3.d0) / sqrt(8.d0)
      box_uc(3,3) = a_uc * factor / sqrt(3.d0)

c     /*   atomic positions   */
      x_uc(1) =  0.d0
      y_uc(1) =  0.d0
      z_uc(1) =  0.d0

      return
      end





c***********************************************************************
      subroutine uc_bcc1
c***********************************************************************
c-----------------------------------------------------------------------
c     /*   shared variables                                           */
c-----------------------------------------------------------------------

      use uc_variables, only :
     &   bohr2ang, a_uc, box_uc, x_uc, y_uc, z_uc,
     &   ikind_uc, natom_uc, nkind_uc, na, nb, nc,
     &   unit_uc, species_uc, spec_uc, char_arg

c-----------------------------------------------------------------------
c     /*   local variables                                            */
c-----------------------------------------------------------------------

c     /*   initialize   */
      implicit none

c     /*   integers   */
      integer :: i, ierr

c     /*   real numbers   */
      real(8) :: factor = 1.d0

c-----------------------------------------------------------------------
c     /*   unit cell                                                  */
c-----------------------------------------------------------------------

c     /*   number of atoms in a unit cell   */
      natom_uc = 1

c     /*   number of atomic kinds   */
      nkind_uc = 1

c-----------------------------------------------------------------------
c     /*   memory allocation                                          */
c-----------------------------------------------------------------------

c     /*   atomic positions in unit cell    */
      allocate( x_uc(natom_uc) )
      allocate( y_uc(natom_uc) )
      allocate( z_uc(natom_uc) )

c     /*   atomic species    */
      allocate( species_uc(natom_uc) )

c     /*   atomic kinds   */
      allocate( ikind_uc(natom_uc) )

c     /*   number of atomic kinds   */
      allocate( spec_uc(nkind_uc) )

c-----------------------------------------------------------------------
c     /*   read arguments                                             */
c-----------------------------------------------------------------------

      do i = 2, 7
         call getarg( i, char_arg(i) )
      end do

c-----------------------------------------------------------------------
c     /*   example:  BCC Ni 3.52 ANGSTROM 4 4 4                       */
c-----------------------------------------------------------------------

c     /*   argument 2:  atomic species   */
      spec_uc(1) = char_arg(2)

c     /*   argument 3:  lattice constant   */
      read ( char_arg(3), *, iostat=ierr ) a_uc

c     /*   argument 4:  unit of lattice constant   */
      unit_uc = char_arg(4)

c     /*   argument 5:  number of lattices in a, b, c directions   */
      read ( char_arg(5), *, iostat=ierr ) na
      read ( char_arg(6), *, iostat=ierr ) nb
      read ( char_arg(7), *, iostat=ierr ) nc

c     /*   unit conversion factor   */
      if ( unit_uc(1:1) .eq. 'A' ) factor = 1.d0 / bohr2ang

c-----------------------------------------------------------------------
c     /*   settings of FCC                                            */
c-----------------------------------------------------------------------

c     /*   atomic species   */
      species_uc(1) = spec_uc(1)

c     /*   atomic kinds   */
      ikind_uc(1)   = 1

c     //   original axis lattice vector in [0,1,1]/sqrt(2)
      box_uc(1,1) =   a_uc * factor
      box_uc(2,1) =   0.d0
      box_uc(3,1) =   0.d0

c     //   original axis lottice vector in [1,0,-1]/sqrt(2)
      box_uc(1,2) =   a_uc * factor / 2.d0
      box_uc(2,2) =   a_uc * factor / 2.d0
      box_uc(3,2) =   a_uc * factor / 2.d0

c     //   original axis lattice vector in [1,-1,0]/sqrt(2)
      box_uc(1,3) =   a_uc * factor / 2.d0
      box_uc(2,3) =   a_uc * factor / 2.d0
      box_uc(3,3) = - a_uc * factor / 2.d0

c     /*   atomic positions   */
      x_uc(1) =  0.d0
      y_uc(1) =  0.d0
      z_uc(1) =  0.d0

      return
      end





c***********************************************************************
      subroutine uc_bcc
c***********************************************************************
c-----------------------------------------------------------------------
c     /*   shared variables                                           */
c-----------------------------------------------------------------------

      use uc_variables, only :
     &   bohr2ang, a_uc, box_uc, x_uc, y_uc, z_uc,
     &   ikind_uc, natom_uc, nkind_uc, na, nb, nc,
     &   unit_uc, species_uc, spec_uc, char_arg

c-----------------------------------------------------------------------
c     /*   local variables                                            */
c-----------------------------------------------------------------------

c     /*   initialize   */
      implicit none

c     /*   integers   */
      integer :: i, ierr

c     /*   real numbers   */
      real(8) :: factor = 1.d0

c-----------------------------------------------------------------------
c     /*   unit cell                                                  */
c-----------------------------------------------------------------------

c     /*   number of atoms in a unit cell   */
      natom_uc = 2

c     /*   number of atomic kinds   */
      nkind_uc = 1

c-----------------------------------------------------------------------
c     /*   memory allocation                                          */
c-----------------------------------------------------------------------

c     /*   atomic positions in unit cell    */
      allocate( x_uc(natom_uc) )
      allocate( y_uc(natom_uc) )
      allocate( z_uc(natom_uc) )

c     /*   atomic species    */
      allocate( species_uc(natom_uc) )

c     /*   atomic kinds   */
      allocate( ikind_uc(natom_uc) )

c     /*   number of atomic kinds   */
      allocate( spec_uc(nkind_uc) )

c-----------------------------------------------------------------------
c     /*   read arguments                                             */
c-----------------------------------------------------------------------

      do i = 2, 7
         call getarg( i, char_arg(i) )
      end do

c-----------------------------------------------------------------------
c     /*   example:  BCC Fe 2.86 ANGSTROM 4 4 4                       */
c-----------------------------------------------------------------------

c     /*   argument 2:  atomic species   */
      spec_uc(1) = char_arg(2)

c     /*   argument 3:  lattice constant   */
      read ( char_arg(3), *, iostat=ierr ) a_uc

c     /*   argument 4:  unit of lattice constant   */
      unit_uc = char_arg(4)

c     /*   argument 5:  number of lattices in a, b, c directions   */
      read ( char_arg(5), *, iostat=ierr ) na
      read ( char_arg(6), *, iostat=ierr ) nb
      read ( char_arg(7), *, iostat=ierr ) nc

c     /*   unit conversion factor   */
      if ( unit_uc(1:1) .eq. 'A' ) factor = 1.d0 / bohr2ang

c-----------------------------------------------------------------------
c     /*   settings of BCC                                            */
c-----------------------------------------------------------------------

c     /*   atomic species   */
      species_uc(1) = spec_uc(1)
      species_uc(2) = spec_uc(1)

c     /*   atomic kinds   */
      ikind_uc(1)   = 1
      ikind_uc(2)   = 1

c     /*   unit cell matrix   */
      box_uc(1,1) = a_uc * factor
      box_uc(1,2) = 0.d0
      box_uc(1,3) = 0.d0
      box_uc(2,1) = 0.d0
      box_uc(2,2) = a_uc * factor
      box_uc(2,3) = 0.d0
      box_uc(3,1) = 0.d0
      box_uc(3,2) = 0.d0
      box_uc(3,3) = a_uc * factor

c     /*   atomic positions   */
      x_uc(1) =  0.d0
      y_uc(1) =  0.d0
      z_uc(1) =  0.d0
      x_uc(2) =  a_uc / 2.d0 * factor
      y_uc(2) =  a_uc / 2.d0 * factor
      z_uc(2) =  a_uc / 2.d0 * factor

      return
      end





c***********************************************************************
      subroutine uc_hcp
c***********************************************************************
c-----------------------------------------------------------------------
c     /*   shared variables                                           */
c-----------------------------------------------------------------------

      use uc_variables, only :
     &   bohr2ang, a_uc, box_uc, x_uc, y_uc, z_uc,
     &   ikind_uc, natom_uc, nkind_uc, na, nb, nc,
     &   unit_uc, species_uc, spec_uc, char_arg

c-----------------------------------------------------------------------
c     /*   local variables                                            */
c-----------------------------------------------------------------------

c     /*   initialize   */
      implicit none

c     /*   integers   */
      integer :: i, ierr

c     /*   real numbers   */
      real(8) :: factor = 1.d0

c-----------------------------------------------------------------------
c     /*   unit cell                                                  */
c-----------------------------------------------------------------------

c     /*   number of atoms in a unit cell   */
      natom_uc = 2

c     /*   number of atomic kinds   */
      nkind_uc = 1

c-----------------------------------------------------------------------
c     /*   memory allocation                                          */
c-----------------------------------------------------------------------

c     /*   atomic positions in unit cell    */
      allocate( x_uc(natom_uc) )
      allocate( y_uc(natom_uc) )
      allocate( z_uc(natom_uc) )

c     /*   atomic species    */
      allocate( species_uc(natom_uc) )

c     /*   atomic kinds   */
      allocate( ikind_uc(natom_uc) )

c     /*   number of atomic kinds   */
      allocate( spec_uc(nkind_uc) )

c-----------------------------------------------------------------------
c     /*   read arguments                                             */
c-----------------------------------------------------------------------

      do i = 2, 7
         call getarg( i, char_arg(i) )
      end do

c-----------------------------------------------------------------------
c     /*   example:  HCP Mg 3.21 ANGSTROM 4 4 4                       */
c-----------------------------------------------------------------------

c     /*   argument 2:  atomic species   */
      spec_uc(1) = char_arg(2)

c     /*   argument 3:  lattice constant   */
      read ( char_arg(3), *, iostat=ierr ) a_uc

c     /*   argument 4:  unit of lattice constant   */
      unit_uc = char_arg(4)

c     /*   argument 5:  number of lattices in a, b, c directions   */
      read ( char_arg(5), *, iostat=ierr ) na
      read ( char_arg(6), *, iostat=ierr ) nb
      read ( char_arg(7), *, iostat=ierr ) nc

c     /*   unit conversion factor   */
      if ( unit_uc(1:1) .eq. 'A' ) factor = 1.d0 / bohr2ang

c-----------------------------------------------------------------------
c     /*   settings of HCP                                            */
c-----------------------------------------------------------------------

c     /*   atomic species   */
      species_uc(1) = spec_uc(1)
      species_uc(2) = spec_uc(1)

c     /*   atomic kinds   */
      ikind_uc(1)   = 1
      ikind_uc(2)   = 1

c     /*   unit cell matrix   */
      box_uc(1,1) = + 0.5d0           * a_uc * factor
      box_uc(2,1) = - sqrt(3.d0)/2.d0 * a_uc * factor
      box_uc(3,1) = + 0.0d0           * a_uc * factor
      box_uc(1,2) = + 0.5d0           * a_uc * factor
      box_uc(2,2) = + sqrt(3.d0)/2.d0 * a_uc * factor
      box_uc(3,2) = + 0.0d0           * a_uc * factor
      box_uc(1,3) = + 0.0d0           * a_uc * factor
      box_uc(2,3) = + 0.0d0           * a_uc * factor
      box_uc(3,3) = + sqrt(8.d0/3.d0) * a_uc * factor

c     /*   atomic positions   */
      x_uc(1) = + 0.d0                * a_uc * factor
      y_uc(1) = + 0.d0                * a_uc * factor
      z_uc(1) = + 0.d0                * a_uc * factor
      x_uc(2) = + 0.5d0               * a_uc * factor
      y_uc(2) = + sqrt(3.d0)/6.d0     * a_uc * factor
      z_uc(2) = + sqrt(2.d0/3.d0)     * a_uc * factor

      return
      end





c***********************************************************************
      subroutine uc_b1
c***********************************************************************
c-----------------------------------------------------------------------
c     /*   shared variables                                           */
c-----------------------------------------------------------------------

      use uc_variables, only :
     &   bohr2ang, a_uc, box_uc, x_uc, y_uc, z_uc,
     &   ikind_uc, natom_uc, nkind_uc, na, nb, nc,
     &   unit_uc, species_uc, spec_uc, char_arg

c-----------------------------------------------------------------------
c     /*   local variables                                            */
c-----------------------------------------------------------------------

c     /*   initialize   */
      implicit none

c     /*   integers   */
      integer :: i, ierr

c     /*   real numbers   */
      real(8) :: factor = 1.d0

c-----------------------------------------------------------------------
c     /*   unit cell                                                  */
c-----------------------------------------------------------------------

c     /*   number of atoms in a unit cell   */
      natom_uc = 8

c     /*   number of atomic kinds   */
      nkind_uc = 2

c-----------------------------------------------------------------------
c     /*   memory allocation                                          */
c-----------------------------------------------------------------------

c     /*   atomic positions in unit cell    */
      allocate( x_uc(natom_uc) )
      allocate( y_uc(natom_uc) )
      allocate( z_uc(natom_uc) )

c     /*   atomic species    */
      allocate( species_uc(natom_uc) )

c     /*   atomic kinds   */
      allocate( ikind_uc(natom_uc) )

c     /*   number of atomic kinds   */
      allocate( spec_uc(nkind_uc) )

c-----------------------------------------------------------------------
c     /*   read arguments                                             */
c-----------------------------------------------------------------------

      do i = 2, 8
         call getarg( i, char_arg(i) )
      end do

c-----------------------------------------------------------------------
c     /*   example:  B1 Na Cl 5.64 ANGSTROM 3 3 3                     */
c-----------------------------------------------------------------------

c     /*   argument 2:  atomic species   */
      spec_uc(1) = char_arg(2)

c     /*   argument 3:  atomic species   */
      spec_uc(2) = char_arg(3)

c     /*   argument 4:  lattice constant   */
      read ( char_arg(4), *, iostat=ierr ) a_uc

c     /*   argument 5:  unit of lattice constant   */
      unit_uc = char_arg(5)

c     /*   argument 5:  number of lattices in a, b, c directions   */
      read ( char_arg(6), *, iostat=ierr ) na
      read ( char_arg(7), *, iostat=ierr ) nb
      read ( char_arg(8), *, iostat=ierr ) nc

c     /*   unit conversion factor   */
      if ( unit_uc(1:1) .eq. 'A' ) factor = 1.d0 / bohr2ang

c-----------------------------------------------------------------------
c     /*   settings of B1                                             */
c-----------------------------------------------------------------------

c     /*   atomic species   */
      species_uc(1) = spec_uc(1)
      species_uc(2) = spec_uc(1)
      species_uc(3) = spec_uc(1)
      species_uc(4) = spec_uc(1)
      species_uc(5) = spec_uc(2)
      species_uc(6) = spec_uc(2)
      species_uc(7) = spec_uc(2)
      species_uc(8) = spec_uc(2)

c     /*   atomic kinds   */
      ikind_uc(1)   = 1
      ikind_uc(2)   = 1
      ikind_uc(3)   = 1
      ikind_uc(4)   = 1
      ikind_uc(5)   = 2
      ikind_uc(6)   = 2
      ikind_uc(7)   = 2
      ikind_uc(8)   = 2

c     /*   unit cell matrix   */
      box_uc(1,1) = a_uc * factor
      box_uc(1,2) = 0.d0
      box_uc(1,3) = 0.d0
      box_uc(2,1) = 0.d0
      box_uc(2,2) = a_uc * factor
      box_uc(2,3) = 0.d0
      box_uc(3,1) = 0.d0
      box_uc(3,2) = 0.d0
      box_uc(3,3) = a_uc * factor

c     /*   atomic positions   */
      x_uc(1) =  0.d0
      y_uc(1) =  0.d0
      z_uc(1) =  0.d0
      x_uc(2) =  0.d0
      y_uc(2) =  a_uc / 2.d0 * factor
      z_uc(2) =  a_uc / 2.d0 * factor
      x_uc(3) =  a_uc / 2.d0 * factor
      y_uc(3) =  0.d0
      z_uc(3) =  a_uc / 2.d0 * factor
      x_uc(4) =  a_uc / 2.d0 * factor
      y_uc(4) =  a_uc / 2.d0 * factor
      z_uc(4) =  0.d0
      x_uc(5) =  a_uc / 2.d0 * factor
      y_uc(5) =  0.d0
      z_uc(5) =  0.d0
      x_uc(6) =  0.d0
      y_uc(6) =  a_uc / 2.d0 * factor
      z_uc(6) =  0.d0
      x_uc(7) =  0.d0
      y_uc(7) =  0.d0
      z_uc(7) =  a_uc / 2.d0 * factor
      x_uc(8) =  a_uc / 2.d0 * factor
      y_uc(8) =  a_uc / 2.d0 * factor
      z_uc(8) =  a_uc / 2.d0 * factor

      return
      end










c***********************************************************************
      subroutine uc_b2
c***********************************************************************
c-----------------------------------------------------------------------
c     /*   shared variables                                           */
c-----------------------------------------------------------------------

      use uc_variables, only :
     &   bohr2ang, a_uc, box_uc, x_uc, y_uc, z_uc,
     &   ikind_uc, natom_uc, nkind_uc, na, nb, nc,
     &   unit_uc, species_uc, spec_uc, char_arg

c-----------------------------------------------------------------------
c     /*   local variables                                            */
c-----------------------------------------------------------------------

c     /*   initialize   */
      implicit none

c     /*   integers   */
      integer :: i, ierr

c     /*   real numbers   */
      real(8) :: factor = 1.d0

c-----------------------------------------------------------------------
c     /*   unit cell                                                  */
c-----------------------------------------------------------------------

c     /*   number of atoms in a unit cell   */
      natom_uc = 2

c     /*   number of atomic kinds   */
      nkind_uc = 2

c-----------------------------------------------------------------------
c     /*   memory allocation                                          */
c-----------------------------------------------------------------------

c     /*   atomic positions in unit cell    */
      allocate( x_uc(natom_uc) )
      allocate( y_uc(natom_uc) )
      allocate( z_uc(natom_uc) )

c     /*   atomic species    */
      allocate( species_uc(natom_uc) )

c     /*   atomic kinds   */
      allocate( ikind_uc(natom_uc) )

c     /*   number of atomic kinds   */
      allocate( spec_uc(nkind_uc) )

c-----------------------------------------------------------------------
c     /*   read arguments                                             */
c-----------------------------------------------------------------------

      do i = 2, 8
         call getarg( i, char_arg(i) )
      end do

c-----------------------------------------------------------------------
c     /*   example:  B2 Cs Cl 4.12 ANGSTROM 4 4 4                     */
c-----------------------------------------------------------------------

c     /*   argument 2:  atomic species   */
      spec_uc(1) = char_arg(2)

c     /*   argument 3:  atomic species   */
      spec_uc(2) = char_arg(3)

c     /*   argument 4:  lattice constant   */
      read ( char_arg(4), *, iostat=ierr ) a_uc

c     /*   argument 5:  unit of lattice constant   */
      unit_uc = char_arg(5)

c     /*   argument 5:  number of lattices in a, b, c directions   */
      read ( char_arg(6), *, iostat=ierr ) na
      read ( char_arg(7), *, iostat=ierr ) nb
      read ( char_arg(8), *, iostat=ierr ) nc

c     /*   unit conversion factor   */
      if ( unit_uc(1:1) .eq. 'A' ) factor = 1.d0 / bohr2ang

c-----------------------------------------------------------------------
c     /*   settings of B2                                             */
c-----------------------------------------------------------------------

c     /*   atomic species   */
      species_uc(1) = spec_uc(1)
      species_uc(2) = spec_uc(2)

c     /*   atomic kinds   */
      ikind_uc(1)   = 1
      ikind_uc(2)   = 2

c     /*   unit cell matrix   */
      box_uc(1,1) = a_uc * factor
      box_uc(1,2) = 0.d0
      box_uc(1,3) = 0.d0
      box_uc(2,1) = 0.d0
      box_uc(2,2) = a_uc * factor
      box_uc(2,3) = 0.d0
      box_uc(3,1) = 0.d0
      box_uc(3,2) = 0.d0
      box_uc(3,3) = a_uc * factor

c     /*   atomic positions   */
      x_uc(1) =  0.d0
      y_uc(1) =  0.d0
      z_uc(1) =  0.d0
      x_uc(2) =  a_uc / 2.d0 * factor
      y_uc(2) =  a_uc / 2.d0 * factor
      z_uc(2) =  a_uc / 2.d0 * factor

      return
      end
