c***********************************************************************
      program tstrate
c***********************************************************************
c-----------------------------------------------------------------------
c     //   variables
c-----------------------------------------------------------------------

c     //   initialize
      implicit none

c     //   rate
      real(8) :: rate

c     //   free energies
      real(8) :: fenergy_ts, fenergy_eq, dfenergy

c     //   temperature
      real(8) :: temperature

c     //   real numbers
      real(8) :: prefactor, exponent, s, halflife

c     //   characters
      character(len=80) :: char, scale

c     //   circular constant
      real(8) :: pi

c     //   boltzmann constant
      real(8) :: boltz = 1.380649e-23

c     //   plank constant
      real(8) :: h = 6.62607015e-34

c     //   conversion factors
      real(8) :: to_kcalmol = 6.94782e-21
      real(8) :: to_cminv   = 1.98644e-23
      real(8) :: to_kjmol   = 1.66057e-21
      real(8) :: to_ev      = 1.60218e-19
      real(8) :: to_hartree = 4.35974e-18

c     //   integers
      integer :: iargc

c-----------------------------------------------------------------------
c     //   initial message
c-----------------------------------------------------------------------

      if ( iargc() .ne. 4 ) then

         write( 6, '(a)' )
         write( 6, '(a)' ) 'Program tstrate'
         write( 6, '(a)' )
         write( 6, '(a)' ) 'Usage: tstrate.x $1 $2 $3 $4'
         write( 6, '(a)' )
         write( 6, '(a)' ) '$1: temperature [K]'
         write( 6, '(a)' ) '$2: free energy at equilibrium state'
         write( 6, '(a)' ) '$3: free energy at transition state'
         write( 6, '(a)' ) '$4: energy unit from following options'
         write( 6, '(a)' ) '    kcalmol: kcal/mol'
         write( 6, '(a)' ) '    kjmol:   kJ/mol'
         write( 6, '(a)' ) '    cminv:   cm**-1'
         write( 6, '(a)' ) '    ev:      eV'
         write( 6, '(a)' ) '    hartree: hartree'
         write( 6, '(a)' )

         stop

      else

         write( 6, '(a)' )
         write( 6, '(a)' ) 'Program tstrate'
         write( 6, '(a)' )

      end if

c-----------------------------------------------------------------------
c     //   constants
c-----------------------------------------------------------------------

c     //   circular constant
      pi = acos(-1.d0)

c-----------------------------------------------------------------------
c     //   read values
c-----------------------------------------------------------------------

c     //   temperature
      call getarg( 1, char )
      read( char, * ) temperature

c     //   free energy at equilibrium state
      call getarg( 2, char )
      read( char, * ) fenergy_eq

c     //   free energy at transition state
      call getarg( 3, char )
      read( char, * ) fenergy_ts

c     //   scale free energy: kcalmol
      call getarg( 4, scale )

      if ( scale(1:7) .eq. 'kcalmol' ) then
         s = to_kcalmol
      else if ( scale(1:7) .eq. 'kjmol  ' ) then
         s = to_kjmol
      else if ( scale(1:7) .eq. 'cminv  ' ) then
         s = to_cminv
      else if ( scale(1:7) .eq. 'ev     ' ) then
         s = to_ev
      else if ( scale(1:7) .eq. 'hartree' ) then
         s = to_hartree
      else
         write( 6, '(a)' ) 'Error in the energy unit.'
         write( 6, '(a)' )
         stop
      end if

c-----------------------------------------------------------------------
c     //   energy conversion
c-----------------------------------------------------------------------

      dfenergy = fenergy_ts - fenergy_eq

      dfenergy = dfenergy * s

c-----------------------------------------------------------------------
c     //   calculate rate
c-----------------------------------------------------------------------

      prefactor = ( boltz * temperature ) / h

      exponent = dfenergy / ( boltz * temperature )

      rate = prefactor * exp( - exponent )

      halflife = log(2.d0) / rate

c-----------------------------------------------------------------------
c     //   print rate
c-----------------------------------------------------------------------

      write( 6, '(a,e12.4)' ) 'rate [/sec]:   ', rate
      write( 6, '(a,e12.4)' ) 'rate [/min]:   ', rate*60
      write( 6, '(a,e12.4)' ) 'rate [/hour]:  ', rate*3600
      write( 6, '(a,e12.4)' ) 'rate [/day]:   ', rate*86400
      write( 6, '(a,e12.4)' ) 'rate [/year]:  ', rate*31557600

c      write( 6, '(a,e12.4)' ) 
c      write( 6, '(a,e12.4)' ) '1/rate [sec]:  ', 1.d0/rate
c      write( 6, '(a,e12.4)' ) '1/rate [min]:  ', 1.d0/rate/60
c      write( 6, '(a,e12.4)' ) '1/rate [hour]: ', 1.d0/rate/3600
c      write( 6, '(a,e12.4)' ) '1/rate [day]:  ', 1.d0/rate/86400
c      write( 6, '(a,e12.4)' ) '1/rate [year]: ', 1.d0/rate/31557600

      write( 6, '(a)' ) 
      if ( halflife/31557600 .gt. 1.d5 ) then
         write( 6, '(a,e10.2)' ) 'half-life [year]:', halflife/31557600
      else if ( halflife/31557600 .gt. 1.d0 ) then
         write( 6, '(a,f10.2)' ) 'half-life [year]:', halflife/31557600
      else if ( halflife/86400 .gt. 1.d0 ) then
         write( 6, '(a,f10.2)' ) 'half-life [day]: ', halflife/86400
      else if ( halflife/3600 .gt. 1.d0 ) then
         write( 6, '(a,f10.2)' ) 'half-life [hour]:', halflife/3600
      else if ( halflife/60 .gt. 1.d0 ) then
         write( 6, '(a,f10.2)' ) 'half-life [min]: ', halflife/60
      else if ( halflife .gt. 1.d-4 ) then
         write( 6, '(a,f10.2)' ) 'half-life [sec]: ', halflife
      else
         write( 6, '(a,e10.2)' ) 'half-life [sec]: ', halflife
      end if
      write( 6, '(a)' ) 

      stop
      end
