c///////////////////////////////////////////////////////////////////////
c
c      Author:          M. Shiga
c      Last updated:    Feb 5, 2023 by M. Shiga
c      Description:     convert trj.out file to xyz and adjust
c
c///////////////////////////////////////////////////////////////////////
c***********************************************************************
      program trj2xyz
c***********************************************************************
c-----------------------------------------------------------------------
c     //   variables
c-----------------------------------------------------------------------

c     //   initialize
      implicit none

c     //   unit conversion factors
      real(8), parameter :: au_time   = 0.024188843d-15
      real(8), parameter :: au_length = 0.529177249d-10
      real(8), parameter :: amu_mass  = 1.6605402d-27
      real(8), parameter :: au_mass   = 9.1093897d-31

c     //   file unit
      integer :: iounit = 10

c     //   number of atoms
      integer :: natom

c     //   number of beads
      integer :: nbead

c     //   step number
      integer :: istep

c     //   atomic coordinates
      real(8), dimension(:,:), allocatable :: x, y, z

c     //   old atomic coordinates
      real(8), dimension(:,:), allocatable :: xold, yold, zold

c     //   boundary condition
      integer :: iboundary

c     //   box size
      real(8), dimension(3,3) :: box, boxinv

c     //   file names
      character(len=80) :: trjfile, strfile, inpfile, deffile, xyzfile

c     //   atomic species
      character(len=8), dimension(:), allocatable :: species

c     //   atomic mass
      real(8), dimension(:), allocatable :: physmass

c     /*   number of atomic symbols   */
      integer :: nsymbol

c     /*   atomic symbols   */
      character(len=8), dimension(:), allocatable :: symbol

c     /*   atomic masses   */
      real(8), dimension(:), allocatable :: physmass_symbol

c     //   number of kinds
      integer :: nkind

c     //   atomic kind
      integer, dimension(:), allocatable :: ikind

c     //   atoms
      integer :: iatom, jatom

c     //   integers
      integer :: ierr, i, j, k, l

c     //   real numbers
      real(8) :: xi, yi, zi, xg, yg, zg, pg, r2, r2min, d, bohr2ang

c     //   real numbers
      real(8) :: huge = 1.d+99

c     //   characters
      character(len=80) :: char

c-----------------------------------------------------------------------
c     //   usage
c-----------------------------------------------------------------------

      if ( iargc() .ne. 7 ) then

         write( 6, '(a)' ) 
         write( 6, '(a)' ) 'Program trj2xyz'
         write( 6, '(a)' ) 
         write( 6, '(a)' ) 'trj.out to xyz with adjustment'
         write( 6, '(a)' ) 
         write( 6, '(a)' ) 'Usage: trj2xyz.x $1 $2 $3 $4 $5 $6 $7'
         write( 6, '(a)' )
         write( 6, '(a)' ) '$1: trj.out (position trajectory)'
         write( 6, '(a)' ) '$2: input.dat (beads, box)'
         write( 6, '(a)' ) '$3: input_default.dat (atom symbols)'
         write( 6, '(a)' ) '$4: structure.dat (atoms, atomic kinds)'
         write( 6, '(a)' ) '$5: output file in xyz format'
         write( 6, '(a)' ) '$6: atoms in center (the first one)'
         write( 6, '(a)' ) '$7: atoms in center (the last one)'
         write( 6, '(a)' ) 
         write( 6, '(a)' )
     &      'Example: trj2xyz.x trj.out input.dat' //
     &      ' input_default.dat structure.dat out.xyz 1 1'
         write( 6, '(a)' )

         stop

      else

         write( 6, '(a)' ) 
         write( 6, '(a)' ) 'Program trj2xyz'
         write( 6, '(a)' ) 

      end if

c-----------------------------------------------------------------------
c     //   read parameters
c-----------------------------------------------------------------------

c     //   trajectory file
      call getarg( 1, trjfile )

c     //   input file
      call getarg( 2, inpfile )

c     //   output xyz file
      call getarg( 3, deffile )

c     //   structure file
      call getarg( 4, strfile )

c     //   output xyz file
      call getarg( 5, xyzfile )

c     //   atoms in center
      call getarg( 6, char )
      read( char, * ) iatom

c     //   atoms in center
      call getarg( 7, char )
      read( char, * ) jatom

c-----------------------------------------------------------------------
c     //   constants
c-----------------------------------------------------------------------

c     //   bohr to angstrom
      bohr2ang = au_length * 1.d+10

c-----------------------------------------------------------------------
c     //   box
c-----------------------------------------------------------------------

c     /*   file open   */
      open ( iounit, file = trim(inpfile) )

c     /*   search for tag    */
      call search_tag ( '<iboundary>', 11, iounit, ierr )

c     /*   number of atomic symbols   */
      read ( iounit, *, iostat=ierr ) char

c     //   periodic boundary condition
      iboundary = 1
      if ( char(1:1) .eq. '0' ) then
         iboundary = 0
      end if

c     /*   box matrix  */
      if ( iboundary .eq. 1 ) then
         read ( iounit, *, iostat=ierr ) box(1,1), box(1,2), box(1,3)
         read ( iounit, *, iostat=ierr ) box(2,1), box(2,2), box(2,3)
         read ( iounit, *, iostat=ierr ) box(3,1), box(3,2), box(3,3)
      end if

c     /*   file close   */
      close( iounit )

c     //   angstrom to bohr
      if ( ( ierr .eq. 0 ) .and. ( char(1:1) .eq. 'A' ) ) then
         box(:,:) = box(:,:) / bohr2ang
      end if

c     //   output
      if ( iboundary .eq. 1 ) then
         write( 6, '(a)' ) 'Box matrix [angstrom]:'
         write( 6, '(3f10.5)' ) box(1,1:3) * bohr2ang
         write( 6, '(3f10.5)' ) box(2,1:3) * bohr2ang
         write( 6, '(3f10.5)' ) box(3,1:3) * bohr2ang
         write( 6, '(a)' )
      end if

c     //   inverse matrix
      call inv3( box, boxinv )

c-----------------------------------------------------------------------
c     //   number of atoms
c-----------------------------------------------------------------------

c     //   open file
      open ( iounit, file = trim(strfile) )

c     //   number of atoms
      read ( iounit, *, iostat=ierr ) natom

c     //   close file
      close( iounit )

c     //   output
      write( 6, '(a,i8)' ) 'Number of atoms:', natom

c-----------------------------------------------------------------------
c     //   number of beads
c-----------------------------------------------------------------------

c     /*   file open   */
      open ( iounit, file = trim(inpfile) )

c     /*   search for tag    */
      call search_tag ( '<nbead>', 7, iounit, ierr )

c     /*   number of atomic symbols   */
      read ( iounit, *, iostat=ierr ) nbead

c     /*   file close   */
      close( iounit )

c     //   output
      write( 6, '(a,i8)' ) 'Number of beads:', nbead

c-----------------------------------------------------------------------
c     //   memory allocation
c-----------------------------------------------------------------------

c     //   atomic coordinates
      allocate( x(natom,nbead) )
      allocate( y(natom,nbead) )
      allocate( z(natom,nbead) )

c     //   atomic coordinates
      allocate( xold(natom,nbead) )
      allocate( yold(natom,nbead) )
      allocate( zold(natom,nbead) )

c     //   atomic masses
      allocate( physmass(natom) )

c     //   atomic kind
      allocate( ikind(natom) )

c-----------------------------------------------------------------------
c     /*   read atomic symbols                                        */
c-----------------------------------------------------------------------

c     //   atomic species
      allocate( species(natom) )

c     //   open file
      open ( iounit, file = trim(strfile) )

c     //   number of atoms
      read ( iounit, *, iostat=ierr )

c     //   comment line
      read ( iounit, *, iostat=ierr )

c     //   number of atomic kinds
      nkind = 1

c     //   loop of atoms
      do i = 1, natom

c        //   atomic symbol
         read( iounit, *, iostat=ierr ) species(i), d, d, d, ikind(i)

c        //   atomic symbol
         if ( ierr .ne. 0 ) then
            read( iounit, *, iostat=ierr ) species(i)
            ikind(i) = 1
         end if

c        //   number of atomic kinds
         nkind = max( ikind(i), nkind )

c     //   loop of atoms
      end do

c     //   close file
      close( iounit )

c     //   output
      if ( ierr .ne. 0 ) then
         write( 6, '(a)' )
         write( 6, '(a)' ) 'Error in atom kinds.'
         write( 6, '(a)' )
         stop
      else
         write( 6, '(a)' ) 'Read kinds from ' // trim(strfile) // '.'
         write( 6, '(a,i8)' ) 'Atom kinds:     ', nkind
      end if

c-----------------------------------------------------------------------
c     /*   read atomic masses                                         */
c-----------------------------------------------------------------------

c     //   atomic masses
      physmass(:) = 0.d0

c     /*   file open   */
      open ( iounit, file = deffile )

c     /*   search for tag    */
      call search_tag ( '<nsymbol>', 9, iounit, ierr )

c     /*   number of atomic symbols   */
      read ( iounit, *, iostat=ierr ) nsymbol

c     /*   file close   */
      close( iounit )

c     /*   memory allocation: atomic symbols   */
      allocate( symbol(nsymbol) )

c     /*   memory allocation: atomic masses   */
      allocate( physmass_symbol(nsymbol) )

c     /*   file open   */
      open ( iounit, file = 'input_default.dat' )

c     /*   search for tag    */
      call search_tag ( '<nsymbol>', 9, iounit, ierr )

c     /*   number of atomic symbols   */
      read ( iounit, *, iostat=ierr )

c     /*   loop of symbols   */
      do i = 1, nsymbol

c        /*   read symbol, atomic number, atomic mass   */
         read ( iounit, *, iostat=ierr )
     &      symbol(i), j, physmass_symbol(i)

c     /*   loop of symbols   */
      end do

c     /*   file close   */
      close( iounit )

c     //   atomic masses
      physmass(:) = 0.d0

c     /*   loop of atoms   */
      do i = 1, natom

c        //   flag
         ierr = 1

c        /*   loop of symbols   */
         do l = 1, nsymbol

c           /*   if symbol matched   */
            if ( species(i)(1:8) .eq. symbol(l)(1:8) ) then

c              /*   substitute mass   */
               physmass(i) = physmass_symbol(l)*amu_mass/au_mass

c              //   flag
               ierr = 0

c              /*   go to next loop   */
               exit

c           /*   if symbol matched   */
            end if

c        /*   loop of symbols   */
         end do

c        /*   go to next loop   */
         if ( ierr .ne. 0 ) exit

c     /*   loop of atoms   */
      end do

c     /*   memory allocation: atomic symbols   */
      deallocate( symbol )

c     /*   memory allocation: atomic masses   */
      deallocate( physmass_symbol )

c     //   output
      if ( ierr .ne. 0 ) then
         write( 6, '(a)' )
         write( 6, '(a)' ) 'Error in atomic mass.'
         write( 6, '(a)' )
         stop
      else
         write( 6, '(a)' ) 'Read masses from ' // trim(deffile) // '.'
      end if

c-----------------------------------------------------------------------
c     //   first step
c-----------------------------------------------------------------------

c     //   open file
      open ( 10, file = trjfile )

c     //   read atomic coordinates
      do j = 1, nbead
      do i = 1, natom
         read( 10, *, iostat=ierr )  k, x(i,j), y(i,j), z(i,j)
      end do
      end do

c     //   close file
      close( 10 )

c     //   output
      if ( ierr .ne. 0 ) then
         write( 6, '(a)' )
         write( 6, '(a)' ) 'Error in trajectory file.'
         write( 6, '(a)' )
         stop
      else
         write( 6, '(a)' ) 'Read ' // trim(trjfile) // '.'
      end if

c-----------------------------------------------------------------------
c     //   center of mass
c-----------------------------------------------------------------------

c     //   center of mass
      xg = 0.d0
      yg = 0.d0
      zg = 0.d0
      pg = 0.d0

      if ( ( iatom .ge. 1 ) .and. ( jatom .le. natom ) ) then

         do i = iatom, jatom
            pg = pg + physmass(i)
            do j = 1, nbead
               xg = xg + physmass(i) * x(i,j)
               yg = yg + physmass(i) * y(i,j)
               zg = zg + physmass(i) * z(i,j)
            end do
         end do

         xg = xg / pg / dble(nbead)
         yg = yg / pg / dble(nbead)
         zg = zg / pg / dble(nbead)

      end if

c     //   output
      write( 6, '(a)' ) 'Center of mass [angstrom]:'
      write( 6, '(3f10.5)' ) xg*bohr2ang, yg*bohr2ang, zg*bohr2ang

c-----------------------------------------------------------------------
c     //   nearest position from the center of mass
c-----------------------------------------------------------------------

c     //   output
      write( 6, '(a)' ) 'Fold all atoms closest to the center of mass.'
 
c     //    periodic boundary condition
      if ( iboundary .eq. 1 ) then

         do j = 1, nbead
         do i = 1, natom
            xi = x(i,j) - xg
            yi = y(i,j) - yg
            zi = z(i,j) - zg
            call pbc_fold( xi, yi, zi, box, boxinv )
            x(i,j) = xi + xg
            y(i,j) = yi + yg
            z(i,j) = zi + zg
         end do
         end do

c     //   periodic boundary condition
      end if

c-----------------------------------------------------------------------
c     //   nearest position from the center of mass
c-----------------------------------------------------------------------

      do l = 1, nbead

         do k = 2, nkind

c          //   output
           write( 6, '(a,i3,a)' ) 'Fold kind', k, '.'

            do i = 1, natom

               if ( ikind(i) .ne. k ) cycle

               r2min = huge

               do j = 1, natom

                  if ( ikind(j) .ge. k ) cycle

                  xi = x(i,l) - x(j,l)
                  yi = y(i,l) - y(j,l)
                  zi = z(i,l) - z(j,l)

                  call pbc_fold( xi, yi, zi, box, boxinv )

                  r2 = xi*xi + yi*yi + zi*zi

                  if ( r2 .lt. r2min ) then

                     x(i,l) = x(j,l) + xi
                     y(i,l) = y(j,l) + yi
                     z(i,l) = z(j,l) + zi

                     r2min = r2

                  end if

               end do

            end do

         end do

      end do

c-----------------------------------------------------------------------
c     //   save coordinates
c-----------------------------------------------------------------------

      xold(:,:) = x(:,:)
      yold(:,:) = y(:,:)
      zold(:,:) = z(:,:)

c-----------------------------------------------------------------------
c     //  open files
c-----------------------------------------------------------------------

      open ( 10, file = trjfile )
      open ( 11, file = xyzfile )

c-----------------------------------------------------------------------
c     //   start adjust
c-----------------------------------------------------------------------

c     //   step
      istep = 0

c-----------------------------------------------------------------------
c     //   loop of trajectory
c-----------------------------------------------------------------------

      do

c        //   read atomic coordinates 
         do j = 1, nbead
         do i = 1, natom
            read( 10, *, iostat=ierr )  k, x(i,j), y(i,j), z(i,j)
         end do
         end do

c-----------------------------------------------------------------------
c        //   stop on error
c-----------------------------------------------------------------------

         if ( ierr .ne. 0 ) exit

c-----------------------------------------------------------------------
c        //   step
c-----------------------------------------------------------------------

         istep = istep + 1

c-----------------------------------------------------------------------
c        //   nearest from old coordinates
c-----------------------------------------------------------------------

c        //   periodic boundary condition
         if ( iboundary .eq. 1 ) then

            do j = 1, nbead
            do i = 1, natom

               xi = x(i,j) - xold(i,j)
               yi = y(i,j) - yold(i,j)
               zi = z(i,j) - zold(i,j)

               call pbc_fold( xi, yi, zi, box, boxinv )

               x(i,j) = xi + xold(i,j)
               y(i,j) = yi + yold(i,j)
               z(i,j) = zi + zold(i,j)

            end do
            end do

c        //   periodic boundary condition
         end if

c-----------------------------------------------------------------------
c        //   nearest from old coordinates
c-----------------------------------------------------------------------

         write( 11, '(i8)' ) natom*nbead
         write( 11, '(i8)' ) istep

         do j = 1, nbead
         do i = 1, natom
            xi = x(i,j) * bohr2ang
            yi = y(i,j) * bohr2ang
            zi = z(i,j) * bohr2ang
            write( 11, '(a3,3f10.4)' )
     &         trim(species(i)), xi, yi, zi
         end do
         end do

c-----------------------------------------------------------------------
c        //   save old coordinates
c-----------------------------------------------------------------------

         xold(:,:) = x(:,:)
         yold(:,:) = y(:,:)
         zold(:,:) = z(:,:)

c-----------------------------------------------------------------------
c     //   loop of trajectory
c-----------------------------------------------------------------------

      end do

c-----------------------------------------------------------------------
c     //   close files
c-----------------------------------------------------------------------

      close( 10 )
      close( 11 )

c-----------------------------------------------------------------------
c     //   end of program
c-----------------------------------------------------------------------

      write( 6, '(a)' ) 'Normal termination of trj2xyz.'
      write( 6, '(a)' )

      stop
      end





c***********************************************************************
      subroutine pbc_fold ( xi, yi, zi, box, boxinv )
c***********************************************************************
c-----------------------------------------------------------------------
c     /*   local variables                                            */
c-----------------------------------------------------------------------

      implicit none

      real(8) :: box(3,3), boxinv(3,3)
      real(8) :: ai, bi, ci, xi, yi, zi

c-----------------------------------------------------------------------
c     /*   apply boundary condition                                   */
c-----------------------------------------------------------------------

      ai = boxinv(1,1)*xi + boxinv(1,2)*yi + boxinv(1,3)*zi
      bi = boxinv(2,1)*xi + boxinv(2,2)*yi + boxinv(2,3)*zi
      ci = boxinv(3,1)*xi + boxinv(3,2)*yi + boxinv(3,3)*zi

      ai = ai - dnint(ai)
      bi = bi - dnint(bi)
      ci = ci - dnint(ci)

      xi = box(1,1)*ai + box(1,2)*bi + box(1,3)*ci
      yi = box(2,1)*ai + box(2,2)*bi + box(2,3)*ci
      zi = box(3,1)*ai + box(3,2)*bi + box(3,3)*ci

      return
      end





c***********************************************************************
      subroutine search_tag ( char_tag, length_tag, iounit, ierr )
c***********************************************************************
c-----------------------------------------------------------------------
c     /*   local variables                                            */
c-----------------------------------------------------------------------

      implicit none

      integer:: k, length_tag, iounit, ierr

      character(len=80) :: char_line

      character(len=length_tag) :: char_tag

c-----------------------------------------------------------------------
c     /*   do loop start                                              */
c-----------------------------------------------------------------------

      do

c-----------------------------------------------------------------------
c        /*   read a line                                             */
c-----------------------------------------------------------------------

         read (iounit,*,iostat=ierr) char_line

c-----------------------------------------------------------------------
c        /*   return if error is found                                */
c-----------------------------------------------------------------------

         if ( ierr .ne. 0 ) return

c-----------------------------------------------------------------------
c        /*   search for the tag                                      */
c-----------------------------------------------------------------------

         k = index(char_line(1:length_tag),char_tag(1:length_tag))

c-----------------------------------------------------------------------
c        /*   return as soon as we find the tag                       */
c-----------------------------------------------------------------------

         if ( k .ge. 1 ) return

c-----------------------------------------------------------------------
c     /*   do loop end                                                */
c-----------------------------------------------------------------------

      end do

      return
      end





c***********************************************************************
      subroutine inv3 ( a, ainv )
c***********************************************************************
c-----------------------------------------------------------------------
c     /*   local variables                                            */
c-----------------------------------------------------------------------

      implicit none

      integer :: i, j

      real(8) :: a(3,3), ainv(3,3), det3, deta

c-----------------------------------------------------------------------
c     /*   start                                                      */
c-----------------------------------------------------------------------

      ainv(1,1) = + a(2,2)*a(3,3) - a(2,3)*a(3,2)
      ainv(1,2) = + a(3,2)*a(1,3) - a(1,2)*a(3,3)
      ainv(1,3) = + a(1,2)*a(2,3) - a(2,2)*a(1,3)

      ainv(2,1) = + a(2,3)*a(3,1) - a(3,3)*a(2,1)
      ainv(2,2) = + a(3,3)*a(1,1) - a(3,1)*a(1,3)
      ainv(2,3) = + a(1,3)*a(2,1) - a(2,3)*a(1,1)

      ainv(3,1) = + a(2,1)*a(3,2) - a(3,1)*a(2,2)
      ainv(3,2) = + a(3,1)*a(1,2) - a(1,1)*a(3,2)
      ainv(3,3) = + a(1,1)*a(2,2) - a(1,2)*a(2,1)

      deta = det3 ( a )

      do j = 1, 3
      do i = 1, 3
         ainv(i,j) = ainv(i,j)/deta
      end do
      end do

      return
      end





c***********************************************************************
      real(8) function det3 ( a )
c***********************************************************************
c-----------------------------------------------------------------------
c     /*   local variables                                            */
c-----------------------------------------------------------------------

      implicit none

      real(8) :: a(3,3)

c-----------------------------------------------------------------------
c     /*   start                                                      */
c-----------------------------------------------------------------------

      det3 = + a(1,1)*a(2,2)*a(3,3) - a(1,1)*a(2,3)*a(3,2)
     &       + a(2,1)*a(3,2)*a(1,3) - a(2,1)*a(1,2)*a(3,3)
     &       + a(3,1)*a(1,2)*a(2,3) - a(3,1)*a(2,2)*a(1,3)

      return
      end
