c///////////////////////////////////////////////////////////////////////
c
c      Author:          M. Shiga
c      Last updated:    Nov 15, 2023 by M. Shiga
c      Description:     convert trj.out file to dipole.out
c
c///////////////////////////////////////////////////////////////////////
c***********************************************************************
      program trj2dip
c***********************************************************************
c-----------------------------------------------------------------------
c     //   variables
c-----------------------------------------------------------------------

c     //   initialize
      implicit none

c     //   unit conversion factors
      real(8), parameter :: au_time   = 0.024188843d-15
      real(8), parameter :: au_length = 0.529177249d-10
      real(8), parameter :: amu_mass  = 1.6605402d-27
      real(8), parameter :: au_mass   = 9.1093897d-31

c     //   file unit
      integer :: iounit = 10

c     //   number of atoms
      integer :: natom

c     //   number of beads
      integer :: nbead

c     //   step number
      integer :: istep, kstep

c     //   atomic coordinates
      real(8), dimension(:,:), allocatable :: x, y, z

c     //   old atomic coordinates
      real(8), dimension(:,:), allocatable :: xold, yold, zold

c     //   dipole moment at first step
      real(8), dimension(:), allocatable :: dipx0, dipy0, dipz0

c     //   dipole moment
      real(8), dimension(:), allocatable :: dipx, dipy, dipz

c     //   boundary condition
      integer :: iboundary

c     //   box size
      real(8), dimension(3,3) :: box, boxinv

c     //   file names
      character(len=80) :: trjfile, strfile, inpfile, deffile, dipfile
      character(len=80) :: char

c     //   atomic charge
      real(8), dimension(:), allocatable :: q

c     //   integers
      integer :: ierr, i, j

c     //   real numbers
      real(8) :: xi, yi, zi

c     //   real numbers
      real(8) :: bohr2ang

c-----------------------------------------------------------------------
c     //   usage
c-----------------------------------------------------------------------

      if ( iargc() .ne. 5 ) then

         write( 6, '(a)' ) 
         write( 6, '(a)' ) 'Program trj2dip'
         write( 6, '(a)' ) 
         write( 6, '(a)' ) 'trj.out to dipole trajectory'
         write( 6, '(a)' ) 
         write( 6, '(a)' ) 'Usage: trj2xyz.x $1 $2 $3 $4 $5'
         write( 6, '(a)' )
         write( 6, '(a)' ) '$1: trj.out (position trajectory)'
         write( 6, '(a)' ) '$2: input.dat (beads, box)'
         write( 6, '(a)' ) '$3: input_default.dat (atom symbols)'
         write( 6, '(a)' ) '$4: structure.dat (number of atoms)'
         write( 6, '(a)' ) '$5: output file (dipole.out)'
         write( 6, '(a)' ) 
         write( 6, '(a)' )
         write( 6, '(a)' ) 'Example:'
         write( 6, '(a)' ) 'trj2dip.x trj.out input.dat ' //
     &                     'input_default.dat structure.dat dipole.out'
         write( 6, '(a)' ) 
         write( 6, '(a)' ) "Example input.dat of charges"
         write( 6, '(a)' ) 
         write( 6, '(a)' ) "<esf>"
         write( 6, '(a)' ) "    atom  charge"
         write( 6, '(a)' ) "       1 -0.8000"
         write( 6, '(a)' ) "       2  0.4000"
         write( 6, '(a)' ) "       3  0.4000"
         write( 6, '(a)' )

         stop

      else

         write( 6, '(a)' ) 
         write( 6, '(a)' ) 'Program trj2dip'
         write( 6, '(a)' ) 

      end if

c-----------------------------------------------------------------------
c     //   read parameters
c-----------------------------------------------------------------------

c     //   trajectory file
      call getarg( 1, trjfile )

c     //   input file
      call getarg( 2, inpfile )

c     //   output xyz file
      call getarg( 3, deffile )

c     //   structure file
      call getarg( 4, strfile )

c     //   output dipole file
      call getarg( 5, dipfile )

c-----------------------------------------------------------------------
c     //   constants
c-----------------------------------------------------------------------

c     //   bohr to angstrom
      bohr2ang = au_length * 1.d+10

c-----------------------------------------------------------------------
c     //   box
c-----------------------------------------------------------------------

c     /*   file open   */
      open ( iounit, file = trim(inpfile) )

c     /*   search for tag    */
      call search_tag ( '<iboundary>', 11, iounit, ierr )

c     /*   number of atomic symbols   */
      read ( iounit, *, iostat=ierr ) char

c     //   periodic boundary condition
      iboundary = 1
      if ( char(1:1) .eq. '0' ) then
         iboundary = 0
      end if

c     /*   box matrix  */
      if ( iboundary .eq. 1 ) then
         read ( iounit, *, iostat=ierr ) box(1,1), box(1,2), box(1,3)
         read ( iounit, *, iostat=ierr ) box(2,1), box(2,2), box(2,3)
         read ( iounit, *, iostat=ierr ) box(3,1), box(3,2), box(3,3)
      end if

c     /*   file close   */
      close( iounit )

c     //   angstrom to bohr
      if ( ( ierr .eq. 0 ) .and. ( char(1:1) .eq. 'A' ) ) then
         box(:,:) = box(:,:) / bohr2ang
      end if

c     //   output
      if ( iboundary .eq. 1 ) then
         write( 6, '(a)' ) 'Box matrix [angstrom]:'
         write( 6, '(3f10.5)' ) box(1,1:3) * bohr2ang
         write( 6, '(3f10.5)' ) box(2,1:3) * bohr2ang
         write( 6, '(3f10.5)' ) box(3,1:3) * bohr2ang
         write( 6, '(a)' )
      end if

c     //   inverse matrix
      call inv3( box, boxinv )

c-----------------------------------------------------------------------
c     //   number of atoms
c-----------------------------------------------------------------------

c     //   open file
      open ( iounit, file = trim(strfile) )

c     //   number of atoms
      read ( iounit, *, iostat=ierr ) natom

c     //   close file
      close( iounit )

c     //   output
      write( 6, '(a,i8)' ) 'Number of atoms:', natom

c-----------------------------------------------------------------------
c     //   number of beads
c-----------------------------------------------------------------------

c     /*   file open   */
      open ( iounit, file = trim(inpfile) )

c     /*   search for tag    */
      call search_tag ( '<nbead>', 7, iounit, ierr )

c     /*   number of atomic symbols   */
      read ( iounit, *, iostat=ierr ) nbead

c     /*   file close   */
      close( iounit )

c     //   output
      write( 6, '(a,i8)' ) 'Number of beads:', nbead

c     /*   file open   */
      open ( iounit, file = inpfile )

c-----------------------------------------------------------------------
c     //   memory allocation
c-----------------------------------------------------------------------

c     //   atomic coordinates
      allocate( x(natom,nbead) )
      allocate( y(natom,nbead) )
      allocate( z(natom,nbead) )

c     //   atomic coordinates
      allocate( xold(natom,nbead) )
      allocate( yold(natom,nbead) )
      allocate( zold(natom,nbead) )

c     //   atomic charges
      allocate( q(natom) )

c     //   dipole moment at first step
      allocate( dipx0(nbead) )
      allocate( dipy0(nbead) )
      allocate( dipz0(nbead) )

c     //   dipole moment
      allocate( dipx(nbead) )
      allocate( dipy(nbead) )
      allocate( dipz(nbead) )

c-----------------------------------------------------------------------
c     //   charges
c-----------------------------------------------------------------------

c     /*   tag   */
      call search_tag ( '<esf>', 5, iounit, ierr )

c     /*   if no error  */
      if ( ierr .eq. 0 ) then

c        /*   electrostatic field   */
         read( iounit, *, iostat=ierr ) 

c        /*   electrostatic field   */
         do i = 1, natom
            read( iounit, *, iostat=ierr ) j, q(j)
         end do

c     /*   if no error  */
      end if

c     //   output
      if ( ierr .ne. 0 ) then
         write( 6, '(a)' )
         write( 6, '(a)' ) 'Error in charges.'
         write( 6, '(a)' )
         stop
      else
         write( 6, '(a)' ) 'Read charges from ' // trim(inpfile) // '.'
      end if

c-----------------------------------------------------------------------
c     //   first step
c-----------------------------------------------------------------------

c     //   open file
      open ( 10, file = trjfile )

c     //   read atomic coordinates
      do j = 1, nbead
      do i = 1, natom
         read( 10, *, iostat=ierr )  kstep, x(i,j), y(i,j), z(i,j)
      end do
      end do

c     //   close file
      close( 10 )

c     //   output
      if ( ierr .ne. 0 ) then
         write( 6, '(a)' )
         write( 6, '(a)' ) 'Error in trajectory file.'
         write( 6, '(a)' )
         stop
      else
         write( 6, '(a)' ) 'Read ' // trim(trjfile) // '.'
      end if

c-----------------------------------------------------------------------
c     //   save coordinates
c-----------------------------------------------------------------------

      xold(:,:) = x(:,:)
      yold(:,:) = y(:,:)
      zold(:,:) = z(:,:)

c-----------------------------------------------------------------------
c     //   dipole moment at first step
c-----------------------------------------------------------------------

c     //   dipole moment
      dipx0(:) = 0.d0
      dipy0(:) = 0.d0
      dipz0(:) = 0.d0

c     //   dipole moment at first step
      do j = 1, nbead
      do i = 1, natom
         dipx0(j) = dipx0(j) + q(i) * x(i,j)
         dipy0(j) = dipy0(j) + q(i) * y(i,j)
         dipz0(j) = dipz0(j) + q(i) * z(i,j)
      end do
      end do

c     //   dipole moment
      dipx(:) = 0.d0
      dipy(:) = 0.d0
      dipz(:) = 0.d0

c-----------------------------------------------------------------------
c     //  open files
c-----------------------------------------------------------------------

      open ( 10, file = trjfile )
      open ( 11, file = dipfile )

c-----------------------------------------------------------------------
c     //   start adjust
c-----------------------------------------------------------------------

c     //   step
      istep = 0

c-----------------------------------------------------------------------
c     //   write dipole moment
c-----------------------------------------------------------------------

      do j = 1, nbead
         write( 11, '(i8,3f12.6)', iostat=ierr )
     &      kstep, dipx(j), dipy(j), dipz(j)
      end do

c-----------------------------------------------------------------------
c     //   loop of trajectory
c-----------------------------------------------------------------------

      do

c        //   read atomic coordinates 
         do j = 1, nbead
         do i = 1, natom
            read( 10, *, iostat=ierr )  kstep, x(i,j), y(i,j), z(i,j)
         end do
         end do

c-----------------------------------------------------------------------
c        //   stop on error
c-----------------------------------------------------------------------

         if ( ierr .ne. 0 ) exit

c-----------------------------------------------------------------------
c        //   step
c-----------------------------------------------------------------------

         istep = istep + 1

c-----------------------------------------------------------------------
c        //   nearest from old coordinates
c-----------------------------------------------------------------------

c        //   periodic boundary condition
         if ( iboundary .eq. 1 ) then

            do j = 1, nbead
            do i = 1, natom

               xi = x(i,j) - xold(i,j)
               yi = y(i,j) - yold(i,j)
               zi = z(i,j) - zold(i,j)

               call pbc_fold( xi, yi, zi, box, boxinv )

               x(i,j) = xi + xold(i,j)
               y(i,j) = yi + yold(i,j)
               z(i,j) = zi + zold(i,j)

            end do
            end do

c        //   periodic boundary condition
         end if

c-----------------------------------------------------------------------
c        //   nearest from old coordinates
c-----------------------------------------------------------------------

c        //   dipole moment
         dipx(:) = - dipx0(:)
         dipy(:) = - dipy0(:)
         dipz(:) = - dipz0(:)

c        //   dipole moment
         do j = 1, nbead
         do i = 1, natom
            dipx(j) = dipx(j) + q(i) * x(i,j)
            dipy(j) = dipy(j) + q(i) * y(i,j)
            dipz(j) = dipz(j) + q(i) * z(i,j)
         end do
         end do

c-----------------------------------------------------------------------
c        //   save old coordinates
c-----------------------------------------------------------------------

         xold(:,:) = x(:,:)
         yold(:,:) = y(:,:)
         zold(:,:) = z(:,:)

c-----------------------------------------------------------------------
c     //   write dipole moment
c-----------------------------------------------------------------------

         do j = 1, nbead
            write( 11, '(i8,3f12.6)', iostat=ierr )
     &         kstep, dipx(j), dipy(j), dipz(j)
         end do

c-----------------------------------------------------------------------
c     //   loop of trajectory
c-----------------------------------------------------------------------

      end do

c-----------------------------------------------------------------------
c     //   close files
c-----------------------------------------------------------------------

      close( 10 )
      close( 11 )

c-----------------------------------------------------------------------
c     //   end of program
c-----------------------------------------------------------------------

      write( 6, '(a)' ) 'Normal termination of trj2xyz.'
      write( 6, '(a)' )

      stop
      end





c***********************************************************************
      subroutine pbc_fold ( xi, yi, zi, box, boxinv )
c***********************************************************************
c-----------------------------------------------------------------------
c     /*   local variables                                            */
c-----------------------------------------------------------------------

      implicit none

      real(8) :: box(3,3), boxinv(3,3)
      real(8) :: ai, bi, ci, xi, yi, zi

c-----------------------------------------------------------------------
c     /*   apply boundary condition                                   */
c-----------------------------------------------------------------------

      ai = boxinv(1,1)*xi + boxinv(1,2)*yi + boxinv(1,3)*zi
      bi = boxinv(2,1)*xi + boxinv(2,2)*yi + boxinv(2,3)*zi
      ci = boxinv(3,1)*xi + boxinv(3,2)*yi + boxinv(3,3)*zi

      ai = ai - dnint(ai)
      bi = bi - dnint(bi)
      ci = ci - dnint(ci)

      xi = box(1,1)*ai + box(1,2)*bi + box(1,3)*ci
      yi = box(2,1)*ai + box(2,2)*bi + box(2,3)*ci
      zi = box(3,1)*ai + box(3,2)*bi + box(3,3)*ci

      return
      end





c***********************************************************************
      subroutine search_tag ( char_tag, length_tag, iounit, ierr )
c***********************************************************************
c-----------------------------------------------------------------------
c     /*   local variables                                            */
c-----------------------------------------------------------------------

      implicit none

      integer:: k, length_tag, iounit, ierr

      character(len=80) :: char_line

      character(len=length_tag) :: char_tag

c-----------------------------------------------------------------------
c     /*   do loop start                                              */
c-----------------------------------------------------------------------

      do

c-----------------------------------------------------------------------
c        /*   read a line                                             */
c-----------------------------------------------------------------------

         read (iounit,*,iostat=ierr) char_line

c-----------------------------------------------------------------------
c        /*   return if error is found                                */
c-----------------------------------------------------------------------

         if ( ierr .ne. 0 ) return

c-----------------------------------------------------------------------
c        /*   search for the tag                                      */
c-----------------------------------------------------------------------

         k = index(char_line(1:length_tag),char_tag(1:length_tag))

c-----------------------------------------------------------------------
c        /*   return as soon as we find the tag                       */
c-----------------------------------------------------------------------

         if ( k .ge. 1 ) return

c-----------------------------------------------------------------------
c     /*   do loop end                                                */
c-----------------------------------------------------------------------

      end do

      return
      end





c***********************************************************************
      subroutine inv3 ( a, ainv )
c***********************************************************************
c-----------------------------------------------------------------------
c     /*   local variables                                            */
c-----------------------------------------------------------------------

      implicit none

      integer :: i, j

      real(8) :: a(3,3), ainv(3,3), det3, deta

c-----------------------------------------------------------------------
c     /*   start                                                      */
c-----------------------------------------------------------------------

      ainv(1,1) = + a(2,2)*a(3,3) - a(2,3)*a(3,2)
      ainv(1,2) = + a(3,2)*a(1,3) - a(1,2)*a(3,3)
      ainv(1,3) = + a(1,2)*a(2,3) - a(2,2)*a(1,3)

      ainv(2,1) = + a(2,3)*a(3,1) - a(3,3)*a(2,1)
      ainv(2,2) = + a(3,3)*a(1,1) - a(3,1)*a(1,3)
      ainv(2,3) = + a(1,3)*a(2,1) - a(2,3)*a(1,1)

      ainv(3,1) = + a(2,1)*a(3,2) - a(3,1)*a(2,2)
      ainv(3,2) = + a(3,1)*a(1,2) - a(1,1)*a(3,2)
      ainv(3,3) = + a(1,1)*a(2,2) - a(1,2)*a(2,1)

      deta = det3 ( a )

      do j = 1, 3
      do i = 1, 3
         ainv(i,j) = ainv(i,j)/deta
      end do
      end do

      return
      end





c***********************************************************************
      real(8) function det3 ( a )
c***********************************************************************
c-----------------------------------------------------------------------
c     /*   local variables                                            */
c-----------------------------------------------------------------------

      implicit none

      real(8) :: a(3,3)

c-----------------------------------------------------------------------
c     /*   start                                                      */
c-----------------------------------------------------------------------

      det3 = + a(1,1)*a(2,2)*a(3,3) - a(1,1)*a(2,3)*a(3,2)
     &       + a(2,1)*a(3,2)*a(1,3) - a(2,1)*a(1,2)*a(3,3)
     &       + a(3,1)*a(1,2)*a(2,3) - a(3,1)*a(2,2)*a(1,3)

      return
      end
