c***********************************************************************
      program sqw_MPI
c***********************************************************************
c-----------------------------------------------------------------------
c     //   variables
c-----------------------------------------------------------------------

c     //   initialize
      implicit none

c     //   number of processes
      integer :: nprocs

c     //   process rank
      integer :: myrank

c     //   circular constant
      real(8) :: pi

c     //   unit conversion factors
      real(8), parameter :: au_time   = 0.024188843d-15
      real(8), parameter :: au_length = 0.529177249d-10

c     /*   speed of light in SI unit   */
      real(8), parameter :: speedlight_SI = 2.99892458d+8

c     //   number of atoms
      integer :: natom

c     //   number of beads
      integer :: nbead

c     //   number of steps
      integer :: nstep

c     //   number of atomic kinds
      integer :: nkind

c     //   file unit
      integer :: iounit = 10

c     //   trajectory file (trj.out)
      character(len=80) :: trjfile

c     //   structure file (structure.dat)
      character(len=80) :: strfile

c     //   input file (input.dat)
      character(len=80) :: inpfile

c     //   character
      character(len=80) :: char
      character(len=3)  :: char3
      character(len=8)  :: method

c     //   number of frequencies
      integer :: nomega

c     //   box matrix
      real(8), dimension(3,3) :: box

c     //   inverse of box matrix
      real(8), dimension(3,3) :: boxinv

c     //   q sampling index
      integer, dimension(3) :: lmax

c     //   q values
      real(8) :: qmin_aa, qmax_aa, qmin, qmax

c     //   Boltzmann constant
      real(8), parameter :: boltz = 0.316682968d-5

c     //   Planck constant divided by 2 pi
      real(8), parameter :: hbar = 1.d0

c     //   temperature
      real(8) :: temperature

c     //   atomic coordinates
      real(8), dimension(:,:), allocatable :: x, y, z

c     //   atomic coordinates
      real(8), dimension(:,:,:), allocatable :: xtrj, ytrj, ztrj

c     //   atomic coordinates of previous step
      real(8), dimension(:,:), allocatable :: xold, yold, zold

c     //   coherent dynamical structure factor
      real(8), dimension(:,:,:), allocatable :: sqw_coh
      real(8), dimension(:,:,:), allocatable :: sqw_coh_smear

c     //   incoherent dynamical structure factor
      real(8), dimension(:,:,:), allocatable :: sqw_inc
      real(8), dimension(:,:,:), allocatable :: sqw_inc_smear

c     //   sum rule for Sinc
      real(8), dimension(:,:), allocatable :: sum_rule

c     //   real numbers
      real(8), dimension(:,:,:), allocatable :: eiqr
      real(8), dimension(:), allocatable :: b1, b2
      real(8), dimension(:,:), allocatable :: a, b

c     //   real numbers
      real(8) :: qx, qy, qz, qr, qabs, omega, t, sinwt, coswt
      real(8) :: xi, yi, zi, d, qcf, bho, domega, factor, f2
      real(8) :: bohr2ang, factor_cminv
      real(8) :: ax, ay, az, bx, by, bz, cx, cy, cz

c     //   integers
      integer :: istep, i, j, k, l, m, n, ierr, la, lb, lc, iq, nq

c     //   integers
      integer, dimension(:), allocatable :: ikind

c     //   step size
      real(8) :: dt, dt_fs

c     //   inverse of temperature
      real(8) :: beta

c     //   q samples
      real(8), dimension(:), allocatable :: qxs
      real(8), dimension(:), allocatable :: qys
      real(8), dimension(:), allocatable :: qzs

c     //   number of samples
      integer :: nsample

c     //   integers
      integer :: iqs, is

c     //   real numbers
      real(8) :: r(1)

c     //   mpi
      include 'mpif.h'

c-----------------------------------------------------------------------
c     //   start MPI
c-----------------------------------------------------------------------

      call MPI_INIT( ierr )
      call MPI_COMM_SIZE( MPI_COMM_WORLD, nprocs, ierr )
      call MPI_COMM_RANK( MPI_COMM_WORLD, myrank, ierr )

c-----------------------------------------------------------------------
c     //   initial message
c-----------------------------------------------------------------------

c     //   comments
      if ( iargc() .ne. 11 ) then

c        //   master rank
         if ( myrank .eq. 0 ) then

            write( 6, '(a)' )
            write( 6, '(a)' ) 'Program sqw_MPI'
            write( 6, '(a)' )
            write( 6, '(a)' ) 'Usage: sqw_inc.x $1 $2 $3 $4 $5 $6 $7'
     &                        // ' $8 $9 $10 $11'
            write( 6, '(a)' )
            write( 6, '(a)' ) '$1:  trj.out (position trajectory)'
            write( 6, '(a)' ) '$2:  structure.dat (atomic kinds)'
            write( 6, '(a)' ) '$3:  input.dat (beads, box, temperature)'
            write( 6, '(a)' ) '$4:  number of steps'
            write( 6, '(a)' ) '$5:  number of samples'
            write( 6, '(a)' ) '$6:  step interval of velocity [fs]'
            write( 6, '(a)' ) '$7:  minimum q [angstrom**-1]'
            write( 6, '(a)' ) '$8:  maximum q [angstrom**-1]'
            write( 6, '(a)' ) '$9:  maximum frequency [cm**-1]'
            write( 6, '(a)' ) '$10: smearing parameter [cm**-1]'
            write( 6, '(a)' ) '$11: method (MD, CMD, RPMD, BCMD)'
            write( 6, '(a)' )
            write( 6, '(a)' ) 'Example:'
            write( 6, '(a)' )
            write( 6, '(a)' ) 'sqw_MPI.x trj.out structure.dat'
     &                        // ' input.dat 10000'
     &                        // ' 10 0.5 6.00 6.02 5000.0 15.0 BCMD'
            write( 6, '(a)' )

         end if

c        //   master rank
         call my_mpi_finalize
         stop

c     //   comments
      else

c        //   master rank
         if ( myrank .eq. 0 ) then

            write( 6, '(a)' )
            write( 6, '(a)' ) 'Program sqw_MPI'
            write( 6, '(a)' )

c        //   master rank
         end if

c     //   comments
      end if

c-----------------------------------------------------------------------
c     //   constants
c-----------------------------------------------------------------------

c     //   circular constant
      pi = acos(-1.d0)

c     //   bohr to angstrom
      bohr2ang = au_length * 1.d+10

c     //   cm**-1 to hartree
      factor_cminv = 1.d0 / (2.d0*pi*speedlight_SI*au_time*100.d0)

c-----------------------------------------------------------------------
c     //   read arguments
c-----------------------------------------------------------------------

c     //   trajectory filename
      call getarg( 1, trjfile )

c     //   structure filename
      call getarg( 2, strfile )

c     //   input filename
      call getarg( 3, inpfile )

c     //   number of steps
      call getarg( 4, char )
      read( char, *, iostat=ierr ) nstep
      call error_handling_MPI( ierr, 'Argument 4', 10 )

c     //   number of samples
      call getarg( 5, char )
      read( char, *, iostat=ierr ) nsample
      call error_handling_MPI( ierr, 'Argument 5', 10 )

c     //   step size
      call getarg( 6, char )
      read( char, *, iostat=ierr ) dt_fs
      call error_handling_MPI( ierr, 'Argument 6', 10 )

c     //   minimum q
      call getarg( 7, char )
      read( char, *, iostat=ierr ) qmin_aa
      call error_handling_MPI( ierr, 'Argument 7', 10 )

c     //   maximum q
      call getarg( 8, char )
      read( char, *, iostat=ierr ) qmax_aa
      call error_handling_MPI( ierr, 'Argument 8', 10 )

c     //   maximum frequency
      call getarg( 9, char )
      read( char, *, iostat=ierr ) omega
      nomega = nint(omega)
      call error_handling_MPI( ierr, 'Argument 9', 10 )

c     //   maximum frequency
      call getarg( 10, char )
      read( char, *, iostat=ierr ) domega
      call error_handling_MPI( ierr, 'Argument 10', 10 )

c     //   maximum frequency
      call getarg( 11, method )

c-----------------------------------------------------------------------
c     //   unit conversions
c-----------------------------------------------------------------------

c     //   step size in au
      dt = dt_fs / au_time * 1.d-15

c     //   /angstrom to /bohr
      qmin = qmin_aa * bohr2ang
      qmax = qmax_aa * bohr2ang

c-----------------------------------------------------------------------
c     //   number of atoms
c-----------------------------------------------------------------------

c     //   master rank
      if ( myrank .eq. 0 ) then

c        //   open file
         open ( iounit, file = trim(strfile) )

c        //   number of atoms
         read ( iounit, *, iostat=ierr ) natom

c        //   close file
         close( iounit )

c     //   master rank
      end if

c     //   communicate
      call my_mpi_bcast_int_0_world( ierr )

c     //   check error
      call error_handling_MPI( ierr, 'Number of atoms', 15 )

c     //   communicate
      call my_mpi_bcast_int_0_world( natom )

c-----------------------------------------------------------------------
c     //   number of beads
c-----------------------------------------------------------------------

c     //   master rank
      if ( myrank .eq. 0 ) then

c        /*   file open   */
         open ( iounit, file = trim(inpfile) )

c        /*   search for tag    */
         call search_tag ( '<nbead>', 7, iounit, ierr )

c        /*   number of atomic symbols   */
         read ( iounit, *, iostat=ierr ) nbead

c        /*   file close   */
         close( iounit )

c     //   master rank
      end if

c     //   communicate
      call my_mpi_bcast_int_0_world( ierr )

c     //   check error
      call error_handling_MPI( ierr, 'Number of beads', 15 )

c     //   communicate
      call my_mpi_bcast_int_0_world( nbead )

c-----------------------------------------------------------------------
c     //   temperature
c-----------------------------------------------------------------------

c     //   master rank
      if ( myrank .eq. 0 ) then

c        /*   file open   */
         open ( iounit, file = trim(inpfile) )

c        /*   search for tag    */
         call search_tag ( '<temperature>', 13, iounit, ierr )

c        /*   number of atomic symbols   */
         read ( iounit, *, iostat=ierr ) temperature

c        /*   file close   */
         close( iounit )

c     //   master rank
      end if

c     //   communicate
      call my_mpi_bcast_int_0_world( ierr )

c     //   check error
      call error_handling_MPI( ierr, 'Temperature', 11 )

c     //   communicate
      call my_mpi_bcast_real_0_world( temperature )

c-----------------------------------------------------------------------
c     //   inverse of temperature
c-----------------------------------------------------------------------

      beta = 1.d0 / ( boltz * temperature )

c-----------------------------------------------------------------------
c     //   memory allocation
c-----------------------------------------------------------------------

c     //   atomic coordinates
      allocate( x(natom,nbead) )
      allocate( y(natom,nbead) )
      allocate( z(natom,nbead) )

c     //   old atomic coordinates
      allocate( xold(natom,nbead) )
      allocate( yold(natom,nbead) )
      allocate( zold(natom,nbead) )

c-----------------------------------------------------------------------
c     //   box
c-----------------------------------------------------------------------

c     //   master rank
      if ( myrank .eq. 0 ) then

c        /*   file open   */
         open ( iounit, file = trim(inpfile) )

c        /*   search for tag    */
         call search_tag ( '<iboundary>', 11, iounit, ierr )

c        /*   number of atomic symbols   */
         read ( iounit, *, iostat=ierr ) char

c        /*   number of atomic symbols   */
         read ( iounit, *, iostat=ierr ) box(1,1), box(1,2), box(1,3)
         read ( iounit, *, iostat=ierr ) box(2,1), box(2,2), box(2,3)
         read ( iounit, *, iostat=ierr ) box(3,1), box(3,2), box(3,3)

c        /*   file close   */
         close( iounit )

c        //   angstrom to bohr
         if ( ( ierr .eq. 0 ) .and. ( char(1:1) .eq. 'A' ) ) then
            box(:,:) = box(:,:) / bohr2ang
         end if

c     //   master rank
      end if

c     //   communicate
      call my_mpi_bcast_int_0_world( ierr )

c     //   check error
      call error_handling_MPI( ierr, 'Box size', 8 )

c     //   communicate
      call my_mpi_bcast_real_2_world( box, 3, 3 )

c     //   inverse matrix
      call inv3( box, boxinv )

c-----------------------------------------------------------------------
c     //   atom kinds
c-----------------------------------------------------------------------

c     //   atom kinds of atom
      allocate( ikind(natom) )

c     //   number of atomic kinds
      nkind = 1

c     //   master rank
      if ( myrank .eq. 0 ) then

c        //   open file
         open ( iounit, file = trim(strfile) )

c        //   skip lines
         read ( iounit, *, iostat=ierr )
         read ( iounit, *, iostat=ierr )

c       //   loop of atoms
         do j = 1, natom

c           //   number of atoms
            read ( iounit, *, iostat=ierr ) char, d, d, d, i

c           //   default kind is 1
            if ( ierr .ne. 0 ) i = 1

c           //   number of atomic kinds
            ikind(j) = i

c           //   number of atomic kinds
            nkind = max( nkind, i )

c        //   loop of atoms
         end do

c        //   close file
         close( iounit )

c        //   error flag
         ierr = 0

c     //   master rank
      end if

c     //   communicate
      call my_mpi_bcast_int_0_world( ierr )

c     //   check error
      call error_handling_MPI( ierr, 'Atom kinds', 10 )

c     //   communicate
      call my_mpi_bcast_int_1_world( ikind, natom )
      call my_mpi_bcast_int_0_world( nkind )

c-----------------------------------------------------------------------
c     //   check samples
c-----------------------------------------------------------------------

c     //   counter
      iq = 0

      ax = 2.d0*pi*boxinv(1,1)
      ay = 2.d0*pi*boxinv(1,2)
      az = 2.d0*pi*boxinv(1,3)
      bx = 2.d0*pi*boxinv(2,1)
      by = 2.d0*pi*boxinv(2,2)
      bz = 2.d0*pi*boxinv(2,3)
      cx = 2.d0*pi*boxinv(3,1)
      cy = 2.d0*pi*boxinv(3,2)
      cz = 2.d0*pi*boxinv(3,3)

      lmax(1) = int( qmax/sqrt(ax*ax+ay*ay+az*az) ) + 1
      lmax(2) = int( qmax/sqrt(bx*bx+by*by+bz*bz) ) + 1
      lmax(3) = int( qmax/sqrt(cx*cx+cy*cy+cz*cz) ) + 1

c     //   loop of wave directions
      do la = -lmax(1), lmax(1)
      do lb = -lmax(2), lmax(2)
      do lc = -lmax(3), lmax(3)

c        //   skip for the case lx=ly=lz=0
         if ( abs(la)+abs(lb)+abs(lc) .eq. 0 ) cycle

c        //   vector
         qx = ax*la + bx*lb + cx*lc
         qy = ay*la + by*lb + cy*lc
         qz = az*la + bz*lb + cz*lc

c        //   norm of vector
         qabs = sqrt( qx*qx + qy*qy + qz*qz )

c        //   skip q out of range
         if ( qabs .lt. qmin ) cycle
         if ( qabs .gt. qmax ) cycle

c        //   counter
         iq = iq + 1

c     //   loop of wave directions
      end do
      end do
      end do

c     //   counter
      nq = iq

c-----------------------------------------------------------------------
c     //   create random samples
c-----------------------------------------------------------------------

c     //   q samples
      allocate( qxs(nsample) )
      allocate( qys(nsample) )
      allocate( qzs(nsample) )

c     //   loop of samples
      do is = 1, nsample

c        //   random number
         call random_number( r )

c        //   choose one of the nq samples
         iqs = int( r(1) * dble(nq) ) + 1

c        //   counter
         iq = 0

c        //   loop of wave directions
         do la = -lmax(1), lmax(1)
         do lb = -lmax(2), lmax(2)
         do lc = -lmax(3), lmax(3)

c           //   skip for the case lx=ly=lz=0
            if ( abs(la)+abs(lb)+abs(lc) .eq. 0 ) cycle

c           //   vector
            qx = ax*la + bx*lb + cx*lc
            qy = ay*la + by*lb + cy*lc
            qz = az*la + bz*lb + cz*lc

c           //   norm of vector
            qabs = sqrt( qx*qx + qy*qy + qz*qz )

c           //   skip q out of range
            if ( qabs .lt. qmin ) cycle
            if ( qabs .gt. qmax ) cycle

c           //   counter
            iq = iq + 1

c           //   save iqs-th data
            if ( iq .eq. iqs ) then
               qxs(is) = qx
               qys(is) = qy
               qzs(is) = qz
               go to 100
            end if

c        //   loop of wave directions
         end do
         end do
         end do

c     //   loop of samples
  100 end do

c     //   communicate
      call my_mpi_bcast_real_1_world( qxs, nsample )
      call my_mpi_bcast_real_1_world( qys, nsample )
      call my_mpi_bcast_real_1_world( qzs, nsample )

c-----------------------------------------------------------------------
c     //   print info
c-----------------------------------------------------------------------

c     //   master rank
      if ( myrank .eq. 0 ) then

         write( 6, '(a,i8)' )    'Number of atoms:       ', natom
         write( 6, '(a,i8)' )    'Number of beads:       ', nbead
         write( 6, '(a,i8)' )    'Number of atom kinds:  ', nkind
         write( 6, '(a,i8)' )    'Number of steps:       ', nstep
         write( 6, '(a,i8)' )    'Q samples exist:       ', nq
         write( 6, '(a,i8)' )    'Q samples chosen:      ', nsample
         write( 6, '(a,f8.2)' )  'Step size [au]:        ', dt
         write( 6, '(a,f8.2)' )  'Temperature [K]:       ', temperature
         write( 6, '(a)' )       'Box matrix:'
         write( 6, '(1x,3f10.4)' )  box(1,1:3)
         write( 6, '(1x,3f10.4)' )  box(2,1:3)
         write( 6, '(1x,3f10.4)' )  box(3,1:3)
         write( 6, '(a)' ) 
         write( 6, '(a)', advance='no' ) 'Reading trajectory ... '

c     //   master rank
      end if

c-----------------------------------------------------------------------
c     //   trajectory
c-----------------------------------------------------------------------

c     //   trajectory
      allocate( xtrj(natom,nbead,nstep) )
      allocate( ytrj(natom,nbead,nstep) )
      allocate( ztrj(natom,nbead,nstep) )

c     //   trajectory file open
      if ( myrank .eq. 0 ) then
         open ( iounit, file = trim(trjfile) )
      end if

c     //   loop of steps
      do istep = 1, nstep

c        //   atomic coordinates
         if ( myrank .eq. 0 ) then
            do j = 1, nbead
            do i = 1, natom
               read( iounit, *, iostat=ierr )
     &            k, x(i,j), y(i,j), z(i,j)
            end do
            end do
         end if

c        //   communicate
         call my_mpi_bcast_int_0_world( ierr )

c        //   check error
         call error_handling_MPI( ierr, 'Trajectory', 10 )

c        //   communicate
         call my_mpi_bcast_real_2_world( x, natom, nbead )
         call my_mpi_bcast_real_2_world( y, natom, nbead )
         call my_mpi_bcast_real_2_world( z, natom, nbead )

c        //   not first step
         if ( istep .ne. 1 ) then

c           //   periodic boundary condition
            do j = 1, nbead
            do i = 1, natom
               xi = x(i,j) - xold(i,j)
               yi = y(i,j) - yold(i,j)
               zi = z(i,j) - zold(i,j)
               call pbc_fold( xi, yi, zi, box, boxinv )
               x(i,j) = xi + xold(i,j)
               y(i,j) = yi + yold(i,j)
               z(i,j) = zi + zold(i,j)
            end do
            end do

c        //   not first step
         end if

c        //   save coordinates
         xold(:,:) = x(:,:)
         yold(:,:) = y(:,:)
         zold(:,:) = z(:,:)

c        //   save coordinates
         xtrj(:,:,istep) = x(:,:)
         ytrj(:,:,istep) = y(:,:)
         ztrj(:,:,istep) = z(:,:)

c     //   loop of steps
      end do

c     //   trajectory file close
      if ( myrank .eq. 0 ) then
         close( iounit )
      end if

c-----------------------------------------------------------------------
c     //   print to screen
c-----------------------------------------------------------------------

c     //   master rank
      if ( myrank .eq. 0 ) then

         write( 6, '(a)' ) 'complete.'
         write( 6, '(a)' ) 

c        //   print wavelengths
         write( 6, '(a40)', advance="no" )
     &      'File         |Q|      Qx      Qy      Qz'

c        //   print to screen
         do k = 1, nkind
            write( 6, '(a8,i2)', advance='no' ) ' SumRule', k
         end do

c        //   print to screen
         write( 6, * )

c     //   master rank
      end if

c-----------------------------------------------------------------------
c     //   memory allocation
c-----------------------------------------------------------------------

c     //   function
      allocate( eiqr(2,natom,nstep) )
      allocate( b1(natom) )
      allocate( b2(natom) )

c     //   coherent dynamical structure factor
      allocate( sqw_coh(nkind,nkind,nomega) )
      allocate( sqw_coh_smear(nkind,nkind,nomega) )

c     //   coherent dynamical structure factor
      allocate( sqw_inc(nkind,nkind,nomega) )
      allocate( sqw_inc_smear(nkind,nkind,nomega) )

c     //   sum rule
      allocate( sum_rule(nkind,nkind) )

c-----------------------------------------------------------------------
c     //   main loop
c-----------------------------------------------------------------------

c     //   initialize
      sqw_coh(:,:,:) = 0.d0
      sqw_inc(:,:,:) = 0.d0

c-----------------------------------------------------------------------

c     //   loop of samples
      do is = 1, nsample

c        //   vector
         qx = qxs(is)
         qy = qys(is)
         qz = qzs(is)

c        //   norm of vector
         qabs = qx*qx + qy*qy + qz*qz

c        //   skip q out of range
         if ( qabs .lt. (qmin*qmin) ) cycle
         if ( qabs .gt. (qmax*qmax) ) cycle

c        //   task of myrank only
         if ( mod(is-1,nprocs) .ne. myrank ) cycle

c        //   norm of vector
         qabs = sqrt( qabs )

c-----------------------------------------------------------------------

c        //   loop of steps
         do istep = 1, nstep

c           //   copy coordinates
            x(:,:) = xtrj(:,:,istep)
            y(:,:) = ytrj(:,:,istep)
            z(:,:) = ztrj(:,:,istep)

c           //   function
            do i = 1, natom
               eiqr(1,i,istep) = 0.d0
               eiqr(2,i,istep) = 0.d0
            end do

c           //   loop of beads and atoms
            do j = 1, nbead
            do i = 1, natom

c              //   q*r
               qr = qx*x(i,j) + qy*y(i,j) + qz*z(i,j)

c              //   function exp(iqr)
               eiqr(1,i,istep) = eiqr(1,i,istep) + cos(qr)
               eiqr(2,i,istep) = eiqr(2,i,istep) + sin(qr)

c           //   loop of beads and atoms
            end do
            end do

c           //   normalized by beads
            do i = 1, natom
               eiqr(1,i,istep) = eiqr(1,i,istep) / dble(nbead)
               eiqr(2,i,istep) = eiqr(2,i,istep) / dble(nbead)
            end do

c        //   loop of steps
         end do

c-----------------------------------------------------------------------

c        //   loop of frequencies
         do m = 1, nomega

c           //   w [cm**-1] to w [au]
            omega = dble(m) / factor_cminv

c           //   loop of atoms
            do i = 1, natom

c              //   initialize
               b1(i) = 0.d0
               b2(i) = 0.d0

c              //   loop of steps
               do istep = 1, nstep

c                 //   time in au
                  t = dble(istep) * dt

c                 //   cos(wt) and sin(wt)
                  coswt = cos(omega*t)
                  sinwt = sin(omega*t)

c                 //   real part of exp(iqr)*exp(iwt)
                  b1(i) = b1(i) + eiqr(1,i,istep) * coswt
     &                          + eiqr(2,i,istep) * sinwt

c                 //   imaginary part of exp(iqr)*exp(iwt)
                  b2(i) = b2(i) - eiqr(1,i,istep) * sinwt
     &                          + eiqr(2,i,istep) * coswt

c              //   loop of steps
               end do

c           //   loop of atoms
            end do

c           //   loop of atoms
            do i = 1, natom

c              //   atomic kind
               k = ikind(i)

c              //   loop of atoms
               do j = 1, natom

c                 //   atomic kind
                  l = ikind(j)

c                 //   s(q,w) = sum_ij exp(iqr)*exp(i*w*t)
                  sqw_coh(k,l,m) = sqw_coh(k,l,m)
     &               + ( b1(i)*b1(j) + b2(i)*b2(j) )

c              //   loop of atoms
               end do

c              //   s(q,w) = sum_i exp(iqr)*exp(i*w*t)
               sqw_inc(k,k,m) = sqw_inc(k,k,m)
     &            + ( b1(i)*b1(i) + b2(i)*b2(i) )

c           //   loop of atoms
            end do

c        //   loop of frequencies
         end do

c-----------------------------------------------------------------------

c        //   constant = dt**2/(nstep*dt) / (2.d0*pi)
         factor = dt / dble(nstep) / (2.d0*pi)

c        //   initialize
         sum_rule(:,:) = 0.d0

c        //   loop of frequencies
         do m = 1, nomega

c           //   S_kubo: normalized by q sampling points
            do k = 1, nkind
            do l = 1, nkind
               sqw_inc(k,l,m) = sqw_inc(k,l,m) * factor
               sqw_coh(k,l,m) = sqw_coh(k,l,m) * factor
            end do
            end do

c           //   w [cm**-1] to w [au]
            omega = dble(m) / factor_cminv

c           //   beta hbar omega
            bho = beta * hbar * omega

c           //   for path integrals
            if ( method(1:3) .ne. 'MD ' ) then

c              //   quantum correction factor
               qcf = bho / ( 1.d0 - exp(-bho) )

c              //   quantum correction
               do k = 1, nkind
               do l = 1, nkind
                  sqw_inc(k,l,m) = sqw_inc(k,l,m) * qcf
                  sqw_coh(k,l,m) = sqw_coh(k,l,m) * qcf
               end do
               end do

c           //   for path integrals
            end if

c           //   constant
            f2 = (1.d0+exp(-bho)) / factor_cminv

c           //   Sum rule: \int_-infty^\infty S(q,w) dw
            do k = 1, nkind
               sum_rule(k,k) = sum_rule(k,k) + sqw_inc(k,k,m) * f2
            end do

c        //   loop of frequencies
         end do

c-----------------------------------------------------------------------

c        //   initialize
         sqw_coh_smear(:,:,:) = 0.d0
         sqw_inc_smear(:,:,:) = 0.d0

c        //   smearing factor
         factor = 1.d0 / sqrt( pi * ( domega * domega ) / 0.5d0 )

c        //   loop of frequencies
         do n = 1, nomega

c           //   loop of frequencies
            do m = 1, nomega

c              //   frequency difference
               d = dble(m-n)

c              //   exponent
               d = 0.5d0 * d * d / ( domega * domega )

c              //   loop of kinds
               do k = 1, nkind
               do l = 1, nkind

c                 //   loop of kinds
                  sqw_coh_smear(k,l,n) = sqw_coh_smear(k,l,n)
     &               + sqw_coh(k,l,m) * factor * exp( - d )

c                 //   loop of kinds
                  sqw_inc_smear(k,l,n) = sqw_inc_smear(k,l,n)
     &               + sqw_inc(k,l,m) * factor * exp( - d )

c              //   loop of kinds
               end do
               end do

c           //   loop of frequencies
            end do

c        //   loop of frequencies
         end do

c-----------------------------------------------------------------------

c        //   number to character
         write( char3, '(i3.3)' ) is

c        //   file name
         char = 'sqw.' // char3

c        //   print to screen
         write( 6, '(a,4f8.4)', advance='no' ) trim(char) // ':',
     &      qabs/bohr2ang, qx/bohr2ang, qy/bohr2ang, qz/bohr2ang

c        //   print to screen
         do k = 1, nkind
            write( 6, '(f10.2)', advance='no' ) sum_rule(k,k)
         end do

c        //   print to screen
         write( 6, * )

c-----------------------------------------------------------------------

c        //   output file open
         open( iounit, file=trim(char) )

c        //   print
         write( iounit, '(a40)', advance='no' )
     &      '----------------------------------------'

c        //   print
         do i = 1, nkind+nkind*(nkind+1)/2
            write( iounit, '(a12)', advance='no' )
     &         '------------'
         end do

c        //   print
         write( iounit, '(a)' )

c        //   print wavelengths
         write( iounit, '(a40)', advance='no' )
     &      '     |Q|      Qx      Qy      Qz       W'

c        //   print incoherent component
         do k = 1, nkind
            write( iounit, '(a10,i2)', advance='no' )
     &         '      Sinc', k
         end do

c        //   print coherent component
         do k = 1, nkind
         do l = k, nkind
            write( iounit, '(a8,2i2)', advance='no' )
     &         '    Scoh', k, l
         end do
         end do

c        //   advance to next line
         write( iounit, * )

c        //   units
         write( iounit, '(a40)', advance='no' )
     &      '   A**-1   A**-1   A**-1   A**-1  cm**-1'

c        //   advance to next line
         write( iounit, * )

c        //   print
         write( iounit, '(a40)', advance='no' )
     &      '----------------------------------------'

c        //   print
         do i = 1, nkind+nkind*(nkind+1)/2
            write( iounit, '(a12)', advance='no' )
     &         '------------'
         end do

c        //   print
         write( iounit, '(a)' )

c        //   loop of frequencies
         do m = 1, nomega

c           //   print wavelengths
            write( iounit, '(4f8.4,f8.1)', advance='no' )
     &         qabs/bohr2ang, qx/bohr2ang, qy/bohr2ang, qz/bohr2ang,
     &         dble(m)

c           //   print incoherent component
            do k = 1, nkind
               write( iounit, '(e12.4)', advance='no' )
     &            sqw_inc_smear(k,k,m)
            end do

c           //   print coherent component
            do k = 1, nkind
            do l = k, nkind
               write( iounit, '(e12.4)', advance='no' )
     &            sqw_coh_smear(k,l,m)
            end do
            end do

c           //   advance to next line
            write( iounit, * )

c        //   loop of frequencies
         end do

c        //   output file close
         close( iounit )

c     //   loop of samples
      end do

c-----------------------------------------------------------------------
c     //   wait until all the processes ends
c-----------------------------------------------------------------------

      call my_mpi_barrier_world

c-----------------------------------------------------------------------
c     //   print to screen
c-----------------------------------------------------------------------

c     //   master rank
      if ( myrank .eq. 0 ) then

         write( 6, '(a)' ) 
         write( 6, '(a)' ) 'Complete.'
         write( 6, '(a)' ) 

c     //   master rank
      end if

c-----------------------------------------------------------------------
c     //   isotropic values
c-----------------------------------------------------------------------

c     //   master rank
      if ( myrank .eq. 0 ) then

c        //   dimension
         n = 5 + nkind + nkind*(nkind+1)/2

c        //   memory allocation
         allocate( a(n,nomega) )
         allocate( b(n,nomega) )

c        //   average
         b(:,:) = 0.d0

c        //   loop of q
         do is = 1, nsample

c           //   number to character
            write( char3, '(i3.3)' ) is

c           //   file name
            char = 'sqw.' // char3

c           //   open file
            open ( iounit, file = trim(char) )

c           //   read data
            read( iounit, * )
            read( iounit, * )
            read( iounit, * )
            read( iounit, * )

c           //   read data
            do m = 1, nomega
               read( iounit, * ) a(1:n,m)
            end do

c           //   close file
            close( iounit )

c           //   average
            b(:,:) = b(:,:) + a(:,:) / dble(nsample)

c        //   loop of q
         end do

c        //   sampling must be greater than 0
         if ( nsample .ge. 1 ) then

c           //   output file open
            open( iounit, file='sqw_iso.out' )

c           //   print
            write( iounit, '(a16)', advance='no' )
     &      '----------------'

c           //   print
            do i = 1, nkind+nkind*(nkind+1)/2
               write( iounit, '(a12)', advance='no' )
     &            '------------'
            end do

c           //   print
            write( iounit, '(a)' )

c           //   print wavelengths
            write( iounit, '(a16)', advance='no' )
     &         '     |Q|       W'

c           //   print incoherent component
            do k = 1, nkind
               write( iounit, '(a10,i2)', advance='no' )
     &            '      Sinc', k
            end do

c           //   print coherent component
            do k = 1, nkind
            do l = k, nkind
               write( iounit, '(a8,2i2)', advance='no' )
     &            '    Scoh', k, l
            end do
            end do

c           //   advance to next line
            write( iounit, * )

c           //   units
            write( iounit, '(a16)', advance='no' )
     &         '   A**-1  cm**-1'

c           //   advance to next line
            write( iounit, * )

c           //   print
            write( iounit, '(a16)', advance='no' )
     &      '----------------'

c           //   print
            do i = 1, nkind+nkind*(nkind+1)/2
               write( iounit, '(a12)', advance='no' )
     &            '------------'
            end do

c           //   print
            write( iounit, '(a)' )

c           //   loop of frequencies
            do m = 1, nomega

c              //   print data
               write( iounit, '(f8.4,f8.1)', advance='no' )
     &            b(1,m), b(5,m)

c              //   print data
               do l = 6, n
                  write( iounit, '(e12.4)', advance='no' ) b(l,m)
               end do

c              //   advance to next line
               write( iounit, * )

c           //   loop of frequencies
            end do

c           //   output file close
            close( iounit )

c        //   sampling must be greater than 0
         end if

c     //   master rank
      end if

c-----------------------------------------------------------------------
c     //   print to screen
c-----------------------------------------------------------------------

c     //   master rank
      if ( myrank .eq. 0 ) then

         write( 6, '(a)' ) 'Isotropic S(Q,W) printed in sqw_iso.out.'
         write( 6, '(a)' ) 
         write( 6, '(a)' ) 'Normal termination of sqw_MPI.'
         write( 6, '(a)' ) 

c     //   master rank
      end if

c-----------------------------------------------------------------------
c     //   end of code
c-----------------------------------------------------------------------

      call my_mpi_finalize

      stop
      end





c***********************************************************************
      subroutine pbc_fold ( xi, yi, zi, box, boxinv )
c***********************************************************************
c-----------------------------------------------------------------------
c     /*   local variables                                            */
c-----------------------------------------------------------------------

      implicit none

      real(8) :: box(3,3), boxinv(3,3)
      real(8) :: ai, bi, ci, xi, yi, zi

c-----------------------------------------------------------------------
c     /*   apply boundary condition                                   */
c-----------------------------------------------------------------------

      ai = boxinv(1,1)*xi + boxinv(1,2)*yi + boxinv(1,3)*zi
      bi = boxinv(2,1)*xi + boxinv(2,2)*yi + boxinv(2,3)*zi
      ci = boxinv(3,1)*xi + boxinv(3,2)*yi + boxinv(3,3)*zi

      ai = ai - dnint(ai)
      bi = bi - dnint(bi)
      ci = ci - dnint(ci)

      xi = box(1,1)*ai + box(1,2)*bi + box(1,3)*ci
      yi = box(2,1)*ai + box(2,2)*bi + box(2,3)*ci
      zi = box(3,1)*ai + box(3,2)*bi + box(3,3)*ci

      return
      end





c***********************************************************************
      subroutine search_tag ( char_tag, length_tag, iounit, ierr )
c***********************************************************************
c-----------------------------------------------------------------------
c     /*   local variables                                            */
c-----------------------------------------------------------------------

      implicit none

      integer:: k, length_tag, iounit, ierr

      character(len=80) :: char_line

      character(len=length_tag) :: char_tag

c-----------------------------------------------------------------------
c     /*   do loop start                                              */
c-----------------------------------------------------------------------

      do

c-----------------------------------------------------------------------
c        /*   read a line                                             */
c-----------------------------------------------------------------------

         read (iounit,*,iostat=ierr) char_line

c-----------------------------------------------------------------------
c        /*   return if error is found                                */
c-----------------------------------------------------------------------

         if ( ierr .ne. 0 ) return

c-----------------------------------------------------------------------
c        /*   search for the tag                                      */
c-----------------------------------------------------------------------

         k = index(char_line(1:length_tag),char_tag(1:length_tag))

c-----------------------------------------------------------------------
c        /*   return as soon as we find the tag                       */
c-----------------------------------------------------------------------

         if ( k .ge. 1 ) return

c-----------------------------------------------------------------------
c     /*   do loop end                                                */
c-----------------------------------------------------------------------

      end do

      return
      end





c***********************************************************************
      subroutine inv3 ( a, ainv )
c***********************************************************************
c-----------------------------------------------------------------------
c     /*   local variables                                            */
c-----------------------------------------------------------------------

      implicit none

      integer :: i, j

      real(8) :: a(3,3), ainv(3,3), det3, deta

c-----------------------------------------------------------------------
c     /*   start                                                      */
c-----------------------------------------------------------------------

      ainv(1,1) = + a(2,2)*a(3,3) - a(2,3)*a(3,2)
      ainv(1,2) = + a(3,2)*a(1,3) - a(1,2)*a(3,3)
      ainv(1,3) = + a(1,2)*a(2,3) - a(2,2)*a(1,3)

      ainv(2,1) = + a(2,3)*a(3,1) - a(3,3)*a(2,1)
      ainv(2,2) = + a(3,3)*a(1,1) - a(3,1)*a(1,3)
      ainv(2,3) = + a(1,3)*a(2,1) - a(2,3)*a(1,1)

      ainv(3,1) = + a(2,1)*a(3,2) - a(3,1)*a(2,2)
      ainv(3,2) = + a(3,1)*a(1,2) - a(1,1)*a(3,2)
      ainv(3,3) = + a(1,1)*a(2,2) - a(1,2)*a(2,1)

      deta = det3 ( a )

      do j = 1, 3
      do i = 1, 3
         ainv(i,j) = ainv(i,j)/deta
      end do
      end do

      return
      end





c***********************************************************************
      real(8) function det3 ( a )
c***********************************************************************
c-----------------------------------------------------------------------
c     /*   local variables                                            */
c-----------------------------------------------------------------------

      implicit none

      real(8) :: a(3,3)

c-----------------------------------------------------------------------
c     /*   start                                                      */
c-----------------------------------------------------------------------

      det3 = + a(1,1)*a(2,2)*a(3,3) - a(1,1)*a(2,3)*a(3,2)
     &       + a(2,1)*a(3,2)*a(1,3) - a(2,1)*a(1,2)*a(3,3)
     &       + a(3,1)*a(1,2)*a(2,3) - a(3,1)*a(2,2)*a(1,3)

      return
      end





c***********************************************************************
      subroutine error_handling_MPI( ierr, char_tag, length_tag )
c***********************************************************************
c-----------------------------------------------------------------------
c     /*   local variables                                            */
c-----------------------------------------------------------------------

      implicit none

      integer:: ierr, length_tag

      character(len=length_tag) :: char_tag

c-----------------------------------------------------------------------
c     /*   start                                                      */
c-----------------------------------------------------------------------

      if ( ierr .ne. 0 ) then

         write( 6, '(a)' ) 'Error termination at: ' // char_tag // '.'

         call my_mpi_finalize

         stop

      end if

      return
      end





c***********************************************************************
      subroutine my_mpi_finalize
c***********************************************************************

      implicit none

      integer :: ierr

      include 'mpif.h'

c     /*   finalize MPI prallelization   */
      call MPI_FINALIZE ( ierr )

      return
      end





c***********************************************************************
      subroutine my_mpi_bcast_int_0_world ( i )
c***********************************************************************

      implicit none

      integer               ::  i, ierr
      integer, dimension(1) ::  j

      include 'mpif.h'

      j(1) = i

      call MPI_BCAST ( j, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr )

      i = j(1)

      return
      end





c***********************************************************************
      subroutine my_mpi_bcast_int_1_world ( i, n )
c***********************************************************************

      implicit none

      integer :: ierr, n
      integer, dimension(n) :: i

      include 'mpif.h'

      call MPI_BCAST ( i, n, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr )

      return
      end





c***********************************************************************
      subroutine my_mpi_bcast_real_0_world ( a )
c***********************************************************************

      implicit none

      integer :: ierr
      real(8) :: a, b(1)

      include 'mpif.h'

      b(1) = a

      call MPI_BCAST ( b, 1, MPI_DOUBLE_PRECISION,
     &                 0, MPI_COMM_WORLD, ierr )

      a = b(1)

      return
      end





c***********************************************************************
      subroutine my_mpi_bcast_real_1_world ( a, n )
c***********************************************************************

      implicit none

      integer :: i, n, ierr
      real(8) :: a(n), b(n)

      include 'mpif.h'

      do i = 1, n
         b(i) = a(i)
      end do

      call MPI_BCAST ( b, n, MPI_DOUBLE_PRECISION,
     &                 0, MPI_COMM_WORLD, ierr )

      do i = 1, n
         a(i) = b(i)
      end do

      return
      end





c***********************************************************************
      subroutine my_mpi_bcast_real_2_world ( a, n1, n2 )
c***********************************************************************

      implicit none

      integer :: i, j, k, n, n1, n2, ierr
      real(8) :: a(n1,n2), b(n1*n2)

      include 'mpif.h'

      k = 0
      do j = 1, n2
      do i = 1, n1
         k = k + 1
         b(k) = a(i,j)
      end do
      end do

      n = n1*n2

      call MPI_BCAST ( b, n, MPI_DOUBLE_PRECISION,
     &                 0, MPI_COMM_WORLD, ierr )

      k = 0
      do j = 1, n2
      do i = 1, n1
         k = k + 1
         a(i,j) = b(k)
      end do
      end do

      return
      end





c***********************************************************************
      subroutine my_mpi_allreduce_real_3_world( a, n1, n2, n3 )
c***********************************************************************
c=======================================================================
c
c     all-reduce communication of three-dimensional real numbers
c
c=======================================================================

      implicit none

      integer :: i, j, k, l, n, n1, n2, n3, ierr
      real(8) :: a(n1,n2,n3), b1(n1*n2*n3), b2(n1*n2*n3)

      include 'mpif.h'

      l = 0
      do k = 1, n3
      do j = 1, n2
      do i = 1, n1
         l = l + 1
         b1(l) = a(i,j,k)
      end do
      end do
      end do

      n = n1*n2*n3

      call MPI_ALLREDUCE ( b1, b2, n, MPI_DOUBLE_PRECISION,
     &                     MPI_SUM, MPI_COMM_WORLD, ierr )

      l = 0
      do k = 1, n3
      do j = 1, n2
      do i = 1, n1
         l = l + 1
         a(i,j,k) = b2(l)
      end do
      end do
      end do

      return
      end





c***********************************************************************
      subroutine my_mpi_barrier_world
c***********************************************************************

      implicit none

      integer :: ierr

      include 'mpif.h'

      call MPI_BARRIER ( MPI_COMM_WORLD, ierr )

      return
      end
