c***********************************************************************
      program refine2d
c***********************************************************************
c-----------------------------------------------------------------------
c     //   variables
c-----------------------------------------------------------------------

c     //   initialize variables
      implicit none

c     //   file unit
      integer :: iounit = 10

c     //   number of steps of cv output
      integer :: nstep

c     //   cv position
      real(8), dimension(:,:), allocatable :: cv

c     //   cv density
      real(8), dimension(:,:), allocatable :: dens

c     //   mesh parameters
      real(8) :: xmin, ymin, xmax, ymax, dx, dy

c     //   density meshes
      integer :: nx, ny

c     //   Boltzmann constant
      real(8) :: boltz = 0.316682968d-5

c     //   unit conversion factors
ccc      real(8) :: bohr2ang = 0.52918d0
      real(8) :: har2kcal = 627.51d0

c     //   inverse of temperature
      real(8) :: beta

c     //   density
      real(8), dimension(:,:), allocatable :: g

c     //   maximum density
      real(8) :: densmax

c     //   density cutoff
      real(8) :: denscut

c     //   matrices
      real(8), dimension(:,:), allocatable :: u
      real(8), dimension(:,:), allocatable :: c

c     //   vector
      real(8), dimension(:), allocatable :: b

c     //   vector: gaussian height
      real(8), dimension(:), allocatable :: h

c     //   gaussian widths
      real(8) :: sigx, sigy

c     //   temperature
      real(8) :: temperature

c     //   cutoff parameter
      real(8) :: cutoff

c     //   number of arguments
      integer :: iargc

c     //   file names
      character(len=120) :: cvfile
      character(len=120) :: hillsfile

c     //   integers
      integer :: i, j, k, l, ij, kl, ierr, istep

c     //   real numbers
      real(8) :: dum, xij, yij, xkl, ykl, rx, ry, h1, h2

c     //   characters
      character(len=80) :: char

c     //   tiny value
      real(8) :: tiny = 1.d-24

c     //   dimension
      integer :: nxy

c     //   factor
      real(8) :: fact = 0.67d0

c     //   new reconstructed hills
      real(8), dimension(:,:), allocatable :: p

c     //   old reconstructed hills
      real(8), dimension(:,:), allocatable :: q

c     //   number of hills
      integer :: nhills_old, nhills_new

c-----------------------------------------------------------------------
c     //   comments
c-----------------------------------------------------------------------

c     //   comments
      if ( iargc() .ne. 10 ) then

         write( 6, '(a)' )
         write( 6, '(a)' ) 'Program refine2d'
         write( 6, '(a)' )
         write( 6, '(a)' )
     &      'Usage: refine2d.x $1 $2 $3 $4 $5 $6 $7 $8 $9 $10'
         write( 6, '(a)' )
         write( 6, '(a)' ) '$1:  cv.out (cv trajectory)'
         write( 6, '(a)' ) '$2:  hills.ini (hills)'
         write( 6, '(a)' ) '$3:  temperature [K]'
         write( 6, '(a)' ) '$4:  cutoff parameter'
         write( 6, '(a)' ) '$5:  mesh parameter: xmin'
         write( 6, '(a)' ) '$6:  mesh parameter: xmax'
         write( 6, '(a)' ) '$7:  mesh parameter: dx'
         write( 6, '(a)' ) '$8:  mesh parameter: ymin'
         write( 6, '(a)' ) '$9:  mesh parameter: ymax'
         write( 6, '(a)' ) '$10: mesh parameter: dy'
         write( 6, '(a)' )
         write( 6, '(a)' ) 'Example:'
         write( 6, '(a)' )
         write( 6, '(a)' )
     &     './refine2d.x cv.out hills.ini ' //
     &     '463.0 0.05 0.0 1.8 0.03 2.0 8.0 0.1'
         write( 6, '(a)' )

         stop

c     //   comments
      else

         write( 6, '(a)' )
         write( 6, '(a)' ) 'Program refine2d'
         write( 6, '(a)' )

c     //   comments
      end if

c-----------------------------------------------------------------------
c     //   read argument
c-----------------------------------------------------------------------

c     //   trajectory filename
      call getarg( 1, cvfile )

c     //   trajectory filename
      call getarg( 2, hillsfile )

c     //   temperature
      call getarg( 3, char )
      read( char, *, iostat=ierr ) temperature

c     //   cutoff
      call getarg( 4, char )
      read( char, *, iostat=ierr ) cutoff

c     //   xmin
      call getarg( 5, char )
      read( char, *, iostat=ierr ) xmin

c     //   xmax
      call getarg( 6, char )
      read( char, *, iostat=ierr ) xmax

c     //   dx
      call getarg( 7, char )
      read( char, *, iostat=ierr ) dx

c     //   ymin
      call getarg( 8, char )
      read( char, *, iostat=ierr ) ymin

c     //   ymax
      call getarg( 9, char )
      read( char, *, iostat=ierr ) ymax

c     //   dy
      call getarg( 10, char )
      read( char, *, iostat=ierr ) dy

c-----------------------------------------------------------------------
c     //   constants
c-----------------------------------------------------------------------

c     //   inverse of temperature
      beta = 1.d0 / ( boltz * temperature )

c-----------------------------------------------------------------------
c     //   mesh parameters
c-----------------------------------------------------------------------

c     //   number of meshes
      nx   = nint( ( xmax - xmin ) / dx ) + 1
      ny   = nint( ( ymax - ymin ) / dy ) + 1

c     //   print to screen
      write( 6, '(a,i8)' ) "Number of x meshes:", nx
      write( 6, '(a,i8)' ) "Number of y meshes:", ny
      write( 6, '(a,i8)' ) "Total meshes:      ", nx*ny
      write( 6, '(a)' )

c     //   error termination
      if ( ( nx .le. 0 ) .or. ( ny .le. 0 ) ) stop

c-----------------------------------------------------------------------
c     //   read number of steps
c-----------------------------------------------------------------------

c     //   open file
      open ( iounit, file = trim(cvfile) )

c     //   skip three lines
      read( iounit, *, iostat=ierr )
      read( iounit, *, iostat=ierr )
      read( iounit, *, iostat=ierr )

c     //   step counter
      istep = 0

c     //   loop of lines
      do

c        //   read two lines
         do i = 1, 2
            read( iounit, *, iostat=ierr ) j, dum, dum
         end do

c        //   detect read error
         if ( ierr .ne. 0 ) exit

c        //   step counter
         istep = istep + 1

c     //   loop of lines
      end do

c     //   final number of steps
      nstep = istep

c     //   close file
      close( iounit )

c     //   print to screen
      write( 6, '(a,i8)' ) "Number of steps:   ", nstep
      write( 6, '(a)' )

c     //   error termination
      if ( nstep .eq. 0 ) stop

c-----------------------------------------------------------------------
c     //   read cv
c-----------------------------------------------------------------------

c     //   memory allocation
      allocate( cv(2,nstep) )

c     //   open file
      open ( iounit, file = trim(cvfile) )

c     //   skip three lines
      read( iounit, *, iostat=ierr )
      read( iounit, *, iostat=ierr )
      read( iounit, *, iostat=ierr )

c     //   loop of lines
      do istep = 1, nstep

c        //   read two lines
         do i = 1, 2
            read( iounit, *, iostat=ierr ) j, cv(i,istep), dum
         end do

c     //   loop of lines
      end do

c     //   close file
      close( iounit )

c-----------------------------------------------------------------------
c     //   density
c-----------------------------------------------------------------------

c     //   memory allocation
      allocate( dens(nx,ny) )

c     //   initialize
      dens(:,:) = 0.d0

c     //   loop of lines
      do istep = 1, nstep

c        //   mesh position
         i = nint( ( cv(1,istep) - xmin ) / dx ) + 1
         j = nint( ( cv(2,istep) - ymin ) / dy ) + 1

c        //   must be in the mesh range
         if ( i .gt. nx ) cycle
         if ( j .gt. ny ) cycle

c        //   update density
         dens(i,j) = dens(i,j) + 1.d0 / (dx*dy) / dble(nstep)

c     //   loop of lines
      end do

c     //   maximum density: initialize
      densmax = 0.d0

c     //   maximum density
      do j = 1, ny
      do i = 1, nx
         densmax = max( dens(i,j), densmax )
      end do
      end do

c-----------------------------------------------------------------------
c     //   free energy correction:  g is negative
c-----------------------------------------------------------------------

c     //   memory allocation
      allocate( g(nx,ny) )

c     //   free energy correction: initialize
      g(:,:) = 0.d0

c     //   density cutoff
      denscut = densmax * cutoff

c     //   free energy correction
      do j = 1, ny
      do i = 1, nx
         dum = max( dens(i,j), denscut )
         g(i,j) = - 1.d0 / beta * log( dum / denscut )
      end do
      end do

c-----------------------------------------------------------------------
c     //   print 2D free energy surface:  g is negative
c-----------------------------------------------------------------------

c     //   open file
      open ( iounit, file = 'rec.add' )

c     //   loop of y meshes
      do j = 1, ny

c        //   loop of x meshes
         do i = 1, nx

c           //   mesh point
            xij = xmin + dble(i-1) * dx
            yij = ymin + dble(j-1) * dy

c           //   x, y, density
            write( iounit, '(2f12.6,e16.8)' )
     &         xij, yij, g(i,j)*har2kcal

c        //   loop of x meshes
         end do

c        //   blank line
         write( iounit, '(a)' )

c     //   loop of y meshes
      end do

c     //   close file
      close( iounit )

c-----------------------------------------------------------------------
c     //   count non-zero points
c-----------------------------------------------------------------------

c     //   counter
      nxy = 0

c     //   loop of y meshes
      do j = 1, ny

c        //   loop of x meshes
         do i = 1, nx

c           //   neglect tiny values
            if ( abs(g(i,j)) .lt. tiny ) cycle

c           //   counter
            nxy = nxy + 1

c        //   loop of x meshes
         end do

c     //   loop of y meshes
      end do

c     //   matrix dimension
      write( 6, '(a,i8)' ) "Matrix dimension:  ", nxy
      write( 6, '(a)' )

c-----------------------------------------------------------------------
c     //   u, c, b, h matrices:  u, c are positive, b is negative
c-----------------------------------------------------------------------

c     //   memory allocation
      allocate( u(nxy,nxy) )
      allocate( c(nxy,nxy) )
      allocate( b(nxy) )
      allocate( h(nxy) )

c     //   initialize
      u(:,:) = 0.d0
      c(:,:) = 0.d0
      b(:)   = 0.d0
      h(:)   = 0.d0

c     //   gaussian width
      sigx = fact * dx
      sigy = fact * dy

c     //   counter
      kl = 0

c     //   loop of mesh points
      do l = 1, ny
      do k = 1, nx

c        //   counter
         if ( abs(g(k,l)) .le. tiny ) cycle

c        //   counter
         kl = kl + 1

c        //   counter
         ij = 0

c        //   loop of mesh points
         do j = 1, ny
         do i = 1, nx

c           //   counter
            if ( abs(g(i,j)) .le. tiny ) cycle

c           //   counter
            ij = ij + 1

c           //   mesh point
            xij = xmin + dble(i-1) * dx
            yij = ymin + dble(j-1) * dy

c           //   mesh point
            xkl = xmin + dble(k-1) * dx
            ykl = ymin + dble(l-1) * dy

c           //   relative distance
            rx = ( xij - xkl ) / sigx
            ry = ( yij - ykl ) / sigy

c           //   u matrix
            u(ij,kl) = exp( - 0.5d0 * ( rx*rx + ry*ry ) )

c           //   b vector
ccc            b(ij) = b(ij) + u(ij,kl) * g(k,l)
            b(kl) = b(kl) + u(kl,ij) * g(i,j)

c        //   loop of mesh points
         end do
         end do

c     //   loop of mesh points
      end do
      end do

c     //   c matrix = u matrix * udagger matrix
      do j = 1, nxy
      do i = 1, nxy
      do k = 1, nxy
         c(i,j) = c(i,j) + u(i,k) * u(j,k)
      end do
      end do
      end do

c     //   solve linear equation
      call lineq ( c, b, h, nxy )

c-----------------------------------------------------------------------
c     //   print gaussian hills:  h is mostly negative
c-----------------------------------------------------------------------

c     //   copy hills
      call system( 'touch ' // trim(hillsfile) )
      call system( 'cp ' // trim(hillsfile) // ' hills.old' )
      call system( 'cp ' // trim(hillsfile) // ' hills.new' )
      call system( 'sleep 0.1' )

c     //   open hills file
      open ( iounit, file = 'hills.old' )

c     //   counter
      ij = 0

c     //   loop of hills
      do

c        //   read line
         read( iounit, *, iostat=ierr ) i, dum, dum, dum

c        //   detect read error
         if ( ierr .ne. 0 ) exit

c        //   counter
         ij = ij + 1

c     //   loop of hills
      end do

c     //   close hills file
      close( iounit )

c     //   counter
      nhills_old = ij

c     //   counter
      ij = 0

c     //   counter
      kl = 0

c     //   open hills file
      open ( iounit, file = 'hills.new', access='append' )

c     //   loop of mesh points
      do j = 1, ny
      do i = 1, nx

c        //   counter
         if ( abs(g(i,j)) .le. tiny ) cycle

         xij = xmin + dble(i-1) * dx
         yij = ymin + dble(j-1) * dy

c        //   counter
         ij = ij + 1

c        //   counter
         if ( abs(h(ij)) .le. tiny ) cycle

c        //   counter
         kl = kl + 1

c        //   step, position, width, height
         write( iounit, '(i8,3e24.16)' ) 0, -h(ij), sigx, xij
         write( iounit, '(i8,3e24.16)' ) 0, -h(ij), sigy, yij

c     //   loop of mesh points
      end do
      end do

c     //   close hills file
      close( iounit )

c     //   counter
      nhills_new = nhills_old + kl

c-----------------------------------------------------------------------
c     //   new reconstructued hills
c-----------------------------------------------------------------------

c     //   memory allocation
      allocate( p(nx,ny) )

c     //   initialize
      p(:,:) = 0.d0

c     //   open hills file
      open ( iounit, file = 'hills.new' )

c     //   loop of hills
      do

c        //   read lines
         read( iounit, *, iostat=ierr ) l, h1, sigx, xkl
         read( iounit, *, iostat=ierr ) l, h1, sigy, ykl

c        //   exit loop when read error occurred
         if ( ierr .ne. 0 ) exit

c        //   loop of mesh points
         do j = 1, ny
         do i = 1, nx

c           //   mesh point
            xij = xmin + dble(i-1) * dx
            yij = ymin + dble(j-1) * dy

c           //   relative distance
            rx = ( xij - xkl ) / sigx
            ry = ( yij - ykl ) / sigy

c           //   gaussian height
            h2 = exp( - 0.5d0 * ( rx*rx + ry*ry ) )

c           //   skip if too short
            if ( h2 .le. tiny ) cycle

c           //   reconstruction
            p(i,j) = p(i,j) + h1 * h2

c        //   loop of mesh points
         end do
         end do

c     //   loop of hills
      end do

c     //   close hills file
      close( iounit )

c     //   open hills file
      open ( iounit, file = 'rec.new' )

c     //   loop of mesh points
      do j = 1, ny

c        //   loop of mesh points
         do i = 1, nx

c           //   mesh point
            xij = xmin + dble(i-1) * dx
            yij = ymin + dble(j-1) * dy

c           //   print x, y, density
            write( iounit, '(2f12.6,e16.8)' )
     &         xij, yij, min(-p(i,j)*har2kcal,0.d0)

c        //   loop of mesh points
         end do

c        //   x, y, density
         write( iounit, '(a)' )

c     //   loop of mesh points
      end do

c     //   close hills file
      close( iounit )

c-----------------------------------------------------------------------
c     //   old reconstructued hills
c-----------------------------------------------------------------------

c     //   memory allocation
      allocate( q(nx,ny) )

c     //   initialize
      q(:,:) = 0.d0

c     //   open hills file
      open ( iounit, file = 'hills.old' )

c     //   loop of hills
      do

c        //   read lines
         read( iounit, *, iostat=ierr ) l, h1, sigx, xkl
         read( iounit, *, iostat=ierr ) l, h1, sigy, ykl

c        //   exit loop when read error occurred
         if ( ierr .ne. 0 ) exit

c        //   loop of mesh points
         do j = 1, ny
         do i = 1, nx

c           //   mesh point
            xij = xmin + dble(i-1) * dx
            yij = ymin + dble(j-1) * dy

c           //   relative distance
            rx = ( xij - xkl ) / sigx
            ry = ( yij - ykl ) / sigy

c           //   gaussian height
            h2 = exp( - 0.5d0 * ( rx*rx + ry*ry ) )

c           //   skip if too short
            if ( h2 .le. tiny ) cycle

c           //   reconstruction
            q(i,j) = q(i,j) + h1 * h2

c        //   loop of mesh points
         end do
         end do

c     //   loop of hills
      end do

c     //   close hills file
      close( iounit )

c     //   open hills file
      open ( iounit, file = 'rec.old' )

c     //   loop of mesh points
      do j = 1, ny

c        //   loop of mesh points
         do i = 1, nx

c           //   mesh point
            xij = xmin + dble(i-1) * dx
            yij = ymin + dble(j-1) * dy

c            //   print x, y, density
             write( iounit, '(2f12.6,e16.8)' )
     &         xij, yij, min(-q(i,j)*har2kcal,0.d0)

c        //   loop of mesh points
         end do

c        //   x, y, density
         write( iounit, '(a)' )

c     //   loop of mesh points
      end do

c     //   close hills file
      close( iounit )

c     //   number of hills
      write( 6, '(a)' )
      write( 6, '(a,i8)' ) "Old hills:         ", nhills_old
      write( 6, '(a,i8)' ) "Added:             ", nhills_new-nhills_old
      write( 6, '(a,i8)' ) "New hills:         ", nhills_new
      write( 6, '(a)' )

c-----------------------------------------------------------------------
c     //   gnuplot 2D surface
c-----------------------------------------------------------------------

c     //   open gnuplot input file
      open ( iounit, file = 'refine2d.plt' )

c     //   gnuplot settings
      write( iounit, '(a)' )
     &   "set pm3d map; set palette model RGB; set palette defined"
      write( iounit, '(a)' )
     &   "set size square"
      write( iounit, '(a)' )
     &   "set term x11 font 'Helvetica,18'"
      write( iounit, '(a)' )
     &   "set xlabel 'CV1'"
      write( iounit, '(a)' )
     &   "set ylabel 'CV2'"
      write( iounit, '(a,f8.3,a,f8.3,a)' )
     &   "set xrange [", xmin, ":", xmax,  "]"
      write( iounit, '(a,f8.3,a,f8.3,a)' ) 
     &   "set yrange [", ymin, ":", ymax,  "]"

c     //   old surface
      write( iounit, '(a)' )
     &   "set title 'OLD SURFACE'"
      write( iounit, '(a)' )
     &   "splot 'rec.old' u 1:2:3"
      write( iounit, '(a)' )
     &   "pause -1"

c     //   new surface
      write( iounit, '(a)' )
     &   "set title 'NEW SURFACE'"
      write( iounit, '(a)' )
     &   "splot 'rec.new' u 1:2:3"
      write( iounit, '(a)' )
     &   "pause -1"

c     //   difference of surfaces
      write( iounit, '(a)' )
     &   "set title 'DIFFERENCE'"
      write( iounit, '(a)' )
     &   "splot 'rec.add' u 1:2:3"
      write( iounit, '(a)' )
     &   "pause -1"

c     //   close gnuplot input file
      close( iounit )

c     //   run gnuplot
      call system( "gnuplot 'refine2d.plt'" )

c-----------------------------------------------------------------------
c     //   print to screen
c-----------------------------------------------------------------------

      write( 6, '(a)' )
      write( 6, '(a,i8)' ) "Normal termination of refine2d."
      write( 6, '(a)' )

      stop
      end






c***********************************************************************
      subroutine lineq ( a, b, x, n )
c***********************************************************************

c-----------------------------------------------------------------------
c     /*   local variables                                            */
c-----------------------------------------------------------------------

      implicit none

      integer          :: i, j, n, info

      character(len=1) :: uplo

      real(8)          :: a(n,n), b(n), c(n,n), e(n,1), x(n)

      integer          :: ipiv(n)
      integer          :: lwork

      real(8), dimension(:), allocatable :: work

c-----------------------------------------------------------------------
c     /*   start                                                      */
c-----------------------------------------------------------------------

      uplo = 'U'

      do j = 1, n
      do i = 1, n
         c(i,j) = a(i,j)
      end do
      end do

      do i = 1, n
         e(i,1) = b(i)
      end do

      allocate( work(1) )

      call dsysv( uplo, n, 1, a, n, ipiv, e, n, work(1), -1, info )

      lwork = min( int(work(1))+1, n*n )

      deallocate( work )

      write( 6, '(a,i8)' ) "Working memory:    ", lwork

      allocate( work(lwork) )

      call dsysv( uplo, n, 1, a, n, ipiv, e, n, work, lwork, info )

      write( 6, '(a,i8)' ) "Error information: ", info

      do j = 1, n
      do i = 1, n
         a(i,j) = c(i,j)
      end do
      end do

      do i = 1, n
         x(i) = e(i,1)
      end do

      return
      end
