c///////////////////////////////////////////////////////////////////////
c
c      Author:          M. Shiga
c      Last updated:    Feb 16, 2020 by M. Shiga
c      Description:     make rec2hills_after.cube from rec.out
c
c///////////////////////////////////////////////////////////////////////
c***********************************************************************
      program rec2hills
c***********************************************************************

c-----------------------------------------------------------------------
c     //   declare variables
c-----------------------------------------------------------------------

c     //   clear all variables
      implicit none

c     //   actual number of meshes
      integer :: nmesh

c     //   number of meshes in rec.out
      integer :: mmesh

c     //   distance of mesh positions
      real(8) :: rx, ry, rz

c     //   distance normalized by width
      real(8) :: rwx, rwy, rwz, rw2

c     //   boundary condition: 0 (free) or 1 (periodic)
      integer :: ipbc_x = 0
      integer :: ipbc_y = 0
      integer :: ipbc_z = 0

c     //   mesh size of the cube
      real(8) :: x_mesh = 0.d0
      real(8) :: y_mesh = 0.d0
      real(8) :: z_mesh = 0.d0

c     //   width
      real(8) :: sigma_x, sigma_y, sigma_z, sigma_2

c     //   vectors
      real(8), dimension(:), allocatable :: b, f, g, h, x, y, z

c     //   vectors
      real(8), dimension(:), allocatable :: frev, grev

c     //   matrix
      real(8), dimension(:,:), allocatable :: c, u

c     //   vectors
      integer, dimension(:), allocatable :: nlist

c     //   matrix
      integer, dimension(:,:), allocatable :: list

c     //   default parameters
      real(8), dimension(8) :: cv_min
      real(8), dimension(8) :: cv_max
      real(8), dimension(8) :: cv_mesh

c     //   integers
      integer :: iounit = 10

c     //   integers
      integer :: i, j, k, ierr, jlist, klist, nlist_max

c     //   cut off of r value
      real(8) :: cutoff = 40.d0

c     //   lower bound of the cube
      real(8) :: x_min = 0.d0
      real(8) :: y_min = 0.d0
      real(8) :: z_min = 0.d0

c     //   upper bound of the cube
      real(8) :: x_max = 0.d0
      real(8) :: y_max = 0.d0
      real(8) :: z_max = 0.d0
      real(8) :: g_max = 0.d0

c     //   number of meshes
      integer :: mesh_x = 0
      integer :: mesh_y = 0
      integer :: mesh_z = 0

c     //   real number
      real(8) :: tiny = 1.d-10

c     //   real number
      real(8) :: small = 1.d-5

c     //   real number
      real(8) :: g_trim = 0.d0

c     //   real number
      real(8) :: ucut, error_max, error_rms

c     //   character
      character(len=24) :: cv1, cv2, cv3, cv4

c     //   logical
      logical :: file_exists

c-----------------------------------------------------------------------
c     //   read parameters
c-----------------------------------------------------------------------

c     //   error flag
      ierr = 1

c     //   file
      inquire( file='rec2hills.dat', exist=file_exists )

c     //   read from file
      if ( file_exists ) then

c        //   open file
         open ( iounit, file='rec2hills.dat' )

c        //   input file
         read( iounit, *, iostat=ierr ) cv1, x_min, x_max, x_mesh
         read( iounit, *, iostat=ierr ) cv2, y_min, y_max, y_mesh
         read( iounit, *, iostat=ierr ) cv3, z_min, z_max, z_mesh
         read( iounit, *, iostat=ierr ) g_trim

c        //   close file
         close( iounit )

c        //   on error, remove file and stop
         if ( ierr .ne. 0 ) then
            call system( 'rm rec2hills.dat' )
            stop
         end if

c        //   free boundary
         if ( cv1(1:5) .ne. 'DIH  ' ) ipbc_x = 0
         if ( cv2(1:5) .ne. 'DIH  ' ) ipbc_y = 0
         if ( cv3(1:5) .ne. 'DIH  ' ) ipbc_z = 0

c        //   periodic boundary
         if ( cv1(1:5) .eq. 'DIH  ' ) ipbc_x = 1
         if ( cv2(1:5) .eq. 'DIH  ' ) ipbc_y = 1
         if ( cv2(1:5) .eq. 'DIH  ' ) ipbc_z = 1

c     //   three arguments
      end if

c-----------------------------------------------------------------------
c     //   read data
c-----------------------------------------------------------------------

c     //   three arguments
      if ( ( .not. file_exists ) .and.
     &     ( ( iargc() .eq. 3 ) .or. ( iargc() .eq. 4 ) ) ) then

         call getarg( 1, cv1 )
         call getarg( 2, cv2 )
         call getarg( 3, cv3 )

         if ( iargc() .eq. 4 ) then
            call getarg( 4, cv4 )
            read( cv4, * ) g_trim
         else
            g_trim = 0.d0
         end if

c-----------------------------------------------------------------------
c        //   set default parameters
c-----------------------------------------------------------------------
c
c        <params_rec_meta>
c        DIST   2.0    5.0   0.1
c        ANGL   0.0  180.0   6.0
c        DIH    0.0  360.0  12.0
c        DIFF   0.0   15.0   0.5
c        CN     0.0    3.0   0.1
c        DCN   -2.0    2.0   0.1
c        XYZ    2.0    5.0   0.1
c        DXYZ   2.0    5.0   0.1

         cv_min(1)  =   2.0d0
         cv_max(1)  =   5.0d0
         cv_mesh(1) =   0.1d0

         cv_min(2)  =   0.0d0
         cv_max(2)  = 180.0d0
         cv_mesh(2) =   6.0d0

         cv_min(3)  =   0.0d0
         cv_max(3)  = 348.0d0
         cv_mesh(3) =  12.0d0
 
         cv_min(4)  =   0.0d0
         cv_max(4)  =  15.0d0
         cv_mesh(4) =   0.5d0

         cv_min(5)  =   0.0d0
         cv_max(5)  =   3.0d0
         cv_mesh(5) =   0.1d0

         cv_min(6)  =  -2.0d0
         cv_max(6)  =   2.0d0
         cv_mesh(6) =   0.1d0

         cv_min(7)  =   2.0d0
         cv_max(7)  =   5.0d0
         cv_mesh(7) =   0.1d0

         cv_min(8)  =   2.0d0
         cv_max(8)  =   5.0d0
         cv_mesh(8) =   0.1d0

         if ( cv1(1:5) .eq. 'DIST ' ) then
            x_min  =   cv_min(1)
            x_max  =   cv_max(1)
            x_mesh =   cv_mesh(1)
            ipbc_x =   0
         else if ( cv1(1:5) .eq. 'ANGL ' ) then
            x_min  =   cv_min(2)
            x_max  =   cv_max(2)
            x_mesh =   cv_mesh(2)
            ipbc_x =   0
         else if ( cv1(1:5) .eq. 'DIH  ' ) then
            x_min  =   cv_min(3)
            x_max  =   cv_max(3)
            x_mesh =   cv_mesh(3)
            ipbc_x =   1
         else if ( cv1(1:5) .eq. 'DIFF ' ) then
            x_min  =   cv_min(4)
            x_max  =   cv_max(4)
            x_mesh =   cv_mesh(4)
            ipbc_x =   0
         else if ( cv1(1:5) .eq. 'CN   ' ) then
            x_min  =   cv_min(5)
            x_max  =   cv_max(5)
            x_mesh =   cv_mesh(5)
            ipbc_x =   0
         else if ( cv1(1:5) .eq. 'DCN  ' ) then
            x_min  =   cv_min(6)
            x_max  =   cv_max(6)
            x_mesh =   cv_mesh(6)
            ipbc_x =   0
         else if ( cv1(1:5) .eq. 'XYZ  ' ) then
            x_min  =   cv_min(7)
            x_max  =   cv_max(7)
            x_mesh =   cv_mesh(7)
            ipbc_x =   0
         else if ( cv1(1:5) .eq. 'DXYZ ' ) then
            x_min  =   cv_min(8)
            x_max  =   cv_max(8)
            x_mesh =   cv_mesh(8)
            ipbc_x =   0
         end if

         if ( cv2(1:5) .eq. 'DIST ' ) then
            y_min  =   cv_min(1)
            y_max  =   cv_max(1)
            y_mesh =   cv_mesh(1)
            ipbc_y =   0
         else if ( cv2(1:5) .eq. 'ANGL ' ) then
            y_min  =   cv_min(2)
            y_max  =   cv_max(2)
            y_mesh =   cv_mesh(2)
            ipbc_y =   0
         else if ( cv2(1:5) .eq. 'DIH  ' ) then
            y_min  =   cv_min(3)
            y_max  =   cv_max(3)
            y_mesh =   cv_mesh(3)
            ipbc_y =   1
         else if ( cv2(1:5) .eq. 'DIFF ' ) then
            y_min  =   cv_min(4)
            y_max  =   cv_max(4)
            y_mesh =   cv_mesh(4)
            ipbc_y =   0
         else if ( cv2(1:5) .eq. 'CN   ' ) then
            y_min  =   cv_min(5)
            y_max  =   cv_max(5)
            y_mesh =   cv_mesh(5)
            ipbc_y =   0
         else if ( cv2(1:5) .eq. 'DCN  ' ) then
            y_min  =   cv_min(6)
            y_max  =   cv_max(6)
            y_mesh =   cv_mesh(6)
            ipbc_y =   0
         else if ( cv2(1:5) .eq. 'XYZ  ' ) then
            y_min  =   cv_min(7)
            y_max  =   cv_max(7)
            y_mesh =   cv_mesh(7)
            ipbc_y =   0
         else if ( cv2(1:5) .eq. 'DXYZ ' ) then
            y_min  =   cv_min(8)
            y_max  =   cv_max(8)
            y_mesh =   cv_mesh(8)
            ipbc_y =   0
         end if

         if ( cv3(1:5) .eq. 'DIST ' ) then
            z_min  =   cv_min(1)
            z_max  =   cv_max(1)
            z_mesh =   cv_mesh(1)
            ipbc_z =   0
         else if ( cv3(1:5) .eq. 'ANGL ' ) then
            z_min  =   cv_min(2)
            z_max  =   cv_max(2)
            z_mesh =   cv_mesh(2)
            ipbc_z =   0
         else if ( cv3(1:5) .eq. 'DIH  ' ) then
            z_min  =   cv_min(3)
            z_max  =   cv_max(3)
            z_mesh =   cv_mesh(3)
            ipbc_z =   1
         else if ( cv3(1:5) .eq. 'DIFF ' ) then
            z_min  =   cv_min(4)
            z_max  =   cv_max(4)
            z_mesh =   cv_mesh(4)
            ipbc_z =   0
         else if ( cv3(1:5) .eq. 'CN   ' ) then
            z_min  =   cv_min(5)
            z_max  =   cv_max(5)
            z_mesh =   cv_mesh(5)
            ipbc_z =   0
         else if ( cv3(1:5) .eq. 'DCN  ' ) then
            z_min  =   cv_min(6)
            z_max  =   cv_max(6)
            z_mesh =   cv_mesh(6)
            ipbc_z =   0
         else if ( cv3(1:5) .eq. 'XYZ  ' ) then
            z_min  =   cv_min(7)
            z_max  =   cv_max(7)
            z_mesh =   cv_mesh(7)
            ipbc_z =   0
         else if ( cv3(1:5) .eq. 'DXYZ ' ) then
            z_min  =   cv_min(8)
            z_max  =   cv_max(8)
            z_mesh =   cv_mesh(8)
            ipbc_z =   0
         end if

c        //   open file
         open ( iounit, file = 'rec2hills.dat' )

c        //   input file
         write( iounit, '(a,3f12.6)' ) cv1, x_min, x_max, x_mesh
         write( iounit, '(a,3f12.6)' ) cv2, y_min, y_max, y_mesh
         write( iounit, '(a,3f12.6)' ) cv3, z_min, z_max, z_mesh
         write( iounit, '(f12.8)'   ) g_trim

c        //   close file
         close( iounit )

c     //   three arguments
      end if

c-----------------------------------------------------------------------
c     //   usage
c-----------------------------------------------------------------------

c     //   three arguments
      if ( ( ierr .ne. 0 ) .and.
     &     ( ( iargc() .ne. 3 ) .and. ( iargc() .ne. 4 ) ) ) then

c        //   print message
         write( 6, * ) 
         write( 6, * ) 'CODE:        rec2hills.x'
         write( 6, * ) 
         write( 6, * ) 'INPUT FILE:  rec.out'
         write( 6, * ) 'OUTPUT FILE: rec2hills_before.cube'
         write( 6, * ) 'OUTPUT FILE: rec2hills_after.cube'
         write( 6, * ) 'OUTPUT FILE: rec2hills.ini'
         write( 6, * ) 'OUTPUT FILE: rec2hills.out'
         write( 6, * ) 
         write( 6, * ) 'USAGE:       rec2hills.x $1 $2 $3'
         write( 6, * ) '$1:          cv type of x.'
         write( 6, * ) '$2:          cv type of y.'
         write( 6, * ) '$3:          cv type of z.'
         write( 6, * ) '$4:          hills trimming in au (0.000000).'
         write( 6, * ) 

c        //   error termination
         stop

c     //   three arguments
      end if

c-----------------------------------------------------------------------
c     //   number of meshes
c-----------------------------------------------------------------------

      mesh_x = nint( ( x_max - x_min ) / x_mesh ) + 1
      mesh_y = nint( ( y_max - y_min ) / y_mesh ) + 1
      mesh_z = nint( ( z_max - z_min ) / z_mesh ) + 1

c-----------------------------------------------------------------------
c     //   print message
c-----------------------------------------------------------------------

      write( 6, '(a)' ) 'CV TYPE OF X:  ' // trim(cv1) // '.'
      write( 6, '(a)' ) 'CV TYPE OF Y:  ' // trim(cv2) // '.'
      write( 6, '(a)' ) 'CV TYPE OF Z:  ' // trim(cv3) // '.'

      write( 6, '(a)' ) 'PARAMETER FILE: rec2hills.dat.'

c-----------------------------------------------------------------------
c     //   set width
c-----------------------------------------------------------------------

c     //   gaussian width
      sigma_x = x_mesh / 1.5d0
      sigma_y = y_mesh / 1.5d0
      sigma_z = z_mesh / 1.5d0

c     //   square of gaussian width
      sigma_2 = sigma_x*sigma_x + sigma_y*sigma_y + sigma_z*sigma_z

c     //   cut off in energy scale
      ucut = exp( - 0.5d0 * cutoff )

c-----------------------------------------------------------------------
c     //   read meshes
c-----------------------------------------------------------------------

c     //   print message
      write( 6, '(a)' ) 'START: read rec.out.'

c     //   open file
      open ( iounit, file = 'rec.out' )

c     //   counter
      i = 0
      j = 0

c     //   maximum barrier
      g_max = 0.d0

c     //   loop start
      do

c        //   read line
         read( iounit, *, iostat=ierr ) rx, ry, rz, rw2

c        //   end of file
         if ( ierr .ne. 0 ) exit

c        //   counter
         j = j + 1

c        //   out of boundary
         if ( rx .gt. x_max ) cycle
         if ( ry .gt. y_max ) cycle
         if ( rz .gt. z_max ) cycle

c        //   counter
         i = i + 1

c        //   maximum barrier
         g_max = max( rw2, g_max )

c     //   loop end
      end do

c     //   close file
      close( iounit )

c     //   number of meshes read
      nmesh = i

c     //   actual number of meshes
      mmesh = j

c     //   print message
      write( 6, '(a)' ) 'DONE:  read rec.out.'

c-----------------------------------------------------------------------
c     //   print messages
c-----------------------------------------------------------------------

c     //   number of meshes
      write( 6, '(a,i8)' ) 'NUMBER OF MESHES IN REC.OUT: ', mmesh

c     //   if number of meshes is relevant
      if ( nmesh .eq. (mesh_x*mesh_y*mesh_z) ) then

c        //   print messages
         write( 6, '(a,i8)' )
     &      'ACTUAL NUMBER OF MESHES IS CONSISTENT: ',
     &       nmesh
         write( 6, '(a,3f10.4)' )
     &       'ACTUAL MIN, MAX, MESH VALUES OF X: ',
     &        x_min, x_max, x_mesh
         write( 6, '(a,3f10.4)' )
     &       'ACTUAL MIN, MAX, MESH VALUES OF Y: ',
     &        y_min, y_max, y_mesh
         write( 6, '(a,3f10.4)' )
     &       'ACTUAL MIN, MAX, MESH VALUES OF Z: ',
     &        z_min, z_max, z_mesh
         write( 6, '(a,3i4)' )
     &       'ACTUAL NUMBER OF MESHES IN X, Y, Z: ',
     &        mesh_x, mesh_y, mesh_z

c     //   if number of meshes is irrelevant
      else

c        //   print messages
         write( 6, '(a,i8,a,i8)' )
     &      'ERROR - NUMBER OF MESHES IS NOT CONSISTENT: ',
     &       nmesh, ' AND ', mesh_x*mesh_y*mesh_z
         write( 6, '(a,3f10.4)' )
     &       'ACTUAL MIN, MAX, MESH VALUES OF X: ',
     &        x_min, x_max, x_mesh
         write( 6, '(a,3f10.4)' )
     &       'ACTUAL MIN, MAX, MESH VALUES OF Y: ',
     &        y_min, y_max, y_mesh
         write( 6, '(a,3f10.4)' )
     &       'ACTUAL MIN, MAX, MESH VALUES OF Z: ',
     &        z_min, z_max, z_mesh
         write( 6, '(a,3i4)' )
     &       'ACTUAL NUMBER OF MESHES IN X, Y, Z: ',
     &        mesh_x, mesh_y, mesh_z
         write( 6, '(a)' )
     &       'MANUALLY CHANGE PARAMETERS IN: rec2hills.dat.'

c         //   error termination
          stop

c     //   end of if statement
      end if

c     //   print message
      write( 6, '(a,f12.8)' ) 'CUTOFF PARAMETER: ', ucut

c     //   print message
      write( 6, '(a,f12.8)' ) 'HILLS TRIMMING:   ', g_trim

c-----------------------------------------------------------------------
c     //   memory allocation
c-----------------------------------------------------------------------

c     //   mesh position
      allocate( x(nmesh) )
      allocate( y(nmesh) )
      allocate( z(nmesh) )

c     //   height
      allocate( g(nmesh) )

c     //   gaussian height
      allocate( h(nmesh) )

c     //   neighbor list
      allocate( nlist(nmesh) )
      allocate( list(nmesh,nmesh) )

c     //   vectors and matrices
      allocate( grev(nmesh) )
      allocate( f(nmesh) )
      allocate( frev(nmesh) )
      allocate( b(nmesh) )
      allocate( u(nmesh,nmesh) )
      allocate( c(nmesh,nmesh) )

c-----------------------------------------------------------------------
c     //   read grids
c-----------------------------------------------------------------------

c     //   print message
      write( 6, '(a)' ) 'START: read rec.out.'

c     //   open file
      open ( iounit, file = 'rec.out' )

c     //   counter
      i = 0

      do j = 1, mmesh

c        //   cv position and height
         read( iounit, * ) rx, ry, rz, rw2

c        //   out of boundary
         if ( rx .gt. x_max ) cycle
         if ( ry .gt. y_max ) cycle
         if ( rz .gt. z_max ) cycle

c        //   counter
         i = i + 1

c        //   cv position
         x(i) = rx
         y(i) = ry
         z(i) = rz

c        //   free energy height (0 < g(i) < g_max)
         g(i) = rw2 

      end do

c     //   close file
      close( iounit )

c     //   print message
      write( 6, '(a)' ) 'DONE:  read rec.out.'

c-----------------------------------------------------------------------
c     //   reverse g
c-----------------------------------------------------------------------

c     //   print message
      write( 6, '(a)' ) 'START: negative of grid values, the G vector.'

c     //   hills height (negative of free energy, 0 < grev(i) < g_max)
      do i = 1, nmesh
         grev(i) = - g(i) + g_max
      end do

c     //   trimmed hills (negative of free energy, 0 < grev(i) < g_max)
      do i = 1, nmesh
         grev(i) = max( grev(i)-g_trim, 0.d0 )
      end do

c     //   print message
      write( 6, '(a)' ) 'DONE:  negative of grid values, the G vector.'

c-----------------------------------------------------------------------
c     //   rec2hills_before.cube
c-----------------------------------------------------------------------

c     //   print message
      write( 6, '(a)' ) 'START: print hills to rec2hills_before.cube.'

c     //   open file
      open ( iounit, file = "rec2hills_before.cube" )

c     //   header
      write( iounit, '(a)' )
     &   "CUBE FILE."

c     //   header
      write( iounit, '(a)' )
     &   "OUTER LOOP: X, MIDDLE LOOP: Y, INNER LOOP: Z"

c     //   header
      write( iounit, '(i5,3e16.8)' )
     &    0, x_max-x_min, y_max-y_min, z_max-z_min

c     //   header
      write( iounit, '(i5,3e16.8)' )
     &    mesh_x, 1.d0, 0.d0, 0.d0

c     //   header
      write( iounit, '(i5,3e16.8)' )
     &    mesh_y, 0.d0, 1.d0, 0.d0

c     //   header
      write( iounit, '(i5,3e16.8)' )
     &    mesh_z, 0.d0, 0.d0, 1.d0

cc     //   shifted hills height (-g_max < grev(i)-g_max < 0)
c      do i = 1, nmesh
c         write( iounit, '(f12.6)' ) grev(i)-g_max
c      end do

c     //   shifted free energy (-g_max < g(i)-g_max < 0)
      do i = 1, nmesh
         write( iounit, '(f12.6)' ) g(i)-g_max
      end do

c     //   close file
      close( iounit )

c     //   print message
      write( 6, '(a)' ) 'DONE:  print hills to rec2hills_before.cube.'

c-----------------------------------------------------------------------
c     //   calculate b
c-----------------------------------------------------------------------

c     //   print message
      write( 6, '(a)' ) 'START: calculation of B vector from G vector.'

c     //   loop of actual meshes
      do i = 1, nmesh

c        //   b vector
         b(i) = 0.d0

c        //   loop of actual meshes
         do j = 1, nmesh

c           //   distance of two cv positions
            rx = x(i) - x(j)
            ry = y(i) - y(j)
            rz = z(i) - z(j)

c           //   apply periodic boundary
            rx = rx - ipbc_x * nint( rx / 360.d0 ) * 360.d0
            ry = ry - ipbc_y * nint( ry / 360.d0 ) * 360.d0
            rz = rz - ipbc_z * nint( rz / 360.d0 ) * 360.d0

c           //   scaled by gaussian width
            rwx = rx / sigma_x
            rwy = ry / sigma_y
            rwz = rz / sigma_z

c           //   scaled distance squared
            rw2 = rwx*rwx + rwy*rwy + rwz*rwz

c           //   u matrix
            u(i,j) = exp( - 0.5d0 * rw2 )

c           //   b vector
            b(i) = b(i) + u(i,j) * grev(j)

c        //   loop of actual meshes
         end do

c     //   loop of actual meshes
      end do

c     //   print message
      write( 6, '(a)' ) 'DONE:  calculation of B vector from G vector.'

c-----------------------------------------------------------------------
c     //   neighbor list
c-----------------------------------------------------------------------

c     //   print message
      write( 6, '(a)' ) 'START: making a neighbor list.'

c     //   maximum number of neighbors
      nlist_max = 0

c     //   loop of actual meshes
      do i = 1, nmesh

c        //   counter
         jlist = 0

c        //   loop of actual meshes
         do j = 1, nmesh

c           //   make sure those within cut off distance
            if ( u(i,j) .lt. ucut ) cycle

c           //   counter
            jlist = jlist + 1

c           //   neighbor list
            list(jlist,i) = j

c        //   loop of actual meshes
         end do

c        //   number of neighbors
         nlist(i) = jlist

c        //   maximum number of neighbors
         nlist_max = max( jlist, nlist_max )

c     //   loop of actual meshes
      end do

c     //   print message
      write( 6, '(a,i6)' ) 'MAXIMUM NUMBER OF NEIGHBORS: ', nlist_max

c     //   print message
      write( 6, '(a)' ) 'DONE:  making a neighbor list.'

c-----------------------------------------------------------------------
c     //   calculate c
c-----------------------------------------------------------------------

c     //   print message
      write( 6, '(a)' ) 'START: calculation of C matrix.'

      c(:,:) = 0.d0

c     //   loop of actual meshes
      do i = 1, nmesh

c        //   loop of neighbors
         do jlist = 1, nlist(i)

c           //   neighbor point
            j = list(jlist,i)

c           //   loop of neighbors
            do klist = 1, nlist(i)

c              //   neighbor point
               k = list(klist,i)

c              //   c matrix
               c(i,j) = c(i,j) + u(i,k) * u(j,k)

c           //   loop of neighbors
            end do

c        //   loop of neighbors
         end do

c     //   loop of actual meshes
      end do

c      do i = 1, nmesh
c
c         do j = 1, nmesh
c
c            c(i,j) = 0.d0
c
c            if ( u(i,j) .gt. ucut ) cycle
c
c            do k = 1, nmesh
c
c               if ( u(i,k) .gt. ucut ) cycle
c               if ( u(j,k) .gt. ucut ) cycle
c
c               c(i,j) = c(i,j) + u(i,k) * u(j,k)
c
c            end do
c
c         end do
c
c         write( 6, '(a,f7.2)' )
c     &      'DONE:  calculation of c (%):', dble(i)/dble(nmesh)*100.d0
c
c      end do

c     //   print message
      write( 6, '(a)' ) 'DONE:  calculation of C matrix.'

c-----------------------------------------------------------------------
c     //   calculate h
c-----------------------------------------------------------------------

c     //   print message
      write( 6, '(a)' ) 'START: solving linear equation CX = B.'

c     //   solve linear equation
      call lineq ( c, b, h, nmesh )

c     //   print message
      write( 6, '(a)' ) 'DONE:  solving linear equation CX = B.'

c-----------------------------------------------------------------------
c     //   correction
c-----------------------------------------------------------------------

c     //   print message
      write( 6, '(a)' ) 'START: correction of X vector.'

c     //   loop of actual meshes
      do i = 1, nmesh

c        //   neglect small gaussian
         if ( ( abs(g(i)) .lt. tiny ) .and.
     &        ( abs(h(i)) .lt. small ) ) h(i) = 0.d0

c     //   loop of actual meshes
      end do

c     //   print message
      write( 6, '(a)' ) 'DONE:  correction of X vector.'

c-----------------------------------------------------------------------
c     //   print gaussians
c-----------------------------------------------------------------------

c     //   print message
      write( 6, '(a)' ) 'START: print X vector to rec2hills.ini.'

c     //   open file
      open ( iounit, file = "rec2hills.ini" )

c     //   counter
      j = 0

c     //   actual number of meshes
      do i = 1, nmesh

c        //   print only those with non-zero gaussian heights
         if ( h(i) .eq. 0.d0 ) cycle

c        //   counter
         j = j + 1

c        //   print data
         write( iounit, '(i8,3e24.16)' ) 0, h(i), sigma_x, x(i)
         write( iounit, '(i8,3e24.16)' ) 0, h(i), sigma_y, y(i)
         write( iounit, '(i8,3e24.16)' ) 0, h(i), sigma_z, z(i)

c     //   actual number of meshes
      end do

c     //   close file
      close( iounit )

c     //   print message
      write( 6, '(a)' ) 'DONE:  print X vector to rec2hills.ini.'

c     //   print message
      write( 6, '(a,i8)' ) 'NUMBER OF HILLS:', j

c-----------------------------------------------------------------------
c     //   calculate hills
c-----------------------------------------------------------------------

c     //   print message
      write( 6, '(a)' ) 'START: hills reconstruction by X.'

c     //   actual number of meshes
      do i = 1, nmesh

c        //   hills
         f(i) = 0.d0

c        //   actual number of meshes
         do j = 1, nmesh

c           //   difference in two cvs
            rx = x(i) - x(j)
            ry = y(i) - y(j)
            rz = z(i) - z(j)

c           //   periodic boundary
            rx = rx - ipbc_x * nint( rx / 360.d0 ) * 360.d0
            ry = ry - ipbc_y * nint( ry / 360.d0 ) * 360.d0
            rz = rz - ipbc_z * nint( rz / 360.d0 ) * 360.d0

c           //   scaled by gaussian width
            rwx = rx / sigma_x
            rwy = ry / sigma_y
            rwz = rz / sigma_z

c           //   scaled difference squared
            rw2 = rwx*rwx + rwy*rwy + rwz*rwz

c           //   hills
            f(i) = f(i) + exp( - 0.5d0 * rw2 ) * h(j)

c         //   actual number of meshes
          end do

c     //   actual number of meshes
      end do

c     //   print message
      write( 6, '(a)' ) 'DONE:  hills reconstruction by X.'

c-----------------------------------------------------------------------
c     //   negative of hills
c-----------------------------------------------------------------------

c     //   print message
      write( 6, '(a)' ) 'START: negative of hills.'

c     //   free energy (0 < frev(i) < g_max)
      do i = 1, nmesh
         frev(i) = - f(i) + g_max
      end do

c     //   print message
      write( 6, '(a)' ) 'DONE:  negative of hills.'

c-----------------------------------------------------------------------
c     //   check.out
c-----------------------------------------------------------------------

c     //   print message
      write( 6, '(a)' ) 'START: print hills to rec2hills.out.'

c     //   open file
      open ( iounit, file = "rec2hills.out" )

c     //   maximum error
      error_max = 0.d0

c     //   rms error
      error_rms = 0.d0

c     //   loop of actual meshes
      do i = 1, nmesh

c        //   cv, negative of hills, height, difference
         write( iounit, '(6f12.6)' )
     &      x(i), y(i), z(i), frev(i), g(i), frev(i)-g(i)

c        //   maximum error
         error_max = max( error_max, abs(frev(i)-g(i)) )

c        //   rms error
         error_rms = error_rms + (frev(i)-g(i))*(frev(i)-g(i))

c     //   loop of actual meshes
      end do

c     //   rms error
      error_rms = sqrt( error_rms / dble(nmesh) )

c     //   close file
      close( iounit )

c     //   print message
      write( 6, '(a)' ) 'DONE:  print hills to rec2hills.out.'

c     //   print message
      write( 6, '(a,f10.6)' ) 'MAXIMUM ERROR: ', error_max
      write( 6, '(a,f10.6)' ) 'RMS ERROR:     ', error_rms

c-----------------------------------------------------------------------
c     //   rec2hills_after.cube
c-----------------------------------------------------------------------

c     //   print message
      write( 6, '(a)' ) 'START: print hills to rec2hills_after.cube.'

c     //   open file
      open ( iounit, file = "rec2hills_after.cube" )

c     //   header
      write( iounit, '(a)' )
     &   "CUBE FILE."

c     //   header
      write( iounit, '(a)' )
     &   "OUTER LOOP: X, MIDDLE LOOP: Y, INNER LOOP: Z"

c     //   header
      write( iounit, '(i5,3e16.8)' )
     &    0, x_max-x_min, y_max-y_min, z_max-z_min

c     //   header
      write( iounit, '(i5,3e16.8)' )
     &    mesh_x, 1.d0, 0.d0, 0.d0

c     //   header
      write( iounit, '(i5,3e16.8)' )
     &    mesh_y, 0.d0, 1.d0, 0.d0

c     //   header
      write( iounit, '(i5,3e16.8)' )
     &    mesh_z, 0.d0, 0.d0, 1.d0

c     //   shifted free energy  (-g_max < frev(i)-g_max < 0)
      do i = 1, nmesh
         write( iounit, '(f12.6)' ) frev(i)-g_max
      end do

c     //   close file
      close( iounit )

c     //   print message
      write( 6, '(a)' ) 'DONE:  print hills to rec2hills_after.cube.'

      stop
      end





c***********************************************************************
      subroutine lineq ( a, b, x, n )
c***********************************************************************

c-----------------------------------------------------------------------
c     /*   local variables                                            */
c-----------------------------------------------------------------------

      implicit none

      integer          :: i, j, n, info

      character(len=1) :: uplo

      real(8)          :: a(n,n), b(n), c(n,n), e(n,1), x(n)

      integer          :: ipiv(n)
      integer          :: lwork

      real(8)          :: dlwork

      real(8), dimension(:), allocatable :: work

c-----------------------------------------------------------------------
c     /*   start                                                      */
c-----------------------------------------------------------------------

      uplo = 'U'

      do j = 1, n
      do i = 1, n
         c(i,j) = a(i,j)
      end do
      end do

      do i = 1, n
         e(i,1) = b(i)
      end do

      call dsysv( uplo, n, 1, a, n, ipiv, e, n, dlwork, -1, info)

      lwork = min( int(dlwork)+1, n*n)

      write( 6, '(a,i8)' ) 'LWORK:', lwork

      allocate( work(lwork) )

      call dsysv( uplo, n, 1, a, n, ipiv, e, n, work, lwork, info)

      do j = 1, n
      do i = 1, n
         a(i,j) = c(i,j)
      end do
      end do

      do i = 1, n
         x(i) = e(i,1)
      end do

      return
      end
