c***********************************************************************
      program poscar2xyz
c***********************************************************************
c-----------------------------------------------------------------------
c     //   variables
c-----------------------------------------------------------------------

c     //   local variables
      implicit none

c     //   number of atomic species
      integer :: nspec

c     //   number of atoms
      integer :: natom

c     //   cartesian coordinates
      real(8), dimension(:), allocatable :: x, y, z

c     //   box matrix
      real(8) :: box(3,3)

c     //   number of atoms per species
      integer, dimension(:), allocatable :: natom_spec

c     //   species
      character(len=4), dimension(:), allocatable :: spec

c     //   species
      character(len=4), dimension(:), allocatable :: species

c     //   real numbers
      real(8) :: sx, sy, sz, det3, s(3), xl, yl, zl

c     //   characters
      character(len=120) :: char_line, char_word(3), char

c     //   file unit
      integer :: iounit = 10

c     //   option
      integer :: ioption = 0

c     //   integers
      integer :: i, j, k, ierr, iargc, la, lb, lc

c     //   boxes
      integer :: na, nb, nc

c     //   file names
      character(len=80) :: inpfile
      character(len=80) :: xyzfile
      character(len=80) :: posfile

c     //   option
      character(len=1)  :: option

c-----------------------------------------------------------------------
c     //   initial message
c-----------------------------------------------------------------------

      if ( iargc() .ne. 7 ) then

         write( 6, '(a)' )
         write( 6, '(a)' ) 'Program poscar2xyz'
         write( 6, '(a)' )
         write( 6, '(a)' ) 'Usage: poscar2xyz.x $1 $2 $3'
         write( 6, '(a)' )
         write( 6, '(a)' ) '$1: poscar file      (POSCAR)'
         write( 6, '(a)' ) '$2: PIMD input file: (input.dat)'
         write( 6, '(a)' ) '$3: xyz file:        (structure.dat)'
         write( 6, '(a)' ) '$4: na:              (1)'
         write( 6, '(a)' ) '$5: nb:              (1)'
         write( 6, '(a)' ) '$6: nc:              (1)'
         write( 6, '(a)' ) '$7: option:          (Y/N)'
         write( 6, '(a)' )
         write( 6, '(a)' ) 'Notes:'
         write( 6, '(a)' )
     &      'Atomic species in line 6 required in poscar file.'
         write( 6, '(a)' )
     &      '$4=Y adds atom number column in the xyz file.'
         write( 6, '(a)' )
         write( 6, '(a)' ) 'Example:'
         write( 6, '(a)' )
     &      'poscar2xyz.x POSCAR input.dat structure.dat 1 1 1 Y'
         write( 6, '(a)' )
         stop

      else

         write( 6, '(a)' )
         write( 6, '(a)' ) 'Program poscar2xyz'
         write( 6, '(a)' )

      end if

c-----------------------------------------------------------------------
c     //   files
c-----------------------------------------------------------------------

      call getarg( 1, posfile )
      call getarg( 2, inpfile )
      call getarg( 3, xyzfile )
      call getarg( 4, char_line )
      read( char_line, * ) na
      call getarg( 5, char_line )
      read( char_line, * ) nb
      call getarg( 6, char_line )
      read( char_line, * ) nc
      call getarg( 7, option  )

c-----------------------------------------------------------------------
c     //   open poscar file
c-----------------------------------------------------------------------

      open ( iounit, file = trim(posfile) )

c-----------------------------------------------------------------------
c     //   1st line
c-----------------------------------------------------------------------

c     //   read 1st line: comment
      read ( iounit, '(a)' ) char_line

c-----------------------------------------------------------------------
c     //   2nd line
c-----------------------------------------------------------------------

c     //   read 2nd line: comment
      read ( iounit, '(a)' ) char_line

c     //   x, y, z scaling factors
      read ( char_line, *, iostat=ierr ) sx
      if ( ierr .ne. 0 ) sx = 0.d0

      read ( char_line, *, iostat=ierr ) sx, sy
      if ( ierr .ne. 0 ) sy = sx

      read ( char_line, *, iostat=ierr ) sx, sy, sz
      if ( ierr .ne. 0 ) sz = sx

c-----------------------------------------------------------------------
c     //   3nd-5th lines
c-----------------------------------------------------------------------

c     //   read 3rd-5th lines: comment
      read ( iounit, *, iostat=ierr ) box(1,1), box(2,1), box(3,1)
      read ( iounit, *, iostat=ierr ) box(1,2), box(2,2), box(3,2)
      read ( iounit, *, iostat=ierr ) box(1,3), box(2,3), box(3,3)

c     //   if sx is negative, abs(sx) is the volume
      if ( sx .lt. 0.d0 ) then
         sx  = ( abs(sx) / det3(box) )**(1.d0/3.d0)
         sy  = sx
         sz  = sx
      end if

c     //   scale box
      box(1,1:3) = box(1,1:3) * sx
      box(2,1:3) = box(2,1:3) * sy
      box(3,1:3) = box(3,1:3) * sz

c-----------------------------------------------------------------------
c     //   6th line
c-----------------------------------------------------------------------

c     //   read 6th line: atomic species
      read ( iounit, '(a)', iostat=ierr ) char_line

c     //   count number of species
      nspec = 0
      i = 1
      do while ( i .le. len_trim(char_line) )
         if ( char_line(i:i) .ne. ' ' ) then
            nspec = nspec + 1
            do while ( ( i .le. len_trim(char_line) ) .and.
     &                 ( char_line(i:i) .ne. ' ' ) )
               i = i + 1
            end do
         else
            i = i + 1
         end if
      end do

c     //   memory allocation
      allocate ( spec(nspec) )
      allocate ( natom_spec(nspec) )

c     //   species
      read ( char_line, * ) spec(1:nspec)

c-----------------------------------------------------------------------
c     //   7th line
c-----------------------------------------------------------------------

c     //   read 7th line: number of atoms per species
      read ( iounit, *, iostat=ierr ) natom_spec(1:nspec)

c     //   total number of atoms
      natom = 0
      do i = 1, nspec
         natom = natom + natom_spec(i)
      end do

c     //   memory allocation
      allocate ( species(natom) )
      allocate ( x(natom) )
      allocate ( y(natom) )
      allocate ( z(natom) )

c     //   atomic species
      k = 0
      do i = 1, nspec
      do j = 1, natom_spec(i)
         k = k + 1
         species(k) = spec(i)
      end do
      end do

c-----------------------------------------------------------------------
c     //   8th line and beyond
c-----------------------------------------------------------------------

c     //   atom counter
      j = 0

c     //   loop of lines
      do

c        //   read 6th line: atomic species
         read ( iounit, '(a)', iostat=ierr ) char_line

c        //   exit on error
         if ( ierr .ne. 0 ) exit

c        //   change letters to upper case
         do i = 1, len_trim(char_line)
            if ( ( char_line(i:i) .ge. 'a' ) .and.
     &           ( char_line(i:i) .le. 'z' ) ) then
               char_line(i:i) = char( ichar(char_line(i:i)) - 32 )
            end if
         end do

c        //   first word
         read ( char_line, *, iostat=ierr ) char_word(1)

c        //   is "selective"
         if      ( char_word(1)(1:9) .eq. 'SELECTIVE' ) then
            cycle
c        //   is "cartesian"
         else if ( char_word(1)(1:9) .eq. 'CARTESIAN' ) then
            ioption = 1
            cycle
c        //   is "direct"
         else if ( char_word(1)(1:6) .eq. 'DIRECT'    ) then
            ioption = 2
            cycle
         end if

c        //   atom counter
         j = j + 1

c        //   read atom
         if ( ioption .eq. 1 ) then

            read( char_line, *, iostat=ierr ) x(j), y(j), z(j)

            x(j) = x(j) * sx
            y(j) = y(j) * sy
            z(j) = z(j) * sz

         else if ( ioption .eq. 2 ) then

            read( char_line, *, iostat=ierr ) s(1), s(2), s(3)

            x(j) = box(1,1)*s(1) + box(1,2)*s(2) + box(1,3)*s(3)
            y(j) = box(2,1)*s(1) + box(2,2)*s(2) + box(2,3)*s(3)
            z(j) = box(3,1)*s(1) + box(3,2)*s(2) + box(3,3)*s(3)

         end if

c        //   all atoms read
         if ( j .eq. natom ) exit

c     //   loop of lines
      end do

c     //   open poscar file
      close( iounit )

c-----------------------------------------------------------------------
c     //   output
c-----------------------------------------------------------------------

      open ( iounit, file = trim(inpfile) )

      write( iounit, '(a)' ) '<method>'
      write( iounit, '(a)' ) 'STATIC'
      write( iounit, '(a)' )
      write( iounit, '(a)' ) '<iboundary>'
      write( iounit, '(a)' ) 'ANGSTROM'
      write( iounit, '(4x,3f12.6)' )
     &   box(1,1)*na, box(1,2)*nb, box(1,3)*nc
      write( iounit, '(4x,3f12.6)' )
     &   box(2,1)*na, box(2,2)*nb, box(2,3)*nc
      write( iounit, '(4x,3f12.6)' )
     &   box(3,1)*na, box(3,2)*nb, box(3,3)*nc
      write( iounit, '(a)' )
      write( iounit, '(a)' ) '<ipotential>'
      write( iounit, '(a)' ) 'MM'

      close( iounit )

c-----------------------------------------------------------------------
c     //   xyz file
c-----------------------------------------------------------------------

      open ( iounit, file = trim(xyzfile) )

      write( iounit, '(i8)' ) natom*na*nb*nc
      write( iounit, '(a)'  ) 'ANGSTROM'

      do la = 0, na-1
      do lb = 0, nb-1
      do lc = 0, nc-1

         k = 0

         do i = 1, nspec
         do j = 1, natom_spec(i)

            k = k + 1

            xl = x(k) + box(1,1)*la + box(1,2)*lb + box(1,3)*lc
            yl = y(k) + box(2,1)*la + box(2,2)*lb + box(2,3)*lc
            zl = z(k) + box(3,1)*la + box(3,2)*lb + box(3,3)*lc

            if ( option(1:1) .eq. 'Y' ) then
               write( iounit, '(a4,3f12.6,i3)' )
     &            species(k), xl, yl, zl, i
            else
               write( iounit, '(a4,3f12.6)' )
     &            species(k), xl, yl, zl
            end if

         end do
         end do

      end do
      end do
      end do

      close( iounit )

      write( 6, '(a)' ) '<iboundary>'
      write( 6, '(a)' ) 'ANGSTROM'
      write( 6, '(3f12.6)' ) 
     &   box(1,1)*na, box(1,2)*nb, box(1,3)*nc
      write( 6, '(3f12.6)' ) 
     &   box(2,1)*na, box(2,2)*nb, box(2,3)*nc
      write( 6, '(3f12.6)' ) 
     &   box(3,1)*na, box(3,2)*nb, box(3,3)*nc
      write( 6, '(a)' ) 

      write( 6, '(a)' ) 'Normal termination of poscar2xyz.'
      write( 6, '(a)' ) 

c-----------------------------------------------------------------------
c     //   end of program
c-----------------------------------------------------------------------

      stop
      end





c***********************************************************************
      subroutine inv3 ( a, ainv )
c***********************************************************************
c-----------------------------------------------------------------------
c     /*   local variables                                            */
c-----------------------------------------------------------------------

      implicit none

      integer :: i, j

      real(8) :: a(3,3), ainv(3,3), det3, deta

c-----------------------------------------------------------------------
c     /*   start                                                      */
c-----------------------------------------------------------------------

      ainv(1,1) = + a(2,2)*a(3,3) - a(2,3)*a(3,2)
      ainv(1,2) = + a(3,2)*a(1,3) - a(1,2)*a(3,3)
      ainv(1,3) = + a(1,2)*a(2,3) - a(2,2)*a(1,3)

      ainv(2,1) = + a(2,3)*a(3,1) - a(3,3)*a(2,1)
      ainv(2,2) = + a(3,3)*a(1,1) - a(3,1)*a(1,3)
      ainv(2,3) = + a(1,3)*a(2,1) - a(2,3)*a(1,1)

      ainv(3,1) = + a(2,1)*a(3,2) - a(3,1)*a(2,2)
      ainv(3,2) = + a(3,1)*a(1,2) - a(1,1)*a(3,2)
      ainv(3,3) = + a(1,1)*a(2,2) - a(1,2)*a(2,1)

      deta = det3 ( a )

      do j = 1, 3
      do i = 1, 3
         ainv(i,j) = ainv(i,j)/deta
      end do
      end do

      return
      end





c***********************************************************************
      real(8) function det3 ( a )
c***********************************************************************
c-----------------------------------------------------------------------
c     /*   local variables                                            */
c-----------------------------------------------------------------------

      implicit none

      real(8) :: a(3,3)

c-----------------------------------------------------------------------
c     /*   start                                                      */
c-----------------------------------------------------------------------

      det3 = + a(1,1)*a(2,2)*a(3,3) - a(1,1)*a(2,3)*a(3,2)
     &       + a(2,1)*a(3,2)*a(1,3) - a(2,1)*a(1,2)*a(3,3)
     &       + a(3,1)*a(1,2)*a(2,3) - a(3,1)*a(2,2)*a(1,3)

      return
      end
