c///////////////////////////////////////////////////////////////////////
c
c      Author:          M. Shiga
c      Last updated:    Jun 11, 2025 by M. Shiga
c      Description:     print xsf file
c
c///////////////////////////////////////////////////////////////////////
c***********************************************************************
      program npf2xsf
c***********************************************************************

c     //   local variables
      implicit none

c     //   atomic coordinates
      integer, parameter :: maxatom = 100000

c     //   atomic coordinates
      real(8), dimension(:), allocatable :: x, y, z

c     //   potential energies
      real(8) :: pot

c     //   atomic forces
      real(8), dimension(:), allocatable :: fx, fy, fz

c     //   atomic forces
      real(8), dimension(3,3) :: box

c     //   total charge
      real(8) :: charge

c     //   atomic symbols
      character(len=4), dimension(:), allocatable :: species

c     //   input and output files
      character(len=120) :: xsf, npf

c     //   characters
      character(len=120) :: char
      character(len=8)   :: char8

c     //   constants
      real(8), parameter:: au_energy  = 4.3597482d-18
      real(8), parameter:: au_charge  = 1.60217646d-19
      real(8), parameter:: au_length  = 0.529177249d-10

c     //   integers
      integer :: i, nstep, natom, iargc, ierr

c     //   real numbers
      real(8) :: const_1, const_2, const_3

c-----------------------------------------------------------------------
c     //   initial message
c-----------------------------------------------------------------------

      write( 6, '(a)' )
      write( 6, '(a)' ) 'Program npf2xsf'
      write( 6, '(a)' )

      if ( iargc() .ne. 2 ) then

         write( 6, '(a)' ) 'Usage: npf2xsf.x $1 $2'
         write( 6, '(a)' )
         write( 6, '(a)' ) '$1: input n2p2 file (trj.npf)'
         write( 6, '(a)' ) '$2: output xsf file (trj.xsf)'
         write( 6, '(a)' )

         stop

      end if

c-----------------------------------------------------------------------
c     //   memory allocation
c-----------------------------------------------------------------------

c     //   memory allocation: atomic symbols
      allocate( species(maxatom) )

c     //   memory allocation: atomic coordinates
      allocate( x(maxatom) )
      allocate( y(maxatom) )
      allocate( z(maxatom) )

c     //   memory allocation: atomic forces
      allocate( fx(maxatom) )
      allocate( fy(maxatom) )
      allocate( fz(maxatom) )

c-----------------------------------------------------------------------
c     //   read values
c-----------------------------------------------------------------------

c     //   input
      call getarg( 1, npf )

c     //   output
      call getarg( 2, xsf )

c-----------------------------------------------------------------------
c     //    constant
c-----------------------------------------------------------------------

c     /*   conversion factor   */
      const_1 = au_length * 1.d+10

c     /*   conversion factor   */
      const_2 =  au_charge / au_energy

c     /*   conversion factor   */
      const_3 =  1.d0 / const_2 / const_1

c-----------------------------------------------------------------------
c     //   list of xsf files
c-----------------------------------------------------------------------

c     //   step
      nstep = 0

c     //   number of atoms
      natom = 0

c     //   open npf file
      open ( 11, file = trim(npf) )

c     //   loop of files
      do

c        //   line 1: 'begin'
         read( 11, *, iostat=ierr ) char

c        //   look for error
         if ( ( char(1:5) .ne. 'begin' ) .or. ( ierr .ne. 0 ) ) exit

c        //   step
         nstep = nstep + 1

c        //   line 2: comment
         read( 11, *, iostat=ierr ) char

c        //   line 3-5: lattice
         read( 11, *, iostat=ierr ) char, box(1:3,1)
         read( 11, *, iostat=ierr ) char, box(1:3,2)
         read( 11, *, iostat=ierr ) char, box(1:3,3)

c        //   iatom index
         i = 0

c        //   line by line
         do

c           //   read keyword
            read( 11, *, iostat=ierr ) char

c           //   look for error
            if ( ierr .ne. 0 ) exit

            if ( char(1:4) .eq. 'atom' ) then

c              //   go back one line
               backspace( 11 )

c              //   atom index
               i = i + 1

c              //   read atomic position, species, forces
               read( 11, *, iostat=ierr )
     &            char,
     &            x(i), y(i), z(i),
     &            species(i),
     &            char, char,
     &            fx(i), fy(i), fz(i)

c           //   potential energy and charge
            else if ( char(1:6) .eq. 'energy' ) then

c              //   go back one line
               backspace( 11 )

c              //   read potential energy
               read( 11, *, iostat=ierr ) char, pot

c              //   read charge
               read( 11, *, iostat=ierr ) char, charge

c           //   detect end
            else if ( char(1:3) .eq. 'end' ) then

c              //   exit loop
               exit

c           //   end of if statement
            end if

c        //   line by line
         end do

c        //   look for error
         if ( ierr .ne. 0 ) exit

c        //   number of atoms
         natom = i

c        //   messeage
         write( 6, '(a,i8,a,i8)' )
     &      "Structure:", nstep, " Atoms:", natom

c        /*   geometry in angstrom   */
         x(:) = x(:) * const_1
         y(:) = y(:) * const_1
         z(:) = z(:) * const_1

c        //   lattice vectors in angstrom   */
         box(:,:) = box(:,:) * const_1

c        //   potential in eV
         pot = pot / const_2

c        /*   force in eV/angstrom   */
         fx(:) = fx(:) * const_3
         fy(:) = fy(:) * const_3
         fz(:) = fz(:) * const_3

c        //   number to character
         call int8_to_char( nstep, char8 )

c        //   open xsf file
         open ( 10, file = trim(xsf) // '.' // char8 )

c        //   potential energy in eV
         write( 10, '(a,f16.8,a)' ) '# total energy =', pot, ' eV'

c        //   read a line
         write( 10, '(a)', iostat=ierr )

c        //   read a line
         write( 10, '(a)', iostat=ierr ) 'CRYSTAL'

c        //   read a line
         write( 10, '(a)', iostat=ierr ) 'PRIMVEC'

c        /*   write three lines   */
         write( 10, '(3f16.8)' ) box(1,1), box(2,1), box(3,1)
         write( 10, '(3f16.8)' ) box(1,2), box(2,2), box(3,2)
         write( 10, '(3f16.8)' ) box(1,3), box(2,3), box(3,3)

c        //   read two lines
         write( 10, '(a)' ) 'PRIMCOORD'
         write( 10, '(2i8)' ) natom, 1

c        /*   loop of atoms   */
         do i = 1, natom

c           /*   read one line   */
            write( 10, '(a5,6f16.8)' ) species(i),
     &         x(i), y(i), z(i), fx(i), fy(i), fz(i)

c        /*   loop of atoms   */
         end do

c        //   close xsf file
         close( 10 )

c     //   loop of files
      end do

c     //   close npf file
      close( 11 )

c-----------------------------------------------------------------------
c     //   final message
c-----------------------------------------------------------------------

      write( 6, '(a)' )
      write( 6, '(a)' ) 'Normal termination of npf2xsf.'
      write( 6, '(a)' )

      stop
      end





!***********************************************************************
      subroutine int8_to_char( num, char_num )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: j, num

      character(len=8) :: char_num

!-----------------------------------------------------------------------
!     /*   integer 7   ->   character '007'                           */
!-----------------------------------------------------------------------

      char_num = '   '

      write( char_num, '(i8)' ) num

      do j = 1, 8
          if ( char_num( j:j ) .eq. ' ' ) char_num( j:j ) = '0'
      end do

      return
      end
