c***********************************************************************
      program msd
c***********************************************************************
c-----------------------------------------------------------------------
c     //   variables
c-----------------------------------------------------------------------

c     //   local variables
      implicit none

c     //   number of beads
      integer :: nbead

c     //   number of steps
      integer :: nstep

c     //   number of atoms
      integer :: natom

c     //   file numbers
      integer :: iounit     = 10
      integer :: iounit_xyz = 11
      integer :: iounit_msd = 12

c     //   atomic element
      character(len=3), dimension(:), allocatable :: symbol

c     //   atomic coordinates
      real(8), dimension(:,:), allocatable :: x, y, z

c     //   centroid coordinates
      real(8), dimension(:,:), allocatable :: xc, yc, zc

c     //   centroid coordinates
      real(8), dimension(:), allocatable :: ux, uy, uz

c     //   mean square displacement
      real(8), dimension(:), allocatable :: d2

c     //   atom kind
      integer :: ikind

c     //   atom kind
      integer, dimension(:), allocatable :: kind

c     //   number of atoms of the atom kind
      integer :: natom_kind

c     //   characters
      character(len=80) :: charline
      character(len=3)  :: char

c     //   print interval
      integer :: iprint_xyz

c     //   step size
      real(8) :: dt

c     //   time interval
      real(8) :: dt_xyz

c     //   integers
      integer :: i, j, k, istep, jstep, ierr

c     //   real numbers
      real(8) :: dx, dy, dz, time_xyz, dcoeff, a8, b8

c-----------------------------------------------------------------------
c     //   initial message
c-----------------------------------------------------------------------

      write( 6, '(a)' )
      write( 6, '(a)' ) 'Program msd'
      write( 6, '(a)' )

      if ( iargc() .ne. 1 ) then

         write( 6, '(a)' ) 'Usage: msd.x $1'
         write( 6, '(a)' )
         write( 6, '(a)' ) '$1: atom kind'
         write( 6, '(a)' )

         stop

      end if

c-----------------------------------------------------------------------
c     //   read values
c-----------------------------------------------------------------------

c     //   atom kind
      call getarg( 1, charline )
      read( charline, * ) ikind

      write( 6, '(a,i4)' ) 'Atom kind: ', ikind

c-----------------------------------------------------------------------
c     //   number of atoms
c-----------------------------------------------------------------------

      open ( iounit, file = 'structure.dat' )
      read ( iounit, * ) natom
      close( iounit )

      write( 6, '(a,i4)' ) 'Number of atom: ', natom

c-----------------------------------------------------------------------
c     //   number of beads
c-----------------------------------------------------------------------

      open ( iounit, file = 'input.dat' )

      nbead = 1

      do

          read( iounit, *, iostat=ierr ) charline

          if ( ierr .ne. 0 ) exit

          if ( charline(1:7) .eq. '<nbead>' ) then
             read ( iounit, * ) nbead
             exit
          end if

      end do

      close( iounit )

      write( 6, '(a,i4)' ) 'Number of beads: ', nbead

c-----------------------------------------------------------------------
c     //   memory allocation
c-----------------------------------------------------------------------

      allocate( kind(natom) )

      allocate( x(natom,nbead) )
      allocate( y(natom,nbead) )
      allocate( z(natom,nbead) )

      allocate( ux(natom) )
      allocate( uy(natom) )
      allocate( uz(natom) )

      allocate( symbol(natom) )

c-----------------------------------------------------------------------
c     //   number of atoms of the kind
c-----------------------------------------------------------------------

      open ( iounit, file = 'structure.dat' )

      read ( iounit, *, iostat=ierr )
      read ( iounit, *, iostat=ierr )

      k = 0

      do i = 1, natom

         read ( iounit, *, iostat=ierr ) symbol(i), dx, dy, dz, kind(i)

         if ( ierr .ne. 0 ) then
            read ( iounit, *, iostat=ierr ) symbol(i), dx, dy, dz
            kind(i) = 1
         end if

         if ( ierr .ne. 0 ) then
            write( 6, '(a)' ) 'Error - reading structure.dat.'
         end if

         if ( kind(i) .eq. ikind ) then
            k = k + 1
         end if

      end do

      close( iounit )

      natom_kind = k

      write( 6, '(a,i4)' ) 'Number of atoms of the kind: ', natom_kind

      if ( natom_kind .eq. 0 ) stop

c-----------------------------------------------------------------------
c     //   print interval
c-----------------------------------------------------------------------

      open ( iounit, file = 'input.dat' )

      iprint_xyz = 10

      do

          read( iounit, *, iostat=ierr ) charline

          if ( ierr .ne. 0 ) exit

          if ( charline(1:12) .eq. '<iprint_xyz>' ) then
             read ( iounit, *, iostat=ierr ) iprint_xyz
             exit
          end if

      end do

      close( iounit )

      write( 6, '(a,i4)' ) 'Print interval of trj.xyz: ', iprint_xyz

c-----------------------------------------------------------------------
c     //   step size
c-----------------------------------------------------------------------

      open ( iounit, file = 'input.dat' )

      do

          read( iounit, *, iostat=ierr ) charline

          if ( ierr .ne. 0 ) dt = 0.25d0

          if ( charline(1:12) .eq. '<dt>' ) then
             read ( iounit, *, iostat=ierr ) dt
             exit
          end if

      end do

      close( iounit )

      write( 6, '(a,f6.2)' ) 'Step size: ', dt

c-----------------------------------------------------------------------
c     //   time interval
c-----------------------------------------------------------------------

      dt_xyz = dt * dble(iprint_xyz) * 0.001d0

c-----------------------------------------------------------------------
c     //   number of structures
c-----------------------------------------------------------------------

      open ( iounit, file = 'trj.xyz' )

      istep = 0

      do

         read ( iounit, *, iostat=ierr )
         read ( iounit, *, iostat=ierr )

         do j = 1, nbead
         do i = 1, natom
            read ( iounit, *, iostat=ierr )
     &         char, dx, dy, dz, dx, dy, dz
         end do
         end do

         if ( ierr .ne. 0 ) exit

         istep = istep + 1

      end do

      nstep = istep

      close( iounit )

      write( 6, '(a,i8)' ) 'Number of structures: ', nstep

c-----------------------------------------------------------------------
c     //   memory allocation
c-----------------------------------------------------------------------

      allocate( xc(natom_kind,nstep) )
      allocate( yc(natom_kind,nstep) )
      allocate( zc(natom_kind,nstep) )

      allocate( d2(0:nstep-1) )

c-----------------------------------------------------------------------
c     //   centroid trajectory
c-----------------------------------------------------------------------

      open ( iounit, file = 'trj.xyz' )
      open ( iounit_xyz, file = 'centroid.xyz' )

      do istep = 1, nstep

         read ( iounit, *, iostat=ierr )
         read ( iounit, *, iostat=ierr )

         do j = 1, nbead
         do i = 1, natom
            read ( iounit, *, iostat=ierr )
     &         char, dx, dy, dz, x(i,j), y(i,j), z(i,j)
         end do
         end do

         if ( ierr .ne. 0 ) then
            write( 6, '(a,i8)' )
     &         'Error - read trj.xyz incorrectly at step:', istep
            stop
         end if

         do i = 1, natom
            ux(i) = 0.d0
            uy(i) = 0.d0
            uz(i) = 0.d0
            do j = 1, nbead
               ux(i) = ux(i) + x(i,j) / dble(nbead)
               uy(i) = uy(i) + y(i,j) / dble(nbead)
               uz(i) = uz(i) + z(i,j) / dble(nbead)
            end do
         end do

         write( iounit_xyz, '(i8)' ) natom
         write( iounit_xyz, '(i8)' ) istep

         do i = 1, natom
            write( iounit_xyz, '(a,3f10.3)' )
     &         symbol(i), ux(i), uy(i), uz(i)
         end do

         j = 0

         do i = 1, natom

            if ( kind(i) .ne. ikind ) cycle

            j = j + 1

            xc(j,istep) = ux(i)
            yc(j,istep) = uy(i)
            zc(j,istep) = uz(i)

         end do

      end do

      close( iounit )
      close( iounit_xyz )

c-----------------------------------------------------------------------
c     //   mean square displacement
c-----------------------------------------------------------------------

      a8 = 0.d0
      b8 = dble(nstep) * dble(nstep+1) / 2.d0

      d2(:) = 0.d0

      do istep = 1, nstep

         if ( mod(istep,max(1,nstep/10)) .eq. 0 ) then
            write( 6, '(a,i5,a)' ) 'Done: ', int((a8/b8)*100), ' %'
         end if

         do jstep = istep, nstep

            i = jstep - istep

            do j = 1, natom_kind

               dx = xc(j,jstep) - xc(j,istep)
               dy = yc(j,jstep) - yc(j,istep)
               dz = zc(j,jstep) - zc(j,istep)

               d2(i) = d2(i) + dx*dx + dy*dy + dz*dz

            end do

            a8 = a8 + 1.d0

         end do

      end do

      do i = 0, nstep-1
         d2(i) = d2(i) / dble(nstep-i) / dble(natom_kind)
      end do

c-----------------------------------------------------------------------
c     //   print mean square displacement
c-----------------------------------------------------------------------

c     //   open file
      open ( iounit_msd, file = 'msd.out' )

c     //   loop of steps
      do i = 1, nstep/2

c        //   time
         time_xyz = dble(i) * dt_xyz

c        //   an estimate of diffusion coefficient
         dcoeff = d2(i) / ( time_xyz * 6.d0 )

c        //   print
         write( iounit_msd, '(f12.4,2f12.4)' )
     &      time_xyz, d2(i), dcoeff

c     //   loop of steps
      end do

c     //   close file
      close( iounit_msd )

      stop
      end
