c=======================================================================
c
c     References:
c
c        Lin, Blanco, Goddard III, J. Chem. Phys. 119, 11792 (2003).
c        Witt, et al, J. Chem. Phys. 130, 194510 (2009).
c
c=======================================================================
c***********************************************************************
      program irspectra
c***********************************************************************
c-----------------------------------------------------------------------

c     //   variables
      implicit none

c     //   step number
      integer :: istep

c     //   number of steps
      integer :: nstep

c     //   number of beads
      integer :: nbead

c     //   number of frequencies
      integer :: nomega

c     //   dipole moment of beads
      real(8), dimension(:), allocatable :: dx, dy, dz

c     //   centroid dipole moment
      real(8) :: dipx, dipy, dipz

c     //   average dipole moment
      real(8) :: dx_avg, dy_avg, dz_avg
      real(8) :: dx_ini, dy_ini, dz_ini
      real(8) :: dx_fin, dy_fin, dz_fin

c     //   complex numbers
      complex(8) :: zi, zexp

c     //   complex numbers
      complex(8), dimension(:), allocatable :: zxsum, zysum, zzsum

c     //   step size
      real(8) :: dt_fs, dt

c     //   time
      real(8) :: t

c     //   circular constant
      real(8) :: pi

c     //   frequency
      real(8) :: omega

c     //   intensity I_K(omega)
      real(8), dimension(:), allocatable :: f_kubo

c     //   intensity I(omega)
      real(8), dimension(:), allocatable :: f_acf

c     //   intensity alpha(omega)
      real(8), dimension(:), allocatable :: f_alpha

c     //   intensity alpha(omega) smeared
      real(8), dimension(:), allocatable :: f_smear

c     //   temperature
      real(8) :: temperature

c     //   characters
      character(len=80) :: char

c     //   filename
      character(len=80) :: filename

c     //   unit conversion factors
      real(8), parameter :: au_time = 0.024188843d-15
      real(8), parameter :: au_length = 0.529177249d-10

c     //   Boltzmann constant
      real(8), parameter :: boltz = 0.316682968d-5

c     //   Planck constant divided by 2 pi
      real(8), parameter :: hbar = 1.d0

c     /*   speed of light in SI unit   */
      real(8), parameter :: speedlight_SI = 2.99892458d+8

c     //   speed of light
      real(8) :: speedlight

c     //   integers
      integer :: i, j, k, l, ierr, iargc, ioption

c     //   real numbers
      real(8) :: beta, bho, factor_cminv, vn, d, dstep

c     //   smearing parameter
      real(8) :: domega

c     //   window function
      real(8), dimension(:), allocatable :: w

c-----------------------------------------------------------------------
c     //   initial message
c-----------------------------------------------------------------------

      if ( iargc() .ne. 7 ) then

         write( 6, '(a)' )
         write( 6, '(a)' ) 'Program irspectra'
         write( 6, '(a)' )
         write( 6, '(a)' ) 'Usage: irspectra.x $1 $2 $3 $4 $5 $6'
         write( 6, '(a)' )
         write( 6, '(a)' ) '$1: dipole trajectory (dipole.out)'
         write( 6, '(a)' ) '$2: 0 (dipole) or 1 (dipole derivative)'
         write( 6, '(a)' ) '$3: number of beads'
         write( 6, '(a)' ) '$4: step interval [fs]'
         write( 6, '(a)' ) '$5: temperature [K]'
         write( 6, '(a)' ) '$6: highest frequency [cm**-1]'
         write( 6, '(a)' ) '$7: smearing parameter [cm**-1]'
         write( 6, '(a)' )
         write( 6, '(a)' ) 'Example:'
         write( 6, '(a)' ) 'irspectra.x dipole.out 0 16 0.25 300.0' //
     &                     ' 5000.0 10.0'
         write( 6, '(a)' )

         stop

      end if

c-----------------------------------------------------------------------
c     //   read values
c-----------------------------------------------------------------------

c     //   input filename
      call getarg( 1, filename )

c     //   0 (dipole) or 1 (dipole derivative)
      call getarg( 2, char )
      read( char, * ) ioption

c     //   number of beads
      call getarg( 3, char )
      read( char, * ) nbead

c     //   step size in femtoseconds
      call getarg( 4, char )
      read( char, * ) dt_fs

c     //   temperature in kelvin
      call getarg( 5, char )
      read( char, * ) temperature

c     //   highest frequency
      call getarg( 6, char )
      read( char, * ) omega
      nomega = nint(omega)

c     //   highest frequency
      call getarg( 7, char )
      read( char, * ) domega

c     //   domega must be greater than 0.00001
      domega = max(domega,0.00001d0)

c-----------------------------------------------------------------------
c     //   constants
c-----------------------------------------------------------------------

c     //   circular constant
      pi = acos( -1.d0 )

c     //   imaginary number
      zi = dcmplx( 0.d0, 1.d0 )

c     //   step size in au
      dt = dt_fs / au_time * 1.d-15

c     //   inverse temperature
      beta = 1.d0 / ( boltz * temperature )

c     //   constant
      factor_cminv = 1.d0 / (2.d0*pi*speedlight_SI*au_time*100.d0)

c     //   speed of light
      speedlight = speedlight_SI / ( au_length / au_time )

c-----------------------------------------------------------------------
c     //   memory allocation
c-----------------------------------------------------------------------

      allocate( dx(nbead) )
      allocate( dy(nbead) )
      allocate( dz(nbead) )

      allocate( zxsum(nomega) )
      allocate( zysum(nomega) )
      allocate( zzsum(nomega) )

      allocate( f_kubo(nomega) )
      allocate( f_acf(nomega) )
      allocate( f_alpha(nomega) )
      allocate( f_smear(nomega) )

c-----------------------------------------------------------------------
c     //   read dipole trajectory, compute average dipoles
c-----------------------------------------------------------------------

      open ( 10, file = trim(filename) )

      istep = 0

      do

         if ( ioption .eq. 0 ) then
            do j = 1, nbead
               read( 10, *, iostat=ierr )
     &            i, dx(j), dy(j), dz(j)
            end do
         else
            do j = 1, nbead
               read( 10, *, iostat=ierr )
     &            i, d, d, d, dx(j), dy(j), dz(j)
            end do
         end if

         if ( ierr .ne. 0 ) exit

         istep = istep + 1

      end do

      nstep = istep

      close( 10 )

c-----------------------------------------------------------------------
c     //   dipole average
c-----------------------------------------------------------------------

      dx_ini = 0.d0
      dy_ini = 0.d0
      dz_ini = 0.d0

      dx_fin = 0.d0
      dy_fin = 0.d0
      dz_fin = 0.d0

      open ( 10, file = trim(filename) )

      do istep = 1, nstep

         if ( ioption .eq. 0 ) then
            do j = 1, nbead
               read( 10, *, iostat=ierr )
     &            i, dx(j), dy(j), dz(j)
            end do
         else
            do j = 1, nbead
               read( 10, *, iostat=ierr )
     &            i, d, d, d, dx(j), dy(j), dz(j)
            end do
         end if

         dx_avg = 0.d0
         dy_avg = 0.d0
         dz_avg = 0.d0

         do j = 1, nbead
            dx_avg = dx_avg + dx(j) / dble(nbead)
            dy_avg = dy_avg + dy(j) / dble(nbead)
            dz_avg = dz_avg + dz(j) / dble(nbead)
         end do

         if ( istep .eq. 1 ) then
            dx_ini = dx_avg
            dy_ini = dy_avg
            dz_ini = dz_avg
         else
            dx_fin = dx_avg
            dy_fin = dy_avg
            dz_fin = dz_avg
         end if

      end do

      close( 10 )

c-----------------------------------------------------------------------
c     //   window function
c-----------------------------------------------------------------------

      allocate( w(nstep) )

      do istep = 1, nstep
ccc         w(istep) = 0.5d0 * cos(2*pi*dble(istep-1)/dble(nstep-1))
         w(istep) = 1.d0
      end do

c-----------------------------------------------------------------------
c     //   read dipole trajectory, compute spectral density
c-----------------------------------------------------------------------

      open ( 10, file = trim(filename) )

      zxsum(:) = dcmplx( 0.d0, 0.d0 )
      zysum(:) = dcmplx( 0.d0, 0.d0 )
      zzsum(:) = dcmplx( 0.d0, 0.d0 )

      do istep = 1, nstep

         t = dble(istep-1) * dt

         if ( ioption .eq. 0 ) then
            do j = 1, nbead
               read( 10, *, iostat=ierr )
     &            i, dx(j), dy(j), dz(j)
            end do
         else
            do j = 1, nbead
               read( 10, *, iostat=ierr )
     &            i, d, d, d, dx(j), dy(j), dz(j)
            end do
         end if

         dipx = 0.d0
         dipy = 0.d0
         dipz = 0.d0

         do j = 1, nbead
            dipx = dipx + dx(j) / dble(nbead)
            dipy = dipy + dy(j) / dble(nbead)
            dipz = dipz + dz(j) / dble(nbead)
         end do

         dstep = dble(istep-1) / dble(nstep-1)

         dx_avg = dx_ini + dstep * ( dx_fin - dx_ini )
         dy_avg = dy_ini + dstep * ( dy_fin - dy_ini )
         dz_avg = dz_ini + dstep * ( dz_fin - dz_ini )

         dipx = dipx - dx_avg
         dipy = dipy - dy_avg
         dipz = dipz - dz_avg

         do k = 1, nomega

            omega = dble(k) / factor_cminv

            zexp = exp( - zi * omega * t )

            zxsum(k) = zxsum(k) + dipx * zexp * w(istep) * dt
            zysum(k) = zysum(k) + dipy * zexp * w(istep) * dt
            zzsum(k) = zzsum(k) + dipz * zexp * w(istep) * dt

         end do

      end do

      close( 10 )

c-----------------------------------------------------------------------
c     //   compute spectra
c-----------------------------------------------------------------------

c     //   loop of frequencies
      do k = 1, nomega

c        //   frequency in au
         omega = dble(k) / factor_cminv

c        //   factor
         bho = beta * hbar * omega

c        //   spectral density of kubo-transformed autocorrelation
         f_kubo(k) = ( dreal( dconjg(zxsum(k)) * zxsum(k) )
     &               + dreal( dconjg(zysum(k)) * zysum(k) )
     &               + dreal( dconjg(zzsum(k)) * zzsum(k) ) )
     &             / ( 2.d0 * pi ) / ( 2.d0 * pi )

c        //   spectral density of normal autocorrelation
         f_acf(k)   = f_kubo(k) * bho / ( 1.d0 - exp(-bho) )

c        //   volume times refraction cofficient assumed unity
         vn = 1.d0

c        //   infrared absorption coefficient
         if ( ioption .eq. 0 ) then
            f_alpha(k) = f_acf(k) * ( 1.d0 - exp(-bho) )
     &                 * ( 4.d0 * pi * pi * omega )
     &                 / ( 3.d0 * hbar * speedlight * vn )
         else
            f_alpha(k) = f_acf(k) * ( 1.d0 - exp(-bho) )
     &                 * ( 4.d0 * pi * pi * omega )
     &                 / ( 3.d0 * hbar * speedlight * vn )
     &                 / ( omega * omega )
         end if

c     //   loop of frequencies
      end do

c-----------------------------------------------------------------------
c     //   print spectra
c-----------------------------------------------------------------------

c     //   open file
      open ( 11, file = 'irspectra.out' )

c     //   header
      write( 6, '(a)' )
     &   '----------------------------' //
     &   '------------------------------------------------'
      write( 6, '(a)' )
     &   ' freq [cm-1]        I_k [au]' //
     &   '          I [au]      alpha [au]   smeared_alpha'
      write( 6, '(a)' )
     &   '----------------------------' //
     &   '------------------------------------------------'
      write( 11, '(a)' )
     &   '----------------------------' //
     &   '------------------------------------------------'
      write( 11, '(a)' )
     &   ' freq [cm-1]        I_k [au]' //
     &   '          I [au]      alpha [au]   smeared_alpha'
      write( 11, '(a)' )
     &   '----------------------------' //
     &   '------------------------------------------------'

c     //   loop of frequencies
      do k = 1, nomega

         f_smear(k) = 0.d0

         do l = 1, nomega

            d = abs( dble(k-l) )

            d = 0.5d0 * d * d / ( domega * domega )

            f_smear(k) = f_smear(k) + f_alpha(l) * exp( - d )

         end do

         f_smear(k) = f_smear(k)
     &      / sqrt ( pi * ( domega * domega ) / 0.5d0 )

c        //   print data
         write( 6, '(f12.1,f16.2,f16.2,2f16.6)' )
     &      dble(k), f_kubo(k), f_acf(k), f_alpha(k), f_smear(k)
         write( 11, '(f12.1,f16.2,f16.2,2f16.6)' )
     &      dble(k), f_kubo(k), f_acf(k), f_alpha(k), f_smear(k)

c     //   loop of frequencies
      end do

c     //   close file
      close( 11 )

c-----------------------------------------------------------------------
c     //   end of code
c-----------------------------------------------------------------------

      stop
      end
