c///////////////////////////////////////////////////////////////////////
c
c      Author:          M. Shiga
c      Last updated:    Feb 16, 2020 by M. Shiga
c      Description:     detect reaction from hills.ini
c
c///////////////////////////////////////////////////////////////////////
c***********************************************************************
      program hills2detect
c***********************************************************************

c     //   clear variables
      implicit none

c     //   integers
      integer :: k, l, ierr, iflag

c     //   real numbers
      real(8) :: gh, sigma_x, sigma_y, sigma_z, xi, yi, zi, xj, yj, zj

c     //   real numbers
      real(8) :: rx, ry, rz, rwx, rwy, rwz, rw2

c     //   real numbers
      real(8) :: rw2cut = 1.5d0

c     //   boundary condition: 0 (free) or 1 (periodic)
      integer :: ipbc_x = 0
      integer :: ipbc_y = 0
      integer :: ipbc_z = 0

c     //   character
      character(len=24) :: cv1, cv2, cv3, cv4, cv5, cv6

c-----------------------------------------------------------------------
c     //   usage
c-----------------------------------------------------------------------

c     //   four arguments
      if ( iargc() .ne. 6 ) then

c        //   print message
         write( 6, * ) 
         write( 6, * ) 'CODE:        hills2detect.x'
         write( 6, * ) 
         write( 6, * ) 'INPUT FILE:  hills.ini'
         write( 6, * ) 'STD OUTPUT:  detection of hill closeby'
         write( 6, * ) 
         write( 6, * ) 'USAGE:       hills2detect.x $1 $2 $3 $4 $5 $6'
         write( 6, * ) '$1:          cv type of x.'
         write( 6, * ) '$2:          cv type of y.'
         write( 6, * ) '$3:          cv type of z.'
         write( 6, * ) '$4:          cv position of x.'
         write( 6, * ) '$5:          cv position of y.'
         write( 6, * ) '$6:          cv position of z.'
         write( 6, * ) 

c        //   error termination
         stop

c     //   four arguments
      end if

c-----------------------------------------------------------------------
c     //   read from command line
c-----------------------------------------------------------------------

      call getarg( 1, cv1 )
      call getarg( 2, cv2 )
      call getarg( 3, cv3 )

      call getarg( 4, cv4 )
      call getarg( 5, cv5 )
      call getarg( 6, cv6 )

      read ( cv4, * ) xj
      read ( cv5, * ) yj
      read ( cv6, * ) zj

c-----------------------------------------------------------------------
c     //   boundary condition
c-----------------------------------------------------------------------

c     //   free boundary
      if ( cv1(1:5) .ne. 'DIH  ' ) ipbc_x = 0
      if ( cv2(1:5) .ne. 'DIH  ' ) ipbc_y = 0
      if ( cv3(1:5) .ne. 'DIH  ' ) ipbc_z = 0

c        //   periodic boundary
      if ( cv1(1:5) .eq. 'DIH  ' ) ipbc_x = 1
      if ( cv2(1:5) .eq. 'DIH  ' ) ipbc_y = 1
      if ( cv2(1:5) .eq. 'DIH  ' ) ipbc_z = 1

c-----------------------------------------------------------------------
c     //   read hills
c-----------------------------------------------------------------------

c     //   comment
      write( 6, '(a)' )

c     //   comment
      write( 6, '(a)' ) 'In hills.ini:'

c     //   flag
      iflag = 0

c     //   open file
      open ( 10, file = 'hills.ini' )

c     //   line counter
      l = 0

c     //   loop of hills
      do

c        //   read a hill
         read( 10, *, iostat=ierr ) k, gh, sigma_x, xi
         read( 10, *, iostat=ierr ) k, gh, sigma_y, yi
         read( 10, *, iostat=ierr ) k, gh, sigma_z, zi

         if ( ierr .ne. 0 ) exit

c        //   line counter
         l = l + 3

c        //   difference in two cvs
         rx = xi - xj
         ry = yi - yj
         rz = zi - zj

c        //   periodic boundary
         rx = rx - ipbc_x * nint( rx / 360.d0 ) * 360.d0
         ry = ry - ipbc_y * nint( ry / 360.d0 ) * 360.d0
         rz = rz - ipbc_z * nint( rz / 360.d0 ) * 360.d0

c        //   scaled by gaussian width
         rwx = rx / sigma_x
         rwy = ry / sigma_y
         rwz = rz / sigma_z

c        //   scaled difference squared
         rw2 = rwx*rwx + rwy*rwy + rwz*rwz

c        //   detected
         if ( rw2 .lt. rw2cut ) then

            write( 6, '(a,i8,a,i8,a,i8,a,3f9.3)' )
     &         'lines ', l-2, ' - ', l, ' step ', k,
     &         ' position ', xi, yi, zi

c        //   flag
         iflag = iflag + 1

c        //   detected
         end if

c     //   loop of hills
      end do

c     //   close file
      close( 10 )

c     //   comment
      if ( iflag .eq. 0 ) then
         write( 6, '(a)' ) 'No hills found nearby the cv.'
      end if

c     //   comment
      write( 6, '(a)' )

      stop
      end
