c///////////////////////////////////////////////////////////////////////
c
c      Author:          M. Shiga
c      Last updated:    Feb 19, 2019 by M. Shiga
c      Description:     make cube file from hills.ini
c
c///////////////////////////////////////////////////////////////////////
c***********************************************************************
      program hills2cube
c***********************************************************************

c-----------------------------------------------------------------------
c     //   declare variables
c-----------------------------------------------------------------------

c     //   clear all variables
      implicit none

c     //   hill step
      integer, dimension(:), allocatable :: istep

c     //   hill height
      real(8), dimension(:), allocatable :: height

c     //   hill width
      real(8), dimension(:,:), allocatable :: width

c     //   hill center
      real(8), dimension(:,:), allocatable :: center

c     //   number of hills
      integer :: nhills

c     //   integers
      integer :: i, j, k, kx, ky, kz, l, ierr

c     //   input and output file numbers
      integer :: iounit = 10

c     //   lower bound of the cube
      real(8) :: x_min, y_min, z_min

c     //   upper bound of the cube
      real(8) :: x_max, y_max, z_max

c     //   mesh size of the cube
      real(8) :: x_mesh, y_mesh, z_mesh

c     //   number of meshes
      integer :: mesh_x, mesh_y, mesh_z

c     //   cut off parameter
      real(8), parameter :: cutoff = 1.d-4

c     //   cut off of exponent
      real(8) :: rw2cut

c     //   mesh point
      real(8) :: xm, ym, zm

c     //   distance between mesh and hill center
      real(8) :: rx, ry, rz

c     //   distance normalized by width
      real(8) :: rwx, rwy, rwz, rw2

c     //   hills potential
      real(8) :: vhills

c     //   height at peak position
      real(8) :: vhills_peak

c     //   peak position
      real(8) :: xm_peak, ym_peak, zm_peak

c     //   boundary condition: 0 (free) or 1 (periodic)
      integer :: ipbc_x = 0
      integer :: ipbc_y = 0
      integer :: ipbc_z = 0

c     //   maximum and minimum values of hill center
      real(8) :: cxmax, cymax, czmax, cxmin, cymin, czmin

c     //   default parameters
      real(8), dimension(8) :: cv_min
      real(8), dimension(8) :: cv_max
      real(8), dimension(8) :: cv_mesh

c     //   character
      character(len=8) :: cv1, cv2, cv3

c     //   character
      logical :: file_exists

c-----------------------------------------------------------------------
c     //   read parameters
c-----------------------------------------------------------------------

c     //   error flag
      ierr = 1

c     //   file
      inquire( file='hills2cube.dat', exist=file_exists )

c     //   read from file
      if ( file_exists ) then

c        //   open file
         open ( iounit, file='hills2cube.dat' )

c        //   input file
         read( iounit, *, iostat=ierr ) cv1, x_min, x_max, x_mesh
         read( iounit, *, iostat=ierr ) cv2, y_min, y_max, y_mesh
         read( iounit, *, iostat=ierr ) cv3, z_min, z_max, z_mesh

c        //   close file
         close( iounit )

c        //   on error, remove file and stop
         if ( ierr .ne. 0 ) then
            call system( 'rm hills2cube.dat' )
            stop
         end if

c        //   free boundary
         if ( cv1(1:5) .ne. 'DIH  ' ) ipbc_x = 0
         if ( cv2(1:5) .ne. 'DIH  ' ) ipbc_y = 0
         if ( cv3(1:5) .ne. 'DIH  ' ) ipbc_z = 0

c        //   periodic boundary
         if ( cv1(1:5) .eq. 'DIH  ' ) ipbc_x = 1
         if ( cv2(1:5) .eq. 'DIH  ' ) ipbc_y = 1
         if ( cv2(1:5) .eq. 'DIH  ' ) ipbc_z = 1

c     //   three arguments
      end if

c-----------------------------------------------------------------------
c     //   read data
c-----------------------------------------------------------------------

c     //   three arguments
      if ( ( .not. file_exists ) .and. ( iargc() .eq. 3 ) ) then

         call getarg( 1, cv1 )
         call getarg( 2, cv2 )
         call getarg( 3, cv3 )

c-----------------------------------------------------------------------
c        //   set default parameters
c-----------------------------------------------------------------------
c
c        <params_rec_meta>
c        DIST   2.0    5.0   0.1
c        ANGL   0.0  180.0   6.0
c        DIH    0.0  360.0  12.0
c        DIFF   0.0   15.0   0.5
c        CN     0.0    3.0   0.1
c        DCN   -2.0    2.0   0.1
c        XYZ    2.0    5.0   0.1
c        DXYZ   2.0    5.0   0.1

         cv_min(1)  =   2.0d0
         cv_max(1)  =   5.0d0
         cv_mesh(1) =   0.1d0

         cv_min(2)  =   0.0d0
         cv_max(2)  = 180.0d0
         cv_mesh(2) =   6.0d0

         cv_min(3)  =   0.0d0
         cv_max(3)  = 348.0d0
         cv_mesh(3) =  12.0d0

         cv_min(4)  =   0.0d0
         cv_max(4)  =  15.0d0
         cv_mesh(4) =   0.5d0

         cv_min(5)  =   0.0d0
         cv_max(5)  =   3.0d0
         cv_mesh(5) =   0.1d0

         cv_min(6)  =  -2.0d0
         cv_max(6)  =   2.0d0
         cv_mesh(6) =   0.1d0

         cv_min(7)  =   2.0d0
         cv_max(7)  =   5.0d0
         cv_mesh(7) =   0.1d0

         cv_min(8)  =   2.0d0
         cv_max(8)  =   5.0d0
         cv_mesh(8) =   0.1d0

         if ( cv1(1:5) .eq. 'DIST ' ) then
            x_min  =   cv_min(1)
            x_max  =   cv_max(1)
            x_mesh =   cv_mesh(1)
            ipbc_x =   0
         else if ( cv1(1:5) .eq. 'ANGL ' ) then
            x_min  =   cv_min(2)
            x_max  =   cv_max(2)
            x_mesh =   cv_mesh(2)
            ipbc_x =   0
         else if ( cv1(1:5) .eq. 'DIH  ' ) then
            x_min  =   cv_min(3)
            x_max  =   cv_max(3)
            x_mesh =   cv_mesh(3)
            ipbc_x =   1
         else if ( cv1(1:5) .eq. 'DIFF ' ) then
            x_min  =   cv_min(4)
            x_max  =   cv_max(4)
            x_mesh =   cv_mesh(4)
            ipbc_x =   0
         else if ( cv1(1:5) .eq. 'CN   ' ) then
            x_min  =   cv_min(5)
            x_max  =   cv_max(5)
            x_mesh =   cv_mesh(5)
            ipbc_x =   0
         else if ( cv1(1:5) .eq. 'DCN  ' ) then
            x_min  =   cv_min(6)
            x_max  =   cv_max(6)
            x_mesh =   cv_mesh(6)
            ipbc_x =   0
         else if ( cv1(1:5) .eq. 'XYZ  ' ) then
            x_min  =   cv_min(7)
            x_max  =   cv_max(7)
            x_mesh =   cv_mesh(7)
            ipbc_x =   0
         else if ( cv1(1:5) .eq. 'DXYZ ' ) then
            x_min  =   cv_min(8)
            x_max  =   cv_max(8)
            x_mesh =   cv_mesh(8)
            ipbc_x =   0
         end if

         if ( cv2(1:5) .eq. 'DIST ' ) then
            y_min  =   cv_min(1)
            y_max  =   cv_max(1)
            y_mesh =   cv_mesh(1)
            ipbc_y =   0
         else if ( cv2(1:5) .eq. 'ANGL ' ) then
            y_min  =   cv_min(2)
            y_max  =   cv_max(2)
            y_mesh =   cv_mesh(2)
            ipbc_y =   0
         else if ( cv2(1:5) .eq. 'DIH  ' ) then
            y_min  =   cv_min(3)
            y_max  =   cv_max(3)
            y_mesh =   cv_mesh(3)
            ipbc_y =   1
         else if ( cv2(1:5) .eq. 'DIFF ' ) then
            y_min  =   cv_min(4)
            y_max  =   cv_max(4)
            y_mesh =   cv_mesh(4)
            ipbc_y =   0
         else if ( cv2(1:5) .eq. 'CN   ' ) then
            y_min  =   cv_min(5)
            y_max  =   cv_max(5)
            y_mesh =   cv_mesh(5)
            ipbc_y =   0
         else if ( cv2(1:5) .eq. 'DCN  ' ) then
            y_min  =   cv_min(6)
            y_max  =   cv_max(6)
            y_mesh =   cv_mesh(6)
            ipbc_y =   0
         else if ( cv2(1:5) .eq. 'XYZ  ' ) then
            y_min  =   cv_min(7)
            y_max  =   cv_max(7)
            y_mesh =   cv_mesh(7)
            ipbc_y =   0
         else if ( cv2(1:5) .eq. 'DXYZ ' ) then
            y_min  =   cv_min(8)
            y_max  =   cv_max(8)
            y_mesh =   cv_mesh(8)
            ipbc_y =   0
         end if

         if ( cv3(1:5) .eq. 'DIST ' ) then
            z_min  =   cv_min(1)
            z_max  =   cv_max(1)
            z_mesh =   cv_mesh(1)
            ipbc_z =   0
         else if ( cv3(1:5) .eq. 'ANGL ' ) then
            z_min  =   cv_min(2)
            z_max  =   cv_max(2)
            z_mesh =   cv_mesh(2)
            ipbc_z =   0
         else if ( cv3(1:5) .eq. 'DIH  ' ) then
            z_min  =   cv_min(3)
            z_max  =   cv_max(3)
            z_mesh =   cv_mesh(3)
            ipbc_z =   1
         else if ( cv3(1:5) .eq. 'DIFF ' ) then
            z_min  =   cv_min(4)
            z_max  =   cv_max(4)
            z_mesh =   cv_mesh(4)
            ipbc_z =   0
         else if ( cv3(1:5) .eq. 'CN   ' ) then
            z_min  =   cv_min(5)
            z_max  =   cv_max(5)
            z_mesh =   cv_mesh(5)
            ipbc_z =   0
         else if ( cv3(1:5) .eq. 'DCN  ' ) then
            z_min  =   cv_min(6)
            z_max  =   cv_max(6)
            z_mesh =   cv_mesh(6)
            ipbc_z =   0
         else if ( cv3(1:5) .eq. 'XYZ  ' ) then
            z_min  =   cv_min(7)
            z_max  =   cv_max(7)
            z_mesh =   cv_mesh(7)
            ipbc_z =   0
         else if ( cv3(1:5) .eq. 'DXYZ ' ) then
            z_min  =   cv_min(8)
            z_max  =   cv_max(8)
            z_mesh =   cv_mesh(8)
            ipbc_z =   0
         end if

c        //   open file
         open ( iounit, file = 'hills2cube.dat' )

c        //   input file
         write( iounit, '(a,3f12.6)' ) trim(cv1), x_min, x_max, x_mesh
         write( iounit, '(a,3f12.6)' ) trim(cv2), y_min, y_max, y_mesh
         write( iounit, '(a,3f12.6)' ) trim(cv3), z_min, z_max, z_mesh

c        //   close file
         close( iounit )

c     //   three arguments
      end if

c-----------------------------------------------------------------------
c     //   usage
c-----------------------------------------------------------------------

c     //   three arguments
      if ( ( ierr .ne. 0 ) .and. ( iargc() .ne. 3 ) ) then

c        //   print message
         write( 6, * ) 
         write( 6, * ) 'CODE:        hills2cube.x'
         write( 6, * ) 
         write( 6, * ) 'INPUT FILE:  hills.ini'
         write( 6, * ) 'OUTPUT FILE: hills2cube.cube'
         write( 6, * ) 
         write( 6, * ) 'USAGE:       hills2cube.x $1 $2 $3'
         write( 6, * ) '$1:          cv type of x.'
         write( 6, * ) '$2:          cv type of y.'
         write( 6, * ) '$3:          cv type of z.'
         write( 6, * ) 

c        //   error termination
         stop

c     //   three arguments
      end if

c-----------------------------------------------------------------------
c     //   number of meshes
c-----------------------------------------------------------------------

      mesh_x = nint( ( x_max - x_min ) / x_mesh ) + 1
      mesh_y = nint( ( y_max - y_min ) / y_mesh ) + 1
      mesh_z = nint( ( z_max - z_min ) / z_mesh ) + 1

c-----------------------------------------------------------------------
c     //   print message
c-----------------------------------------------------------------------

      write( 6, '(a)' ) 'CV TYPE OF X:  ' // trim(cv1) // '.'
      write( 6, '(a)' ) 'CV TYPE OF Y:  ' // trim(cv2) // '.'
      write( 6, '(a)' ) 'CV TYPE OF Z:  ' // trim(cv3) // '.'

      write( 6, '(a)' ) 'PARAMETER FILE: hills2cube.dat.'

c-----------------------------------------------------------------------
c     //   count number of hills
c-----------------------------------------------------------------------

      write( 6, '(a)' ) 'Reading file: hills.ini.'
      write( 6, '(a)' ) 

      open ( iounit, file = "hills.ini" )

      nhills = 0

      do

         do j = 1, 3
            read ( iounit, *, iostat=ierr ) l
         end do

         if ( ierr .ne. 0 ) exit

         nhills = nhills + 1

      end do

      close( iounit )

      write( 6, '(a,i6)' ) 'Number of hills:', nhills
      write( 6, '(a)' ) 

      if ( nhills .eq. 0 ) then
         write( 6, '(a)' ) 'Error termination of hills2cube.'
         write( 6, '(a)' )
         stop
      end if

c-----------------------------------------------------------------------
c     //   memory allocation
c-----------------------------------------------------------------------

c     //   hill height
      allocate( istep(nhills) )

c     //   hill height
      allocate( height(nhills) )

c     //   hill width
      allocate( width(3,nhills) )

c     //   hill center
      allocate( center(3,nhills) )

c-----------------------------------------------------------------------
c     //   read hills
c-----------------------------------------------------------------------

      cxmax = 0.d0
      cymax = 0.d0
      czmax = 0.d0
      cxmin = 0.d0
      cymin = 0.d0
      czmin = 0.d0

      open ( iounit, file = "hills.ini" )

      k = 0

      do i = 1, nhills

         do j = 1, 3

            read ( iounit, *, iostat=ierr )
     &         istep(i), height(i), width(j,i), center(j,i)

            if ( ierr .ne. 0 ) then
               write( 6, '(a)' ) 'Error in reading hills.ini.'
               write( 6, '(a)' )
               write( 6, '(a)' ) 'Error termination of hills2cube.'
               write( 6, '(a)' )
               stop
            end if

         end do

c        //   artificial wall
         if ( istep(i) .le. 0 ) cycle

         if ( k .eq. 0 ) then
            cxmax = center(1,i)
            cymax = center(2,i)
            czmax = center(3,i)
            cxmin = center(1,i)
            cymin = center(2,i)
            czmin = center(3,i)
            k = 1
         else
            cxmax = max(cxmax,center(1,i))
            cymax = max(cymax,center(2,i))
            czmax = max(czmax,center(3,i))
            cxmin = min(cxmin,center(1,i))
            cymin = min(cymin,center(2,i))
            czmin = min(czmin,center(3,i))
         end if

      end do

      close( iounit )

c-----------------------------------------------------------------------
c     //   write hills
c-----------------------------------------------------------------------

      open ( iounit, file = "center.out" )

      do i = 1, nhills
         write ( iounit, '(i6,3f9.3)' ) i, center(1:3,i)
      end do

      close( iounit )

c-----------------------------------------------------------------------
c     //   print results
c-----------------------------------------------------------------------

      write( 6, '(a,f9.3,a,f9.3)' )
     &   'CV1 minimum:', cxmin, '  maximum:', cxmax

      write( 6, '(a,f9.3,a,f9.3)' )
     &   'CV2 minimum:', cymin, '  maximum:', cymax

      write( 6, '(a,f9.3,a,f9.3)' )
     &   'CV3 minimum:', czmin, '  maximum:', czmax

      write( 6, '(a)' )

c-----------------------------------------------------------------------
c     //   print header part
c-----------------------------------------------------------------------

      write( 6, '(a)' ) 'Printing header info: hills2cube.cube.'
      write( 6, '(a)' )

      open ( iounit, file = "hills2cube.cube" )

      write( iounit, '(a)' )
     &   "HILLS2CUBE CUBE FILE."

      write( iounit, '(a)' )
     &   "OUTER LOOP: X, MIDDLE LOOP: Y, INNER LOOP: Z"

      write( iounit, '(i5,3e16.8)' )
     &    0, x_max-x_min, y_max-y_min, z_max-z_min

      write( iounit, '(i5,3e16.8)' )
     &    mesh_x, 1.e0, 0.e0, 0.e0

      write( iounit, '(i5,3e16.8)' )
     &    mesh_y, 0.e0, 1.e0, 0.e0

      write( iounit, '(i5,3e16.8)' )
     &    mesh_z, 0.e0, 0.e0, 1.e0

c-----------------------------------------------------------------------
c     //   calculate sum of hills
c-----------------------------------------------------------------------

      write( 6, '(a)' ) 'Entering main loop.'
      write( 6, '(a)' )

c     //   cut off of exponent

      rw2cut = - log( cutoff )

c     //   mesh interval

      x_mesh = ( x_max - x_min ) / dble( mesh_x - 1 )
      y_mesh = ( y_max - y_min ) / dble( mesh_y - 1 )
      z_mesh = ( z_max - z_min ) / dble( mesh_z - 1 )

c     //   initialize peak position

      vhills_peak = 0.e0

      xm_peak = 0.e0
      ym_peak = 0.e0
      zm_peak = 0.e0

c     //   loop of meshes: start

      do kx = 1, mesh_x

         do ky = 1, mesh_y

            do kz = 1, mesh_z

               vhills = 0.e0

               xm = x_min
               ym = y_min
               zm = z_min

               do i = 1, nhills

c                 //   artificial wall
                  if ( istep(i) .lt. 0 ) cycle

c                 //   mesh point

                  xm = x_min + x_mesh * ( kx - 1 )
                  ym = y_min + y_mesh * ( ky - 1 )
                  zm = z_min + z_mesh * ( kz - 1 )

c                 //   distance between from mesh point

                  rx = center(1,i) - xm
                  ry = center(2,i) - ym
                  rz = center(3,i) - zm

c                 //   apply periodic boundary condition if ipbc = 1

                  rx = rx - ipbc_x * nint( rx / 360.e0 ) * 360.e0
                  ry = ry - ipbc_y * nint( ry / 360.e0 ) * 360.e0
                  rz = rz - ipbc_z * nint( rz / 360.e0 ) * 360.e0

c                 //   normalize distance by width

                  rwx = rx / width(1,i)
                  rwy = ry / width(2,i)
                  rwz = rz / width(3,i)

c                 //   exponent value

                  rw2 = 0.5e0 * ( rwx*rwx + rwy*rwy + rwz*rwz )

c                 //   add to vhills if exponent value is small

                  if ( rw2 .lt. rw2cut ) then

                     vhills = vhills + height(i) * exp( - rw2 )

                  end if

               end do

c              //   print hills value

c               write( iounit, '(f10.6)' ) vhills
               write( iounit, '(f10.6)' ) -vhills
c               write( iounit, '(f10.6)' ) vhills*627.51e0

c              //   peak position

               if ( vhills .gt. vhills_peak ) then
                  vhills_peak = vhills
                  xm_peak = xm
                  ym_peak = ym
                  zm_peak = zm
               end if

            end do

         end do

c        //   print data

         write( 6, '(a,f6.1)' )
     &      'done (%):', float(kx)/float(mesh_x)*1.e2

      end do

c     //   loop of meshes: end

      close( iounit )

c-----------------------------------------------------------------------
c     //   print results
c-----------------------------------------------------------------------

      write( 6, '(a)' )

      write( 6, '(a,3f9.3)' )
     &   'Min position:', xm_peak, ym_peak, zm_peak

      write( 6, '(a)' )

      write( 6, '(a,f9.5)' )
     &   'Min depth (hartree): ', vhills_peak

      write( 6, '(a)' )

      write( 6, '(a,f9.2)' )
     &   'Min depth (kcal/mol):', vhills_peak*627.51e0

      write( 6, '(a)' )

      write( 6, '(a)' ) 'Normal termination of hills2cube.'
      write( 6, '(a)' )

      stop
      end
