c***********************************************************************
      program fspimd
c***********************************************************************
c-----------------------------------------------------------------------
c     //   variables
c-----------------------------------------------------------------------

c     //   initialize
      implicit none

c     //   file unit
      integer :: iounit = 10

c     //   trajectory file (trj.out)
      character(len=80) :: trjfile

c     //   structure file (structure.dat)
      character(len=80) :: strfile

c     //   input file (input.dat)
      character(len=80) :: inpfile

c     //   Boltzmann constant
      real(8), parameter :: boltz = 0.316682968d-5

c     //   Planck constant divided by 2 pi
      real(8), parameter :: hbar = 1.d0

c     //   unit conversion factors
      real(8), parameter :: au_length = 0.529177249d-10

c     //   intermediate scattering function
      real(8), dimension(:), allocatable :: fs

c     //   atomic coordinates
      real(8), dimension(:,:), allocatable :: x, y, z

c     //   box matrix
      real(8), dimension(3,3) :: box

c     //   inverse of box matrix
      real(8), dimension(3,3) :: boxinv

c     //   q sampling index
      integer, dimension(3) :: lmax

c     //   q values
      real(8) :: qmin_aa, qmax_aa, qmin, qmax

c     //   q samples
      real(8), dimension(:), allocatable :: qxs
      real(8), dimension(:), allocatable :: qys
      real(8), dimension(:), allocatable :: qzs

c     //   number of samples
      integer :: nsample

c     //   number of steps
      integer :: nstep

c     //   step
      integer :: istep

c     //   number of beads
      integer :: nbead

c     //   number of atoms
      integer :: natom

c     //   temperature
      real(8) :: temperature

c     //   integers
      integer :: i, j, k, l, m, iatom, ierr, la, lb, lc, iargc, iqs, is
      integer :: iq, nq

c     //   real numbers
      real(8) :: rx, ry, rz, beta, tau, r(1), bohr2ang, pi, qabs, qr
      real(8) :: qx, qy, qz

c     //   box vectors
      real(8) :: ax, ay, az, bx, by, bz, cx, cy, cz

c     //   character
      character(len=80) :: char

c-----------------------------------------------------------------------
c     //   initial message
c-----------------------------------------------------------------------

      if ( iargc() .ne. 7 ) then

         write( 6, '(a)' )
         write( 6, '(a)' ) 'Program fspimd'
         write( 6, '(a)' )
         write( 6, '(a)' ) 'Usage: fspimd.x $1 $2 $3 $4 $5 $6 $7'
         write( 6, '(a)' )
         write( 6, '(a)' ) '$1:  trj.out (position trajectory)'
         write( 6, '(a)' ) '$2:  structure.dat (atomic kinds)'
         write( 6, '(a)' ) '$3:  input.dat (beads, box, temperature)'
         write( 6, '(a)' ) '$4:  number of k-point sampling'
         write( 6, '(a)' ) '$5:  minimum q [angstrom**-1]'
         write( 6, '(a)' ) '$6:  maximum q [angstrom**-1]'
         write( 6, '(a)' ) '$7: hydrogen atom'
         write( 6, '(a)' )
         write( 6, '(a)' ) 'Example:'
         write( 6, '(a)' )
         write( 6, '(a)' )
     &      'fspimd.x trj.out structure.dat input.dat 30 4.0 8.0 109'

         stop

      end if

c-----------------------------------------------------------------------
c     //   read arguments
c-----------------------------------------------------------------------

c     //   trajectory filename
      call getarg( 1, trjfile )

c     //   structure filename
      call getarg( 2, strfile )

c     //   input filename
      call getarg( 3, inpfile )

c     //   number of samples
      call getarg( 4, char )
      read( char, *, iostat=ierr ) nsample

c     //   minimum q
      call getarg( 5, char )
      read( char, *, iostat=ierr ) qmin_aa

c     //   maximum q
      call getarg( 6, char )
      read( char, *, iostat=ierr ) qmax_aa

c     //   hydrogen atom
      call getarg( 7, char )
      read( char, * ) iatom

c-----------------------------------------------------------------------
c     //   constants
c-----------------------------------------------------------------------

c     //   circular constant
      pi = acos(-1.d0)

c     //   bohr to angstrom
      bohr2ang = au_length * 1.d+10

c-----------------------------------------------------------------------
c     //   unit conversions
c-----------------------------------------------------------------------

c     //   bohr to angstrom
      bohr2ang = au_length * 1.d+10

c     //   /angstrom to /bohr
      qmin = qmin_aa * bohr2ang
      qmax = qmax_aa * bohr2ang

c-----------------------------------------------------------------------
c     //   number of atoms
c-----------------------------------------------------------------------

c     //   open file
      open ( iounit, file = trim(strfile) )

c     //   number of atoms
      read ( iounit, *, iostat=ierr ) natom

c     //   close file
      close( iounit )

c-----------------------------------------------------------------------
c     //   number of beads
c-----------------------------------------------------------------------

c     /*   file open   */
      open ( iounit, file = trim(inpfile) )

c     /*   search for tag    */
      call search_tag ( '<nbead>', 7, iounit, ierr )

c     /*   number of atomic symbols   */
      read ( iounit, *, iostat=ierr ) nbead

c     /*   file close   */
      close( iounit )

c-----------------------------------------------------------------------
c     //   temperature
c-----------------------------------------------------------------------

c     /*   file open   */
      open ( iounit, file = trim(inpfile) )

c     /*   search for tag    */
      call search_tag ( '<temperature>', 13, iounit, ierr )

c     /*   number of atomic symbols   */
      read ( iounit, *, iostat=ierr ) temperature

c     /*   file close   */
      close( iounit )

c     //   inverse of temperature
      beta = 1.d0 / ( boltz * temperature )

c-----------------------------------------------------------------------

      allocate( fs(nbead) )

      allocate( x(natom,nbead) )
      allocate( y(natom,nbead) )
      allocate( z(natom,nbead) )

c-----------------------------------------------------------------------
c     //   box
c-----------------------------------------------------------------------

c     /*   file open   */
      open ( iounit, file = trim(inpfile) )

c     /*   search for tag    */
      call search_tag ( '<iboundary>', 11, iounit, ierr )

c     /*   number of atomic symbols   */
      read ( iounit, *, iostat=ierr ) char

c     /*   number of atomic symbols   */
      read ( iounit, *, iostat=ierr ) box(1,1), box(1,2), box(1,3)
      read ( iounit, *, iostat=ierr ) box(2,1), box(2,2), box(2,3)
      read ( iounit, *, iostat=ierr ) box(3,1), box(3,2), box(3,3)

c     /*   file close   */
      close( iounit )

c     //   angstrom to bohr
      if ( ( ierr .eq. 0 ) .and. ( char(1:1) .eq. 'A' ) ) then
         box(:,:) = box(:,:) / bohr2ang
      end if

c     //   inverse matrix
      call inv3( box, boxinv )

c-----------------------------------------------------------------------
c     //   check samples
c-----------------------------------------------------------------------

c     //   counter
      iq = 0

      ax = 2.d0*pi*boxinv(1,1)
      ay = 2.d0*pi*boxinv(1,2)
      az = 2.d0*pi*boxinv(1,3)
      bx = 2.d0*pi*boxinv(2,1)
      by = 2.d0*pi*boxinv(2,2)
      bz = 2.d0*pi*boxinv(2,3)
      cx = 2.d0*pi*boxinv(3,1)
      cy = 2.d0*pi*boxinv(3,2)
      cz = 2.d0*pi*boxinv(3,3)

      lmax(1) = int( qmax/sqrt(ax*ax+ay*ay+az*az) ) + 1
      lmax(2) = int( qmax/sqrt(bx*bx+by*by+bz*bz) ) + 1
      lmax(3) = int( qmax/sqrt(cx*cx+cy*cy+cz*cz) ) + 1

c     //   loop of wave directions
      do la = -lmax(1), lmax(1)
      do lb = -lmax(2), lmax(2)
      do lc = -lmax(3), lmax(3)

c        //   skip for the case lx=ly=lz=0
         if ( abs(la)+abs(lb)+abs(lc) .eq. 0 ) cycle

c        //   vector
         qx = ax*la + bx*lb + cx*lc
         qy = ay*la + by*lb + cy*lc
         qz = az*la + bz*lb + cz*lc

c        //   norm of vector
         qabs = sqrt( qx*qx + qy*qy + qz*qz )

c        //   skip q out of range
         if ( qabs .lt. qmin ) cycle
         if ( qabs .gt. qmax ) cycle

c        //   counter
         iq = iq + 1

c     //   loop of wave directions
      end do
      end do
      end do

c     //   counter
      nq = iq

c-----------------------------------------------------------------------
c     //   create random samples
c-----------------------------------------------------------------------

c     //   q samples
      allocate( qxs(nsample) )
      allocate( qys(nsample) )
      allocate( qzs(nsample) )

c     //   loop of samples
      do is = 1, nsample

c        //   random number
         call random_number( r )

c        //   choose one of the nq samples
         iqs = int( r(1) * dble(nq) ) + 1

c        //   counter
         iq = 0

c        //   loop of wave directions
         do la = -lmax(1), lmax(1)
         do lb = -lmax(2), lmax(2)
         do lc = -lmax(3), lmax(3)

c           //   skip for the case lx=ly=lz=0
            if ( abs(la)+abs(lb)+abs(lc) .eq. 0 ) cycle

c           //   vector
            qx = ax*la + bx*lb + cx*lc
            qy = ay*la + by*lb + cy*lc
            qz = az*la + bz*lb + cz*lc

c           //   norm of vector
            qabs = sqrt( qx*qx + qy*qy + qz*qz )

c           //   skip q out of range
            if ( qabs .lt. qmin ) cycle
            if ( qabs .gt. qmax ) cycle

c           //   counter
            iq = iq + 1

c           //   save iqs-th data
            if ( iq .eq. iqs ) then
               qxs(is) = qx
               qys(is) = qy
               qzs(is) = qz
               go to 100
            end if

c        //   loop of wave directions
         end do
         end do
         end do

c     //   loop of samples
  100 end do

c-----------------------------------------------------------------------
c     //   check samples
c-----------------------------------------------------------------------

      fs(:) = 0.d0

      open ( iounit, file = 'trj.out' )

      istep = 0

      do

         do j = 1, nbead
         do i = 1, natom
            read( iounit, *, iostat=ierr ) k, x(i,j), y(i,j), z(i,j)
         end do
         end do

         if ( ierr .ne. 0 ) exit

         istep = istep + 1

c        //   loop of samples
         do is = 1, nsample

c           //   vector
            qx = qxs(is)
            qy = qys(is)
            qz = qzs(is)

c           //   norm of vector
            qabs = qx*qx + qy*qy + qz*qz

c           //   skip q out of range
            if ( qabs .lt. (qmin*qmin) ) cycle
            if ( qabs .gt. (qmax*qmax) ) cycle

c           //   norm of vector
            qabs = sqrt( qabs )

            do m = 1, nbead
            do k = 1, nbead

               l = mod(k+m-1,nbead)

               if ( l .eq. 0 ) l = nbead

               rx = x(iatom,k) - x(iatom,l)
               ry = y(iatom,k) - y(iatom,l)
               rz = z(iatom,k) - z(iatom,l)

               qr = qx*rx + qy*ry + qz*rz

               fs(m) = fs(m) + cos( qr ) / dble(nbead*nsample)

            end do
            end do

         end do

      end do

      close( iounit )

      nstep = istep

      fs(:) = fs(:) / nstep

c-----------------------------------------------------------------------
c     //   print output
c-----------------------------------------------------------------------

      open ( iounit, file = 'fspimd.out' )

      do j = 1, nbead

         tau = beta * hbar / dble(nbead) * dble(j-1)

         write( 10, '(i6,f12.2,f12.6)' ) j-1, tau, fs(j)

      end do

      write( 10, '(i6,f12.2,f12.6)' ) nbead, tau, 1.d0

      close( iounit )

      write( 6, '(a)' ) 
      write( 6, '(a)' ) 'Normal termination of fspimd.'
      write( 6, '(a)' ) 

c-----------------------------------------------------------------------
c     //   end of code
c-----------------------------------------------------------------------

      stop
      end





c***********************************************************************
      subroutine search_tag ( char_tag, length_tag, iounit, ierr )
c***********************************************************************
c-----------------------------------------------------------------------
c     /*   local variables                                            */
c-----------------------------------------------------------------------

      implicit none

      integer:: k, length_tag, iounit, ierr

      character(len=80) :: char_line

      character(len=length_tag) :: char_tag

c-----------------------------------------------------------------------
c     /*   do loop start                                              */
c-----------------------------------------------------------------------

      do

c-----------------------------------------------------------------------
c        /*   read a line                                             */
c-----------------------------------------------------------------------

         read (iounit,*,iostat=ierr) char_line

c-----------------------------------------------------------------------
c        /*   return if error is found                                */
c-----------------------------------------------------------------------

         if ( ierr .ne. 0 ) return

c-----------------------------------------------------------------------
c        /*   search for the tag                                      */
c-----------------------------------------------------------------------

         k = index(char_line(1:length_tag),char_tag(1:length_tag))

c-----------------------------------------------------------------------
c        /*   return as soon as we find the tag                       */
c-----------------------------------------------------------------------

         if ( k .ge. 1 ) return

c-----------------------------------------------------------------------
c     /*   do loop end                                                */
c-----------------------------------------------------------------------

      end do

      return
      end





c***********************************************************************
      subroutine inv3 ( a, ainv )
c***********************************************************************
c-----------------------------------------------------------------------
c     /*   local variables                                            */
c-----------------------------------------------------------------------

      implicit none

      integer :: i, j

      real(8) :: a(3,3), ainv(3,3), det3, deta

c-----------------------------------------------------------------------
c     /*   start                                                      */
c-----------------------------------------------------------------------

      ainv(1,1) = + a(2,2)*a(3,3) - a(2,3)*a(3,2)
      ainv(1,2) = + a(3,2)*a(1,3) - a(1,2)*a(3,3)
      ainv(1,3) = + a(1,2)*a(2,3) - a(2,2)*a(1,3)

      ainv(2,1) = + a(2,3)*a(3,1) - a(3,3)*a(2,1)
      ainv(2,2) = + a(3,3)*a(1,1) - a(3,1)*a(1,3)
      ainv(2,3) = + a(1,3)*a(2,1) - a(2,3)*a(1,1)

      ainv(3,1) = + a(2,1)*a(3,2) - a(3,1)*a(2,2)
      ainv(3,2) = + a(3,1)*a(1,2) - a(1,1)*a(3,2)
      ainv(3,3) = + a(1,1)*a(2,2) - a(1,2)*a(2,1)

      deta = det3 ( a )

      do j = 1, 3
      do i = 1, 3
         ainv(i,j) = ainv(i,j)/deta
      end do
      end do

      return
      end





c***********************************************************************
      real(8) function det3 ( a )
c***********************************************************************
c-----------------------------------------------------------------------
c     /*   local variables                                            */
c-----------------------------------------------------------------------

      implicit none

      real(8) :: a(3,3)

c-----------------------------------------------------------------------
c     /*   start                                                      */
c-----------------------------------------------------------------------

      det3 = + a(1,1)*a(2,2)*a(3,3) - a(1,1)*a(2,3)*a(3,2)
     &       + a(2,1)*a(3,2)*a(1,3) - a(2,1)*a(1,2)*a(3,3)
     &       + a(3,1)*a(1,2)*a(2,3) - a(3,1)*a(2,2)*a(1,3)

      return
      end
