c***********************************************************************
      program fsbcmd
c***********************************************************************

c     //   initialize
      implicit none

c     //   Boltzmann constant
      real(8), parameter :: boltz = 0.316682968d-5

c     //   unit conversion factors
      real(8), parameter :: au_time   = 0.024188843d-15
      real(8), parameter :: au_length = 0.529177249d-10

c     /*   speed of light in SI unit   */
      real(8), parameter :: speedlight_SI = 2.99892458d+8

c     //   Planck constant divided by 2 pi
      real(8), parameter :: hbar = 1.d0

c     //   temperature
      real(8) :: temperature

c     //   circular constant
      real(8) :: pi

c     //   number of bead
      integer :: nbead

c     //   real numbers
      real(8) :: beta, omega, domega, bho, tau, bohr2ang, factor
      real(8) :: sumrule

c     //   intermediate scattering function
      real(8), dimension(:), allocatable :: fs

c     //   dynamic structure factor
      real(8), dimension(:), allocatable :: skw

c     //   integers
      integer :: i, j, nomega, iargc, ierr

c     //   file unit
      integer :: iounit = 10

c     //   real numbers
      real(8) :: dum, factor_cminv

c     //   characters
      character(len=80) :: char

c-----------------------------------------------------------------------
c     //   initial message
c-----------------------------------------------------------------------

      if ( iargc() .ne. 3 ) then

         write( 6, '(a)' )
         write( 6, '(a)' ) 'Program fsbcmd'
         write( 6, '(a)' )
         write( 6, '(a)' ) 'Usage: fsbcmd.x $1 $2 $3'
         write( 6, '(a)' )
         write( 6, '(a)' ) '$1: max frequency [cm**-1]'
         write( 6, '(a)' ) '$2: number of beads'
         write( 6, '(a)' ) '$3: temperature [K]'
         write( 6, '(a)' )
         write( 6, '(a)' ) 'Uses sqw_iso.out'
         write( 6, '(a)' )

         stop

      end if

c-----------------------------------------------------------------------
c     //   constants
c-----------------------------------------------------------------------

c     //   circular constant
      pi = acos(-1.d0)

c     //   bohr to angstrom
      bohr2ang = au_length * 1.d+10

c     //   cm**-1 to hartree
      factor_cminv = 1.d0 / (2.d0*pi*speedlight_SI*au_time*100.d0)

c-----------------------------------------------------------------------
c     //   read arguments
c-----------------------------------------------------------------------

      call getarg( 1, char )
      read( char, * ) nomega

      call getarg( 2, char )
      read( char, * ) nbead

      call getarg( 3, char )
      read( char, * ) temperature

c-----------------------------------------------------------------------
c     //   inverse of temperature
c-----------------------------------------------------------------------

      beta = 1.d0 / ( boltz * temperature )

c-----------------------------------------------------------------------
c     //   cm**-1 in au
c-----------------------------------------------------------------------

      domega = 1.d0 / factor_cminv

c-----------------------------------------------------------------------
c     //   memory allocation
c-----------------------------------------------------------------------

      allocate( fs(0:nbead) )

      allocate( skw(-nomega:nomega) )

c-----------------------------------------------------------------------
c     //   read skw
c-----------------------------------------------------------------------

      open ( iounit, file = 'sqw_iso.out' )

      read( iounit, *, iostat=ierr )
      read( iounit, *, iostat=ierr )
      read( iounit, *, iostat=ierr )
      read( iounit, *, iostat=ierr )
      do i = 1, nomega
         read( iounit, *, iostat=ierr ) dum, dum, dum, skw(i)
      end do

      close( iounit )

      call error_handling( ierr, 'Error: sqw_iso.out', 19 )

c-----------------------------------------------------------------------
c     //   skw of negative frequency: detailed balance condition
c-----------------------------------------------------------------------

      skw(0) = skw(1)

      do i = 1, nomega

         omega = dble(i) * domega

         bho = beta * hbar * omega

         skw(-i) = exp( - bho ) * skw(i)

      end do

c-----------------------------------------------------------------------
c     //   check sum rule
c-----------------------------------------------------------------------

      sumrule = 0.d0

      do i = 1, nomega-1

         omega = dble(i) * domega

         bho = beta * hbar * omega

         sumrule = sumrule + (1.d0+exp(-bho)) * skw(i) * domega

      end do

      write( 6, '(a,f10.5)' ) "sumrule", sumrule
      write( 6, '(a)' )

c-----------------------------------------------------------------------
c     //   compute fs and print output
c-----------------------------------------------------------------------

      open ( iounit, file = 'fsbcmd.out' )

      do j = 0, nbead

         tau = dble(j) * beta * hbar / dble(nbead)

         fs(j) = 0.d0

         do i = -nomega, nomega-1

            omega = (dble(i)+0.5d0) * domega

            bho = beta * hbar * omega

            factor = 0.5d0 * (skw(i)+skw(i+1))
     &             * exp( - 0.5d0 * bho )
     &             * cosh( 0.5d0*bho - omega*tau )
     &             * domega

            fs(j) = fs(j) + factor

         end do

         tau = tau * au_time / 1.d-15

         write( iounit, '(i6,f12.2,2f12.6)' )
     &      j, tau, fs(j), fs(j)/fs(0)

      end do

      close( iounit )

c-----------------------------------------------------------------------
c     //   gnuplot
c-----------------------------------------------------------------------

      open ( iounit, file = 'fsbcmd.plt' )

      write( iounit, '(a)' ) 'set size square'
      write( iounit, '(a)' ) "set term qt font 'helvetica,20'"
      write( iounit, '(a)' ) 'unset key'
      write( iounit, '(a)' ) "set xlabel '{/Symbol t} [fs]'"
      write( iounit, '(a)' )
     &   "set ylabel 'F@_s@(k,{/Symbol t})/F@_s(k,0)'"
      write( iounit, '(a,f8.2,a)' ) "set xrange [0:", tau, "]"
      write( iounit, '(a)' ) "plot 'fsbcmd.out' u 2:4 w l lw 3"
      write( iounit, '(a)' ) 'pause -1'

      close( iounit )

      call system( 'gnuplot fsbcmd.plt' )

c-----------------------------------------------------------------------
c     //   end of code
c-----------------------------------------------------------------------

      stop
      end





c***********************************************************************
      subroutine error_handling ( ierr, char_tag, length_tag )
c***********************************************************************

c-----------------------------------------------------------------------
c     /*   local variables                                            */
c-----------------------------------------------------------------------

      implicit none

      integer:: ierr, length_tag

      character(len=length_tag) :: char_tag

c-----------------------------------------------------------------------
c     /*   start                                                      */
c-----------------------------------------------------------------------

      if ( ierr .ne. 0 ) then

         write(6,'(a)') 'Error termination at: ' // char_tag // '.'

         write(6,'(a)')

         stop

      end if

      return
      end

-