c**********************************************************************
      module dvr1d_variables
c**********************************************************************

c     /*   number of grid points   */
      integer, parameter :: ndvr = 99

c     /*   maximum value of the grid   */
      real(8) :: dvrmax =  5.d0

c     /*   minimum value of the grid   */
      real(8) :: dvrmin = -5.d0

c     /*   grid spacing   */
      real(8) :: dvrmesh

c     /*   reduced mass   */
      real(8) :: redmass = 1.d0

c     /*   grid points   */
      real(8), dimension(:), allocatable :: xdvr

c     /*   potential energies   */
      real(8), dimension(:), allocatable :: vdvr

c     /*   hamiltonian matrix   */
      real(8), dimension(:,:), allocatable :: hdvr

c     /*   eigenvectors   */
      real(8), dimension(:,:), allocatable :: cdvr

c     /*   eigenvalues   */
      real(8), dimension(:), allocatable :: edvr

c     /*   number of states   */
      integer :: nvib

c     /*   number of beads   */
      integer :: nbead = 32

c     /*   number of steps   */
      integer :: nstep = 3000

c     /*   inverse temperature   */
      real(8) :: beta

c**********************************************************************
      end module dvr1d_variables
c**********************************************************************





c**********************************************************************
      program dvr1d
c**********************************************************************

c-----------------------------------------------------------------------
c     /*   shared variables                                           */
c-----------------------------------------------------------------------

      use dvr1d_variables, only :
     &   dvrmin, dvrmax, dvrmesh, redmass, hdvr, vdvr, edvr,
     &   cdvr, xdvr, beta, ndvr, nvib, nbead, nstep

c-----------------------------------------------------------------------
c     /*   local variables                                            */
c-----------------------------------------------------------------------

c     /*   initialize   */
      implicit none

c     /*   integer   */
      integer :: i, j, k, ix, jx, ivib, istep, ioption, joption

c     /*   real numbers   */
      real(8) :: const, pm, sinm, sinp, di, dn, t0, x1, x3, xn

c     /*   real numbers   */
      real(8) :: wi, factor, hij, eij, zq, cxx, pi, tau, q, cij

c     /*   real numbers   */
      real(8) :: d(ndvr,ndvr), t(ndvr,ndvr)

c     /*   real numbers   */
      real(8) :: tiny = 1.d-16

c     /*   character   */
      character(len=80) :: char

c-----------------------------------------------------------------------
c     //   initial message
c-----------------------------------------------------------------------

      if ( iargc() .ne. 3 ) then

         write( 6, '(a)' )
         write( 6, '(a)' ) 'Program dvr1d'
         write( 6, '(a)' )
         write( 6, '(a)' ) 'Usage: dvr1dx $1 $2 $3'
         write( 6, '(a)' )
         write( 6, '(a)' ) '$1: potential = 0 : harmonic'
         write( 6, '(a)' ) '              = 1 : weak'
         write( 6, '(a)' ) '              = 2 : strong'
         write( 6, '(a)' ) '$2: function  = 1 : x'
         write( 6, '(a)' ) '              = 3 : x**3'
         write( 6, '(a)' ) '$3: beta      = 1.0'
         write( 6, '(a)' ) '              = 8.0'
         write( 6, '(a)' )

         stop

      end if

c-----------------------------------------------------------------------
c     //   read data
c-----------------------------------------------------------------------

c     //   potential
      call getarg( 1, char )
      read( char, * ) ioption

c     //   function
      call getarg( 2, char )
      read( char, * ) joption

c     //   beta
      call getarg( 3, char )
      read( char, * ) beta

c-----------------------------------------------------------------------
c     /*   circular constant                                          */
c-----------------------------------------------------------------------

      pi = acos( - 1.d0 )

c-----------------------------------------------------------------------
c     /*   grid size                                                  */
c-----------------------------------------------------------------------

      dvrmesh = ( dvrmax - dvrmin ) / ( ndvr + 1 )

c-----------------------------------------------------------------------
c     /*   memory allocation                                          */
c-----------------------------------------------------------------------

c     /*   grid   */
      if ( .not. allocated(xdvr) ) allocate( xdvr(ndvr) )

c     /*   hamiltonian   */
      if ( .not. allocated(hdvr) ) allocate( hdvr(ndvr,ndvr) )

c     /*   eigenstate   */
      if ( .not. allocated(cdvr) ) allocate( cdvr(ndvr,ndvr) )

c     /*   eigenvalue   */
      if ( .not. allocated(edvr) ) allocate( edvr(ndvr) )

c     /*   potential   */
      if ( .not. allocated(vdvr) ) allocate( vdvr(ndvr) )

c-----------------------------------------------------------------------
c     /*   generate grids:  one dimensional case                      */
c-----------------------------------------------------------------------

      do i = 1, ndvr
         xdvr(i) = dvrmin + dvrmesh * dble(i)
      end do

c-----------------------------------------------------------------------
c     /*   potential                                                  */
c-----------------------------------------------------------------------

      if ( ioption .eq. 0 ) then

         do i = 1, ndvr
            q = xdvr(i)
            vdvr(i) = 0.5d0*q**2
         end do

      else if ( ioption .eq. 1 ) then

         do i = 1, ndvr
            q = xdvr(i)
            vdvr(i) = 0.5d0*q**2 + 0.1d0*q**3 + 0.01*q**4
         end do

      else if ( ioption .eq. 2 ) then

         do i = 1, ndvr
            q = xdvr(i)
            vdvr(i) = 0.25d0*q**4
         end do

      else

         stop

      end if

c-----------------------------------------------------------------------
c     /*   constant                                                   */
c-----------------------------------------------------------------------

      const = 1.d0 / (2.d0*redmass) / (dvrmax-dvrmin)**2 * pi**2 / 2.d0

c-----------------------------------------------------------------------
c     /*   kronecker delta                                            */
c-----------------------------------------------------------------------

      d(:,:) = 0.d0

      do i = 1, ndvr
         d(i,i) = 1.d0
      end do

c-----------------------------------------------------------------------
c     /*   generate kinetic energy matrix                             */
c-----------------------------------------------------------------------

      do i = 1, ndvr
      do j = 1, ndvr

         if ( i .eq. j ) then

            di = dble( i )
            dn = dble( ndvr + 1 )

            sinm = dsin( pi*di/dn )

            t(i,i) = const * ( (2.d0*dn**2+1.d0)/3.d0 - 1.d0/sinm**2 )

         else

            if ( mod(iabs(i-j),2) .eq. 0 ) then
               pm =   1.d0
            else
               pm = - 1.d0
            end if

            sinm = sin( pi * dble(i-j) / dble(2*ndvr+2) )
            sinp = sin( pi * dble(i+j) / dble(2*ndvr+2) )

            t(i,j) = const * pm * ( 1.d0/sinm**2 - 1.d0/sinp**2 )

         end if

      end do
      end do

c-----------------------------------------------------------------------
c     /*   generate hamiltonian matrix:  one dimensional case         */
c-----------------------------------------------------------------------

      i = 0

      do ix = 1, ndvr

         i = i + 1

         j = 0

         do jx = 1, ndvr

            j = j + 1

            hdvr(i,j) =  t(ix,jx) + vdvr(ix)*d(ix,jx)

         end do

      end do

c-----------------------------------------------------------------------
c     /*   diagonalize hamiltonian                                    */
c-----------------------------------------------------------------------

      call ddiag ( hdvr, edvr, cdvr, ndvr )

c-----------------------------------------------------------------------
c     /*   normalize wavefunctions                                    */
c-----------------------------------------------------------------------

      cdvr(:,:) = cdvr(:,:) / sqrt(dvrmesh)

c-----------------------------------------------------------------------
c     /*   partition function                                         */
c-----------------------------------------------------------------------

      zq = 0.d0

      do i = 1, ndvr

         wi = exp( - beta * ( edvr(i) - edvr(1) ) )

         if ( wi .le. tiny ) cycle

         zq = zq + wi

         nvib = i

      end do

c-----------------------------------------------------------------------
c     /*   matrix                                                     */
c-----------------------------------------------------------------------

      factor = (dvrmax-dvrmin) / dble(ndvr+1)

      d(:,:) = 0.d0

      if ( joption .eq. 1 ) then

         do i = 1, nvib
         do j = 1, nvib
            do k = 1, ndvr
               x1 = xdvr(k)
               d(i,j) = d(i,j) + cdvr(k,i) * x1 * cdvr(k,j) * factor
            end do
         end do
         end do

      else if ( joption .eq. 3 ) then

         do i = 1, nvib
         do j = 1, nvib
            do k = 1, ndvr
               x1 = xdvr(k)
               x3 = x1 * x1 * x1
               d(i,j) = d(i,j) + cdvr(k,i) * x3 * cdvr(k,j) * factor
            end do
         end do
         end do

      else

         do i = 1, nvib
         do j = 1, nvib
            do k = 1, ndvr
               x1 = xdvr(k)
               xn = x1**joption
               d(i,j) = d(i,j) + cdvr(k,i) * xn * cdvr(k,j) * factor
            end do
         end do
         end do

      end if

c-----------------------------------------------------------------------
c     /*   correlation function                                       */
c-----------------------------------------------------------------------

      do istep = 1, nstep

         t0 = (istep-1) * 0.01d0

         cxx = 0.d0

         do i = 1, nvib

            wi = exp( - beta * ( edvr(i) - edvr(1) ) )

            if ( wi .le. tiny ) cycle

            do j = 1, nvib

               eij = edvr(i) - edvr(j)

               cij = cos( ( edvr(i) - edvr(j) ) * t0 )

               factor = cij * wi * d(i,j) * d(i,j) / dble(nbead) / zq

               do k = 1, nbead

                  tau = beta * dble(k) / dble(nbead)

                  cxx = cxx + factor * exp( eij * tau )

               end do

            end do

         end do

         write( 6, '(2f10.5)' ) t0, cxx

      end do

      return
      end





c***********************************************************************
      subroutine ddiag ( a, w, v, n )
c***********************************************************************

      implicit none

c-----------------------------------------------------------------------
c     /*   local variables                                            */
c-----------------------------------------------------------------------

      integer          :: i, j, n, info

      character(len=1) :: jobz, uplo

      real(8)          :: a(n,n), b(n,n), w(n), work(4*n), v(n,n)

c-----------------------------------------------------------------------
c     /*   start                                                      */
c-----------------------------------------------------------------------

      jobz = 'V'
      uplo = 'U'

      do j = 1, n
      do i = 1, n
         b(i,j) = a(i,j)
      end do
      end do

      call dsyev( jobz, uplo, n, a, n, w, work, 4*n, info )

      do j = 1, n
      do i = 1, n
         v(i,j) = a(i,j)
         a(i,j) = b(i,j)
      end do
      end do

      return
      end
