c***********************************************************************
      module dvr_variables
c***********************************************************************

c     /*   circular constant   */
      real(8) :: pi

c     /*   grid point range   */
      real(8) :: dvrmax, dvrmin, dvrmesh

c     /*   data point range   */
      real(8) :: rmax, rmin, rmesh

c     /*   number of grid points   */
      integer :: ndvr

c     /*   number of dimension   */
      integer :: ndim

c     /*   matrix dimension   */
      integer :: nmat

c     /*   physical mass   */
      real(8) :: physmass

c     /*   amu mass   */
      real(8) :: amumass

c     /*   grid points   */
      real(8), dimension(:), allocatable :: xdvr
      real(8), dimension(:), allocatable :: ydvr
      real(8), dimension(:), allocatable :: zdvr

c     /*   potential energies   */
      real(8), dimension(:),     allocatable :: v1
      real(8), dimension(:,:),   allocatable :: v2
      real(8), dimension(:,:,:), allocatable :: v3

c     /*   hamiltonian matrix   */
      real(8), dimension(:,:), allocatable :: hdvr

c     /*   eigenvectors   */
      real(8), dimension(:,:), allocatable :: cdvr

c     /*   eigenvalues   */
      real(8), dimension(:), allocatable :: edvr

c     //   potential filename
      character(len=80) :: potfile

c     //   unit conversion factor: length
      real(8), parameter :: bohr2ang  = 0.529177249d0

c     //   unit conversion factor: mass
      real(8), parameter :: amu2au    = 1822.88853006d0

c     /*   file unit   */
      integer :: iounit = 10

c     /*   length unit   */
      character(len=8)  :: unitlength

c***********************************************************************
      end module dvr_variables
c***********************************************************************





c***********************************************************************
      program dvr
c***********************************************************************
c-----------------------------------------------------------------------
c     /*   local variables                                            */
c-----------------------------------------------------------------------

c     /*   common variables   */
      use dvr_variables, only : hdvr, edvr, cdvr, nmat

c     /*   initialize   */
      implicit none

c-----------------------------------------------------------------------
c     //   initial message
c-----------------------------------------------------------------------

c     /*   memory allocation   */
      call dvr_set

c     /*   grid   */
      call dvr_grid

c     /*   potential   */
      call dvr_pot

c     /*   hamiltonian   */
      call dvr_hamiltonian

c     /*   diagonalize hamiltonian   */
      call ddiag ( hdvr, edvr, cdvr, nmat )

c     /*   write output   */
      call dvr_write

      stop
      end





c**********************************************************************
      subroutine dvr_set
c**********************************************************************

c     /*   common variables   */
      use dvr_variables, only :
     &   pi, bohr2ang, amu2au, amumass, physmass, dvrmax, dvrmin,
     &   dvrmesh, potfile, bohr2ang, amu2au, xdvr, ydvr, zdvr, hdvr,
     &   cdvr, edvr, rmax, rmin, rmesh, ndim, ndvr, nmat, unitlength

c     /*   local variables   */
      implicit none

c     /*   integer   */
      integer :: iargc

c     /*   characters   */
      character(len=80) :: char

c-----------------------------------------------------------------------
c     //   initial message
c-----------------------------------------------------------------------

      if ( iargc() .ne. 7 ) then

         write( 6, '(a)' )
         write( 6, '(a)' ) 'Program dvr'
         write( 6, '(a)' )
         write( 6, '(a)' ) 'Usage: dvr.x $1 $2 $3 $4 $5 $6 $7'
         write( 6, '(a)' )
         write( 6, '(a)' ) '$1: dimension (1-3)'
         write( 6, '(a)' ) '$2: mass [amu]'
         write( 6, '(a)' ) '$3: potential filename'
         write( 6, '(a)' ) '$4: DVR mesh: rmin'
         write( 6, '(a)' ) '$5: DVR mesh: rmax'
         write( 6, '(a)' ) '$6: DVR mesh: dr'
         write( 6, '(a)' ) '$7: DVR mesh: A [angstrom] or B [bohr]'
         write( 6, '(a)' )
         write( 6, '(a)' ) 'Example:'
         write( 6, '(a)' ) 'dvr.x 3 1.0078 dvr.dat -1.25 1.25 0.25 A'
         write( 6, '(a)' )
         stop

      end if

c-----------------------------------------------------------------------
c     /*   circular constant                                          */
c-----------------------------------------------------------------------

      pi = acos( - 1.d0 )

c-----------------------------------------------------------------------
c     //   read data
c-----------------------------------------------------------------------

c     //   dimension
      call getarg( 1, char )
      read( char, * ) ndim

c     //   function
      call getarg( 2, char )
      read( char, * ) amumass

c     //   potential file
      call getarg( 3, char )
      potfile = char

c     //   mesh
      call getarg( 4, char )
      read( char, * ) rmin

c     //   mesh
      call getarg( 5, char )
      read( char, * ) rmax

c     //   mesh
      call getarg( 6, char )
      read( char, * ) rmesh

c     //   mesh
      call getarg( 7, char )
      unitlength = char

c-----------------------------------------------------------------------
c     //   unit conversion
c-----------------------------------------------------------------------

c     //   length
      if ( unitlength(1:1) .eq. 'A' ) then
         rmin  = rmin  / bohr2ang
         rmax  = rmax  / bohr2ang
         rmesh = rmesh / bohr2ang
      end if

c     /*   mass   */
      physmass = amumass * amu2au

c-----------------------------------------------------------------------
c     //   meshes
c-----------------------------------------------------------------------

c     /*   number of meshes   */
      dvrmesh = rmesh

c     /*   number of meshes   */
      ndvr = nint( abs( rmax - rmin ) / dvrmesh ) + 1

c     /*   edges   */
      dvrmin = 0.d0

c     /*   edges   */
      dvrmax = dble(ndvr+1) * dvrmesh

c-----------------------------------------------------------------------
c     //   matrix size
c-----------------------------------------------------------------------

c     /*   matrix size   */
      nmat = ndvr**ndim

c-----------------------------------------------------------------------
c     //   meshes
c-----------------------------------------------------------------------

c     /*   memory allocation   */

      allocate( xdvr(ndvr) )
      allocate( ydvr(ndvr) )
      allocate( zdvr(ndvr) )

c     /*   memory allocation   */

      allocate( hdvr(nmat,nmat) )
      allocate( cdvr(nmat,nmat) )
      allocate( edvr(nmat) )

c-----------------------------------------------------------------------
c     //   print
c-----------------------------------------------------------------------

c     //   blank line
      write( 6, '(a)' )

c     //   blank line
      if ( unitlength(1:1) .eq. 'A' ) then
         write( 6, '(a,a)' )  'Units:      angstrom'
      else
         write( 6, '(a,a)' )  'Units:          Bohr'
      end if

c     /*   number of meshes   */
      write( 6, '(a,i8)' ) 'Mesh points:', ndvr

c     //   matrix size
      write( 6, '(a,i8)' ) 'Matrix size:', nmat

c     //   blank line
      write( 6, '(a)' )

      return
      end





c***********************************************************************
      subroutine dvr_grid
c***********************************************************************
c-----------------------------------------------------------------------
c     /*   variables                                                  */
c-----------------------------------------------------------------------

c     /*   common variables   */
      use dvr_variables

c     /*   local variables   */
      implicit none

c     /*   integer   */
      integer :: i

c     /*   real numbers   */
      real(8) :: a, b

c-----------------------------------------------------------------------
c     /*   generate grids:  one dimensional case                      */
c-----------------------------------------------------------------------

      if ( ndim .ge. 1 ) then

c        /*   generate grid points: xdvr   */

         a = dvrmin
         b = dvrmax

         do i = 1, ndvr
            xdvr(i) = a + (b-a) * dble(i) / dble(ndvr+1)
         end do

      end if

c-----------------------------------------------------------------------
c     /*   generate grids:  two dimensional case                      */
c-----------------------------------------------------------------------

      if ( ndim .ge. 2 ) then

c        /*   generate grid points: ydvr   */

         a = dvrmin
         b = dvrmax

         do i = 1, ndvr
            ydvr(i) = a + (b-a) * dble(i) / dble(ndvr+1)
         end do

      end if

c-----------------------------------------------------------------------
c     /*   generate grids:  three dimensional case                    */
c-----------------------------------------------------------------------

      if ( ndim .ge. 3 ) then

c        /*   generate grid points: zdvr   */

         a = dvrmin
         b = dvrmax

         do i = 1, ndvr
            zdvr(i) = a + (b-a) * dble(i) / dble(ndvr+1)
         end do

      end if

      return
      end





c***********************************************************************
      subroutine dvr_pot
c***********************************************************************
c-----------------------------------------------------------------------
c     /*   variables                                                  */
c-----------------------------------------------------------------------

c     /*   common variables   */
      use dvr_variables, only : v1, v2, v3, ndim, ndvr, iounit, potfile

c     /*   local variables   */
      implicit none

c     /*   integer   */
      integer :: i, j, k

c     /*   integer   */
      integer :: ierr = 1

c     /*   real numbers   */
      real(8) :: dum

c-----------------------------------------------------------------------
c     /*   potential:  one dimensional case                           */
c-----------------------------------------------------------------------

      if ( ndim .eq. 1 ) then

         allocate( v1(ndvr) )

         open ( iounit, file=potfile )

         do i = 1, ndvr
            read( iounit, *, iostat=ierr ) dum, v1(i)
         end do

         close( iounit )

c-----------------------------------------------------------------------
c     /*   potential:  two dimensional case                           */
c-----------------------------------------------------------------------

      else if ( ndim .eq. 2 ) then

         allocate( v2(ndvr,ndvr) )

         open ( iounit, file=potfile )

         do j = 1, ndvr
         do i = 1, ndvr
            read( iounit, *, iostat=ierr ) dum, v2(i,j)
         end do
         end do

         close( iounit )

c-----------------------------------------------------------------------
c     /*   potential:  three dimensional case                         */
c-----------------------------------------------------------------------

      else if ( ndim .eq. 3 ) then

         allocate( v3(ndvr,ndvr,ndvr) )

         open ( iounit, file=potfile )

         do k = 1, ndvr
         do j = 1, ndvr
         do i = 1, ndvr
            read( iounit, *, iostat=ierr ) dum, v3(i,j,k)
         end do
         end do
         end do

         close( iounit )

      end if

      call error_handling( ierr, 'potential file', 14 )

      return
      end





c***********************************************************************
      subroutine dvr_hamiltonian
c***********************************************************************
c-----------------------------------------------------------------------
c     /*   variables                                                  */
c-----------------------------------------------------------------------

c     /*   common variables   */
      use dvr_variables, only :
     &   dvrmin, dvrmax, physmass, pi, v1, v2, v3, hdvr, ndvr, ndim

c     /*   local variables   */
      implicit none

c     /*   integer   */
      integer :: i, j, ix, iy, iz, jx, jy, jz

c     /*   real numbers   */
      real(8) :: a, b, const, pm, sinm, sinp, di, dn

c     /*   real numbers   */
      real(8) :: d(ndvr,ndvr), t(ndvr,ndvr)

c-----------------------------------------------------------------------
c     /*   constant                                                   */
c-----------------------------------------------------------------------

      a = dvrmin
      b = dvrmax

      const = 1.d0 / (2.d0*physmass) / (b-a)**2 * pi**2 / 2.d0

c-----------------------------------------------------------------------
c     /*   kronecker delta                                            */
c-----------------------------------------------------------------------

      d(:,:) = 0.d0

      do i = 1, ndvr
         d(i,i) = 1.d0
      end do

c-----------------------------------------------------------------------
c     /*   generate kinetic energy matrix                             */
c-----------------------------------------------------------------------

      do i = 1, ndvr
      do j = 1, ndvr

         if ( i .eq. j ) then

            di = dble( i )
            dn = dble( ndvr + 1 )

            sinm = dsin( pi*di/dn )

            t(i,i) = const * ( (2.d0*dn**2+1.d0)/3.d0 - 1.d0/sinm**2 )

         else

            if ( mod(iabs(i-j),2) .eq. 0 ) then
               pm =   1.d0
            else
               pm = - 1.d0
            end if

            sinm = sin( pi * dble(i-j) / dble(2*ndvr+2) )
            sinp = sin( pi * dble(i+j) / dble(2*ndvr+2) )

            t(i,j) = const * pm * ( 1.d0/sinm**2 - 1.d0/sinp**2 )

         end if

      end do
      end do

c-----------------------------------------------------------------------
c     /*   generate hamiltonian matrix:  one dimensional case         */
c-----------------------------------------------------------------------

      if ( ndim .eq. 1 ) then

         i = 0

         do ix = 1, ndvr

            i = i + 1

            j = 0

            do jx = 1, ndvr

               j = j + 1

               hdvr(i,j) =  t(ix,jx) + v1(ix)*d(ix,jx)

            end do

         end do

c-----------------------------------------------------------------------
c     /*   generate hamiltonian matrix:  two dimensional case         */
c-----------------------------------------------------------------------

      else if ( ndim .eq. 2 ) then

         i = 0

         do ix = 1, ndvr
         do iy = 1, ndvr

            i = i + 1

            j = 0

            do jx = 1, ndvr
            do jy = 1, ndvr

               j = j + 1

               hdvr(i,j) =  t(ix,jx)*d(iy,jy)
     $                   +  t(iy,jy)*d(ix,jx)
     $                   + v2(ix,iy)*d(ix,jx)*d(iy,jy)

            end do
            end do

         end do
         end do

c-----------------------------------------------------------------------
c     /*   generate hamiltonian matrix:  three dimensional case       */
c-----------------------------------------------------------------------

      else if ( ndim .eq. 3 ) then

         i = 0

         do ix = 1, ndvr
         do iy = 1, ndvr
         do iz = 1, ndvr

            i = i + 1

            j = 0

            do jx = 1, ndvr
            do jy = 1, ndvr
            do jz = 1, ndvr

               j = j + 1

               hdvr(i,j) =  t(ix,jx)*d(iy,jy)*d(iz,jz)
     $                   +  t(iy,jy)*d(iz,jz)*d(ix,jx)
     $                   +  t(iz,jz)*d(ix,jx)*d(iy,jy)
     $                   + v3(ix,iy,iz)*d(ix,jx)*d(iy,jy)*d(iz,jz)

               end do
               end do

            end do
            end do

         end do
         end do

      end if

      return
      end





c**********************************************************************
      subroutine dvr_write
c**********************************************************************

c-----------------------------------------------------------------------
c     /*   variables                                                  */
c-----------------------------------------------------------------------

c     /*   common variables   */
      use dvr_variables, only : edvr, nmat

c     /*   local variables   */
      implicit none

c     /*   integer   */
      integer :: i

      real(8), parameter :: har2kcal = 627.50960d0
      real(8), parameter :: har2ev   = 27.211399d0
      real(8), parameter :: har2kj   = 2625.5002d0

      real(8) :: ei

c-----------------------------------------------------------------------
c     /*   eigenvalues                                                */
c-----------------------------------------------------------------------

      write( 6, '(a)' ) '------------------------------------------'
      write( 6, '(a)' ) '     i     Ei [au]  Ei-E0 [au] Ei-E0 [meV]'
      write( 6, '(a)' ) '------------------------------------------'

      do i = 1, 10

         ei = edvr(i) - edvr(1)

         write ( 6, '(i6,2f12.6,f12.3)' )
     &      i-1, edvr(i), ei, ei*har2ev*1000.d0

      end do

      write( 6, '(a)' )

      return
      end





c***********************************************************************
      subroutine ddiag ( a, w, v, n )
c***********************************************************************

      implicit none

c-----------------------------------------------------------------------
c     /*   local variables                                            */
c-----------------------------------------------------------------------

      integer          :: i, j, n, info

      character(len=1) :: jobz, uplo

      real(8)          :: a(n,n), b(n,n), w(n), work(4*n), v(n,n)

c-----------------------------------------------------------------------
c     /*   start                                                      */
c-----------------------------------------------------------------------

      jobz = 'V'
      uplo = 'U'

      do j = 1, n
      do i = 1, n
         b(i,j) = a(i,j)
      end do
      end do

      call dsyev( jobz, uplo, n, a, n, w, work, 4*n, info )

      do j = 1, n
      do i = 1, n
         v(i,j) = a(i,j)
         a(i,j) = b(i,j)
      end do
      end do

      return
      end





c***********************************************************************
      subroutine error_handling ( ierr, char_tag, length_tag )
c***********************************************************************

c-----------------------------------------------------------------------
c     /*   local variables                                            */
c-----------------------------------------------------------------------

      implicit none

      integer:: ierr, length_tag

      character(len=length_tag) :: char_tag

c-----------------------------------------------------------------------
c     /*   start                                                      */
c-----------------------------------------------------------------------

      if ( ierr .ne. 0 ) then

         write(6,'(a)') 'Error termination at: ' // char_tag // '.'

         write(6,'(a)')

         stop

      end if

      return
      end
