c***********************************************************************
      program dcdbox2box
c***********************************************************************
c-----------------------------------------------------------------------
c     /*   local variables                                            */
c-----------------------------------------------------------------------

c     /*   initialize   */
      implicit none

c     /*   number of steps   */
      integer :: nstep

c     /*   number of atoms   */
      integer :: natom

c     /*   number of beads   */
      integer :: nbead

c     /*   file units   */
      integer :: iounit     = 10
      integer :: iounit_box = 11
      integer :: iounit_vel = 12
      integer :: iounit_out = 13

c     /*   box   */
      real(8), dimension(3,3) :: box, box_avg

c     /*   virial   */
      real(8), dimension(3,3) :: vir, vir_avg

c     /*   volume   */
      real(8) :: volume, volume_avg

c     /*   pressure   */
      real(8), dimension(3,3) :: pres, pres_avg

c     /*   isotropic pressure   */
      real(8) :: pres_iso, pres_iso_avg

c     /*   determinant   */
      real(8) :: det3

c     //   filenames
      character(len=80) :: boxfile
      character(len=80) :: velfile
      character(len=80) :: strfile
      character(len=80) :: outfile
      character(len=80) :: deffile

c     /*   velocity   */
      real(8), dimension(:,:), allocatable :: vx, vy, vz

c     //   atomic species
      character(len=8), dimension(:), allocatable :: species

c     //   atomic mass
      real(8), dimension(:), allocatable :: physmass

c     /*   number of atomic symbols   */
      integer :: nsymbol

c     /*   atomic symbols   */
      character(len=8), dimension(:), allocatable :: symbol

c     /*   atomic masses   */
      real(8), dimension(:), allocatable :: physmass_symbol

c     //   unit conversion factors
      real(8), parameter :: au_time   = 0.024188843d-15
      real(8), parameter :: amu_mass  = 1.6605402d-27
      real(8), parameter :: au_mass   = 9.1093897d-31
      real(8), parameter :: au_length = 0.529177249d-10

c     /*   integers   */
      integer :: i, j, l, istep, ierr

c     //   characters
      character(len=80) :: char

c     //   boundary condition
      integer :: iboundary = 1

c     /*   real numbers   */
      real(8) :: vxc, vyc, vzc, const_1

c     /*   real numbers   */
      real(4) :: ax, ay, az

c     //   header variables
      real(4) :: vreal4
      real(8) :: vreal8
      integer(4) :: vint4
      character(len=4) :: vchar4
      character(len=80) :: vchar80

c-----------------------------------------------------------------------
c     //   initial message
c-----------------------------------------------------------------------

      if ( iargc() .ne. 7 ) then

         write( 6, '(a)' )
         write( 6, '(a)' ) 'Program dcdbox2box'
         write( 6, '(a)' )
         write( 6, '(a)' ) 'Usage: dcdbox2box.x $1 $2 $3'
         write( 6, '(a)' )
         write( 6, '(a)' ) '$1: dcd box file (box.dcd)'
         write( 6, '(a)' ) '$2: dcd velocity file (vel.dcd)'
         write( 6, '(a)' ) '$3: structure file (structure.dat)'
         write( 6, '(a)' ) '$4: default file (input_default.dat)'
         write( 6, '(a)' ) '$5: output box file (box.out)'
         write( 6, '(a)' ) '$6: number of steps'
         write( 6, '(a)' ) '$7: number of beads'
         write( 6, '(a)' )

         stop

      else

         write( 6, '(a)' )
         write( 6, '(a)' ) 'Program dcdbox2box'
         write( 6, '(a)' )

      end if

c-----------------------------------------------------------------------
c     //   read arguments
c-----------------------------------------------------------------------

      call getarg( 1, boxfile )
      call getarg( 2, velfile )
      call getarg( 3, strfile )
      call getarg( 4, deffile )
      call getarg( 5, outfile )

      call getarg( 6, char )
      read( char, * ) nstep

      call getarg( 7, char )
      read( char, * ) nbead

c-----------------------------------------------------------------------
c     //   number of atoms
c-----------------------------------------------------------------------

c     //   open file
      open ( iounit, file = trim(strfile) )

c     //   number of atoms
      read ( iounit, *, iostat=ierr ) natom

c     //   close file
      close( iounit )

c-----------------------------------------------------------------------
c     //   memory allocation
c-----------------------------------------------------------------------

c     //   atomic mass
      allocate( physmass(natom) )

c     //   velocity
      allocate( vx(natom,nbead) )
      allocate( vy(natom,nbead) )
      allocate( vz(natom,nbead) )

c-----------------------------------------------------------------------
c     /*   read atomic symbols                                        */
c-----------------------------------------------------------------------

c     //   atomic species
      allocate( species(natom) )

c     //   open file
      open ( iounit, file = trim(strfile) )

c     //   number of atoms
      read ( iounit, *, iostat=ierr )

c     //   comment line
      read ( iounit, *, iostat=ierr )

c     //   loop of atoms
      do i = 1, natom

c        //   atomic symbol
         read( iounit, *, iostat=ierr ) species(i)

c     //   loop of atoms
      end do

c     //   close file
      close( iounit )

c     //   atomic masses
      physmass(:) = 0.d0

c     /*   file open   */
      open ( iounit, file = trim(deffile) )

c     /*   search for tag    */
      call search_tag ( '<nsymbol>', 9, iounit, ierr )

c     /*   number of atomic symbols   */
      read ( iounit, *, iostat=ierr ) nsymbol

c     /*   file close   */
      close( iounit )

c     /*   memory allocation: atomic symbols   */
      allocate( symbol(nsymbol) )

c     /*   memory allocation: atomic masses   */
      allocate( physmass_symbol(nsymbol) )

c     /*   file open   */
      open ( iounit, file = trim(deffile) )

c     /*   search for tag    */
      call search_tag ( '<nsymbol>', 9, iounit, ierr )

c     /*   number of atomic symbols   */
      read ( iounit, *, iostat=ierr )

c     /*   loop of symbols   */
      do i = 1, nsymbol

c        /*   read symbol, atomic number, atomic mass   */
         read ( iounit, *, iostat=ierr )
     &      symbol(i), j, physmass_symbol(i)

c     /*   loop of symbols   */
      end do

c     /*   file close   */
      close( iounit )

c     //   atomic masses
      physmass(:) = 0.d0

c     /*   loop of atoms   */
      do i = 1, natom

c        //   flag
         ierr = 1

c        /*   loop of symbols   */
         do l = 1, nsymbol

c           /*   if symbol matched   */
            if ( species(i)(1:8) .eq. symbol(l)(1:8) ) then

c              /*   substitute mass   */
               physmass(i) = physmass_symbol(l)*amu_mass/au_mass

c              //   flag
               ierr = 0

c              /*   go to next loop   */
               exit

c           /*   if symbol matched   */
            end if

c        /*   loop of symbols   */
         end do

c        /*   go to next loop   */
         if ( ierr .ne. 0 ) exit

c     /*   loop of atoms   */
      end do

c     /*   memory allocation: atomic symbols   */
      deallocate( symbol )

c     /*   memory allocation: atomic masses   */
      deallocate( physmass_symbol )

c     //   atomic species
      deallocate( species )

c-----------------------------------------------------------------------
c     //   check number of steps
c-----------------------------------------------------------------------

c     //   read number of steps
      if ( nstep .le. 0 ) then

c        //   open file
         open ( iounit_box, file = trim(boxfile), form = 'unformatted',
     &          access = 'stream', status = 'old' )

c        //   error flag
         ierr = 0

c        //   read header
         read( iounit_box, iostat=ierr ) vint4

c        //   check endian
         if ( vint4 .ne. 84 ) ierr = 1

c        //   detect error
         call error_handling( ierr, 'endian', 6 )

         read( iounit_box, iostat=ierr ) vchar4
         read( iounit_box, iostat=ierr ) vint4
         read( iounit_box, iostat=ierr ) vint4
         read( iounit_box, iostat=ierr ) vint4
         read( iounit_box, iostat=ierr ) vint4
         read( iounit_box, iostat=ierr ) vint4
         read( iounit_box, iostat=ierr ) vint4, vint4, vint4
         read( iounit_box, iostat=ierr ) vint4
         read( iounit_box, iostat=ierr ) vreal4
         read( iounit_box, iostat=ierr ) vint4
         read( iounit_box, iostat=ierr ) vint4, vint4, vint4, vint4,
     &                                   vint4, vint4, vint4, vint4
         read( iounit_box, iostat=ierr ) vint4
         read( iounit_box, iostat=ierr ) vint4
         read( iounit_box, iostat=ierr ) vint4
         read( iounit_box, iostat=ierr ) vint4
         read( iounit_box, iostat=ierr ) vchar80
         read( iounit_box, iostat=ierr ) vchar80
         read( iounit_box, iostat=ierr ) vint4
         read( iounit_box, iostat=ierr ) vint4
         read( iounit_box, iostat=ierr ) vint4
         read( iounit_box, iostat=ierr ) vint4

c        //   detect error
         call error_handling( ierr, 'dcd box file', 12 )

c        //   step counter
         istep = 0

c        //   counts
         do

c           //   read lines
            read( iounit_box, iostat=ierr ) j
            read( iounit_box, iostat=ierr ) box(1,1:3)
            read( iounit_box, iostat=ierr ) box(2,1:3)
            read( iounit_box, iostat=ierr ) box(3,1:3)
            read( iounit_box, iostat=ierr ) vir(1,1:3)
            read( iounit_box, iostat=ierr ) vir(2,1:3)
            read( iounit_box, iostat=ierr ) vir(3,1:3)

c           //   exit on read error
            if ( ierr .ne. 0 ) exit

c           //   update counter
            istep = istep + 1

c        //   counts
         end do

c        //   close file
         close( iounit_box )

c        //   number of steps
         nstep = istep

c        //   print
         write( 6, '(a,i8)' ) 'Number of steps:', nstep
         write( 6, '(a)' )

c     //   read number of steps
      end if

c-----------------------------------------------------------------------
c     //   open files
c-----------------------------------------------------------------------

c     //   open dcd box file
      open ( iounit_box, file = trim(boxfile), form = 'unformatted',
     &       access = 'stream', status = 'old' )

c     //   open dcd velocity file
      open ( iounit_vel, file = trim(velfile), form = 'unformatted',
     &       access = 'stream', status = 'old' )

c     //   open output file
      open ( iounit_out, file = trim(outfile) )

c-----------------------------------------------------------------------
c     //   skip dcd header
c-----------------------------------------------------------------------

c     //   file units
      iounit = 0

c     //   two files
      do i = 1, 2

c        //   file units
         if ( i .eq. 1 ) iounit = iounit_box
         if ( i .eq. 2 ) iounit = iounit_vel

c        //   read header
         read( iounit, iostat=ierr ) vint4
         read( iounit, iostat=ierr ) vchar4
         read( iounit, iostat=ierr ) vint4
         read( iounit, iostat=ierr ) vint4
         read( iounit, iostat=ierr ) vint4
         read( iounit, iostat=ierr ) vint4
         read( iounit, iostat=ierr ) vint4
         read( iounit, iostat=ierr ) vint4, vint4, vint4
         read( iounit, iostat=ierr ) vint4
         read( iounit, iostat=ierr ) vreal4
         read( iounit, iostat=ierr ) vint4
         read( iounit, iostat=ierr ) vint4, vint4, vint4, vint4,
     &                               vint4, vint4, vint4, vint4
         read( iounit, iostat=ierr ) vint4
         read( iounit, iostat=ierr ) vint4
         read( iounit, iostat=ierr ) vint4
         read( iounit, iostat=ierr ) vint4
         read( iounit, iostat=ierr ) vchar80
         read( iounit, iostat=ierr ) vchar80
         read( iounit, iostat=ierr ) vint4
         read( iounit, iostat=ierr ) vint4
         read( iounit, iostat=ierr ) vint4
         read( iounit, iostat=ierr ) vint4

c        //   detect error
         if ( i .eq. 1 ) call error_handling( ierr, 'dcd box file', 12)
         if ( i .eq. 2 ) call error_handling( ierr, 'dcd vel file', 12)

c     //   two files
      end do

c-----------------------------------------------------------------------
c     //   main loop
c-----------------------------------------------------------------------

c     //   volume average
      volume_avg = 0.d0

c     //   box average
      box_avg(:,:) = 0.d0

c     //   virial average
      vir_avg(:,:) = 0.d0

c     //   pressure average
      pres_avg(:,:) = 0.d0

c     //   isotropic pressure average
      pres_iso_avg = 0.d0

c     //   loop of steps
      do istep = 1, nstep

c-----------------------------------------------------------------------
c        //   read velocity
c-----------------------------------------------------------------------

c        //   read line
         read( iounit_box, iostat=ierr ) j

c        //   read box
         read( iounit_box, iostat=ierr ) box(1,1:3)
         read( iounit_box, iostat=ierr ) box(2,1:3)
         read( iounit_box, iostat=ierr ) box(3,1:3)

c        //   read virial
         read( iounit_box, iostat=ierr ) vir(1,1:3)
         read( iounit_box, iostat=ierr ) vir(2,1:3)
         read( iounit_box, iostat=ierr ) vir(3,1:3)

c        //   detect error
         call error_handling( ierr, 'dcd box file', 12)

c-----------------------------------------------------------------------
c        //   read velocity
c-----------------------------------------------------------------------

c        //   skip lines
         if ( iboundary .ne. 0 ) then
            read( iounit_vel, iostat=ierr ) vint4
            read( iounit_vel, iostat=ierr )
     &         vreal8, vreal8, vreal8, vreal8, vreal8, vreal8
            read( iounit_vel, iostat=ierr ) vint4
         end if

c        //   unit conversion factor
         const_1 = 1.d-2 * au_length / au_time

c        //   skip line
         read( iounit_vel, iostat=ierr ) vint4

c        /*   loop of beads   */
         do j = 1, nbead

c        /*   loop of atoms   */
         do i = 1, natom

c           /*   position in angstrom   */
            read( iounit_vel, iostat=ierr ) ax

c           //   unit conversion
            vx(i,j) = dble( ax ) / const_1

c        /*   loop of atoms   */
         end do

c        /*   loop of beads   */
         end do

c        //   skip lines
         read( iounit_vel, iostat=ierr ) vint4
         read( iounit_vel, iostat=ierr ) vint4

c        /*   loop of beads   */
         do j = 1, nbead

c        /*   loop of atoms   */
         do i = 1, natom

c           /*   position in angstrom   */
            read( iounit_vel ) ay

c           //   unit conversion
            vy(i,j) = dble( ay ) / const_1

c        /*   loop of atoms   */
         end do

c        /*   loop of beads   */
         end do

c        //   skip lines
         read( iounit_vel, iostat=ierr ) vint4
         read( iounit_vel, iostat=ierr ) vint4

c        /*   loop of beads   */
         do j = 1, nbead

c        /*   loop of atoms   */
         do i = 1, natom

c           /*   position in angstrom   */
            read( iounit_vel ) az

c           //   unit conversion
            vz(i,j) = dble( az ) / const_1

c        /*   loop of atoms   */
         end do

c        /*   loop of beads   */
         end do

c        //   skip line
         read( iounit_vel, iostat=ierr ) vint4

c        //   detect error
         call error_handling( ierr, 'dcd vel file', 12 )

c-----------------------------------------------------------------------
c        //   volume
c-----------------------------------------------------------------------

c        //   volume
         volume = det3( box )

c-----------------------------------------------------------------------
c        //   pressure
c-----------------------------------------------------------------------

c        //   pressure
         pres(:,:) =  vir(:,:)

c        //   loop of atoms
         do i = 1, natom

c           //   centroid velocity
            vxc = 0.d0
            vyc = 0.d0
            vzc = 0.d0
            do j = 1, nbead
               vxc = vxc + vx(i,j) / dble(nbead)
               vyc = vyc + vy(i,j) / dble(nbead)
               vzc = vzc + vz(i,j) / dble(nbead)
           end do

c          //   kinetic energy contribution
           pres(1,1) = pres(1,1) + physmass(i)*vxc*vxc
           pres(1,2) = pres(1,2) + physmass(i)*vxc*vyc
           pres(1,3) = pres(1,3) + physmass(i)*vxc*vzc
           pres(2,1) = pres(2,1) + physmass(i)*vyc*vxc
           pres(2,2) = pres(2,2) + physmass(i)*vyc*vyc
           pres(2,3) = pres(2,3) + physmass(i)*vyc*vzc
           pres(3,1) = pres(3,1) + physmass(i)*vzc*vxc
           pres(3,2) = pres(3,2) + physmass(i)*vzc*vyc
           pres(3,3) = pres(3,3) + physmass(i)*vzc*vzc

c        //   loop of atoms
         end do

c        //   divide by volume
         pres(:,:) = pres(:,:) / volume

c        //   isotropic pressure
         pres_iso = ( pres(1,1) + pres(2,2) + pres(3,3) ) / 3.d0

c-----------------------------------------------------------------------
c        //   averages
c-----------------------------------------------------------------------

c        //   volume average
         volume_avg = ( volume_avg*dble(istep-1) + volume )
     &              / dble(istep)

c        //   box average
         box_avg(:,:) = ( box_avg(:,:)*dble(istep-1) + box(:,:) )
     &                / dble(istep)

c        //   pressure average
         pres_avg(:,:) = ( pres_avg(:,:)*dble(istep-1) + pres(:,:) )
     &                / dble(istep)

c        //   isotropic pressure average
         pres_iso_avg = ( pres_iso_avg*dble(istep-1) + pres_iso )
     &                / dble(istep)

c-----------------------------------------------------------------------
c        //   print output
c-----------------------------------------------------------------------

         write(iounit_out, '(i8,2f16.2)' ) istep, volume, volume_avg
         write(iounit_out, '(8x,6f16.6)' ) box(1,1:3), box_avg(1,1:3)
         write(iounit_out, '(8x,6f16.6)' ) box(2,1:3), box_avg(2,1:3)
         write(iounit_out, '(8x,6f16.6)' ) box(3,1:3), box_avg(3,1:3)
         write(iounit_out, '(i8,2f16.12)') istep, pres_iso, pres_iso_avg
         write(iounit_out, '(8x,6f16.12)') pres(1,1:3), pres_avg(1,1:3)
         write(iounit_out, '(8x,6f16.12)') pres(2,1:3), pres_avg(2,1:3)
         write(iounit_out, '(8x,6f16.12)') pres(3,1:3), pres_avg(3,1:3)

c     //   loop of steps
      end do

c-----------------------------------------------------------------------
c     //  close files
c-----------------------------------------------------------------------

      close( iounit_box )
      close( iounit_vel )
      close( iounit_out )

c-----------------------------------------------------------------------
c     //   end of program
c-----------------------------------------------------------------------

      write( 6, '(a)' ) 'Normal termination of dcdbox2box.'
      write( 6, '(a)' )

      return
      end






c***********************************************************************
      real(8) function det3 ( a )
c***********************************************************************

c-----------------------------------------------------------------------
c     /*   local variables                                            */
c-----------------------------------------------------------------------

      implicit none

      real(8) :: a(3,3)

c-----------------------------------------------------------------------
c     /*   start                                                      */
c-----------------------------------------------------------------------

      det3 = + a(1,1)*a(2,2)*a(3,3) - a(1,1)*a(2,3)*a(3,2)
     &       + a(2,1)*a(3,2)*a(1,3) - a(2,1)*a(1,2)*a(3,3)
     &       + a(3,1)*a(1,2)*a(2,3) - a(3,1)*a(2,2)*a(1,3)

      return
      end





c***********************************************************************
      subroutine error_handling ( ierr, char_tag, length_tag )
c***********************************************************************

c-----------------------------------------------------------------------
c     /*   local variables                                            */
c-----------------------------------------------------------------------

      implicit none

      integer:: ierr, length_tag

      character(len=length_tag) :: char_tag

c-----------------------------------------------------------------------
c     /*   start                                                      */
c-----------------------------------------------------------------------

      if ( ierr .ne. 0 ) then

         write(6,'(a)') 'Error termination at: ' // char_tag // '.'

         write(6,'(a)')

         stop

      end if

      return
      end





c***********************************************************************
      subroutine search_tag ( char_tag, length_tag, iounit, ierr )
c***********************************************************************

c-----------------------------------------------------------------------
c     /*   local variables                                            */
c-----------------------------------------------------------------------

      implicit none

      integer:: k, length_tag, iounit, ierr

      character(len=80) :: char_line

      character(len=length_tag) :: char_tag

c-----------------------------------------------------------------------
c     /*   do loop start                                              */
c-----------------------------------------------------------------------

      do

c-----------------------------------------------------------------------
c        /*   read a line                                             */
c-----------------------------------------------------------------------

c         read (iounit,'(a80)',iostat=ierr) char_line
         read (iounit,*,iostat=ierr) char_line

c-----------------------------------------------------------------------
c        /*   return if error is found                                */
c-----------------------------------------------------------------------

         if ( ierr .ne. 0 ) return

c-----------------------------------------------------------------------
c        /*   search for the tag                                      */
c-----------------------------------------------------------------------

         k = index(char_line(1:length_tag),char_tag(1:length_tag))

c-----------------------------------------------------------------------
c        /*   return as soon as we find the tag                       */
c-----------------------------------------------------------------------

         if ( k .ge. 1 ) return

c-----------------------------------------------------------------------
c     /*   do loop end                                                */
c-----------------------------------------------------------------------

      end do

      return
      end
