c***********************************************************************
      program dcd2trj
c***********************************************************************
c-----------------------------------------------------------------------
c     /*   local variables                                            */
c-----------------------------------------------------------------------

c     /*   initialize   */
      implicit none

c     /*   integers   */
      integer :: natom, nbead, nstep

c     /*   file units   */
      integer :: iounit_pos = 11
      integer :: iounit_vel = 12
      integer :: iounit_frc = 13
      integer :: iounit_pot = 14
      integer :: iounit_trj = 15

c     /*   position, velocity, force, potential   */
      real(8), dimension(:,:), allocatable :: x, y, z
      real(8), dimension(:,:), allocatable :: vx, vy, vz
      real(8), dimension(:,:), allocatable :: fx, fy, fz
      real(8), dimension(:), allocatable :: pot

c     //   constants
      real(8), parameter:: au_length  = 0.529177249d-10
      real(8), parameter:: au_time    = 0.024188843d-15

c     /*   boundary condtion   */
      integer :: iboundary = 0

c     /*   character   */
      character(len=80) :: posfile, velfile, frcfile, potfile, trjfile

c     /*   real numbers   */
      real(4) :: ax, ay, az

c     /*   real numbers   */
      real(8) :: const_1, b

c     /*   integers   */
      integer :: i, j, istep, iounit, ierr

c     //   characters
      character(len=80) :: char

c     //   header variables
      real(4) :: vreal4
      real(8) :: vreal8
      integer(4) :: vint4
      character(len=4) :: vchar4
      character(len=80) :: vchar80

c-----------------------------------------------------------------------
c     //   initial message
c-----------------------------------------------------------------------

      if ( iargc() .ne. 9 ) then

         write( 6, '(a)' )
         write( 6, '(a)' ) 'Program dcd2trj'
         write( 6, '(a)' )
         write( 6, '(a)' ) 'Usage: dcd2trj.x $1 $2 $3 $4 $5 $6 $7 $8 $9'
         write( 6, '(a)' )
         write( 6, '(a)' ) '$1: dcd position file (trj.dcd)'
         write( 6, '(a)' ) '$2: dcd velocity file (vel.dcd)'
         write( 6, '(a)' ) '$3: dcd force file (force.dcd)'
         write( 6, '(a)' ) '$4: dcd potential file (pot.dcd)'
         write( 6, '(a)' ) '$5: output trajectory file (trj.out)'
         write( 6, '(a)' ) '$6: number of atoms'
         write( 6, '(a)' ) '$7: number of beads'
         write( 6, '(a)' ) '$8: number of steps'
         write( 6, '(a)' ) '$9: boundary = 0: free, 1: periodic'
         write( 6, '(a)' )

         stop

      else

         write( 6, '(a)' )
         write( 6, '(a)' ) 'Program dcd2trj'
         write( 6, '(a)' )

      end if

c-----------------------------------------------------------------------
c     //   read arguments
c-----------------------------------------------------------------------

      call getarg( 1, posfile )
      call getarg( 2, velfile )
      call getarg( 3, frcfile )
      call getarg( 4, potfile )
      call getarg( 5, trjfile )

      call getarg( 6, char )
      read( char, * ) natom

      call getarg( 7, char )
      read( char, * ) nbead

      call getarg( 8, char )
      read( char, * ) nstep

      call getarg( 9, char )
      read( char, * ) iboundary

c-----------------------------------------------------------------------
c     //   memory allocation
c-----------------------------------------------------------------------

c     //   position
      allocate( x(natom,nbead) )
      allocate( y(natom,nbead) )
      allocate( z(natom,nbead) )

c     //   velocity
      allocate( vx(natom,nbead) )
      allocate( vy(natom,nbead) )
      allocate( vz(natom,nbead) )

c     //   force
      allocate( fx(natom,nbead) )
      allocate( fy(natom,nbead) )
      allocate( fz(natom,nbead) )

c     //   potential
      allocate( pot(nbead) )

c-----------------------------------------------------------------------
c     //   open file
c-----------------------------------------------------------------------

      open( iounit_pos, file = posfile, form = 'unformatted',
     &      access = 'stream', status = 'old' )

      open( iounit_vel, file = velfile, form = 'unformatted',
     &      access = 'stream', status = 'old' )

      open( iounit_frc, file = frcfile, form = 'unformatted',
     &      access = 'stream', status = 'old' )

      open( iounit_pot, file = potfile, form = 'unformatted',
     &      access = 'stream', status = 'old' )

      open( iounit_trj, file = trjfile )

c-----------------------------------------------------------------------
c     //   read header
c-----------------------------------------------------------------------

c     //   file units
      iounit = 0

c     //   four files
      do i = 1, 4

c        //   file units
         if ( i .eq. 1 ) iounit = iounit_pos
         if ( i .eq. 2 ) iounit = iounit_vel
         if ( i .eq. 3 ) iounit = iounit_frc
         if ( i .eq. 4 ) iounit = iounit_pot

c        //   read header
         read( iounit, iostat=ierr ) vint4

c        //   error flag
         ierr = 0

c        //   read header
         read( iounit, iostat=ierr ) vint4

c        //   check endian
         if ( vint4 .ne. 84 ) ierr = 1

c        //   detect error
         call error_handling( ierr, 'endian', 6 )

c        //   read header
         read( iounit, iostat=ierr ) vchar4
         read( iounit, iostat=ierr ) vint4
         read( iounit, iostat=ierr ) vint4
         read( iounit, iostat=ierr ) vint4
         read( iounit, iostat=ierr ) vint4
         read( iounit, iostat=ierr ) vint4
         read( iounit, iostat=ierr ) vint4, vint4, vint4
         read( iounit, iostat=ierr ) vint4
         read( iounit, iostat=ierr ) vreal4
         read( iounit, iostat=ierr ) vint4
         read( iounit, iostat=ierr ) vint4, vint4, vint4, vint4,
     &                               vint4, vint4, vint4, vint4
         read( iounit, iostat=ierr ) vint4
         read( iounit, iostat=ierr ) vint4
         read( iounit, iostat=ierr ) vint4
         read( iounit, iostat=ierr ) vint4
         read( iounit, iostat=ierr ) vchar80
         read( iounit, iostat=ierr ) vchar80
         read( iounit, iostat=ierr ) vint4
         read( iounit, iostat=ierr ) vint4
         read( iounit, iostat=ierr ) vint4
         read( iounit, iostat=ierr ) vint4

c     //   four files
      end do

c-----------------------------------------------------------------------
c     //   main loop
c-----------------------------------------------------------------------

      do istep = 1, nstep

c-----------------------------------------------------------------------
c        //   read position
c-----------------------------------------------------------------------

c        //   skip lines
         if ( iboundary .ne. 0 ) then
            read( iounit_pos, iostat=ierr ) vint4
            read( iounit_pos, iostat=ierr )
     &         vreal8, vreal8, vreal8, vreal8, vreal8, vreal8
            read( iounit_pos, iostat=ierr ) vint4
         end if

c        //   unit conversion factor
         const_1 = au_length * 1.d+10

c        //   skip line
         read( iounit_pos, iostat=ierr ) vint4

c        /*   loop of beads   */
         do j = 1, nbead

c        /*   loop of atoms   */
         do i = 1, natom

c           /*   position in angstrom   */
            read( iounit_pos, iostat=ierr ) ax

c           //   unit conversion
            x(i,j) = dble( ax ) / const_1

c        /*   loop of atoms   */
         end do

c        /*   loop of beads   */
         end do

c        //   skip lines
         read( iounit_pos, iostat=ierr ) vint4
         read( iounit_pos, iostat=ierr ) vint4

c        /*   loop of beads   */
         do j = 1, nbead

c        /*   loop of atoms   */
         do i = 1, natom

c           /*   position in angstrom   */
            read( iounit_pos, iostat=ierr ) ay

c           //   unit conversion
            y(i,j) = dble( ay ) / const_1

c        /*   loop of atoms   */
         end do

c        /*   loop of beads   */
         end do

c        //   skip lines
         read( iounit_pos, iostat=ierr ) vint4
         read( iounit_pos, iostat=ierr ) vint4

c        /*   loop of beads   */
         do j = 1, nbead

c        /*   loop of atoms   */
         do i = 1, natom

c           /*   position in angstrom   */
            read( iounit_pos, iostat=ierr ) az

c           //   unit conversion
            z(i,j) = dble( az ) / const_1

c        /*   loop of atoms   */
         end do

c        /*   loop of beads   */
         end do

c        //   skip line
         read( iounit_pos, iostat=ierr ) vint4

c-----------------------------------------------------------------------
c        //   read velocity
c-----------------------------------------------------------------------

c        //   skip lines
         if ( iboundary .ne. 0 ) then
            read( iounit_vel, iostat=ierr ) vint4
            read( iounit_vel, iostat=ierr )
     &         vreal8, vreal8, vreal8, vreal8, vreal8, vreal8
            read( iounit_vel, iostat=ierr ) vint4
         end if

c        //   unit conversion factor
         const_1 = 1.d-2 * au_length / au_time

c        //   skip line
         read( iounit_vel, iostat=ierr ) vint4

c        /*   loop of beads   */
         do j = 1, nbead

c        /*   loop of atoms   */
         do i = 1, natom

c           /*   position in angstrom   */
            read( iounit_vel ) ax

c           //   unit conversion
            vx(i,j) = dble( ax ) / const_1

c        /*   loop of atoms   */
         end do

c        /*   loop of beads   */
         end do

c        //   skip lines
         read( iounit_vel, iostat=ierr ) vint4
         read( iounit_vel, iostat=ierr ) vint4

c        /*   loop of beads   */
         do j = 1, nbead

c        /*   loop of atoms   */
         do i = 1, natom

c           /*   position in angstrom   */
            read( iounit_vel ) ay

c           //   unit conversion
            vy(i,j) = dble( ay ) / const_1

c        /*   loop of atoms   */
         end do

c        /*   loop of beads   */
         end do

c        //   skip lines
         read( iounit_vel, iostat=ierr ) vint4
         read( iounit_vel, iostat=ierr ) vint4

c        /*   loop of beads   */
         do j = 1, nbead

c        /*   loop of atoms   */
         do i = 1, natom

c           /*   position in angstrom   */
            read( iounit_vel ) az

c           //   unit conversion
            vz(i,j) = dble( az ) / const_1

c        /*   loop of atoms   */
         end do

c        /*   loop of beads   */
         end do

c        //   skip line
         read( iounit_vel, iostat=ierr ) vint4

c-----------------------------------------------------------------------
c        //   read force
c-----------------------------------------------------------------------

c        //   skip lines
         if ( iboundary .ne. 0 ) then
            read( iounit_frc, iostat=ierr ) vint4
            read( iounit_frc, iostat=ierr )
     &         vreal8, vreal8, vreal8, vreal8, vreal8, vreal8
            read( iounit_frc, iostat=ierr ) vint4
         end if

c        //   unit conversion factor
         const_1 = 1.d0

c        //   skip line
         read( iounit_frc, iostat=ierr ) vint4

c        /*   loop of beads   */
         do j = 1, nbead

c        /*   loop of atoms   */
         do i = 1, natom

c           /*   position in angstrom   */
            read( iounit_frc ) ax

c           //   unit conversion
            fx(i,j) = dble( ax ) / const_1

c        /*   loop of atoms   */
         end do

c        /*   loop of beads   */
         end do

c        //   skip lines
         read( iounit_frc, iostat=ierr ) vint4
         read( iounit_frc, iostat=ierr ) vint4

c        /*   loop of beads   */
         do j = 1, nbead

c        /*   loop of atoms   */
         do i = 1, natom

c           /*   position in angstrom   */
            read( iounit_frc ) ay

c           //   unit conversion
            fy(i,j) = dble( ay ) / const_1

c        /*   loop of atoms   */
         end do

c        /*   loop of beads   */
         end do

c        //   skip lines
         read( iounit_frc, iostat=ierr ) vint4
         read( iounit_frc, iostat=ierr ) vint4

c        /*   loop of beads   */
         do j = 1, nbead

c        /*   loop of atoms   */
         do i = 1, natom

c           /*   position in angstrom   */
            read( iounit_frc ) az

c           //   unit conversion
            fz(i,j) = dble( az ) / const_1

c        /*   loop of atoms   */
         end do

c        /*   loop of beads   */
         end do

c        //   skip line
         read( iounit_frc, iostat=ierr ) vint4

c-----------------------------------------------------------------------
c        //   read potential
c-----------------------------------------------------------------------

c        //   unit conversion factor
         const_1 = 1.d0

c        //   skip line
         read( iounit_pot, iostat=ierr ) vint4

c        /*   loop of beads   */
         do j = 1, nbead

c           /*   position in angstrom   */
            read( iounit_pot ) b

c           //   angstrom to bohr
            pot(j) = dble( b ) / const_1

c        /*   loop of beads   */
         end do

c        /*   temperature and hamiltonian   */
         read( iounit_pot ) b
         read( iounit_pot ) b

c-----------------------------------------------------------------------
c        //   print
c-----------------------------------------------------------------------

         do j = 1, nbead
         do i = 1, natom
            write( iounit_trj, '(i8,10e24.16)' )
     &        istep, x(i,j), y(i,j), z(i,j), vx(i,j), vy(i,j), vz(i,j), 
     &               fx(i,j), fy(i,j), fz(i,j), pot(j)
         end do
         end do

c-----------------------------------------------------------------------
c     //   main loop
c-----------------------------------------------------------------------

      end do

c     //   close files
      close ( iounit_pos )
      close ( iounit_vel )
      close ( iounit_frc )
      close ( iounit_pot )
      close ( iounit_trj )

c-----------------------------------------------------------------------
c     //   end of program
c-----------------------------------------------------------------------

      write( 6, '(a)' ) 'Normal termination of dcd2trj.'
      write( 6, '(a)' )

      stop
      end





c***********************************************************************
      subroutine error_handling ( ierr, char_tag, length_tag )
c***********************************************************************

c-----------------------------------------------------------------------
c     /*   local variables                                            */
c-----------------------------------------------------------------------

      implicit none

      integer:: ierr, length_tag

      character(len=length_tag) :: char_tag

c-----------------------------------------------------------------------
c     /*   start                                                      */
c-----------------------------------------------------------------------

      if ( ierr .ne. 0 ) then

         write(6,'(a)') 'Error termination at: ' // char_tag // '.'

         write(6,'(a)')

         stop

      end if

      return
      end

