c///////////////////////////////////////////////////////////////////////
c
c      Author:          M. Shiga
c      Last updated:    Feb 16, 2020 by M. Shiga
c      Description:     make cv2rec.out, cv2rec.cube from cv.out
c
c///////////////////////////////////////////////////////////////////////
c***********************************************************************
      program cv2rec
c***********************************************************************
c-----------------------------------------------------------------------
c     //   declare variables
c-----------------------------------------------------------------------

c     //   clear all variables
      implicit none

c     //   lower bound of the cube
      real(8) :: x_min, y_min, z_min

c     //   upper bound of the cube
      real(8) :: x_max, y_max, z_max

c     //   boundary condition: 0 (free) or 1 (periodic)
      integer :: ipbc_x = 0
      integer :: ipbc_y = 0
      integer :: ipbc_z = 0

c     //   mesh size of the cube
      real(8) :: x_mesh, y_mesh, z_mesh

c     //   number of meshes
      integer :: mesh_x, mesh_y, mesh_z

c     //   distance between mesh and hill center
      real(8) :: rx, ry, rz

c     //   number of equilibration steps
      integer :: nskip

c     //   temperature
      real(8) :: temperature

c     //   boltzmann constant
      real(8), parameter :: boltz = 0.316682968d-5

c     //   counter
      real(8) :: counter

c     //   density
      real(8), dimension(:,:,:), allocatable :: dens

c     //   maximum free energy in hartree
      real(8) :: a_max

c     //   minimum free energy in hartree
      real(8) :: a_min

c     //   maximum free energy in kT
      real(8) :: a_max_kt

c     //   file unit
      integer :: iounit = 10

c     //   real numbers
      real(8) :: xijk, yijk, zijk, aijk, dmin, dmax

c     //   integers
      integer :: i, j, k, l, ierr

c     //   character
      character(len=24) :: cv1, cv2, cv3, cv4, cv5, cv6

c     //   character
      logical :: file_exists

c-----------------------------------------------------------------------
c     //   read parameters
c-----------------------------------------------------------------------

c     //   error flag
      ierr = 1

c     //   file
      inquire( file='cv2rec.dat', exist=file_exists )

c     //   read from file
      if ( file_exists ) then

c        //   open file
         open ( iounit, file='cv2rec.dat' )

c        //   input file
         read( iounit, *, iostat=ierr ) cv1, x_min, x_max, x_mesh
         read( iounit, *, iostat=ierr ) cv2, y_min, y_max, y_mesh
         read( iounit, *, iostat=ierr ) cv3, z_min, z_max, z_mesh
         read( iounit, *, iostat=ierr ) cv4
         read( iounit, *, iostat=ierr ) cv5
         read( iounit, *, iostat=ierr ) cv6

c        //   close file
         close( iounit )

c        //   remove file
         if ( ierr .ne. 0 ) call system( 'rm cv2rec.dat' )

c     //   three arguments
      end if

c-----------------------------------------------------------------------
c     //   usage
c-----------------------------------------------------------------------

c     //   five arguments
      if ( ( .not. file_exists ) .and. ( iargc() .ne. 6 ) ) then

c        //   print message
         write( 6, * ) 
         write( 6, * ) 'CODE:        cv2rec.x'
         write( 6, * ) 
         write( 6, * ) 'INPUT FILE:  cv.out'
         write( 6, * ) 'OUTPUT FILE: cv2rec.out'
         write( 6, * ) 'OUTPUT FILE: cv2rec.cube'
         write( 6, * ) 
         write( 6, * ) 'USAGE:       cv2rec.x $1 $2 $3 $4 $5 $6'
         write( 6, * ) '$1:          cv type of x.'
         write( 6, * ) '$2:          cv type of y.'
         write( 6, * ) '$3:          cv type of z.'
         write( 6, * ) '$4:          skipped steps.'
         write( 6, * ) '$5:          temperature.'
         write( 6, * ) '$6:          maximum free energy (kcal/mol).'
         write( 6, * ) 

c        //   error termination
         stop

c     //   five arguments
      end if

c-----------------------------------------------------------------------
c     //   read data
c-----------------------------------------------------------------------

c     //   three arguments
      if ( .not. file_exists ) then

         call getarg( 1, cv1 )
         call getarg( 2, cv2 )
         call getarg( 3, cv3 )
         call getarg( 4, cv4 )
         call getarg( 5, cv5 )
         call getarg( 6, cv6 )

c     //   three arguments
      end if

c-----------------------------------------------------------------------
c     //   transfer data type
c-----------------------------------------------------------------------

c     //   transform to integer
      read ( cv4, * ) nskip

c     //   transform to real number
      read ( cv5, * ) temperature

c     //   transform to real number
      read ( cv6, * ) a_max_kt

c-----------------------------------------------------------------------
c     //   print message
c-----------------------------------------------------------------------

      write( 6, '(a)' ) 'CV TYPE OF X:  ' // trim(cv1)
      write( 6, '(a)' ) 'CV TYPE OF Y:  ' // trim(cv2)
      write( 6, '(a)' ) 'CV TYPE OF Z:  ' // trim(cv3)
      write( 6, '(a)' ) 'SKIPPED STEPS: ' // trim(cv4)
      write( 6, '(a)' ) 'TEMPERATURE:   ' // trim(cv5)

c-----------------------------------------------------------------------
c     //   set parameters
c-----------------------------------------------------------------------
c
c     <params_rec_meta>
c     DIST   2.0    5.0   0.1
c     ANGL   0.0  180.0   6.0
c     DIH    0.0  360.0  12.0
c     DIFF   0.0   15.0   0.5
c     CN     0.0    3.0   0.1
c     DCN   -2.0    2.0   0.1
c     XYZ    2.0    5.0   0.1
c     DXYZ   2.0    5.0   0.1
c
c-----------------------------------------------------------------------
c     //    set mesh parameters
c-----------------------------------------------------------------------

c     //   three arguments
      if ( .not. file_exists ) then

         if ( cv1(1:5) .eq. 'DIST ' ) then
            x_min  =   2.0d0
            x_max  =   5.0d0
            x_mesh =   0.1d0
         else if ( cv1(1:5) .eq. 'ANGL ' ) then
            x_min  =   0.0d0
            x_max  = 180.0d0
            x_mesh =   6.0d0
         else if ( cv1(1:5) .eq. 'DIH  ' ) then
            x_min  =   0.0d0
            x_max  = 348.0d0
            x_mesh =  12.0d0
         else if ( cv1(1:5) .eq. 'DIFF ' ) then
            x_min  =   0.0d0
            x_max  =  15.0d0
            x_mesh =   0.5d0
         else if ( cv1(1:5) .eq. 'CN   ' ) then
            x_min  =   0.0d0
            x_max  =   3.0d0
            x_mesh =   0.1d0
         else if ( cv1(1:5) .eq. 'DCN  ' ) then
            x_min  =  -2.0d0
            x_max  =   2.0d0
            x_mesh =   0.1d0
         else if ( cv1(1:5) .eq. 'XYZ  ' ) then
            x_min  =   2.0d0
            x_max  =   5.0d0
            x_mesh =   0.1d0
         else if ( cv1(1:5) .eq. 'DXYZ ' ) then
            x_min  =   2.0d0
            x_max  =   5.0d0
            x_mesh =   0.1d0
         end if

         if ( cv2(1:5) .eq. 'DIST ' ) then
            y_min  =   2.0d0
            y_max  =   5.0d0
            y_mesh =   0.1d0
         else if ( cv2(1:5) .eq. 'ANGL ' ) then
            y_min  =   0.0d0
            y_max  = 180.0d0
            y_mesh =   6.0d0
         else if ( cv2(1:5) .eq. 'DIH  ' ) then
            y_min  =   0.0d0
            y_max  = 348.0d0
            y_mesh =  12.0d0
         else if ( cv2(1:5) .eq. 'DIFF ' ) then
            y_min  =   0.0d0
            y_max  =  15.0d0
            y_mesh =   0.5d0
         else if ( cv2(1:5) .eq. 'CN   ' ) then
            y_min  =   0.0d0
            y_max  =   3.0d0
            y_mesh =   0.1d0
         else if ( cv2(1:5) .eq. 'DCN  ' ) then
            y_min  =  -2.0d0
            y_max  =   2.0d0
            y_mesh =   0.1d0
         else if ( cv2(1:5) .eq. 'XYZ  ' ) then
            y_min  =   2.0d0
            y_max  =   5.0d0
            y_mesh =   0.1d0
         else if ( cv2(1:5) .eq. 'DXYZ ' ) then
            y_min  =   2.0d0
            y_max  =   5.0d0
            y_mesh =   0.1d0
         end if

         if ( cv3(1:5) .eq. 'DIST ' ) then
            z_min  =   2.0d0
            z_max  =   5.0d0
            z_mesh =   0.1d0
         else if ( cv3(1:5) .eq. 'ANGL ' ) then
            z_min  =   0.0d0
            z_max  = 180.0d0
            z_mesh =   6.0d0
         else if ( cv3(1:5) .eq. 'DIH  ' ) then
            z_min  =   0.0d0
            z_max  = 348.0d0
            z_mesh =  12.0d0
         else if ( cv3(1:5) .eq. 'DIFF ' ) then
            z_min  =   0.0d0
            z_max  =  15.0d0
            z_mesh =   0.5d0
         else if ( cv3(1:5) .eq. 'CN   ' ) then
            z_min  =   0.0d0
            z_max  =   3.0d0
            z_mesh =   0.1d0
         else if ( cv3(1:5) .eq. 'DCN  ' ) then
            z_min  =  -2.0d0
            z_max  =   2.0d0
            z_mesh =   0.1d0
         else if ( cv3(1:5) .eq. 'XYZ  ' ) then
            z_min  =   2.0d0
            z_max  =   5.0d0
            z_mesh =   0.1d0
         else if ( cv3(1:5) .eq. 'DXYZ ' ) then
            z_min  =   2.0d0
            z_max  =   5.0d0
            z_mesh =   0.1d0
         end if

      end if

c     //   free boundary
      ipbc_x = 0
      ipbc_y = 0
      ipbc_z = 0

c     //   periodic boundary
      if ( cv1(1:5) .eq. 'DIH  ' ) ipbc_x = 1
      if ( cv2(1:5) .eq. 'DIH  ' ) ipbc_y = 1
      if ( cv3(1:5) .eq. 'DIH  ' ) ipbc_z = 1

c     //   number of meshes
      mesh_x = nint( ( x_max - x_min ) / x_mesh ) + 1
      mesh_y = nint( ( y_max - y_min ) / y_mesh ) + 1
      mesh_z = nint( ( z_max - z_min ) / z_mesh ) + 1

c-----------------------------------------------------------------------
c     //   write parameters
c-----------------------------------------------------------------------

c     //   new file
      if ( .not. file_exists ) then

c        //   open file
         open ( iounit, file = 'cv2rec.dat' )

c        //   input file
         write( iounit, '(a,3f9.3)' ) trim(cv1), x_min, x_max, x_mesh
         write( iounit, '(a,3f9.3)' ) trim(cv2), y_min, y_max, y_mesh
         write( iounit, '(a,3f9.3)' ) trim(cv3), z_min, z_max, z_mesh
         write( iounit, '(a)' ) trim(cv4)
         write( iounit, '(a)' ) trim(cv5)
         write( iounit, '(a)' ) trim(cv6)

c        //   close file
         close( iounit )

c     //   new file
      end if

c-----------------------------------------------------------------------
c     //   print info
c-----------------------------------------------------------------------

       write( 6, '(a,3f10.4)' )
     &       'MIN, MAX, MESH VALUES OF X: ',
     &        x_min, x_max, x_mesh
       write( 6, '(a,3f10.4)' )
     &       'MIN, MAX, MESH VALUES OF Y: ',
     &        y_min, y_max, y_mesh
       write( 6, '(a,3f10.4)' )
     &       'MIN, MAX, MESH VALUES OF Z: ',
     &        z_min, z_max, z_mesh
       write( 6, '(a,3i4)' )
     &       'NUMBER OF MESHES IN X, Y, Z: ',
     &        mesh_x, mesh_y, mesh_z
       write( 6, '(a,f10.4)' )
     &       'TEMPERATURE: ', temperature
       write( 6, '(a,f10.4)' )
     &       'MAXIMUM FREE ENERGY (KCAL/MOL): ', a_max_kt

c-----------------------------------------------------------------------
c     //   maximum free energy
c-----------------------------------------------------------------------

c     //   in atomic units
      a_max = a_max_kt * boltz * temperature

c     //   print message
      write( 6, '(a,f10.6)' ) 'MAXIMUM FREE ENERGY: ', a_max

c-----------------------------------------------------------------------
c     //   memory allocation
c-----------------------------------------------------------------------

c     //   cv density
      allocate( dens(mesh_x,mesh_y,mesh_z) )

c-----------------------------------------------------------------------
c     //   calculate density
c-----------------------------------------------------------------------

c     //   print message
      write( 6, '(a)' ) 'START: read cv.out.'

c     //   counter
      counter = 0.d0

c     //   density
      dens(:,:,:) = 0.d0

c     //   open file
      open ( iounit, file = 'cv.out' )

c        //   read header
         do i = 1, 3
            read( iounit, *, iostat=ierr )
         end do

c        //   find EOF
         if ( ierr .ne. 0 ) stop

c        //   loop start
         do

c           //   read data
            read( iounit, *, iostat=ierr ) l, rx
            read( iounit, *, iostat=ierr ) l, ry
            read( iounit, *, iostat=ierr ) l, rz

c           //   find EOF
            if ( ierr .ne. 0 ) exit

c           //   skip data
            if ( l .le. nskip ) cycle

c           //   apply periodic boundary condition if ipbc = 1
            rx = rx - ipbc_x * ( nint( (rx-180.d0) / 360.d0 ) * 360.d0 )
            ry = ry - ipbc_y * ( nint( (ry-180.d0) / 360.d0 ) * 360.d0 )
            rz = rz - ipbc_z * ( nint( (rz-180.d0) / 360.d0 ) * 360.d0 )

c           //   grid position
            i = nint( ( rx - x_min ) / x_mesh ) + 1
            j = nint( ( ry - y_min ) / y_mesh ) + 1
            k = nint( ( rz - z_min ) / z_mesh ) + 1

c           //   apply periodic boundary condition if ipbc = 1
            if ( (ipbc_x.eq.1) .and. (i.eq.(mesh_x+1)) ) i = 1
            if ( (ipbc_y.eq.1) .and. (i.eq.(mesh_y+1)) ) j = 1
            if ( (ipbc_z.eq.1) .and. (i.eq.(mesh_z+1)) ) k = 1

c           //   out of range
            if ( ( i .lt. 1 ) .or. ( i .gt. mesh_x ) ) cycle
            if ( ( j .lt. 1 ) .or. ( j .gt. mesh_y ) ) cycle
            if ( ( k .lt. 1 ) .or. ( k .gt. mesh_z ) ) cycle

c           //   density
            dens(i,j,k) = dens(i,j,k) + 1.d0

c           //   counter
            counter = counter + 1.d0

c        //   loop end
         end do

c     //   close file
      close( iounit )

c     //   maximum density
      dmax = maxval( dens(:,:,:) )

c     //   minimum density
      dmin = 1.d0

ccc     //   maximum free energy
ccc      a_max = - boltz * temperature * log( dmin )

c     //   minimum free energy
      a_min = - boltz * temperature * log( dmax )

c     //   print message
      write( 6, '(a,i8)' ) 'READ CV POSITIONS: ', nint(counter)

c     //   print message
      write( 6, '(a)' ) 'END: read cv.out.'

c-----------------------------------------------------------------------
c     //   print cv2rec.out
c-----------------------------------------------------------------------

c     //   print message
      write( 6, '(a)' ) 'START: print cv2rec.out.'

c     //   open file
      open ( iounit, file = 'cv2rec.out' )

c     //   loop start
      do i = 1, mesh_x
      do j = 1, mesh_y
      do k = 1, mesh_z

c        //   grid position
         xijk = x_min + dble(i-1) * x_mesh
         yijk = y_min + dble(j-1) * y_mesh
         zijk = z_min + dble(k-1) * z_mesh

c        //   if density is positive
         if ( dens(i,j,k) .gt. 0.d0 ) then

c           //   free energy is proportional to the log of density
            aijk = - boltz * temperature * log( dens(i,j,k) ) - a_min

c           //   free energy is a maximum value (cutoff)
            aijk =   min( a_max, aijk )

c        //   if density is zero
         else

c           //   free energy is a maximum value (cutoff)
            aijk =   a_max

c        //   end of if statement
         end if

c        //   print grid position and free energy
         write( iounit, '(4f16.8)' ) xijk, yijk, zijk, aijk

c     //   loop end
      end do
      end do
      end do

c     //   close file
      close( iounit )

c     //   print message
      write( 6, '(a)' ) 'END: print cv2rec.out.'

c-----------------------------------------------------------------------
c     //   print cv2rec.cube
c-----------------------------------------------------------------------

c     //   print message
      write( 6, '(a)' ) 'START: print cv2rec.cube.'

c     //   open file
      open ( iounit, file = 'cv2rec.cube' )

      write( iounit, '(a)' )
     &   "CUBE FILE BY CV2REC."

      write( iounit, '(a)' )
     &   "OUTER LOOP: X, MIDDLE LOOP: Y, INNER LOOP: Z"

      write( iounit, '(i5,3e16.8)' )
     &    0, x_max-x_min, y_max-y_min, z_max-z_min

      write( iounit, '(i5,3e16.8)' )
     &    mesh_x, 1.d0, 0.d0, 0.d0

      write( iounit, '(i5,3e16.8)' )
     &    mesh_y, 0.d0, 1.d0, 0.d0

      write( iounit, '(i5,3e16.8)' )
     &    mesh_z, 0.d0, 0.d0, 1.d0

c     //   loop start
      do i = 1, mesh_x
      do j = 1, mesh_y
      do k = 1, mesh_z

c        //   grid position
         xijk = x_min + dble(i-1) * x_mesh
         yijk = y_min + dble(j-1) * y_mesh
         zijk = z_min + dble(k-1) * z_mesh

c        //   if density is positive
         if ( dens(i,j,k) .gt. 0.d0 ) then

c           //   free energy is proportional to the log of density
            aijk = - boltz * temperature * log( dens(i,j,k) ) - a_min

c           //   free energy is a maximum value (cutoff)
            aijk =   min( a_max, aijk )

c        //   if density is zero
         else

c           //   free energy is a maximum value (cutoff)
            aijk =   a_max

c        //   end of if statement
         end if

c        //   print shifted free energy
         write( iounit, '(f10.6)' ) aijk-a_max

c     //   loop end
      end do
      end do
      end do

c     //   close file
      close( iounit )

c     //   print message
      write( 6, '(a)' ) 'END: print cv2rec.cube.'

      stop
      end
