c///////////////////////////////////////////////////////////////////////
c
c      Author:          M. Shiga
c      Last updated:    Feb 16, 2020 by M. Shiga
c      Description:     make cube2rec.out from rec.cube
c
c///////////////////////////////////////////////////////////////////////
c***********************************************************************
      program cube2rec
c***********************************************************************
c-----------------------------------------------------------------------
c     //   declare variables
c-----------------------------------------------------------------------

c     //   clear all variables
      implicit none

c     //   lower bound of the cube
      real(8) :: x_min, y_min, z_min

c     //   upper bound of the cube
      real(8) :: x_max, y_max, z_max

c     //   boundary condition: 0 (free) or 1 (periodic)
      integer :: ipbc_x = 0
      integer :: ipbc_y = 0
      integer :: ipbc_z = 0

c     //   mesh size of the cube
      real(8) :: x_mesh, y_mesh, z_mesh

c     //   number of meshes
      integer :: mesh_x, mesh_y, mesh_z

c     //   maximum free energy in hartree
      real(8) :: a_max

c     //   file unit
      integer :: iounit = 10

c     //   real numbers
      real(8) :: xijk, yijk, zijk

c     //   real numbers
      real(8), dimension(:,:,:), allocatable :: a

c     //   integers
      integer :: i, j, k, ierr

c     //   character
      character(len=24) :: cv1, cv2, cv3

c     //   character
      logical :: file_exists

c-----------------------------------------------------------------------
c     //   read parameters
c-----------------------------------------------------------------------

c     //   error flag
      ierr = 1

c     //   file
      inquire( file='cube2rec.dat', exist=file_exists )

c     //   read from file
      if ( file_exists ) then

c        //   open file
         open ( iounit, file='cube2rec.dat' )

c        //   input file
         read( iounit, *, iostat=ierr ) cv1, x_min, x_max, x_mesh
         read( iounit, *, iostat=ierr ) cv2, y_min, y_max, y_mesh
         read( iounit, *, iostat=ierr ) cv3, z_min, z_max, z_mesh

c        //   close file
         close( iounit )

c        //   remove file
         if ( ierr .ne. 0 ) call system( 'rm cube2rec.dat' )

c     //   three arguments
      end if

c-----------------------------------------------------------------------
c     //   usage
c-----------------------------------------------------------------------

c     //   five arguments
      if ( ( .not. file_exists ) .and. ( iargc() .ne. 3 ) ) then

c        //   print message
         write( 6, * ) 
         write( 6, * ) 'CODE:        cube2rec.x'
         write( 6, * ) 
         write( 6, * ) 'INPUT FILE:  rec.cube'
         write( 6, * ) 'OUTPUT FILE: cube2rec.out'
         write( 6, * ) 
         write( 6, * ) 'USAGE:       cube2rec.x $1 $2 $3'
         write( 6, * ) '$1:          cv type of x.'
         write( 6, * ) '$2:          cv type of y.'
         write( 6, * ) '$3:          cv type of z.'
         write( 6, * ) 

c        //   error termination
         stop

c     //   five arguments
      end if

c-----------------------------------------------------------------------
c     //   print message
c-----------------------------------------------------------------------

      write( 6, '(a)' ) 'CV TYPE OF X:  ' // trim(cv1)
      write( 6, '(a)' ) 'CV TYPE OF Y:  ' // trim(cv2)
      write( 6, '(a)' ) 'CV TYPE OF Z:  ' // trim(cv3)

c-----------------------------------------------------------------------
c     //   set parameters
c-----------------------------------------------------------------------
c
c     <params_rec_meta>
c     DIST   2.0    5.0   0.1
c     ANGL   0.0  180.0   6.0
c     DIH    0.0  360.0  12.0
c     DIFF   0.0   15.0   0.5
c     CN     0.0    3.0   0.1
c     DCN   -2.0    2.0   0.1
c     XYZ    2.0    5.0   0.1
c     DXYZ   2.0    5.0   0.1
c
c-----------------------------------------------------------------------
c     //    set mesh parameters
c-----------------------------------------------------------------------

c     //   three arguments
      if ( .not. file_exists ) then

         if ( cv1(1:5) .eq. 'DIST ' ) then
            x_min  =   2.0d0
            x_max  =   5.0d0
            x_mesh =   0.1d0
         else if ( cv1(1:5) .eq. 'ANGL ' ) then
            x_min  =   0.0d0
            x_max  = 180.0d0
            x_mesh =   6.0d0
         else if ( cv1(1:5) .eq. 'DIH  ' ) then
            x_min  =   0.0d0
            x_max  = 348.0d0
            x_mesh =  12.0d0
         else if ( cv1(1:5) .eq. 'DIFF ' ) then
            x_min  =   0.0d0
            x_max  =  15.0d0
            x_mesh =   0.5d0
         else if ( cv1(1:5) .eq. 'CN   ' ) then
            x_min  =   0.0d0
            x_max  =   3.0d0
            x_mesh =   0.1d0
         else if ( cv1(1:5) .eq. 'DCN  ' ) then
            x_min  =  -2.0d0
            x_max  =   2.0d0
            x_mesh =   0.1d0
         else if ( cv1(1:5) .eq. 'XYZ  ' ) then
            x_min  =   2.0d0
            x_max  =   5.0d0
            x_mesh =   0.1d0
         else if ( cv1(1:5) .eq. 'DXYZ ' ) then
            x_min  =   2.0d0
            x_max  =   5.0d0
            x_mesh =   0.1d0
         end if

         if ( cv2(1:5) .eq. 'DIST ' ) then
            y_min  =   2.0d0
            y_max  =   5.0d0
            y_mesh =   0.1d0
         else if ( cv2(1:5) .eq. 'ANGL ' ) then
            y_min  =   0.0d0
            y_max  = 180.0d0
            y_mesh =   6.0d0
         else if ( cv2(1:5) .eq. 'DIH  ' ) then
            y_min  =   0.0d0
            y_max  = 348.0d0
            y_mesh =  12.0d0
         else if ( cv2(1:5) .eq. 'DIFF ' ) then
            y_min  =   0.0d0
            y_max  =  15.0d0
            y_mesh =   0.5d0
         else if ( cv2(1:5) .eq. 'CN   ' ) then
            y_min  =   0.0d0
            y_max  =   3.0d0
            y_mesh =   0.1d0
         else if ( cv2(1:5) .eq. 'DCN  ' ) then
            y_min  =  -2.0d0
            y_max  =   2.0d0
            y_mesh =   0.1d0
         else if ( cv2(1:5) .eq. 'XYZ  ' ) then
            y_min  =   2.0d0
            y_max  =   5.0d0
            y_mesh =   0.1d0
         else if ( cv2(1:5) .eq. 'DXYZ ' ) then
            y_min  =   2.0d0
            y_max  =   5.0d0
            y_mesh =   0.1d0
         end if

         if ( cv3(1:5) .eq. 'DIST ' ) then
            z_min  =   2.0d0
            z_max  =   5.0d0
            z_mesh =   0.1d0
         else if ( cv3(1:5) .eq. 'ANGL ' ) then
            z_min  =   0.0d0
            z_max  = 180.0d0
            z_mesh =   6.0d0
         else if ( cv3(1:5) .eq. 'DIH  ' ) then
            z_min  =   0.0d0
            z_max  = 348.0d0
            z_mesh =  12.0d0
         else if ( cv3(1:5) .eq. 'DIFF ' ) then
            z_min  =   0.0d0
            z_max  =  15.0d0
            z_mesh =   0.5d0
         else if ( cv3(1:5) .eq. 'CN   ' ) then
            z_min  =   0.0d0
            z_max  =   3.0d0
            z_mesh =   0.1d0
         else if ( cv3(1:5) .eq. 'DCN  ' ) then
            z_min  =  -2.0d0
            z_max  =   2.0d0
            z_mesh =   0.1d0
         else if ( cv3(1:5) .eq. 'XYZ  ' ) then
            z_min  =   2.0d0
            z_max  =   5.0d0
            z_mesh =   0.1d0
         else if ( cv3(1:5) .eq. 'DXYZ ' ) then
            z_min  =   2.0d0
            z_max  =   5.0d0
            z_mesh =   0.1d0
         end if

      end if

c     //   free boundary
      ipbc_x = 0
      ipbc_y = 0
      ipbc_z = 0

c     //   periodic boundary
      if ( cv1(1:5) .eq. 'DIH  ' ) ipbc_x = 1
      if ( cv2(1:5) .eq. 'DIH  ' ) ipbc_y = 1
      if ( cv3(1:5) .eq. 'DIH  ' ) ipbc_z = 1

c     //   number of meshes
      mesh_x = nint( ( x_max - x_min ) / x_mesh ) + 1
      mesh_y = nint( ( y_max - y_min ) / y_mesh ) + 1
      mesh_z = nint( ( z_max - z_min ) / z_mesh ) + 1

c     // memory allocation
      allocate( a(mesh_x,mesh_y,mesh_z) )

c-----------------------------------------------------------------------
c     //   write parameters
c-----------------------------------------------------------------------

c     //   new file
      if ( .not. file_exists ) then

c        //   open file
         open ( iounit, file = 'cube2rec.dat' )

c        //   input file
         write( iounit, '(a,3f9.3)' ) trim(cv1), x_min, x_max, x_mesh
         write( iounit, '(a,3f9.3)' ) trim(cv2), y_min, y_max, y_mesh
         write( iounit, '(a,3f9.3)' ) trim(cv3), z_min, z_max, z_mesh

c        //   close file
         close( iounit )

c     //   new file
      end if

c-----------------------------------------------------------------------
c     //   print info
c-----------------------------------------------------------------------

       write( 6, '(a,3f10.4)' )
     &       'MIN, MAX, MESH VALUES OF X: ', x_min, x_max, x_mesh
       write( 6, '(a,3f10.4)' )
     &       'MIN, MAX, MESH VALUES OF Y: ', y_min, y_max, y_mesh
       write( 6, '(a,3f10.4)' )
     &       'MIN, MAX, MESH VALUES OF Z: ', z_min, z_max, z_mesh

c-----------------------------------------------------------------------
c     //   print cv2rec.cube
c-----------------------------------------------------------------------

c     //   print message
      write( 6, '(a)' ) 'START: read rec.cube.'

c     //   open file
      open ( iounit, file = 'rec.cube' )

      read( iounit, '(a)', iostat=ierr )
      read( iounit, '(a)', iostat=ierr )
      read( iounit, '(a)', iostat=ierr )
      read( iounit, '(a)', iostat=ierr )
      read( iounit, '(a)', iostat=ierr )
      read( iounit, '(a)', iostat=ierr )

c     //   maximum free energy
      a_max = 0.d0

c     //   loop start
      do i = 1, mesh_x
      do j = 1, mesh_y
      do k = 1, mesh_z

c        //   grid position
         xijk = x_min + dble(i-1) * x_mesh
         yijk = y_min + dble(j-1) * y_mesh
         zijk = z_min + dble(k-1) * z_mesh

c        //   print shifted free energy
         read( iounit, *, iostat=ierr ) a(i,j,k)

c        //  stop on error
         if ( ierr .ne. 0 ) exit

c        //   maximum free energy
         a_max = min( a(i,j,k), a_max )

c     //   loop end
      end do
      end do
      end do

c     //  stop on error
      if ( ierr .ne. 0 ) then
         write( 6, '(a)' )
     &     'ERROR - rec.cube read incorrectly. Check cube2rec.dat.'
         write( 6, '(a)' )
         stop
      end if

c     //   close file
      close( iounit )

c     //   print message
      write( 6, '(a)' ) 'END: read rec.cube.'

c-----------------------------------------------------------------------
c     //   print cv2rec.out
c-----------------------------------------------------------------------

c     //   print message
      write( 6, '(a,f10.5)' ) 'MAXIMUM FREE ENERGY:', a_max

c     //   print message
      write( 6, '(a)' ) 'START: print cube2rec.out.'

c     //   open file
      open ( iounit, file = 'cube2rec.out' )

c     //   loop start
      do i = 1, mesh_x
      do j = 1, mesh_y
      do k = 1, mesh_z

c        //   grid position
         xijk = x_min + dble(i-1) * x_mesh
         yijk = y_min + dble(j-1) * y_mesh
         zijk = z_min + dble(k-1) * z_mesh

c        //   print grid position and free energy
         write( iounit, '(4f16.8)' ) xijk, yijk, zijk, a(i,j,k)-a_max

c     //   loop end
      end do
      end do
      end do

c     //   close file
      close( iounit )

c     //   print message
      write( 6, '(a)' ) 'END: print cube2rec.out.'

      stop
      end
