c***********************************************************************
      program chgtrj2dip
c***********************************************************************
c-----------------------------------------------------------------------
c     //   variables
c-----------------------------------------------------------------------

c     //   initialize
      implicit none

c     //   unit conversion factors
      real(8), parameter :: au_time   = 0.024188843d-15
      real(8), parameter :: au_length = 0.529177249d-10

c     //   file unit
      integer :: iounit = 10

c     //   number of atoms
      integer :: natom

c     //   number of beads
      integer :: nbead

c     //   step number
      integer :: istep

c     //   number of steps
      integer :: nstep

c     //   atomic coordinates
      real(8), dimension(:,:), allocatable :: x, y, z

c     //   old atomic coordinates
      real(8), dimension(:,:), allocatable :: xold, yold, zold

c     //   velocities
      real(8), dimension(:,:), allocatable :: vx, vy, vz

c     //   charges
      real(8), dimension(:,:), allocatable :: q

c     //   charge derivatives
      real(8), dimension(:,:), allocatable :: qdot

c     //   old and charges
      real(8), dimension(:,:), allocatable :: qold
      real(8), dimension(:,:), allocatable :: qnew

c     //   dipole moments
      real(8), dimension(:), allocatable :: dipx, dipy, dipz

c     //   dipole derivatives wrt time
      real(8), dimension(:), allocatable :: ddipx, ddipy, ddipz

c     //   box size
      real(8), dimension(3,3) :: box, boxinv

c     //   file names
      character(len=80) :: trjfile, chgfile, dipfile, strfile, inpfile

c     //   charge type: Mulliken or Hirshfeld
      character(len=10) :: chgtype

c     //   boundary condition
      integer :: iboundary

c     //   step size
      real(8) :: dt, dt_fs

c     //   integers
      integer :: i, j, k, ierr

c     //   real numbers
      real(8) :: xi, yi, zi, bohr2ang, d

c     //   characters
      character(len=80) :: char

c-----------------------------------------------------------------------
c     //   initial message
c-----------------------------------------------------------------------

c     //   comments
      if ( iargc() .ne. 8 ) then

         write( 6, '(a)' )
         write( 6, '(a)' ) 'Program chgtrj2dip'
         write( 6, '(a)' )
         write( 6, '(a)' ) 'Usage: chgtrj2dip.x $1 $2 $3 $4 $5 $6 $7'
         write( 6, '(a)' )
         write( 6, '(a)' ) '$1: trj.out (position trajectory)'
         write( 6, '(a)' ) '$2: charges.out (charge trajectory)'
         write( 6, '(a)' ) '$3: M: Mulliken or H: Hirshfeld'
         write( 6, '(a)' ) '$4: structure.dat (atomic kinds)'
         write( 6, '(a)' ) '$5: input.dat (beads, box, temperature)'
         write( 6, '(a)' ) '$6: output file for dipoles (dipole.out)'
         write( 6, '(a)' ) '$7: number of steps'
         write( 6, '(a)' ) '$8: step interval of velocity [fs]'
         write( 6, '(a)' )
         stop

c     //   comments
      else

         write( 6, '(a)' )
         write( 6, '(a)' ) 'Program chgtrj2dip'
         write( 6, '(a)' )

c     //   comments
      end if

c-----------------------------------------------------------------------
c     //   read arguments
c-----------------------------------------------------------------------

c     //   trajectory file
      call getarg( 1, trjfile )

c     //   charge trajectory file
      call getarg( 2, chgfile )

c     //   charge trajectory file
      call getarg( 3, chgtype )

c     //   structure file
      call getarg( 4, strfile )

c     //   input file
      call getarg( 5, inpfile )

c     //   dipole trajectory file
      call getarg( 6, dipfile )

c     //   number of steps
      call getarg( 7, char )
      read( char, * ) nstep

c     //   step size in fs
      call getarg( 8, char )
      read( char, * ) dt_fs
      dt = dt_fs / ( au_time * 1.d+15 )

c-----------------------------------------------------------------------
c     //   constants
c-----------------------------------------------------------------------

c     //   bohr to angstrom
      bohr2ang = au_length * 1.d+10

c-----------------------------------------------------------------------
c     //   number of atoms
c-----------------------------------------------------------------------

c     //   open file
      open ( iounit, file = trim(strfile) )

c     //   number of atoms
      read ( iounit, *, iostat=ierr ) natom

c     //   close file
      close( iounit )

c-----------------------------------------------------------------------
c     //   number of beads
c-----------------------------------------------------------------------

c     /*   file open   */
      open ( iounit, file = trim(inpfile) )

c     /*   search for tag    */
      call search_tag ( '<nbead>', 7, iounit, ierr )

c     /*   number of atomic symbols   */
      read ( iounit, *, iostat=ierr ) nbead

c     /*   file close   */
      close( iounit )

c-----------------------------------------------------------------------
c     //   box
c-----------------------------------------------------------------------

c     /*   file open   */
      open ( iounit, file = trim(inpfile) )

c     /*   search for tag    */
      call search_tag ( '<iboundary>', 11, iounit, ierr )

c     /*   number of atomic symbols   */
      read ( iounit, *, iostat=ierr ) char

c     //   periodic boundary condition
      iboundary = 1
      if ( char(1:1) .eq. '0' ) then
         iboundary = 0
      end if

c     /*   box matrix  */
      if ( iboundary .eq. 1 ) then
         read ( iounit, *, iostat=ierr ) box(1,1), box(1,2), box(1,3)
         read ( iounit, *, iostat=ierr ) box(2,1), box(2,2), box(2,3)
         read ( iounit, *, iostat=ierr ) box(3,1), box(3,2), box(3,3)
      end if

c     /*   file close   */
      close( iounit )

c     //   angstrom to bohr
      if ( ( ierr .eq. 0 ) .and. ( char(1:1) .eq. 'A' ) ) then
         box(:,:) = box(:,:) / bohr2ang
      end if

c     //   inverse matrix
      call inv3( box, boxinv )

c-----------------------------------------------------------------------
c     //   memory allocation
c-----------------------------------------------------------------------

c     //   atomic coordinates
      allocate( x(natom,nbead) )
      allocate( y(natom,nbead) )
      allocate( z(natom,nbead) )

c     //   old coordinates
      allocate( xold(natom,nbead) )
      allocate( yold(natom,nbead) )
      allocate( zold(natom,nbead) )

c     //   atomic velocities
      allocate( vx(natom,nbead) )
      allocate( vy(natom,nbead) )
      allocate( vz(natom,nbead) )

c     //   atomic charges
      allocate( q(natom,nbead) )

c     //   old and new charges
      allocate( qold(natom,nbead) )
      allocate( qnew(natom,nbead) )

c     //   charge derivative
      allocate( qdot(natom,nbead) )

c     //   dipole moment
      allocate( dipx(nbead) )
      allocate( dipy(nbead) )
      allocate( dipz(nbead) )

c     //   dipole derivative
      allocate( ddipx(nbead) )
      allocate( ddipy(nbead) )
      allocate( ddipz(nbead) )

c-----------------------------------------------------------------------
c     //   open files
c-----------------------------------------------------------------------

c     //   open files
      open ( 11, file = trjfile )
      open ( 12, file = chgfile )
      open ( 13, file = chgfile )
      open ( 14, file = chgfile )
      open ( 20, file = dipfile )

c-----------------------------------------------------------------------
c     //   first step
c-----------------------------------------------------------------------

c     //   skip reading atomic coordinates and velocities
      do j = 1, nbead
      do i = 1, natom
         read( 11, *, iostat=ierr )
     &      k, x(i,j), y(i,j), z(i,j), vx(i,j), vy(i,j), vz(i,j)
      end do
      end do

c     //   save coordinates
      xold(:,:) = x(:,:)
      yold(:,:) = y(:,:)
      zold(:,:) = z(:,:)

c     //   read atomic charges of current step
      if ( chgtype(1:1) .eq. 'M' ) then
         do j = 1, nbead
         do i = 1, natom
            read( 13, *, iostat=ierr ) k, q(i,j)
         end do
         end do
      else if ( chgtype(1:1) .eq. 'H' ) then
         do j = 1, nbead
         do i = 1, natom
            read( 13, *, iostat=ierr ) k, d, q(i,j)
         end do
         end do
      end if

c     //   read atomic charges of next step
      if ( chgtype(1:1) .eq. 'M' ) then
         do j = 1, nbead
         do i = 1, natom
            read( 14, *, iostat=ierr ) k, q(i,j)
            read( 14, *, iostat=ierr ) k, qnew(i,j)
         end do
         end do
      else if ( chgtype(1:1) .eq. 'H' ) then
         do j = 1, nbead
         do i = 1, natom
            read( 14, *, iostat=ierr ) k, d, q(i,j)
            read( 14, *, iostat=ierr ) k, d, qnew(i,j)
         end do
         end do
      end if

c     //   if no error
      if ( ierr .ne. 0 ) then

c        //   step
         istep = 1

c        //   periodic boundary condition
         if ( iboundary .eq. 1 ) then
            do j = 1, nbead
            do i = 1, natom
               xi = x(i,j) - xold(i,j)
               yi = y(i,j) - yold(i,j)
               zi = z(i,j) - zold(i,j)
               call pbc_fold( xi, yi, zi, box, boxinv )
               x(i,j) = xi + xold(i,j)
               y(i,j) = yi + yold(i,j)
               z(i,j) = zi + zold(i,j)
            end do
            end do
         end if

c        //   charge derivative
         do j = 1, nbead
         do i = 1, natom
            qdot(i,j) = ( qnew(i,j) - q(i,j) ) / dt
         end do
         end do

c        //   initialize
         dipx(:) = 0.d0
         dipy(:) = 0.d0
         dipz(:) = 0.d0

c        //   dipole moment
         do j = 1, nbead
         do i = 1, natom
            dipx(j) = dipx(j) + q(i,j) * x(i,j)
            dipy(j) = dipy(j) + q(i,j) * y(i,j)
            dipz(j) = dipz(j) + q(i,j) * z(i,j)
         end do
         end do

c        //   initialize
         ddipx(:) = 0.d0
         ddipy(:) = 0.d0
         ddipz(:) = 0.d0

c        //   dipole derivative
         do j = 1, nbead
         do i = 1, natom
            ddipx(j) = ddipx(j) + q(i,j) * vx(i,j) + qdot(i,j) * x(i,j)
            ddipy(j) = ddipy(j) + q(i,j) * vy(i,j) + qdot(i,j) * y(i,j)
            ddipz(j) = ddipz(j) + q(i,j) * vz(i,j) + qdot(i,j) * z(i,j)
         end do
         end do

c        //   print step, dipole, dipole derivative
         do j = 1, nbead
            write( 20, '(i8,6e16.8)' )
     &         istep, dipx(j), dipy(j), dipz(j),
     &                ddipx(j), ddipy(j), ddipz(j)
         end do

c     //   if no error
      end if

c-----------------------------------------------------------------------
c     //   main loop
c-----------------------------------------------------------------------

c     //   loop of steps
      do istep = 2, nstep-1

c        //   read atomic coordinates and velocities
         do j = 1, nbead
         do i = 1, natom
            read( 11, *, iostat=ierr )
     &         k, x(i,j), y(i,j), z(i,j), vx(i,j), vy(i,j), vz(i,j)
         end do
         end do

c        //   until end of file
         if ( ierr .ne. 0 ) exit

c        //   read atomic charges of previous step
         if ( chgtype(1:1) .eq. 'M' ) then
            do j = 1, nbead
            do i = 1, natom
               read( 12, *, iostat=ierr ) k, qold(i,j)
            end do
            end do
         else if ( chgtype(1:1) .eq. 'H' ) then
            do j = 1, nbead
            do i = 1, natom
               read( 12, *, iostat=ierr ) k, d, qold(i,j)
            end do
            end do
         end if

c        //   read atomic charges of current step
         if ( chgtype(1:1) .eq. 'M' ) then
            do j = 1, nbead
            do i = 1, natom
               read( 13, *, iostat=ierr ) k, q(i,j)
            end do
            end do
         else if ( chgtype(1:1) .eq. 'H' ) then
            do j = 1, nbead
            do i = 1, natom
               read( 13, *, iostat=ierr ) k, d, q(i,j)
            end do
            end do
         end if

c        //   read atomic charges of next step
         if ( chgtype(1:1) .eq. 'M' ) then
            do j = 1, nbead
            do i = 1, natom
               read( 14, *, iostat=ierr ) k, qnew(i,j)
            end do
            end do
         else if ( chgtype(1:1) .eq. 'H' ) then
            do j = 1, nbead
            do i = 1, natom
               read( 14, *, iostat=ierr ) k, d, qnew(i,j)
            end do
            end do
         end if

c        //   until end of file
         if ( ierr .ne. 0 ) exit

c        //   periodic boundary condition
         if ( iboundary .eq. 1 ) then
            do j = 1, nbead
            do i = 1, natom
               xi = x(i,j) - xold(i,j)
               yi = y(i,j) - yold(i,j)
               zi = z(i,j) - zold(i,j)
               call pbc_fold( xi, yi, zi, box, boxinv )
               x(i,j) = xi + xold(i,j)
               y(i,j) = yi + yold(i,j)
               z(i,j) = zi + zold(i,j)
            end do
            end do
         end if

c        //   charge derivative
         do j = 1, nbead
         do i = 1, natom
            qdot(i,j) = ( qnew(i,j) - qold(i,j) ) / (2.d0*dt)
         end do
         end do

c        //   initialize
         dipx(:) = 0.d0
         dipy(:) = 0.d0
         dipz(:) = 0.d0

c        //   dipole moment
         do j = 1, nbead
         do i = 1, natom
            dipx(j) = dipx(j) + q(i,j) * x(i,j)
            dipy(j) = dipy(j) + q(i,j) * y(i,j)
            dipz(j) = dipz(j) + q(i,j) * z(i,j)
         end do
         end do

c        //   initialize
         ddipx(:) = 0.d0
         ddipy(:) = 0.d0
         ddipz(:) = 0.d0

c        //   dipole derivative
         do j = 1, nbead
         do i = 1, natom
            ddipx(j) = ddipx(j) + q(i,j) * vx(i,j) + qdot(i,j) * x(i,j)
            ddipy(j) = ddipy(j) + q(i,j) * vy(i,j) + qdot(i,j) * y(i,j)
            ddipz(j) = ddipz(j) + q(i,j) * vz(i,j) + qdot(i,j) * z(i,j)
         end do
         end do

c        //   print step, dipole, dipole derivative
         do j = 1, nbead
            write( 20, '(i8,6e16.8)' )
     &         istep, dipx(j), dipy(j), dipz(j),
     &                ddipx(j), ddipy(j), ddipz(j)
         end do

c     //   loop of steps
      end do

c-----------------------------------------------------------------------
c     //   final step
c-----------------------------------------------------------------------

c     //   if no error
      if ( ierr .ne. 0 ) then

c        //   step
         istep = nstep

c        //   read atomic coordinates and velocities
         do j = 1, nbead
         do i = 1, natom
            read( 11, *, iostat=ierr )
     &         k, x(i,j), y(i,j), z(i,j), vx(i,j), vy(i,j), vz(i,j)
         end do
         end do

c        //   until end of file
         if ( ierr .ne. 0 ) go to 100

c        //   read atomic charges of current step
         if ( chgtype(1:1) .eq. 'M' ) then
            do j = 1, nbead
            do i = 1, natom
               read( 13, *, iostat=ierr ) k, q(i,j)
            end do
            end do
         else if ( chgtype(1:1) .eq. 'H' ) then
            do j = 1, nbead
            do i = 1, natom
               read( 13, *, iostat=ierr ) k, d, q(i,j)
            end do
            end do
         end if

c        //   until end of file
         if ( ierr .ne. 0 ) go to 100

c        //   periodic boundary condition
         if ( iboundary .eq. 1 ) then
            do j = 1, nbead
            do i = 1, natom
               xi = x(i,j) - xold(i,j)
               yi = y(i,j) - yold(i,j)
               zi = z(i,j) - zold(i,j)
               call pbc_fold( xi, yi, zi, box, boxinv )
               x(i,j) = xi + xold(i,j)
               y(i,j) = yi + yold(i,j)
               z(i,j) = zi + zold(i,j)
            end do
            end do
         end if

c        //   charge derivative
         do j = 1, nbead
         do i = 1, natom
            qdot(i,j) = ( q(i,j) - qold(i,j) ) / dt
         end do
         end do

c        //   initialize
         dipx(:) = 0.d0
         dipy(:) = 0.d0
         dipz(:) = 0.d0

c        //   dipole moment
         do j = 1, nbead
         do i = 1, natom
            dipx(j) = dipx(j) + q(i,j) * x(i,j)
            dipy(j) = dipy(j) + q(i,j) * y(i,j)
            dipz(j) = dipz(j) + q(i,j) * z(i,j)
         end do
         end do

c        //   initialize
         ddipx(:) = 0.d0
         ddipy(:) = 0.d0
         ddipz(:) = 0.d0

c        //   dipole derivative
         do j = 1, nbead
         do i = 1, natom
            ddipx(j) = ddipx(j) + q(i,j) * vx(i,j) + qdot(i,j) * x(i,j)
            ddipy(j) = ddipy(j) + q(i,j) * vy(i,j) + qdot(i,j) * y(i,j)
            ddipz(j) = ddipz(j) + q(i,j) * vz(i,j) + qdot(i,j) * z(i,j)
         end do
         end do

c        //   print step, dipole, dipole derivative
         do j = 1, nbead
            write( 20, '(i8,6e16.8)' )
     &         istep, dipx(j), dipy(j), dipz(j),
     &                ddipx(j), ddipy(j), ddipz(j)
         end do

c     //   if no error
      end if

c     //   continue
  100 continue

c-----------------------------------------------------------------------
c     //   close files
c-----------------------------------------------------------------------

      close( 11 )
      close( 12 )
      close( 13 )
      close( 14 )
      close( 20 )

c-----------------------------------------------------------------------
c     //   end of program
c-----------------------------------------------------------------------

      write( 6, '(a)' )
      write( 6, '(a)' ) 'Normal termination of chgtrj2dip.'
      write( 6, '(a)' )

      stop
      end





c***********************************************************************
      subroutine pbc_fold ( xi, yi, zi, box, boxinv )
c***********************************************************************
c-----------------------------------------------------------------------
c     /*   local variables                                            */
c-----------------------------------------------------------------------

      implicit none

      real(8) :: box(3,3), boxinv(3,3)
      real(8) :: ai, bi, ci, xi, yi, zi

c-----------------------------------------------------------------------
c     /*   apply boundary condition                                   */
c-----------------------------------------------------------------------

      ai = boxinv(1,1)*xi + boxinv(1,2)*yi + boxinv(1,3)*zi
      bi = boxinv(2,1)*xi + boxinv(2,2)*yi + boxinv(2,3)*zi
      ci = boxinv(3,1)*xi + boxinv(3,2)*yi + boxinv(3,3)*zi

      ai = ai - dnint(ai)
      bi = bi - dnint(bi)
      ci = ci - dnint(ci)

      xi = box(1,1)*ai + box(1,2)*bi + box(1,3)*ci
      yi = box(2,1)*ai + box(2,2)*bi + box(2,3)*ci
      zi = box(3,1)*ai + box(3,2)*bi + box(3,3)*ci

      return
      end





c***********************************************************************
      subroutine search_tag ( char_tag, length_tag, iounit, ierr )
c***********************************************************************
c-----------------------------------------------------------------------
c     /*   local variables                                            */
c-----------------------------------------------------------------------

      implicit none

      integer:: k, length_tag, iounit, ierr

      character(len=80) :: char_line

      character(len=length_tag) :: char_tag

c-----------------------------------------------------------------------
c     /*   do loop start                                              */
c-----------------------------------------------------------------------

      do

c-----------------------------------------------------------------------
c        /*   read a line                                             */
c-----------------------------------------------------------------------

         read (iounit,*,iostat=ierr) char_line

c-----------------------------------------------------------------------
c        /*   return if error is found                                */
c-----------------------------------------------------------------------

         if ( ierr .ne. 0 ) return

c-----------------------------------------------------------------------
c        /*   search for the tag                                      */
c-----------------------------------------------------------------------

         k = index(char_line(1:length_tag),char_tag(1:length_tag))

c-----------------------------------------------------------------------
c        /*   return as soon as we find the tag                       */
c-----------------------------------------------------------------------

         if ( k .ge. 1 ) return

c-----------------------------------------------------------------------
c     /*   do loop end                                                */
c-----------------------------------------------------------------------

      end do

      return
      end





c***********************************************************************
      subroutine inv3 ( a, ainv )
c***********************************************************************
c-----------------------------------------------------------------------
c     /*   local variables                                            */
c-----------------------------------------------------------------------

      implicit none

      integer :: i, j

      real(8) :: a(3,3), ainv(3,3), det3, deta

c-----------------------------------------------------------------------
c     /*   start                                                      */
c-----------------------------------------------------------------------

      ainv(1,1) = + a(2,2)*a(3,3) - a(2,3)*a(3,2)
      ainv(1,2) = + a(3,2)*a(1,3) - a(1,2)*a(3,3)
      ainv(1,3) = + a(1,2)*a(2,3) - a(2,2)*a(1,3)

      ainv(2,1) = + a(2,3)*a(3,1) - a(3,3)*a(2,1)
      ainv(2,2) = + a(3,3)*a(1,1) - a(3,1)*a(1,3)
      ainv(2,3) = + a(1,3)*a(2,1) - a(2,3)*a(1,1)

      ainv(3,1) = + a(2,1)*a(3,2) - a(3,1)*a(2,2)
      ainv(3,2) = + a(3,1)*a(1,2) - a(1,1)*a(3,2)
      ainv(3,3) = + a(1,1)*a(2,2) - a(1,2)*a(2,1)

      deta = det3 ( a )

      do j = 1, 3
      do i = 1, 3
         ainv(i,j) = ainv(i,j)/deta
      end do
      end do

      return
      end





c***********************************************************************
      real(8) function det3 ( a )
c***********************************************************************
c-----------------------------------------------------------------------
c     /*   local variables                                            */
c-----------------------------------------------------------------------

      implicit none

      real(8) :: a(3,3)

c-----------------------------------------------------------------------
c     /*   start                                                      */
c-----------------------------------------------------------------------

      det3 = + a(1,1)*a(2,2)*a(3,3) - a(1,1)*a(2,3)*a(3,2)
     &       + a(2,1)*a(3,2)*a(1,3) - a(2,1)*a(1,2)*a(3,3)
     &       + a(3,1)*a(1,2)*a(2,3) - a(3,1)*a(2,2)*a(1,3)

      return
      end
