#!/bin/python3

import struct
from os import path
from os import walk


def ReadHeaderAndGetNumberOfBeads(file):
    endian = "little"
    byte = file.read(4)
    if int.from_bytes(byte, endian) != 84:
        endian = "big"
    # fast forward
    file.read(4 * 3)

    # stepsize
    byte = file.read(4)
    stepsize = int.from_bytes(byte, endian)

    # fast forward
    file.read(4 * 62)

    # number of beads
    byte = file.read(4)

    # fast forward
    file.read(4)

    return endian, stepsize, int.from_bytes(byte, endian)


def ConvPotential():
    file = open("pot.dcd", "rb")
    endian, stepsize, numbeads = ReadHeaderAndGetNumberOfBeads(file)
    conv = "<d"
    if endian == "big":
        conv = ">d"
    # size of next read (discarded)
    byte = file.read(4)
    step = stepsize
    file_out = open("pot.dat", "w")
    file_out.write("-" * 66 + "\n")
    file_out.write(
        "  Step           Potential           Temp.     Hamiltonian         \n"
    )
    file_out.write("-" * 66 + "\n")
    while byte:
        beadpot = []
        for i in range(numbeads):
            byte = file.read(8)
            beadpot.append(struct.unpack(conv, byte)[0])
        # Temperature
        byte = file.read(8)
        temp = struct.unpack(conv, byte)[0]
        # Hamiltonian
        byte = file.read(8)
        ham = struct.unpack(conv, byte)[0]

        for pot in beadpot:
            file_out.write(
                str("%10d" % step)
                + " "
                + str("%10.16E" % (pot))
                + " "
                + str("%7.2f" % (temp))
                + " "
                + str("%10.16E" % (ham))
                + "\n"
            )
        step += stepsize
        # size of next read (discarded)
        byte = file.read(4)

    file.close()
    file_out.close()


def ConvCharges():
    file = open("charges.dcd", "rb")
    endian, stepsize, numbeads = ReadHeaderAndGetNumberOfBeads(file)
    conv = "<d"
    if endian == "big":
        conv = ">d"
    # size of next read numbeads*numatoms*2*8 [size of double]
    byte = file.read(4)
    numatoms = int(int.from_bytes(byte, endian) / (numbeads * 8 * 2))
    step = stepsize
    file_out = open("charges.dat", "w")
    file_out.write("-" * 100 + "\n")
    file_out.write(
        "        Step       Bead Number       Atom Number      Mulliken Charge       Hirshfeld Charge\n"
    )
    file_out.write("-" * 100 + "\n")
    while byte:
        for i in range(numbeads):
            mull_charge = []
            for _ in range(numatoms):
                byte = file.read(4)
                mull_charge.append(struct.unpack(conv, byte)[0])
            hirsh_charge = []
            for _ in range(numatoms):
                byte = file.read(4)
                hirsh_charge.append(struct.unpack(conv, byte)[0])
            for j in range(1, numatoms + 1):
                file_out.write(
                    str("%10d" % step)
                    + "    "
                    + str("%10d" % int(i + 1))
                    + "       "
                    + str("%10d" % j)
                    + "           "
                    + str("%12.8f" % (mull_charge[j - 1]))
                    + "           "
                    + str("%12.8f" % (hirsh_charge[j - 1]))
                    + "\n"
                )
        step += stepsize
        # size of next read (discarded)
        byte = file.read(4)

    file.close()
    file_out.close()


def ConvDipoles():
    file = open("dip.dcd", "rb")
    endian, stepsize, numbeads = ReadHeaderAndGetNumberOfBeads(file)
    conv = "<d"
    if endian == "big":
        conv = ">d"
    # size of next read (discarded)
    byte = file.read(4)
    step = stepsize
    file_out = open("dip.dat", "w")
    file_out.write("-" * 79 + "\n")
    file_out.write(
        "    Step        x dipole                y dipole                 z dipole      \n"
    )
    file_out.write("-" * 79 + "\n")
    while byte:
        for i in range(numbeads):
            byte = file.read(8)
            xdip = struct.unpack(conv, byte)[0]
            byte = file.read(8)
            ydip = struct.unpack(conv, byte)[0]
            byte = file.read(8)
            zdip = struct.unpack(conv, byte)[0]
            file_out.write(
                str("%10d" % step)
                + " "
                + str("%10.16E" % (xdip))
                + " "
                + str("%10.16E" % (ydip))
                + " "
                + str("%10.16E" % (zdip))
                + "\n"
            )
        step += stepsize
        # size of next read (discarded)
        byte = file.read(4)

    file.close()
    file_out.close()


def ConvBox(filename, outputname):
    file = open(filename, "rb")
    endian, stepsize, _ = ReadHeaderAndGetNumberOfBeads(file)
    conv = "<d"
    if endian == "big":
        conv = ">d"
    # size of next read (discarded)
    byte = file.read(4)
    file_out = open(outputname, "w")
    file_out.write("-" * 106 + "\n")
    file_out.write(
        "    Step       Box(A)         Box(B)         Box(C)         Virial(A)        Virial(B)        Virial(C)   \n"
    )
    file_out.write("-" * 106 + "\n")
    step = stepsize
    while byte:
        box = []
        for i in range(3):
            for j in range(3):
                byte = file.read(8)
                box.append(struct.unpack(conv, byte)[0])

        vir = []
        for i in range(3):
            for j in range(3):
                byte = file.read(8)
                vir.append(struct.unpack(conv, byte)[0])

        for i in range(3):
            file_out.write(
                str("%10d" % step)
                + " "
                + str("%14.8E" % (box[0 + i * 3]))
                + " "
                + str("%14.8E" % (box[1 + i * 3]))
                + " "
                + str("%14.8E" % (box[2 + i * 3]))
                + " "
                + str("%16.8E" % (vir[0 + i * 3]))
                + " "
                + str("%16.8E" % (vir[1 + i * 3]))
                + " "
                + str("%16.8E" % (vir[2 + i * 3]))
                + "\n"
            )

        step += stepsize
        # size of next read (discarded)
        byte = file.read(4)

    file.close()
    file_out.close()


def main():
    ConvPotential()
    ConvDipoles()
    ConvCharges()
    if path.exists("box.dcd"):
        ConvBox("box.dcd", "box.dat")
    if path.exists("box.001.dcd"):
        filenames = next(walk("."), (None, None, []))[2]
        for name in filenames:
            namesplit = name.split(".")
            if len(namesplit) == 3:
                if (namesplit[0] == "box") & (namesplit[2] == "dcd"):
                    ConvBox(name, namesplit[0] + "." + namesplit[1] + ".dat")


if __name__ == "__main__":
    main()
