!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     update box velocities in NPH ensemble
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine update_vel_box_nph_c1
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   vux, vuy, vuz, pres, vir, volume, pres_iso, vvol, volmass, &
     &   dt, boxdot, pressure, fictmass, natom

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i

!-----------------------------------------------------------------------
!     /*   pressure                                                   */
!-----------------------------------------------------------------------

      pres(:,:) =  vir(:,:)

      do i = 1, natom

         pres(1,1) = pres(1,1) + fictmass(i,1)*vux(i,1)*vux(i,1)
         pres(1,2) = pres(1,2) + fictmass(i,1)*vux(i,1)*vuy(i,1)
         pres(1,3) = pres(1,3) + fictmass(i,1)*vux(i,1)*vuz(i,1)
         pres(2,1) = pres(2,1) + fictmass(i,1)*vuy(i,1)*vux(i,1)
         pres(2,2) = pres(2,2) + fictmass(i,1)*vuy(i,1)*vuy(i,1)
         pres(2,3) = pres(2,3) + fictmass(i,1)*vuy(i,1)*vuz(i,1)
         pres(3,1) = pres(3,1) + fictmass(i,1)*vuz(i,1)*vux(i,1)
         pres(3,2) = pres(3,2) + fictmass(i,1)*vuz(i,1)*vuy(i,1)
         pres(3,3) = pres(3,3) + fictmass(i,1)*vuz(i,1)*vuz(i,1)

      end do

      pres(:,:) = pres(:,:) / volume

!-----------------------------------------------------------------------
!     /*   isotropic pressure                                         */
!-----------------------------------------------------------------------

      pres_iso = ( pres(1,1) + pres(2,2) + pres(3,3) ) / 3.d0

!-----------------------------------------------------------------------
!     /*   update volume velocity                                     */
!-----------------------------------------------------------------------

      vvol = vvol + ( pres_iso - pressure ) / volmass * 0.5d0*dt

!-----------------------------------------------------------------------
!     /*   update box velocity                                        */
!-----------------------------------------------------------------------

      boxdot(1,1) = vvol / ( 3.d0*volume**(2.d0/3.d0) )
      boxdot(1,2) = 0.d0
      boxdot(1,3) = 0.d0
      boxdot(2,1) = 0.d0
      boxdot(2,2) = vvol / ( 3.d0*volume**(2.d0/3.d0) )
      boxdot(2,3) = 0.d0
      boxdot(3,1) = 0.d0
      boxdot(3,2) = 0.d0
      boxdot(3,3) = vvol / ( 3.d0*volume**(2.d0/3.d0) )

      return
      end





!***********************************************************************
      subroutine update_vel_box_nph_c2
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   vux, vuy, vuz, pres_iso, vir, volume, pres_iso, vlog, boxmass, &
     &   dt, boxdot, pressure, vir_iso, fictmass, natom

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      real(8) :: dkin, factor

      integer :: i

!-----------------------------------------------------------------------
!     /*   kinetic energy of centroids                                */
!-----------------------------------------------------------------------

      dkin = 0.d0

      do i = 1, natom

         dkin = dkin + fictmass(i,1)*vux(i,1)*vux(i,1) &
     &               + fictmass(i,1)*vuy(i,1)*vuy(i,1) &
     &               + fictmass(i,1)*vuz(i,1)*vuz(i,1)

      end do

      dkin = 0.5d0*dkin

!-----------------------------------------------------------------------
!     /*   isotropic virial                                           */
!-----------------------------------------------------------------------

      vir_iso = vir(1,1) + vir(2,2) + vir(3,3)

!-----------------------------------------------------------------------
!     /*   isotropic pressure                                         */
!-----------------------------------------------------------------------

      pres_iso = ( 2.d0*dkin + vir_iso ) / ( 3.d0 * volume )

!-----------------------------------------------------------------------
!     /*   volume force                                               */
!-----------------------------------------------------------------------

      factor = 3.d0 * volume *( pres_iso - pressure ) &
     &       + 3.d0 / dble(3.d0*natom) * 2.d0*dkin

!-----------------------------------------------------------------------
!     /*   update volume velocity                                     */
!-----------------------------------------------------------------------

      vlog = vlog + factor / boxmass(1,1) * 0.5d0*dt

!-----------------------------------------------------------------------
!     /*   box velocity                                               */
!-----------------------------------------------------------------------

      boxdot(1,1) = vlog * volume**(1.d0/3.d0)
      boxdot(1,2) = 0.d0
      boxdot(1,3) = 0.d0
      boxdot(2,1) = 0.d0
      boxdot(2,2) = vlog * volume**(1.d0/3.d0)
      boxdot(2,3) = 0.d0
      boxdot(3,1) = 0.d0
      boxdot(3,2) = 0.d0
      boxdot(3,3) = vlog * volume**(1.d0/3.d0)

      return
      end





!***********************************************************************
      subroutine update_vel_box_nph_pp
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   vux, vuy, vuz, volume, boxmass, pressure, cmtk, &
     &   dt, vbox, pres, vir, boxdot, box, fictmass, natom

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      real(8) :: ckin(3,3), factor(3,3), dkin

      integer :: i, j, k

!-----------------------------------------------------------------------
!     /*   kinetic energy matrix of centroids                         */
!-----------------------------------------------------------------------

      ckin(:,:) = 0.d0

      do i = 1, natom

         ckin(1,1) = ckin(1,1) + fictmass(i,1)*vux(i,1)*vux(i,1)
         ckin(1,2) = ckin(1,2) + fictmass(i,1)*vux(i,1)*vuy(i,1)
         ckin(1,3) = ckin(1,3) + fictmass(i,1)*vux(i,1)*vuz(i,1)
         ckin(2,1) = ckin(2,1) + fictmass(i,1)*vuy(i,1)*vux(i,1)
         ckin(2,2) = ckin(2,2) + fictmass(i,1)*vuy(i,1)*vuy(i,1)
         ckin(2,3) = ckin(2,3) + fictmass(i,1)*vuy(i,1)*vuz(i,1)
         ckin(3,1) = ckin(3,1) + fictmass(i,1)*vuz(i,1)*vux(i,1)
         ckin(3,2) = ckin(3,2) + fictmass(i,1)*vuz(i,1)*vuy(i,1)
         ckin(3,3) = ckin(3,3) + fictmass(i,1)*vuz(i,1)*vuz(i,1)

      end do

      ckin(:,:) = 0.5d0*ckin(:,:)

!-----------------------------------------------------------------------
!     /*   kinetic energy of centroids                                */
!-----------------------------------------------------------------------

      dkin = ckin(1,1) + ckin(2,2) + ckin(3,3)

!-----------------------------------------------------------------------
!     /*   isotropic pressure                                         */
!-----------------------------------------------------------------------

      pres(:,:) = ( 2.d0*ckin(:,:) + vir(:,:) ) / volume

!-----------------------------------------------------------------------
!     /*   box force                                                  */
!-----------------------------------------------------------------------

      factor(:,:) = volume * pres(:,:)

      factor(1,1) = factor(1,1) - volume * pressure
      factor(2,2) = factor(2,2) - volume * pressure
      factor(3,3) = factor(3,3) - volume * pressure

      factor(1,1) = factor(1,1) + cmtk / dble(3.d0*natom) * 2.d0*dkin
      factor(2,2) = factor(2,2) + cmtk / dble(3.d0*natom) * 2.d0*dkin
      factor(3,3) = factor(3,3) + cmtk / dble(3.d0*natom) * 2.d0*dkin

!-----------------------------------------------------------------------
!     /*   update logarithm of box velocity                           */
!-----------------------------------------------------------------------

      vbox(:,:) = vbox(:,:) + factor(:,:) / boxmass(:,:) * 0.5d0*dt

!-----------------------------------------------------------------------
!     /*   apply rotational correction                                */
!-----------------------------------------------------------------------

      do i = 1, 3
      do j = i, 3
         vbox(i,j) = 0.5d0 * ( vbox(j,i) + vbox(i,j) )
         vbox(j,i) = vbox(i,j)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   box velocity                                               */
!-----------------------------------------------------------------------

      boxdot(:,:) = 0.d0

      do k = 1, 3
      do j = 1, 3
      do i = 1, 3
         boxdot(i,j) = boxdot(i,j) + vbox(i,k)*box(k,j)
      end do
      end do
      end do

      return
      end

