!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     update centroid positions in path integrals
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine update_pos_cent_nph_c1
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   ux, uy, uz, vux, vuy, vuz, dt_ref, volume, vvol

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      real(8) :: factor

!-----------------------------------------------------------------------
!     /*   update due to cell transform                               */
!-----------------------------------------------------------------------

      factor = exp( vvol/(3.d0*volume) * 0.5d0*dt_ref )

      ux(:,1) = ux(:,1) * factor
      uy(:,1) = uy(:,1) * factor
      uz(:,1) = uz(:,1) * factor

!-----------------------------------------------------------------------
!     /*   update due to cell transform                               */
!-----------------------------------------------------------------------

      ux(:,1) = ux(:,1) + vux(:,1) * dt_ref
      uy(:,1) = uy(:,1) + vuy(:,1) * dt_ref
      uz(:,1) = uz(:,1) + vuz(:,1) * dt_ref

!-----------------------------------------------------------------------
!     /*   update due to cell transform                               */
!-----------------------------------------------------------------------

      factor = exp( vvol/(3.d0*volume) * 0.5d0*dt_ref )

      ux(:,1) = ux(:,1) * factor
      uy(:,1) = uy(:,1) * factor
      uz(:,1) = uz(:,1) * factor

!-----------------------------------------------------------------------
!     /*   apply boundary condition                                   */
!-----------------------------------------------------------------------

      call pbc_cent

      return
      end





!***********************************************************************
      subroutine update_pos_cent_nph_c2
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   ux, uy, uz, vux, vuy, vuz, dt_ref, vlog

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      real(8) :: factor

!-----------------------------------------------------------------------
!     /*   update centroid due to box transform                       */
!-----------------------------------------------------------------------

      factor = exp( vlog * 0.5d0*dt_ref )

      ux(:,1) = ux(:,1) * factor
      uy(:,1) = uy(:,1) * factor
      uz(:,1) = uz(:,1) * factor

!-----------------------------------------------------------------------
!     /*   update centroid due to interatomic force                   */
!-----------------------------------------------------------------------

      ux(:,1) = ux(:,1) + vux(:,1) * dt_ref
      uy(:,1) = uy(:,1) + vuy(:,1) * dt_ref
      uz(:,1) = uz(:,1) + vuz(:,1) * dt_ref

!-----------------------------------------------------------------------
!     /*   update centroid due to box transform                       */
!-----------------------------------------------------------------------

      factor = exp( vlog * 0.5d0*dt_ref )

      ux(:,1) = ux(:,1) * factor
      uy(:,1) = uy(:,1) * factor
      uz(:,1) = uz(:,1) * factor

!-----------------------------------------------------------------------
!     /*   apply boundary condition                                   */
!-----------------------------------------------------------------------

      call pbc_cent

      return
      end





!***********************************************************************
      subroutine update_pos_cent_nph_pp
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   ux, uy, uz, vux, vuy, vuz, dt_ref, vbox, natom

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k

      real(8) :: b(3,3), a(3,3), e(3), factor

!-----------------------------------------------------------------------
!     /*   diagonalize box velocity                                   */
!-----------------------------------------------------------------------

      call ddiag ( vbox, e, b, 3 )

!-----------------------------------------------------------------------
!     /*   transform matrix                                           */
!-----------------------------------------------------------------------

      a(:,:) = 0.d0

      do k = 1, 3

         factor = exp( e(k)*0.5d0*dt_ref )

         do i = 1, 3
         do j = 1, 3
            a(i,j) = a(i,j) + b(i,k) * factor * b(j,k)
         end do
         end do

      end do

!-----------------------------------------------------------------------
!     /*   update centroid due to box transform                       */
!-----------------------------------------------------------------------

      do i = 1, natom

         e(1) = a(1,1)*ux(i,1) + a(1,2)*uy(i,1) + a(1,3)*uz(i,1)
         e(2) = a(2,1)*ux(i,1) + a(2,2)*uy(i,1) + a(2,3)*uz(i,1)
         e(3) = a(3,1)*ux(i,1) + a(3,2)*uy(i,1) + a(3,3)*uz(i,1)

         ux(i,1) = e(1)
         uy(i,1) = e(2)
         uz(i,1) = e(3)

      end do

!-----------------------------------------------------------------------
!     /*   update centroid due to atom velocity                       */
!-----------------------------------------------------------------------

      ux(:,1) = ux(:,1) + vux(:,1) * dt_ref
      uy(:,1) = uy(:,1) + vuy(:,1) * dt_ref
      uz(:,1) = uz(:,1) + vuz(:,1) * dt_ref

!-----------------------------------------------------------------------
!     /*   update centroid due to box transform                       */
!-----------------------------------------------------------------------

      do i = 1, natom

         e(1) = a(1,1)*ux(i,1) + a(1,2)*uy(i,1) + a(1,3)*uz(i,1)
         e(2) = a(2,1)*ux(i,1) + a(2,2)*uy(i,1) + a(2,3)*uz(i,1)
         e(3) = a(3,1)*ux(i,1) + a(3,2)*uy(i,1) + a(3,3)*uz(i,1)

         ux(i,1) = e(1)
         uy(i,1) = e(2)
         uz(i,1) = e(3)

      end do

!-----------------------------------------------------------------------
!     /*   apply boundary condition                                   */
!-----------------------------------------------------------------------

      call pbc_cent

      return
      end
