!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     update box positions in NPH ensemble
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine update_pos_box_nph_c1
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   volume, vvol, dt_ref, box, boxinv

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   update volume                                              */
!-----------------------------------------------------------------------

      volume = volume + vvol * 0.5d0*dt_ref

!-----------------------------------------------------------------------
!     /*   update box                                                 */
!-----------------------------------------------------------------------

      box(1,1) = volume**(1.d0/3.d0)
      box(1,2) = 0.d0
      box(1,3) = 0.d0
      box(2,1) = 0.d0
      box(2,2) = volume**(1.d0/3.d0)
      box(2,3) = 0.d0
      box(3,1) = 0.d0
      box(3,2) = 0.d0
      box(3,3) = volume**(1.d0/3.d0)

!-----------------------------------------------------------------------
!     /*   inverse matrix of cell matrix                              */
!-----------------------------------------------------------------------

      call inv3 ( box, boxinv )

      return
      end





!***********************************************************************
      subroutine update_pos_box_nph_c2
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   volume, vlog, dt_ref, box, boxinv

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      real(8) :: factor

!-----------------------------------------------------------------------
!     /*   scaling factor                                             */
!-----------------------------------------------------------------------

      factor = exp( 3.d0 * vlog * 0.5d0*dt_ref )

!-----------------------------------------------------------------------
!     /*   update volume due to box transform                         */
!-----------------------------------------------------------------------

      volume = volume * factor

!-----------------------------------------------------------------------
!     /*   box matrix                                                 */
!-----------------------------------------------------------------------

      box(1,1) = volume**(1.d0/3.d0)
      box(1,2) = 0.d0
      box(1,3) = 0.d0
      box(2,1) = 0.d0
      box(2,2) = volume**(1.d0/3.d0)
      box(2,3) = 0.d0
      box(3,1) = 0.d0
      box(3,2) = 0.d0
      box(3,3) = volume**(1.d0/3.d0)

!-----------------------------------------------------------------------
!     /*   inverse matrix of box matrix                               */
!-----------------------------------------------------------------------

      call inv3 ( box, boxinv )

      return
      end





!***********************************************************************
      subroutine update_pos_box_nph_pp
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   volume, vbox, dt_ref, box, boxinv

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k

      real(8) :: b(3,3), a(3,3), e(3), factor, det3

!-----------------------------------------------------------------------
!     /*   diagonalize box velocity                                   */
!-----------------------------------------------------------------------

      call ddiag ( vbox, e, b, 3 )

!-----------------------------------------------------------------------
!     /*   transform matrix                                           */
!-----------------------------------------------------------------------

      a(:,:) = 0.d0

      do k = 1, 3

         factor = exp( e(k)*0.5d0*dt_ref )

         do j = 1, 3
         do i = 1, 3
            a(i,j) = a(i,j) + b(i,k) * factor * b(j,k)
         end do
         end do

      end do

      do j = 1, 3-1
      do i = j+1, 3
         a(i,j) = 0.5d0 * ( a(i,j) + a(j,i) )
         a(j,i) = a(i,j)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   update box due to box transform                            */
!-----------------------------------------------------------------------

      b(1,:) = a(1,1)*box(1,:) + a(1,2)*box(2,:) + a(1,3)*box(3,:)
      b(2,:) = a(2,1)*box(1,:) + a(2,2)*box(2,:) + a(2,3)*box(3,:)
      b(3,:) = a(3,1)*box(1,:) + a(3,2)*box(2,:) + a(3,3)*box(3,:)

      box(:,:) = b(:,:)

!-----------------------------------------------------------------------
!     /*   update volume                                              */
!-----------------------------------------------------------------------

      volume = det3( box )

!-----------------------------------------------------------------------
!     /*   inverse matrix of cell matrix                              */
!-----------------------------------------------------------------------

      call inv3 ( box, boxinv )

      return
      end

