!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     update massive Nose-Hoover chain on centroids
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine update_mnhc_sys_cart( ds )
!***********************************************************************
!=======================================================================
!
!     integrate Nose-Hoover chain thermostat
!
!     reference:  G. J. Martyna et al., Mol. Phys., 87, 1117 (1996).
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   ysweight, gkt, fictmass, qmass, fxbath, fybath, fzbath, &
     &   vxbath, vybath, vzbath, xbath, ybath, zbath, &
     &   vux, vuy, vuz, nys, nbead, natom, nnhc

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, iys, j, k

      real(8) :: dt_ys, dkinx, dkiny, dkinz, scalex, scaley, scalez, &
     &           vxfact, vyfact, vzfact, pvxfact, pvyfact, pvzfact, ds

!-----------------------------------------------------------------------
!     /*   main loop start                                            */
!-----------------------------------------------------------------------

      do iys = 1, nys

!-----------------------------------------------------------------------
!     /*   step size                                                  */
!-----------------------------------------------------------------------

      dt_ys = ds*ysweight(iys) /2.d0

!-----------------------------------------------------------------------
!     /*   massive nhc start                                          */
!-----------------------------------------------------------------------

      do k = 1, nbead
      do j = 1, natom

!-----------------------------------------------------------------------
!        /*   kinetic energy for each atom                            */
!-----------------------------------------------------------------------

         dkinx = fictmass(j,k)*vux(j,k)*vux(j,k)
         dkiny = fictmass(j,k)*vuy(j,k)*vuy(j,k)
         dkinz = fictmass(j,k)*vuz(j,k)*vuz(j,k)

         scalex = 1.d0
         scaley = 1.d0
         scalez = 1.d0

!-----------------------------------------------------------------------
!        /*   update thermostat forces                                */
!-----------------------------------------------------------------------

         fxbath(j,1,k) = (dkinx - gkt)/qmass(k)
         fybath(j,1,k) = (dkiny - gkt)/qmass(k)
         fzbath(j,1,k) = (dkinz - gkt)/qmass(k)

         do i = 2, nnhc
            fxbath(j,i,k) &
     &      = (qmass(k)*vxbath(j,i-1,k)*vxbath(j,i-1,k) - gkt)/qmass(k)
            fybath(j,i,k) &
     &      = (qmass(k)*vybath(j,i-1,k)*vybath(j,i-1,k) - gkt)/qmass(k)
            fzbath(j,i,k) &
     &      = (qmass(k)*vzbath(j,i-1,k)*vzbath(j,i-1,k) - gkt)/qmass(k)
         end do

!-----------------------------------------------------------------------
!       /*   update thermostat velocities                             */
!-----------------------------------------------------------------------

         vxbath(j,nnhc,k) = vxbath(j,nnhc,k) &
     &                    + 0.25d0*fxbath(j,nnhc,k)*dt_ys
         vybath(j,nnhc,k) = vybath(j,nnhc,k) &
     &                    + 0.25d0*fybath(j,nnhc,k)*dt_ys
         vzbath(j,nnhc,k) = vzbath(j,nnhc,k) &
     &                    + 0.25d0*fzbath(j,nnhc,k)*dt_ys

         do i = 1, nnhc-1

            vxfact = exp(-0.125d0*vxbath(j,nnhc-i+1,k)*dt_ys)
            vyfact = exp(-0.125d0*vybath(j,nnhc-i+1,k)*dt_ys)
            vzfact = exp(-0.125d0*vzbath(j,nnhc-i+1,k)*dt_ys)

            vxbath(j,nnhc-i,k) = vxbath(j,nnhc-i,k)*vxfact*vxfact &
     &                         + 0.25d0*fxbath(j,nnhc-i,k)*vxfact*dt_ys
            vybath(j,nnhc-i,k) = vybath(j,nnhc-i,k)*vyfact*vyfact &
     &                         + 0.25d0*fybath(j,nnhc-i,k)*vyfact*dt_ys
            vzbath(j,nnhc-i,k) = vzbath(j,nnhc-i,k)*vzfact*vzfact &
     &                         + 0.25d0*fzbath(j,nnhc-i,k)*vzfact*dt_ys

         end do

!-----------------------------------------------------------------------
!        /*   update atomic velocities                                */
!-----------------------------------------------------------------------

         pvxfact = exp(-0.5d0*vxbath(j,1,k)*dt_ys)
         pvyfact = exp(-0.5d0*vybath(j,1,k)*dt_ys)
         pvzfact = exp(-0.5d0*vzbath(j,1,k)*dt_ys)

         scalex = scalex*pvxfact
         scaley = scaley*pvyfact
         scalez = scalez*pvzfact

!-----------------------------------------------------------------------
!        /*   update thermostat forces                                */
!-----------------------------------------------------------------------

         fxbath(j,1,k) = (scalex*scalex*dkinx - gkt)/qmass(k)
         fybath(j,1,k) = (scaley*scaley*dkiny - gkt)/qmass(k)
         fzbath(j,1,k) = (scalez*scalez*dkinz - gkt)/qmass(k)

!-----------------------------------------------------------------------
!        /*   update thermostat positions                             */
!-----------------------------------------------------------------------

         do i = 1, nnhc
            xbath(j,i,k) = xbath(j,i,k) + 0.5d0*vxbath(j,i,k)*dt_ys
            ybath(j,i,k) = ybath(j,i,k) + 0.5d0*vybath(j,i,k)*dt_ys
            zbath(j,i,k) = zbath(j,i,k) + 0.5d0*vzbath(j,i,k)*dt_ys
         end do

!-----------------------------------------------------------------------
!       /*   update thermostat velocities                             */
!-----------------------------------------------------------------------

         do i = 1, nnhc-1

            vxfact = exp(-0.125d0*vxbath(j,i+1,k)*dt_ys)
            vyfact = exp(-0.125d0*vybath(j,i+1,k)*dt_ys)
            vzfact = exp(-0.125d0*vzbath(j,i+1,k)*dt_ys)

            vxbath(j,i,k) = vxbath(j,i,k)*vxfact*vxfact &
     &                    + 0.25d0*fxbath(j,i,k)*vxfact*dt_ys
            vybath(j,i,k) = vybath(j,i,k)*vyfact*vyfact &
     &                    + 0.25d0*fybath(j,i,k)*vyfact*dt_ys
            vzbath(j,i,k) = vzbath(j,i,k)*vzfact*vzfact &
     &                    + 0.25d0*fzbath(j,i,k)*vzfact*dt_ys

            fxbath(j,i+1,k) &
     &      = (qmass(k)*vxbath(j,i,k)*vxbath(j,i,k) - gkt)/qmass(k)
            fybath(j,i+1,k) &
     &      = (qmass(k)*vybath(j,i,k)*vybath(j,i,k) - gkt)/qmass(k)
            fzbath(j,i+1,k) &
     &      = (qmass(k)*vzbath(j,i,k)*vzbath(j,i,k) - gkt)/qmass(k)

         end do

         vxbath(j,nnhc,k) = vxbath(j,nnhc,k) &
     &                    + 0.25d0*fxbath(j,nnhc,k)*dt_ys
         vybath(j,nnhc,k) = vybath(j,nnhc,k) &
     &                    + 0.25d0*fybath(j,nnhc,k)*dt_ys
         vzbath(j,nnhc,k) = vzbath(j,nnhc,k) &
     &                    + 0.25d0*fzbath(j,nnhc,k)*dt_ys

!-----------------------------------------------------------------------
!        /*   update atomic velocities                                */
!-----------------------------------------------------------------------

         vux(j,k) = vux(j,k)*scalex
         vuy(j,k) = vuy(j,k)*scaley
         vuz(j,k) = vuz(j,k)*scalez

!-----------------------------------------------------------------------
!     /*   massive nhc end                                            */
!-----------------------------------------------------------------------

      end do
      end do

!-----------------------------------------------------------------------
!     /*   massive nhc start                                          */
!-----------------------------------------------------------------------

      do k = nbead, 1, -1
      do j = natom, 1, -1

!-----------------------------------------------------------------------
!        /*   kinetic energy for each atom                            */
!-----------------------------------------------------------------------

         dkinx = fictmass(j,k)*vux(j,k)*vux(j,k)
         dkiny = fictmass(j,k)*vuy(j,k)*vuy(j,k)
         dkinz = fictmass(j,k)*vuz(j,k)*vuz(j,k)

         scalex = 1.d0
         scaley = 1.d0
         scalez = 1.d0

!-----------------------------------------------------------------------
!        /*   update thermostat forces                                */
!-----------------------------------------------------------------------

         fxbath(j,1,k) = (dkinx - gkt)/qmass(k)
         fybath(j,1,k) = (dkiny - gkt)/qmass(k)
         fzbath(j,1,k) = (dkinz - gkt)/qmass(k)

         do i = 2, nnhc
            fxbath(j,i,k) &
     &      = (qmass(k)*vxbath(j,i-1,k)*vxbath(j,i-1,k) - gkt)/qmass(k)
            fybath(j,i,k) &
     &      = (qmass(k)*vybath(j,i-1,k)*vybath(j,i-1,k) - gkt)/qmass(k)
            fzbath(j,i,k) &
     &      = (qmass(k)*vzbath(j,i-1,k)*vzbath(j,i-1,k) - gkt)/qmass(k)
         end do

!-----------------------------------------------------------------------
!       /*   update thermostat velocities                             */
!-----------------------------------------------------------------------

         vxbath(j,nnhc,k) = vxbath(j,nnhc,k) &
     &                    + 0.25d0*fxbath(j,nnhc,k)*dt_ys
         vybath(j,nnhc,k) = vybath(j,nnhc,k) &
     &                    + 0.25d0*fybath(j,nnhc,k)*dt_ys
         vzbath(j,nnhc,k) = vzbath(j,nnhc,k) &
     &                    + 0.25d0*fzbath(j,nnhc,k)*dt_ys

         do i = 1, nnhc-1

            vxfact = exp(-0.125d0*vxbath(j,nnhc-i+1,k)*dt_ys)
            vyfact = exp(-0.125d0*vybath(j,nnhc-i+1,k)*dt_ys)
            vzfact = exp(-0.125d0*vzbath(j,nnhc-i+1,k)*dt_ys)

            vxbath(j,nnhc-i,k) = vxbath(j,nnhc-i,k)*vxfact*vxfact &
     &                         + 0.25d0*fxbath(j,nnhc-i,k)*vxfact*dt_ys
            vybath(j,nnhc-i,k) = vybath(j,nnhc-i,k)*vyfact*vyfact &
     &                         + 0.25d0*fybath(j,nnhc-i,k)*vyfact*dt_ys
            vzbath(j,nnhc-i,k) = vzbath(j,nnhc-i,k)*vzfact*vzfact &
     &                         + 0.25d0*fzbath(j,nnhc-i,k)*vzfact*dt_ys

         end do

!-----------------------------------------------------------------------
!        /*   update atomic velocities                                */
!-----------------------------------------------------------------------

         pvxfact = exp(-0.5d0*vxbath(j,1,k)*dt_ys)
         pvyfact = exp(-0.5d0*vybath(j,1,k)*dt_ys)
         pvzfact = exp(-0.5d0*vzbath(j,1,k)*dt_ys)

         scalex = scalex*pvxfact
         scaley = scaley*pvyfact
         scalez = scalez*pvzfact

!-----------------------------------------------------------------------
!        /*   update thermostat forces                                */
!-----------------------------------------------------------------------

         fxbath(j,1,k) = (scalex*scalex*dkinx - gkt)/qmass(k)
         fybath(j,1,k) = (scaley*scaley*dkiny - gkt)/qmass(k)
         fzbath(j,1,k) = (scalez*scalez*dkinz - gkt)/qmass(k)

!-----------------------------------------------------------------------
!        /*   update thermostat positions                             */
!-----------------------------------------------------------------------

         do i = 1, nnhc
            xbath(j,i,k) = xbath(j,i,k) + 0.5d0*vxbath(j,i,k)*dt_ys
            ybath(j,i,k) = ybath(j,i,k) + 0.5d0*vybath(j,i,k)*dt_ys
            zbath(j,i,k) = zbath(j,i,k) + 0.5d0*vzbath(j,i,k)*dt_ys
         end do

!-----------------------------------------------------------------------
!       /*   update thermostat velocities                             */
!-----------------------------------------------------------------------

         do i = 1, nnhc-1

            vxfact = exp(-0.125d0*vxbath(j,i+1,k)*dt_ys)
            vyfact = exp(-0.125d0*vybath(j,i+1,k)*dt_ys)
            vzfact = exp(-0.125d0*vzbath(j,i+1,k)*dt_ys)

            vxbath(j,i,k) = vxbath(j,i,k)*vxfact*vxfact &
     &                    + 0.25d0*fxbath(j,i,k)*vxfact*dt_ys
            vybath(j,i,k) = vybath(j,i,k)*vyfact*vyfact &
     &                    + 0.25d0*fybath(j,i,k)*vyfact*dt_ys
            vzbath(j,i,k) = vzbath(j,i,k)*vzfact*vzfact &
     &                    + 0.25d0*fzbath(j,i,k)*vzfact*dt_ys

            fxbath(j,i+1,k) &
     &      = (qmass(k)*vxbath(j,i,k)*vxbath(j,i,k) - gkt)/qmass(k)
            fybath(j,i+1,k) &
     &      = (qmass(k)*vybath(j,i,k)*vybath(j,i,k) - gkt)/qmass(k)
            fzbath(j,i+1,k) &
     &      = (qmass(k)*vzbath(j,i,k)*vzbath(j,i,k) - gkt)/qmass(k)

         end do

         vxbath(j,nnhc,k) = vxbath(j,nnhc,k) &
     &                    + 0.25d0*fxbath(j,nnhc,k)*dt_ys
         vybath(j,nnhc,k) = vybath(j,nnhc,k) &
     &                    + 0.25d0*fybath(j,nnhc,k)*dt_ys
         vzbath(j,nnhc,k) = vzbath(j,nnhc,k) &
     &                    + 0.25d0*fzbath(j,nnhc,k)*dt_ys

!-----------------------------------------------------------------------
!        /*   update atomic velocities                                */
!-----------------------------------------------------------------------

         vux(j,k) = vux(j,k)*scalex
         vuy(j,k) = vuy(j,k)*scaley
         vuz(j,k) = vuz(j,k)*scalez

!-----------------------------------------------------------------------
!     /*   massive nhc end                                            */
!-----------------------------------------------------------------------

      end do
      end do

!-----------------------------------------------------------------------
!     /*   main loop end                                              */
!-----------------------------------------------------------------------

      end do

      return
      end





!***********************************************************************
      subroutine update_mnhc_sys_cart_revised( ds )
!***********************************************************************
!=======================================================================
!
!     Update Nose-Hoover chain thermostat attached to centroid.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   fictmass, vux, vuy, vuz, ysweight, qmass_cart, &
     &   fxbath_cart, fybath_cart, fzbath_cart, gkt, vxbath_cart, &
     &   vybath_cart, vzbath_cart, xbath_cart, ybath_cart, zbath_cart, &
     &   nys, nnhc, ncolor, natom, nbead

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, iys, inhc, j, k, l

      real(8) :: dt_ys, dkinx, dkiny, dkinz, scalex, scaley, scalez, &
     &           vxfact, vyfact, vzfact, pvxfact, pvyfact, pvzfact, ds

!-----------------------------------------------------------------------
!     /*   main loop start                                            */
!-----------------------------------------------------------------------

      do iys = 1, nys

!-----------------------------------------------------------------------
!     /*   step size                                                  */
!-----------------------------------------------------------------------

      dt_ys = ds*ysweight(iys) /2.d0

!-----------------------------------------------------------------------
!        /*   massive nhc start                                       */
!-----------------------------------------------------------------------

         do k = 1, nbead
         do l = 1, ncolor
         do j = 1, natom

!           /*   calculate total kinetic energy of the system   */

            dkinx = fictmass(j,k)*vux(j,k)*vux(j,k)
            dkiny = fictmass(j,k)*vuy(j,k)*vuy(j,k)
            dkinz = fictmass(j,k)*vuz(j,k)*vuz(j,k)

            scalex = 1.d0
            scaley = 1.d0
            scalez = 1.d0

            fxbath_cart(j,1,l,k) = (dkinx - gkt)/qmass_cart(1,l)
            fybath_cart(j,1,l,k) = (dkiny - gkt)/qmass_cart(1,l)
            fzbath_cart(j,1,l,k) = (dkinz - gkt)/qmass_cart(1,l)

            do i = 2, nnhc
              fxbath_cart(j,i,l,k) = &
     &           (qmass_cart(i-1,l)*vxbath_cart(j,i-1,l,k) &
     &           *vxbath_cart(j,i-1,l,k) - gkt)/qmass_cart(i,l)
              fybath_cart(j,i,l,k) = &
     &           (qmass_cart(i-1,l)*vybath_cart(j,i-1,l,k) &
     &           *vybath_cart(j,i-1,l,k) - gkt)/qmass_cart(i,l)
              fzbath_cart(j,i,l,k) = &
     &           (qmass_cart(i-1,l)*vzbath_cart(j,i-1,l,k) &
     &           *vzbath_cart(j,i-1,l,k) - gkt)/qmass_cart(i,l)
            end do

!           /*   update the thermostat velocities   */

            vxbath_cart(j,nnhc,l,k) = vxbath_cart(j,nnhc,l,k) &
     &         + 0.25d0*fxbath_cart(j,nnhc,l,k)*dt_ys
            vybath_cart(j,nnhc,l,k) = vybath_cart(j,nnhc,l,k) &
     &         + 0.25d0*fybath_cart(j,nnhc,l,k)*dt_ys
            vzbath_cart(j,nnhc,l,k) = vzbath_cart(j,nnhc,l,k) &
     &         + 0.25d0*fzbath_cart(j,nnhc,l,k)*dt_ys

            do inhc = 1, nnhc-1

               vxfact &
     &            = exp(-0.125d0*vxbath_cart(j,nnhc-inhc+1,l,k)*dt_ys)
               vyfact &
     &            = exp(-0.125d0*vybath_cart(j,nnhc-inhc+1,l,k)*dt_ys)
               vzfact &
     &            = exp(-0.125d0*vzbath_cart(j,nnhc-inhc+1,l,k)*dt_ys)

               vxbath_cart(j,nnhc-inhc,l,k) = &
     &            vxbath_cart(j,nnhc-inhc,l,k)*vxfact*vxfact &
     &            + 0.25d0*fxbath_cart(j,nnhc-inhc,l,k)*vxfact*dt_ys
               vybath_cart(j,nnhc-inhc,l,k) = &
     &            vybath_cart(j,nnhc-inhc,l,k)*vyfact*vyfact &
     &            + 0.25d0*fybath_cart(j,nnhc-inhc,l,k)*vyfact*dt_ys
               vzbath_cart(j,nnhc-inhc,l,k) = &
     &            vzbath_cart(j,nnhc-inhc,l,k)*vzfact*vzfact &
     &            + 0.25d0*fzbath_cart(j,nnhc-inhc,l,k)*vzfact*dt_ys

            end do

!           /*   update the particle velocities   */

            pvxfact = exp(-0.5d0*vxbath_cart(j,1,l,k)*dt_ys)
            pvyfact = exp(-0.5d0*vybath_cart(j,1,l,k)*dt_ys)
            pvzfact = exp(-0.5d0*vzbath_cart(j,1,l,k)*dt_ys)

            scalex = scalex*pvxfact
            scaley = scaley*pvyfact
            scalez = scalez*pvzfact

!           /*   update the force   */

            fxbath_cart(j,1,l,k) &
     &         = (scalex*scalex*dkinx - gkt)/qmass_cart(1,l)
            fybath_cart(j,1,l,k) &
     &         = (scaley*scaley*dkiny - gkt)/qmass_cart(1,l)
            fzbath_cart(j,1,l,k) &
     &         = (scalez*scalez*dkinz - gkt)/qmass_cart(1,l)

!           /*   update the thermostat positions   */
            do inhc = 1, nnhc
               xbath_cart(j,inhc,l,k) = xbath_cart(j,inhc,l,k) &
     &            + 0.5d0*vxbath_cart(j,inhc,l,k)*dt_ys
               ybath_cart(j,inhc,l,k) = ybath_cart(j,inhc,l,k) &
     &            + 0.5d0*vybath_cart(j,inhc,l,k)*dt_ys
               zbath_cart(j,inhc,l,k) = zbath_cart(j,inhc,l,k) &
     &            + 0.5d0*vzbath_cart(j,inhc,l,k)*dt_ys
            end do

!          /*   update the thermostat velocities   */

            do inhc = 1, nnhc-1

               vxfact = exp(-0.125d0*vxbath_cart(j,inhc+1,l,k)*dt_ys)
               vyfact = exp(-0.125d0*vybath_cart(j,inhc+1,l,k)*dt_ys)
               vzfact = exp(-0.125d0*vzbath_cart(j,inhc+1,l,k)*dt_ys)

               vxbath_cart(j,inhc,l,k) = &
     &            vxbath_cart(j,inhc,l,k)*vxfact*vxfact &
     &            + 0.25d0*fxbath_cart(j,inhc,l,k)*vxfact*dt_ys
               vybath_cart(j,inhc,l,k) = &
     &            vybath_cart(j,inhc,l,k)*vyfact*vyfact &
     &            + 0.25d0*fybath_cart(j,inhc,l,k)*vyfact*dt_ys
               vzbath_cart(j,inhc,l,k) = &
     &            vzbath_cart(j,inhc,l,k)*vzfact*vzfact &
     &            + 0.25d0*fzbath_cart(j,inhc,l,k)*vzfact*dt_ys

               fxbath_cart(j,inhc+1,l,k) = &
     &            (qmass_cart(inhc,l)*vxbath_cart(j,inhc,l,k) &
     &            *vxbath_cart(j,inhc,l,k) - gkt)/qmass_cart(inhc+1,l)
               fybath_cart(j,inhc+1,l,k) = &
     &            (qmass_cart(inhc,l)*vybath_cart(j,inhc,l,k) &
     &            *vybath_cart(j,inhc,l,k) - gkt)/qmass_cart(inhc+1,l)
               fzbath_cart(j,inhc+1,l,k) = &
     &            (qmass_cart(inhc,l)*vzbath_cart(j,inhc,l,k) &
     &            *vzbath_cart(j,inhc,l,k) - gkt)/qmass_cart(inhc+1,l)

            end do

            vxbath_cart(j,nnhc,l,k) = vxbath_cart(j,nnhc,l,k) &
     &         + 0.25d0*fxbath_cart(j,nnhc,l,k)*dt_ys
            vybath_cart(j,nnhc,l,k) = vybath_cart(j,nnhc,l,k) &
     &         + 0.25d0*fybath_cart(j,nnhc,l,k)*dt_ys
            vzbath_cart(j,nnhc,l,k) = vzbath_cart(j,nnhc,l,k) &
     &         + 0.25d0*fzbath_cart(j,nnhc,l,k)*dt_ys

!           /*   update the paricle velocities   */

            vux(j,k) = vux(j,k)*scalex
            vuy(j,k) = vuy(j,k)*scaley
            vuz(j,k) = vuz(j,k)*scalez

!-----------------------------------------------------------------------
!     /*   massive nhc end                                            */
!-----------------------------------------------------------------------

         end do
         end do
         end do

!-----------------------------------------------------------------------
!        /*   massive nhc start                                       */
!-----------------------------------------------------------------------

         do k = nbead,  1, -1
         do l = ncolor, 1, -1
         do j = natom,  1, -1

!           /*   calculate total kinetic energy of the system   */

            dkinx = fictmass(j,k)*vux(j,k)*vux(j,k)
            dkiny = fictmass(j,k)*vuy(j,k)*vuy(j,k)
            dkinz = fictmass(j,k)*vuz(j,k)*vuz(j,k)

            scalex = 1.d0
            scaley = 1.d0
            scalez = 1.d0

            fxbath_cart(j,1,l,k) = (dkinx - gkt)/qmass_cart(1,l)
            fybath_cart(j,1,l,k) = (dkiny - gkt)/qmass_cart(1,l)
            fzbath_cart(j,1,l,k) = (dkinz - gkt)/qmass_cart(1,l)

            do i = 2, nnhc
              fxbath_cart(j,i,l,k) = &
     &           (qmass_cart(i-1,l)*vxbath_cart(j,i-1,l,k) &
     &           *vxbath_cart(j,i-1,l,k) - gkt)/qmass_cart(i,l)
              fybath_cart(j,i,l,k) = &
     &           (qmass_cart(i-1,l)*vybath_cart(j,i-1,l,k) &
     &           *vybath_cart(j,i-1,l,k) - gkt)/qmass_cart(i,l)
              fzbath_cart(j,i,l,k) = &
     &           (qmass_cart(i-1,l)*vzbath_cart(j,i-1,l,k) &
     &           *vzbath_cart(j,i-1,l,k) - gkt)/qmass_cart(i,l)
            end do

!           /*   update the thermostat velocities   */

            vxbath_cart(j,nnhc,l,k) = vxbath_cart(j,nnhc,l,k) &
     &         + 0.25d0*fxbath_cart(j,nnhc,l,k)*dt_ys
            vybath_cart(j,nnhc,l,k) = vybath_cart(j,nnhc,l,k) &
     &         + 0.25d0*fybath_cart(j,nnhc,l,k)*dt_ys
            vzbath_cart(j,nnhc,l,k) = vzbath_cart(j,nnhc,l,k) &
     &         + 0.25d0*fzbath_cart(j,nnhc,l,k)*dt_ys

            do inhc = 1, nnhc-1

               vxfact &
     &            = exp(-0.125d0*vxbath_cart(j,nnhc-inhc+1,l,k)*dt_ys)
               vyfact &
     &            = exp(-0.125d0*vybath_cart(j,nnhc-inhc+1,l,k)*dt_ys)
               vzfact &
     &            = exp(-0.125d0*vzbath_cart(j,nnhc-inhc+1,l,k)*dt_ys)

               vxbath_cart(j,nnhc-inhc,l,k) = &
     &            vxbath_cart(j,nnhc-inhc,l,k)*vxfact*vxfact &
     &            + 0.25d0*fxbath_cart(j,nnhc-inhc,l,k)*vxfact*dt_ys
               vybath_cart(j,nnhc-inhc,l,k) = &
     &            vybath_cart(j,nnhc-inhc,l,k)*vyfact*vyfact &
     &            + 0.25d0*fybath_cart(j,nnhc-inhc,l,k)*vyfact*dt_ys
               vzbath_cart(j,nnhc-inhc,l,k) = &
     &            vzbath_cart(j,nnhc-inhc,l,k)*vzfact*vzfact &
     &            + 0.25d0*fzbath_cart(j,nnhc-inhc,l,k)*vzfact*dt_ys

            end do

!           /*   update the particle velocities   */

            pvxfact = exp(-0.5d0*vxbath_cart(j,1,l,k)*dt_ys)
            pvyfact = exp(-0.5d0*vybath_cart(j,1,l,k)*dt_ys)
            pvzfact = exp(-0.5d0*vzbath_cart(j,1,l,k)*dt_ys)

            scalex = scalex*pvxfact
            scaley = scaley*pvyfact
            scalez = scalez*pvzfact

!           /*   update the force   */

            fxbath_cart(j,1,l,k) &
     &         = (scalex*scalex*dkinx - gkt)/qmass_cart(1,l)
            fybath_cart(j,1,l,k) &
     &         = (scaley*scaley*dkiny - gkt)/qmass_cart(1,l)
            fzbath_cart(j,1,l,k) &
     &         = (scalez*scalez*dkinz - gkt)/qmass_cart(1,l)

!           /*   update the thermostat positions   */
            do inhc = 1, nnhc
               xbath_cart(j,inhc,l,k) = xbath_cart(j,inhc,l,k) &
     &            + 0.5d0*vxbath_cart(j,inhc,l,k)*dt_ys
               ybath_cart(j,inhc,l,k) = ybath_cart(j,inhc,l,k) &
     &            + 0.5d0*vybath_cart(j,inhc,l,k)*dt_ys
               zbath_cart(j,inhc,l,k) = zbath_cart(j,inhc,l,k) &
     &            + 0.5d0*vzbath_cart(j,inhc,l,k)*dt_ys
            end do

!          /*   update the thermostat velocities   */

            do inhc = 1, nnhc-1

               vxfact = exp(-0.125d0*vxbath_cart(j,inhc+1,l,k)*dt_ys)
               vyfact = exp(-0.125d0*vybath_cart(j,inhc+1,l,k)*dt_ys)
               vzfact = exp(-0.125d0*vzbath_cart(j,inhc+1,l,k)*dt_ys)

               vxbath_cart(j,inhc,l,k) = &
     &            vxbath_cart(j,inhc,l,k)*vxfact*vxfact &
     &            + 0.25d0*fxbath_cart(j,inhc,l,k)*vxfact*dt_ys
               vybath_cart(j,inhc,l,k) = &
     &            vybath_cart(j,inhc,l,k)*vyfact*vyfact &
     &            + 0.25d0*fybath_cart(j,inhc,l,k)*vyfact*dt_ys
               vzbath_cart(j,inhc,l,k) = &
     &            vzbath_cart(j,inhc,l,k)*vzfact*vzfact &
     &            + 0.25d0*fzbath_cart(j,inhc,l,k)*vzfact*dt_ys

               fxbath_cart(j,inhc+1,l,k) = &
     &            (qmass_cart(inhc,l)*vxbath_cart(j,inhc,l,k) &
     &            *vxbath_cart(j,inhc,l,k) - gkt)/qmass_cart(inhc+1,l)
               fybath_cart(j,inhc+1,l,k) = &
     &            (qmass_cart(inhc,l)*vybath_cart(j,inhc,l,k) &
     &            *vybath_cart(j,inhc,l,k) - gkt)/qmass_cart(inhc+1,l)
               fzbath_cart(j,inhc+1,l,k) = &
     &            (qmass_cart(inhc,l)*vzbath_cart(j,inhc,l,k) &
     &            *vzbath_cart(j,inhc,l,k) - gkt)/qmass_cart(inhc+1,l)

            end do

            vxbath_cart(j,nnhc,l,k) = vxbath_cart(j,nnhc,l,k) &
     &         + 0.25d0*fxbath_cart(j,nnhc,l,k)*dt_ys
            vybath_cart(j,nnhc,l,k) = vybath_cart(j,nnhc,l,k) &
     &         + 0.25d0*fybath_cart(j,nnhc,l,k)*dt_ys
            vzbath_cart(j,nnhc,l,k) = vzbath_cart(j,nnhc,l,k) &
     &         + 0.25d0*fzbath_cart(j,nnhc,l,k)*dt_ys

!           /*   update the paricle velocities   */

            vux(j,k) = vux(j,k)*scalex
            vuy(j,k) = vuy(j,k)*scaley
            vuz(j,k) = vuz(j,k)*scalez

!-----------------------------------------------------------------------
!     /*   massive nhc end                                            */
!-----------------------------------------------------------------------

         end do
         end do
         end do

!-----------------------------------------------------------------------
!     /*   main loop end                                              */
!-----------------------------------------------------------------------

      end do

      return
      end

