!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Jan 19, 2024 by M. Shiga
!      Description:     update thermostats on noncentroids
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine update_mnhc_mode_me_p_MPI( ds_ref )
!***********************************************************************
!=======================================================================
!
!     integrate Nose-Hoover chain thermostat attached to non-centroid
!     normal modes.
!
!     reference:  G. J. Martyna et al., Mol. Phys., 87, 1117 (1996).
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   fictmass, vux, vuy, vuz, fxbath, fybath, fzbath, gkt, &
     &   vxbath, vybath, vzbath, xbath, ybath, zbath, &
     &   ysweight, nys, nbead, nnhc, nprocs, myrank

      use qmmm_variables, only : &
     &   qmass_multi_a, natom_p

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, iys, j, k

      real(8) :: dt_ys, dkinx, dkiny, dkinz, scalex, scaley, scalez, &
     &           vxfact, vyfact, vzfact, pvxfact, pvyfact, pvzfact, &
     &           ds_ref

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      do k = 2, nbead

         if ( mod( k-1, nprocs ) .ne. myrank ) cycle

      do j = 1, natom_p

      do iys = 1, nys

         dt_ys = ds_ref*ysweight(iys)

!-----------------------------------------------------------------------
!        /*   kinetic energy for each atom                            */
!-----------------------------------------------------------------------

         dkinx = fictmass(j,k)*vux(j,k)*vux(j,k)
         dkiny = fictmass(j,k)*vuy(j,k)*vuy(j,k)
         dkinz = fictmass(j,k)*vuz(j,k)*vuz(j,k)

         scalex = 1.d0
         scaley = 1.d0
         scalez = 1.d0

!-----------------------------------------------------------------------
!        /*   update thermostat forces                                */
!-----------------------------------------------------------------------

         fxbath(j,1,k) = (dkinx - gkt)/qmass_multi_a(k)
         fybath(j,1,k) = (dkiny - gkt)/qmass_multi_a(k)
         fzbath(j,1,k) = (dkinz - gkt)/qmass_multi_a(k)

         do i = 2, nnhc
            fxbath(j,i,k) &
     &      = (qmass_multi_a(k)*vxbath(j,i-1,k)*vxbath(j,i-1,k) - gkt) &
     &        /qmass_multi_a(k)
            fybath(j,i,k) &
     &      = (qmass_multi_a(k)*vybath(j,i-1,k)*vybath(j,i-1,k) - gkt) &
     &        /qmass_multi_a(k)
            fzbath(j,i,k) &
     &      = (qmass_multi_a(k)*vzbath(j,i-1,k)*vzbath(j,i-1,k) - gkt) &
     &        /qmass_multi_a(k)
         end do

!-----------------------------------------------------------------------
!       /*   update thermostat velocities                             */
!-----------------------------------------------------------------------

         vxbath(j,nnhc,k) = vxbath(j,nnhc,k) &
     &                    + 0.25d0*fxbath(j,nnhc,k)*dt_ys
         vybath(j,nnhc,k) = vybath(j,nnhc,k) &
     &                    + 0.25d0*fybath(j,nnhc,k)*dt_ys
         vzbath(j,nnhc,k) = vzbath(j,nnhc,k) &
     &                    + 0.25d0*fzbath(j,nnhc,k)*dt_ys

         do i = 1, nnhc-1

            vxfact = exp(-0.125d0*vxbath(j,nnhc-i+1,k)*dt_ys)
            vyfact = exp(-0.125d0*vybath(j,nnhc-i+1,k)*dt_ys)
            vzfact = exp(-0.125d0*vzbath(j,nnhc-i+1,k)*dt_ys)

            vxbath(j,nnhc-i,k) = vxbath(j,nnhc-i,k)*vxfact*vxfact &
     &                         + 0.25d0*fxbath(j,nnhc-i,k)*vxfact*dt_ys
            vybath(j,nnhc-i,k) = vybath(j,nnhc-i,k)*vyfact*vyfact &
     &                         + 0.25d0*fybath(j,nnhc-i,k)*vyfact*dt_ys
            vzbath(j,nnhc-i,k) = vzbath(j,nnhc-i,k)*vzfact*vzfact &
     &                         + 0.25d0*fzbath(j,nnhc-i,k)*vzfact*dt_ys

         end do

!-----------------------------------------------------------------------
!        /*   update atomic velocities                                */
!-----------------------------------------------------------------------

         pvxfact = exp(-0.5d0*vxbath(j,1,k)*dt_ys)
         pvyfact = exp(-0.5d0*vybath(j,1,k)*dt_ys)
         pvzfact = exp(-0.5d0*vzbath(j,1,k)*dt_ys)

         scalex = scalex*pvxfact
         scaley = scaley*pvyfact
         scalez = scalez*pvzfact

!-----------------------------------------------------------------------
!        /*   update thermostat forces                                */
!-----------------------------------------------------------------------

         fxbath(j,1,k) = (scalex*scalex*dkinx - gkt)/qmass_multi_a(k)
         fybath(j,1,k) = (scaley*scaley*dkiny - gkt)/qmass_multi_a(k)
         fzbath(j,1,k) = (scalez*scalez*dkinz - gkt)/qmass_multi_a(k)

!-----------------------------------------------------------------------
!        /*   update thermostat positions                             */
!-----------------------------------------------------------------------

         do i = 1, nnhc
            xbath(j,i,k) = xbath(j,i,k) + 0.5d0*vxbath(j,i,k)*dt_ys
            ybath(j,i,k) = ybath(j,i,k) + 0.5d0*vybath(j,i,k)*dt_ys
            zbath(j,i,k) = zbath(j,i,k) + 0.5d0*vzbath(j,i,k)*dt_ys
         end do

!-----------------------------------------------------------------------
!       /*   update thermostat velocities                             */
!-----------------------------------------------------------------------

         do i = 1, nnhc-1

            vxfact = exp(-0.125d0*vxbath(j,i+1,k)*dt_ys)
            vyfact = exp(-0.125d0*vybath(j,i+1,k)*dt_ys)
            vzfact = exp(-0.125d0*vzbath(j,i+1,k)*dt_ys)

            vxbath(j,i,k) = vxbath(j,i,k)*vxfact*vxfact &
     &                    + 0.25d0*fxbath(j,i,k)*vxfact*dt_ys
            vybath(j,i,k) = vybath(j,i,k)*vyfact*vyfact &
     &                    + 0.25d0*fybath(j,i,k)*vyfact*dt_ys
            vzbath(j,i,k) = vzbath(j,i,k)*vzfact*vzfact &
     &                    + 0.25d0*fzbath(j,i,k)*vzfact*dt_ys

            fxbath(j,i+1,k) &
     &      = (qmass_multi_a(k)*vxbath(j,i,k)*vxbath(j,i,k) - gkt) &
     &        /qmass_multi_a(k)
            fybath(j,i+1,k) &
     &      = (qmass_multi_a(k)*vybath(j,i,k)*vybath(j,i,k) - gkt) &
     &        /qmass_multi_a(k)
            fzbath(j,i+1,k) &
     &      = (qmass_multi_a(k)*vzbath(j,i,k)*vzbath(j,i,k) - gkt) &
     &        /qmass_multi_a(k)

         end do

         vxbath(j,nnhc,k) = vxbath(j,nnhc,k) &
     &                    + 0.25d0*fxbath(j,nnhc,k)*dt_ys
         vybath(j,nnhc,k) = vybath(j,nnhc,k) &
     &                    + 0.25d0*fybath(j,nnhc,k)*dt_ys
         vzbath(j,nnhc,k) = vzbath(j,nnhc,k) &
     &                    + 0.25d0*fzbath(j,nnhc,k)*dt_ys

!-----------------------------------------------------------------------
!        /*   update atomic velocities                                */
!-----------------------------------------------------------------------

         vux(j,k) = vux(j,k)*scalex
         vuy(j,k) = vuy(j,k)*scaley
         vuz(j,k) = vuz(j,k)*scalez

      end do

      end do
      end do

!-----------------------------------------------------------------------
!     /*   mpi communication                                          */
!-----------------------------------------------------------------------

      call my_mpi_allreduce_mnhc

      return
      end





!***********************************************************************
      subroutine update_mnhc_mode_me_s_MPI( ds_ref )
!***********************************************************************
!=======================================================================
!
!     integrate Nose-Hoover chain thermostat attached to non-centroid
!     normal modes.
!
!     reference:  G. J. Martyna et al., Mol. Phys., 87, 1117 (1996).
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   fictmass, vux, vuy, vuz, fxbath, fybath, fzbath, gkt, &
     &   vxbath, vybath, vzbath, xbath, ybath, zbath, &
     &   ysweight, nys, nbead, nnhc, nprocs, myrank

      use qmmm_variables, only : &
     &   qmass_multi_b, natom_p, natom_s

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, iys, j, k

      real(8) :: dt_ys, dkinx, dkiny, dkinz, scalex, scaley, scalez, &
     &           vxfact, vyfact, vzfact, pvxfact, pvyfact, pvzfact, &
     &           ds_ref

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      do k = 2, nbead

         if ( mod( k-1, nprocs ) .ne. myrank ) cycle

      do j = 1, natom_p+1, natom_p+natom_s

      do iys = 1, nys

         dt_ys = ds_ref*ysweight(iys)

!-----------------------------------------------------------------------
!        /*   kinetic energy for each atom                            */
!-----------------------------------------------------------------------

         dkinx = fictmass(j,k)*vux(j,k)*vux(j,k)
         dkiny = fictmass(j,k)*vuy(j,k)*vuy(j,k)
         dkinz = fictmass(j,k)*vuz(j,k)*vuz(j,k)

         scalex = 1.d0
         scaley = 1.d0
         scalez = 1.d0

!-----------------------------------------------------------------------
!        /*   update thermostat forces                                */
!-----------------------------------------------------------------------

         fxbath(j,1,k) = (dkinx - gkt)/qmass_multi_b(k)
         fybath(j,1,k) = (dkiny - gkt)/qmass_multi_b(k)
         fzbath(j,1,k) = (dkinz - gkt)/qmass_multi_b(k)

         do i = 2, nnhc
            fxbath(j,i,k) &
     &      = (qmass_multi_b(k)*vxbath(j,i-1,k)*vxbath(j,i-1,k) - gkt) &
     &        /qmass_multi_b(k)
            fybath(j,i,k) &
     &      = (qmass_multi_b(k)*vybath(j,i-1,k)*vybath(j,i-1,k) - gkt) &
     &        /qmass_multi_b(k)
            fzbath(j,i,k) &
     &      = (qmass_multi_b(k)*vzbath(j,i-1,k)*vzbath(j,i-1,k) - gkt) &
     &        /qmass_multi_b(k)
         end do

!-----------------------------------------------------------------------
!       /*   update thermostat velocities                             */
!-----------------------------------------------------------------------

         vxbath(j,nnhc,k) = vxbath(j,nnhc,k) &
     &                    + 0.25d0*fxbath(j,nnhc,k)*dt_ys
         vybath(j,nnhc,k) = vybath(j,nnhc,k) &
     &                    + 0.25d0*fybath(j,nnhc,k)*dt_ys
         vzbath(j,nnhc,k) = vzbath(j,nnhc,k) &
     &                    + 0.25d0*fzbath(j,nnhc,k)*dt_ys

         do i = 1, nnhc-1

            vxfact = exp(-0.125d0*vxbath(j,nnhc-i+1,k)*dt_ys)
            vyfact = exp(-0.125d0*vybath(j,nnhc-i+1,k)*dt_ys)
            vzfact = exp(-0.125d0*vzbath(j,nnhc-i+1,k)*dt_ys)

            vxbath(j,nnhc-i,k) = vxbath(j,nnhc-i,k)*vxfact*vxfact &
     &                         + 0.25d0*fxbath(j,nnhc-i,k)*vxfact*dt_ys
            vybath(j,nnhc-i,k) = vybath(j,nnhc-i,k)*vyfact*vyfact &
     &                         + 0.25d0*fybath(j,nnhc-i,k)*vyfact*dt_ys
            vzbath(j,nnhc-i,k) = vzbath(j,nnhc-i,k)*vzfact*vzfact &
     &                         + 0.25d0*fzbath(j,nnhc-i,k)*vzfact*dt_ys

         end do

!-----------------------------------------------------------------------
!        /*   update atomic velocities                                */
!-----------------------------------------------------------------------

         pvxfact = exp(-0.5d0*vxbath(j,1,k)*dt_ys)
         pvyfact = exp(-0.5d0*vybath(j,1,k)*dt_ys)
         pvzfact = exp(-0.5d0*vzbath(j,1,k)*dt_ys)

         scalex = scalex*pvxfact
         scaley = scaley*pvyfact
         scalez = scalez*pvzfact

!-----------------------------------------------------------------------
!        /*   update thermostat forces                                */
!-----------------------------------------------------------------------

         fxbath(j,1,k) = (scalex*scalex*dkinx - gkt)/qmass_multi_b(k)
         fybath(j,1,k) = (scaley*scaley*dkiny - gkt)/qmass_multi_b(k)
         fzbath(j,1,k) = (scalez*scalez*dkinz - gkt)/qmass_multi_b(k)

!-----------------------------------------------------------------------
!        /*   update thermostat positions                             */
!-----------------------------------------------------------------------

         do i = 1, nnhc
            xbath(j,i,k) = xbath(j,i,k) + 0.5d0*vxbath(j,i,k)*dt_ys
            ybath(j,i,k) = ybath(j,i,k) + 0.5d0*vybath(j,i,k)*dt_ys
            zbath(j,i,k) = zbath(j,i,k) + 0.5d0*vzbath(j,i,k)*dt_ys
         end do

!-----------------------------------------------------------------------
!       /*   update thermostat velocities                             */
!-----------------------------------------------------------------------

         do i = 1, nnhc-1

            vxfact = exp(-0.125d0*vxbath(j,i+1,k)*dt_ys)
            vyfact = exp(-0.125d0*vybath(j,i+1,k)*dt_ys)
            vzfact = exp(-0.125d0*vzbath(j,i+1,k)*dt_ys)

            vxbath(j,i,k) = vxbath(j,i,k)*vxfact*vxfact &
     &                    + 0.25d0*fxbath(j,i,k)*vxfact*dt_ys
            vybath(j,i,k) = vybath(j,i,k)*vyfact*vyfact &
     &                    + 0.25d0*fybath(j,i,k)*vyfact*dt_ys
            vzbath(j,i,k) = vzbath(j,i,k)*vzfact*vzfact &
     &                    + 0.25d0*fzbath(j,i,k)*vzfact*dt_ys

            fxbath(j,i+1,k) &
     &      = (qmass_multi_b(k)*vxbath(j,i,k)*vxbath(j,i,k) - gkt) &
     &        /qmass_multi_b(k)
            fybath(j,i+1,k) &
     &      = (qmass_multi_b(k)*vybath(j,i,k)*vybath(j,i,k) - gkt) &
     &        /qmass_multi_b(k)
            fzbath(j,i+1,k) &
     &      = (qmass_multi_b(k)*vzbath(j,i,k)*vzbath(j,i,k) - gkt) &
     &        /qmass_multi_b(k)

         end do

         vxbath(j,nnhc,k) = vxbath(j,nnhc,k) &
     &                    + 0.25d0*fxbath(j,nnhc,k)*dt_ys
         vybath(j,nnhc,k) = vybath(j,nnhc,k) &
     &                    + 0.25d0*fybath(j,nnhc,k)*dt_ys
         vzbath(j,nnhc,k) = vzbath(j,nnhc,k) &
     &                    + 0.25d0*fzbath(j,nnhc,k)*dt_ys

!-----------------------------------------------------------------------
!        /*   update atomic velocities                                */
!-----------------------------------------------------------------------

         vux(j,k) = vux(j,k)*scalex
         vuy(j,k) = vuy(j,k)*scaley
         vuz(j,k) = vuz(j,k)*scalez

      end do

      end do
      end do

!-----------------------------------------------------------------------
!     /*   mpi communication                                          */
!-----------------------------------------------------------------------

      call my_mpi_allreduce_mnhc

      return
      end
