!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     update thermostats on noncentroids
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine update_mnhc_mode_MPI
!***********************************************************************
!=======================================================================
!
!     integrate Nose-Hoover chain thermostat attached to non-centroid
!     normal modes.
!
!     reference:  G. J. Martyna et al., Mol. Phys., 87, 1117 (1996).
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   fictmass, vux, vuy, vuz, fxbath, fybath, fzbath, gkt, qmass, &
     &   vxbath, vybath, vzbath, xbath, ybath, zbath, dt_ref, &
     &   ysweight, nys, natom, nbead, nnhc, myrank, nprocs

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, iys, j, k

      real(8) :: dt_ys, dkinx, dkiny, dkinz, scalex, scaley, scalez, &
     &           vxfact, vyfact, vzfact, pvxfact, pvyfact, pvzfact

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      do k = 2, nbead

         if ( mod( k-1, nprocs ) .ne. myrank ) cycle

      do j = 1, natom

      do iys = 1, nys

         dt_ys = dt_ref*ysweight(iys)


!-----------------------------------------------------------------------
!        /*   kinetic energy for each atom                            */
!-----------------------------------------------------------------------

         dkinx = fictmass(j,k)*vux(j,k)*vux(j,k)
         dkiny = fictmass(j,k)*vuy(j,k)*vuy(j,k)
         dkinz = fictmass(j,k)*vuz(j,k)*vuz(j,k)

         scalex = 1.d0
         scaley = 1.d0
         scalez = 1.d0

!-----------------------------------------------------------------------
!        /*   update thermostat forces                                */
!-----------------------------------------------------------------------

         fxbath(j,1,k) = (dkinx - gkt)/qmass(k)
         fybath(j,1,k) = (dkiny - gkt)/qmass(k)
         fzbath(j,1,k) = (dkinz - gkt)/qmass(k)

         do i = 2, nnhc
            fxbath(j,i,k) &
     &      = (qmass(k)*vxbath(j,i-1,k)*vxbath(j,i-1,k) - gkt)/qmass(k)
            fybath(j,i,k) &
     &      = (qmass(k)*vybath(j,i-1,k)*vybath(j,i-1,k) - gkt)/qmass(k)
            fzbath(j,i,k) &
     &      = (qmass(k)*vzbath(j,i-1,k)*vzbath(j,i-1,k) - gkt)/qmass(k)
         end do

!-----------------------------------------------------------------------
!       /*   update thermostat velocities                             */
!-----------------------------------------------------------------------

         vxbath(j,nnhc,k) = vxbath(j,nnhc,k) &
     &                    + 0.25d0*fxbath(j,nnhc,k)*dt_ys
         vybath(j,nnhc,k) = vybath(j,nnhc,k) &
     &                    + 0.25d0*fybath(j,nnhc,k)*dt_ys
         vzbath(j,nnhc,k) = vzbath(j,nnhc,k) &
     &                    + 0.25d0*fzbath(j,nnhc,k)*dt_ys

         do i = 1, nnhc-1

            vxfact = exp(-0.125d0*vxbath(j,nnhc-i+1,k)*dt_ys)
            vyfact = exp(-0.125d0*vybath(j,nnhc-i+1,k)*dt_ys)
            vzfact = exp(-0.125d0*vzbath(j,nnhc-i+1,k)*dt_ys)

            vxbath(j,nnhc-i,k) = vxbath(j,nnhc-i,k)*vxfact*vxfact &
     &                         + 0.25d0*fxbath(j,nnhc-i,k)*vxfact*dt_ys
            vybath(j,nnhc-i,k) = vybath(j,nnhc-i,k)*vyfact*vyfact &
     &                         + 0.25d0*fybath(j,nnhc-i,k)*vyfact*dt_ys
            vzbath(j,nnhc-i,k) = vzbath(j,nnhc-i,k)*vzfact*vzfact &
     &                         + 0.25d0*fzbath(j,nnhc-i,k)*vzfact*dt_ys

         end do

!-----------------------------------------------------------------------
!        /*   update atomic velocities                                */
!-----------------------------------------------------------------------

         pvxfact = exp(-0.5d0*vxbath(j,1,k)*dt_ys)
         pvyfact = exp(-0.5d0*vybath(j,1,k)*dt_ys)
         pvzfact = exp(-0.5d0*vzbath(j,1,k)*dt_ys)

         scalex = scalex*pvxfact
         scaley = scaley*pvyfact
         scalez = scalez*pvzfact

!-----------------------------------------------------------------------
!        /*   update thermostat forces                                */
!-----------------------------------------------------------------------

         fxbath(j,1,k) = (scalex*scalex*dkinx - gkt)/qmass(k)
         fybath(j,1,k) = (scaley*scaley*dkiny - gkt)/qmass(k)
         fzbath(j,1,k) = (scalez*scalez*dkinz - gkt)/qmass(k)

!-----------------------------------------------------------------------
!        /*   update thermostat positions                             */
!-----------------------------------------------------------------------

         do i = 1, nnhc
            xbath(j,i,k) = xbath(j,i,k) + 0.5d0*vxbath(j,i,k)*dt_ys
            ybath(j,i,k) = ybath(j,i,k) + 0.5d0*vybath(j,i,k)*dt_ys
            zbath(j,i,k) = zbath(j,i,k) + 0.5d0*vzbath(j,i,k)*dt_ys
         end do

!-----------------------------------------------------------------------
!       /*   update thermostat velocities                             */
!-----------------------------------------------------------------------

         do i = 1, nnhc-1

            vxfact = exp(-0.125d0*vxbath(j,i+1,k)*dt_ys)
            vyfact = exp(-0.125d0*vybath(j,i+1,k)*dt_ys)
            vzfact = exp(-0.125d0*vzbath(j,i+1,k)*dt_ys)

            vxbath(j,i,k) = vxbath(j,i,k)*vxfact*vxfact &
     &                    + 0.25d0*fxbath(j,i,k)*vxfact*dt_ys
            vybath(j,i,k) = vybath(j,i,k)*vyfact*vyfact &
     &                    + 0.25d0*fybath(j,i,k)*vyfact*dt_ys
            vzbath(j,i,k) = vzbath(j,i,k)*vzfact*vzfact &
     &                    + 0.25d0*fzbath(j,i,k)*vzfact*dt_ys

            fxbath(j,i+1,k) &
     &      = (qmass(k)*vxbath(j,i,k)*vxbath(j,i,k) - gkt)/qmass(k)
            fybath(j,i+1,k) &
     &      = (qmass(k)*vybath(j,i,k)*vybath(j,i,k) - gkt)/qmass(k)
            fzbath(j,i+1,k) &
     &      = (qmass(k)*vzbath(j,i,k)*vzbath(j,i,k) - gkt)/qmass(k)

         end do

         vxbath(j,nnhc,k) = vxbath(j,nnhc,k) &
     &                    + 0.25d0*fxbath(j,nnhc,k)*dt_ys
         vybath(j,nnhc,k) = vybath(j,nnhc,k) &
     &                    + 0.25d0*fybath(j,nnhc,k)*dt_ys
         vzbath(j,nnhc,k) = vzbath(j,nnhc,k) &
     &                    + 0.25d0*fzbath(j,nnhc,k)*dt_ys

!-----------------------------------------------------------------------
!        /*   update atomic velocities                                */
!-----------------------------------------------------------------------

         vux(j,k) = vux(j,k)*scalex
         vuy(j,k) = vuy(j,k)*scaley
         vuz(j,k) = vuz(j,k)*scalez

      end do

      end do

      end do

!-----------------------------------------------------------------------
!     /*   mpi communication                                          */
!-----------------------------------------------------------------------

      call my_mpi_allreduce_mnhc

      return
      end

