!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     update thermostat attached to centroids
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine update_nhc_cent
!***********************************************************************
!=======================================================================
!
!     Update Nose-Hoover chain thermostat attached to centroid.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   fictmass, vux, vuy, vuz, ysweight, fbath_cent, qmass_cent, dt, &
     &   vbath_cent, rbath_cent, gkt, gnkt, nys, nnhc, ncolor, natom

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, iys, j, k

      real(8) :: dt_ys, dkin, scale, vfact, pvfact

!-----------------------------------------------------------------------
!     /*   calculate total kinetic energy of the system               */
!-----------------------------------------------------------------------

      dkin = 0.d0

      do j = 1, natom

         dkin = dkin + fictmass(j,1)*vux(j,1)*vux(j,1) &
     &               + fictmass(j,1)*vuy(j,1)*vuy(j,1) &
     &               + fictmass(j,1)*vuz(j,1)*vuz(j,1)

      end do

!-----------------------------------------------------------------------
!     /*   velocity scaling factor                                    */
!-----------------------------------------------------------------------

      scale = 1.d0

!-----------------------------------------------------------------------
!     /*   main loop start                                            */
!-----------------------------------------------------------------------

      do iys   = 1, nys

!-----------------------------------------------------------------------
!     /*   step size                                                  */
!-----------------------------------------------------------------------

      dt_ys = dt*ysweight(iys) /2.d0

!-----------------------------------------------------------------------
!     /*   massive nhc start                                          */
!-----------------------------------------------------------------------

      do k = 1, ncolor

!-----------------------------------------------------------------------
!        /*   update the force                                        */
!-----------------------------------------------------------------------

         fbath_cent(1,k) = (dkin - gnkt)/qmass_cent(1,k)

         do i = 2, nnhc

            fbath_cent(i,k) = &
     &         (qmass_cent(i-1,k)*vbath_cent(i-1,k) &
     &         *vbath_cent(i-1,k) - gkt)/qmass_cent(i,k)

         end do

!-----------------------------------------------------------------------
!        /*   update the thermostat velocities                        */
!-----------------------------------------------------------------------

         vbath_cent(nnhc,k) = vbath_cent(nnhc,k) &
     &      + 0.25d0*fbath_cent(nnhc,k)*dt_ys

         do i = 1, nnhc-1

            vfact = exp(-0.125d0*vbath_cent(nnhc-i+1,k)*dt_ys)

            vbath_cent(nnhc-i,k) = vbath_cent(nnhc-i,k)*vfact*vfact &
     &         + 0.25d0*fbath_cent(nnhc-i,k)*vfact*dt_ys

         end do

!-----------------------------------------------------------------------
!        /*   update the particle velocities                          */
!-----------------------------------------------------------------------

         pvfact = exp(-0.5d0*vbath_cent(1,k)*dt_ys)

         scale = scale*pvfact

!-----------------------------------------------------------------------
!        /*   calculate kinetic energy                                */
!-----------------------------------------------------------------------

         dkin = dkin*pvfact*pvfact

!-----------------------------------------------------------------------
!        /*   update the force                                        */
!-----------------------------------------------------------------------

         fbath_cent(1,k) = (dkin - gnkt)/qmass_cent(1,k)

!-----------------------------------------------------------------------
!        /*   update the thermostat position                          */
!-----------------------------------------------------------------------

         do i = 1, nnhc

            rbath_cent(i,k) = rbath_cent(i,k) &
     &                      + 0.5d0*vbath_cent(i,k)*dt_ys

         end do

!-----------------------------------------------------------------------
!        /*   update the thermostat velocities                        */
!-----------------------------------------------------------------------

         do i = 1, nnhc-1

            vfact = exp(-0.125d0*vbath_cent(i+1,k)*dt_ys)

            vbath_cent(i,k) = vbath_cent(i,k)*vfact*vfact &
     &         + 0.25d0*fbath_cent(i,k)*vfact*dt_ys

            fbath_cent(i+1,k) = (qmass_cent(i,k)*vbath_cent(i,k) &
     &         *vbath_cent(i,k) - gkt)/qmass_cent(i+1,k)

         end do

         vbath_cent(nnhc,k) &
     &      = vbath_cent(nnhc,k) + 0.25d0*fbath_cent(nnhc,k)*dt_ys

!-----------------------------------------------------------------------
!     /*   massive nhc end                                            */
!-----------------------------------------------------------------------

      end do

!-----------------------------------------------------------------------
!     /*   massive nhc start                                          */
!-----------------------------------------------------------------------

      do k = ncolor, 1, -1

!-----------------------------------------------------------------------
!        /*   update the force                                        */
!-----------------------------------------------------------------------

         fbath_cent(1,k) = (dkin - gnkt)/qmass_cent(1,k)

         do i = 2, nnhc

            fbath_cent(i,k) = &
     &         (qmass_cent(i-1,k)*vbath_cent(i-1,k) &
     &         *vbath_cent(i-1,k) - gkt)/qmass_cent(i,k)

         end do

!-----------------------------------------------------------------------
!        /*   update the thermostat velocities                        */
!-----------------------------------------------------------------------

         vbath_cent(nnhc,k) = vbath_cent(nnhc,k) &
     &      + 0.25d0*fbath_cent(nnhc,k)*dt_ys

         do i = 1, nnhc-1

            vfact = exp(-0.125d0*vbath_cent(nnhc-i+1,k)*dt_ys)

            vbath_cent(nnhc-i,k) = vbath_cent(nnhc-i,k)*vfact*vfact &
     &         + 0.25d0*fbath_cent(nnhc-i,k)*vfact*dt_ys

         end do

!-----------------------------------------------------------------------
!        /*   update the particle velocities                          */
!-----------------------------------------------------------------------

         pvfact = exp(-0.5d0*vbath_cent(1,k)*dt_ys)

         scale = scale*pvfact

!-----------------------------------------------------------------------
!        /*   calculate kinetic energy                                */
!-----------------------------------------------------------------------

         dkin = dkin*pvfact*pvfact

!-----------------------------------------------------------------------
!        /*   update the force                                        */
!-----------------------------------------------------------------------

         fbath_cent(1,k) = (dkin - gnkt)/qmass_cent(1,k)

!-----------------------------------------------------------------------
!        /*   update the thermostat position                          */
!-----------------------------------------------------------------------

         do i = 1, nnhc

            rbath_cent(i,k) = rbath_cent(i,k) &
     &                      + 0.5d0*vbath_cent(i,k)*dt_ys

         end do

!-----------------------------------------------------------------------
!        /*   update the thermostat velocities                        */
!-----------------------------------------------------------------------

         do i = 1, nnhc-1

            vfact = exp(-0.125d0*vbath_cent(i+1,k)*dt_ys)

            vbath_cent(i,k) = vbath_cent(i,k)*vfact*vfact &
     &         + 0.25d0*fbath_cent(i,k)*vfact*dt_ys

            fbath_cent(i+1,k) = (qmass_cent(i,k)*vbath_cent(i,k) &
     &         *vbath_cent(i,k) - gkt)/qmass_cent(i+1,k)

         end do

         vbath_cent(nnhc,k) &
     &      = vbath_cent(nnhc,k) + 0.25d0*fbath_cent(nnhc,k)*dt_ys

!-----------------------------------------------------------------------
!     /*   massive nhc end                                            */
!-----------------------------------------------------------------------

      end do

!-----------------------------------------------------------------------
!     /*   main loop end                                              */
!-----------------------------------------------------------------------

      end do

!-----------------------------------------------------------------------
!     /*   update the paricle velocities                              */
!-----------------------------------------------------------------------

      do j = 1, natom

         vux(j,1) = vux(j,1)*scale
         vuy(j,1) = vuy(j,1)*scale
         vuz(j,1) = vuz(j,1)*scale

      end do

      return
      end





!***********************************************************************
      subroutine update_nhcs_cent
!***********************************************************************
!=======================================================================
!
!     Update Nose-Hoover chain thermostat attached to centroid.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   fictmass, vux, vuy, vuz, ysweight, qmass_cent, dt, &
     &   fxbath_cent, fybath_cent, fzbath_cent, gkt, vxbath_cent, &
     &   vybath_cent, vzbath_cent, xbath_cent, ybath_cent, zbath_cent, &
     &   nys, nnhc, ncolor, natom

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, iys, j, k

      real(8) :: dt_ys, dkinx, dkiny, dkinz, scale_x, scale_y, scale_z, &
     &           vxfact, vyfact, vzfact, pvxfact, pvyfact, pvzfact

!-----------------------------------------------------------------------
!     /*   calculate total kinetic energy of the system               */
!-----------------------------------------------------------------------

      dkinx = 0.d0
      dkiny = 0.d0
      dkinz = 0.d0

      do j = 1, natom

         dkinx = dkinx + fictmass(j,1)*vux(j,1)*vux(j,1)
         dkiny = dkiny + fictmass(j,1)*vuy(j,1)*vuy(j,1)
         dkinz = dkinz + fictmass(j,1)*vuz(j,1)*vuz(j,1)

      end do

!-----------------------------------------------------------------------
!     /*   velocity scaling factor                                    */
!-----------------------------------------------------------------------

      scale_x = 1.d0
      scale_y = 1.d0
      scale_z = 1.d0

!-----------------------------------------------------------------------
!     /*   main loop start                                            */
!-----------------------------------------------------------------------

      do iys   = 1, nys

!-----------------------------------------------------------------------
!     /*   step size                                                  */
!-----------------------------------------------------------------------

      dt_ys = dt*ysweight(iys) /2.d0

!-----------------------------------------------------------------------
!     /*   massive nhc start                                          */
!-----------------------------------------------------------------------

      do k = 1, ncolor

!-----------------------------------------------------------------------
!        /*   update the force                                        */
!-----------------------------------------------------------------------

         fxbath_cent(1,1,k) = (dkinx - natom*gkt)/qmass_cent(1,k)
         fybath_cent(1,1,k) = (dkiny - natom*gkt)/qmass_cent(1,k)
         fzbath_cent(1,1,k) = (dkinz - natom*gkt)/qmass_cent(1,k)

         do i = 2, nnhc

            fxbath_cent(1,i,k) = &
     &         (qmass_cent(i-1,k)*vxbath_cent(1,i-1,k) &
     &         *vxbath_cent(1,i-1,k) - gkt)/qmass_cent(i,k)
            fybath_cent(1,i,k) = &
     &         (qmass_cent(i-1,k)*vybath_cent(1,i-1,k) &
     &         *vybath_cent(1,i-1,k) - gkt)/qmass_cent(i,k)
            fzbath_cent(1,i,k) = &
     &         (qmass_cent(i-1,k)*vzbath_cent(1,i-1,k) &
     &         *vzbath_cent(1,i-1,k) - gkt)/qmass_cent(i,k)

         end do

!-----------------------------------------------------------------------
!        /*   update the thermostat velocities                        */
!-----------------------------------------------------------------------

         vxbath_cent(1,nnhc,k) = vxbath_cent(1,nnhc,k) &
     &      + 0.25d0*fxbath_cent(1,nnhc,k)*dt_ys
         vybath_cent(1,nnhc,k) = vybath_cent(1,nnhc,k) &
     &      + 0.25d0*fybath_cent(1,nnhc,k)*dt_ys
         vzbath_cent(1,nnhc,k) = vzbath_cent(1,nnhc,k) &
     &      + 0.25d0*fzbath_cent(1,nnhc,k)*dt_ys

         do i = 1, nnhc-1

            vxfact = exp(-0.125d0*vxbath_cent(1,nnhc-i+1,k)*dt_ys)
            vyfact = exp(-0.125d0*vybath_cent(1,nnhc-i+1,k)*dt_ys)
            vzfact = exp(-0.125d0*vzbath_cent(1,nnhc-i+1,k)*dt_ys)

            vxbath_cent(1,nnhc-i,k) &
     &         = vxbath_cent(1,nnhc-i,k)*vxfact*vxfact &
     &         + 0.25d0*fxbath_cent(1,nnhc-i,k)*vxfact*dt_ys
            vybath_cent(1,nnhc-i,k) &
     &         = vybath_cent(1,nnhc-i,k)*vyfact*vyfact &
     &         + 0.25d0*fybath_cent(1,nnhc-i,k)*vyfact*dt_ys
            vzbath_cent(1,nnhc-i,k) &
     &         = vzbath_cent(1,nnhc-i,k)*vzfact*vzfact &
     &         + 0.25d0*fzbath_cent(1,nnhc-i,k)*vzfact*dt_ys

         end do

!-----------------------------------------------------------------------
!        /*   update the particle velocities                          */
!-----------------------------------------------------------------------

         pvxfact = exp(-0.5d0*vxbath_cent(1,1,k)*dt_ys)
         pvyfact = exp(-0.5d0*vybath_cent(1,1,k)*dt_ys)
         pvzfact = exp(-0.5d0*vzbath_cent(1,1,k)*dt_ys)

         scale_x = scale_x*pvxfact
         scale_y = scale_y*pvyfact
         scale_z = scale_z*pvzfact

!-----------------------------------------------------------------------
!        /*   calculate kinetic energy                                */
!-----------------------------------------------------------------------

         dkinx = dkinx*pvxfact*pvxfact
         dkiny = dkiny*pvyfact*pvyfact
         dkinz = dkinz*pvzfact*pvzfact

!-----------------------------------------------------------------------
!        /*   update the force                                        */
!-----------------------------------------------------------------------

         fxbath_cent(1,1,k) = (dkinx - natom*gkt)/qmass_cent(1,k)
         fybath_cent(1,1,k) = (dkiny - natom*gkt)/qmass_cent(1,k)
         fzbath_cent(1,1,k) = (dkinz - natom*gkt)/qmass_cent(1,k)

!-----------------------------------------------------------------------
!        /*   update the thermostat position                          */
!-----------------------------------------------------------------------

         do i = 1, nnhc

            xbath_cent(1,i,k) &
     &         = xbath_cent(1,i,k) + 0.5d0*vxbath_cent(1,i,k)*dt_ys
            ybath_cent(1,i,k) &
     &         = ybath_cent(1,i,k) + 0.5d0*vybath_cent(1,i,k)*dt_ys
            zbath_cent(1,i,k) &
     &         = zbath_cent(1,i,k) + 0.5d0*vzbath_cent(1,i,k)*dt_ys

         end do

!-----------------------------------------------------------------------
!        /*   update the thermostat velocities                        */
!-----------------------------------------------------------------------

         do i = 1, nnhc-1

            vxfact = exp(-0.125d0*vxbath_cent(1,i+1,k)*dt_ys)
            vyfact = exp(-0.125d0*vybath_cent(1,i+1,k)*dt_ys)
            vzfact = exp(-0.125d0*vzbath_cent(1,i+1,k)*dt_ys)

            vxbath_cent(1,i,k) = vxbath_cent(1,i,k)*vxfact*vxfact &
     &                  + 0.25d0*fxbath_cent(1,i,k)*vxfact*dt_ys
            vybath_cent(1,i,k) = vybath_cent(1,i,k)*vyfact*vyfact &
     &                  + 0.25d0*fybath_cent(1,i,k)*vyfact*dt_ys
            vzbath_cent(1,i,k) = vzbath_cent(1,i,k)*vzfact*vzfact &
     &                  + 0.25d0*fzbath_cent(1,i,k)*vzfact*dt_ys

            fxbath_cent(1,i+1,k) = (qmass_cent(i,k)*vxbath_cent(1,i,k) &
     &         *vxbath_cent(1,i,k) - gkt)/qmass_cent(i+1,k)
            fybath_cent(1,i+1,k) = (qmass_cent(i,k)*vybath_cent(1,i,k) &
     &         *vybath_cent(1,i,k) - gkt)/qmass_cent(i+1,k)
            fzbath_cent(1,i+1,k) = (qmass_cent(i,k)*vzbath_cent(1,i,k) &
     &         *vzbath_cent(1,i,k) - gkt)/qmass_cent(i+1,k)

         end do

         vxbath_cent(1,nnhc,k) = vxbath_cent(1,nnhc,k) &
     &      + 0.25d0*fxbath_cent(1,nnhc,k)*dt_ys
         vybath_cent(1,nnhc,k) = vybath_cent(1,nnhc,k) &
     &      + 0.25d0*fybath_cent(1,nnhc,k)*dt_ys
         vzbath_cent(1,nnhc,k) = vzbath_cent(1,nnhc,k) &
     &      + 0.25d0*fzbath_cent(1,nnhc,k)*dt_ys

!-----------------------------------------------------------------------
!     /*   massive nhc end                                            */
!-----------------------------------------------------------------------

      end do

!-----------------------------------------------------------------------
!     /*   massive nhc start                                          */
!-----------------------------------------------------------------------

      do k = ncolor, 1, -1

!-----------------------------------------------------------------------
!        /*   update the force                                        */
!-----------------------------------------------------------------------

         fxbath_cent(1,1,k) = (dkinx - natom*gkt)/qmass_cent(1,k)
         fybath_cent(1,1,k) = (dkiny - natom*gkt)/qmass_cent(1,k)
         fzbath_cent(1,1,k) = (dkinz - natom*gkt)/qmass_cent(1,k)

         do i = 2, nnhc

            fxbath_cent(1,i,k) = &
     &         (qmass_cent(i-1,k)*vxbath_cent(1,i-1,k) &
     &         *vxbath_cent(1,i-1,k) - gkt)/qmass_cent(i,k)
            fybath_cent(1,i,k) = &
     &         (qmass_cent(i-1,k)*vybath_cent(1,i-1,k) &
     &         *vybath_cent(1,i-1,k) - gkt)/qmass_cent(i,k)
            fzbath_cent(1,i,k) = &
     &         (qmass_cent(i-1,k)*vzbath_cent(1,i-1,k) &
     &         *vzbath_cent(1,i-1,k) - gkt)/qmass_cent(i,k)

         end do

!-----------------------------------------------------------------------
!        /*   update the thermostat velocities                        */
!-----------------------------------------------------------------------

         vxbath_cent(1,nnhc,k) = vxbath_cent(1,nnhc,k) &
     &      + 0.25d0*fxbath_cent(1,nnhc,k)*dt_ys
         vybath_cent(1,nnhc,k) = vybath_cent(1,nnhc,k) &
     &      + 0.25d0*fybath_cent(1,nnhc,k)*dt_ys
         vzbath_cent(1,nnhc,k) = vzbath_cent(1,nnhc,k) &
     &      + 0.25d0*fzbath_cent(1,nnhc,k)*dt_ys

         do i = 1, nnhc-1

            vxfact = exp(-0.125d0*vxbath_cent(1,nnhc-i+1,k)*dt_ys)
            vyfact = exp(-0.125d0*vybath_cent(1,nnhc-i+1,k)*dt_ys)
            vzfact = exp(-0.125d0*vzbath_cent(1,nnhc-i+1,k)*dt_ys)

            vxbath_cent(1,nnhc-i,k) &
     &         = vxbath_cent(1,nnhc-i,k)*vxfact*vxfact &
     &         + 0.25d0*fxbath_cent(1,nnhc-i,k)*vxfact*dt_ys
            vybath_cent(1,nnhc-i,k) &
     &         = vybath_cent(1,nnhc-i,k)*vyfact*vyfact &
     &         + 0.25d0*fybath_cent(1,nnhc-i,k)*vyfact*dt_ys
            vzbath_cent(1,nnhc-i,k) &
     &         = vzbath_cent(1,nnhc-i,k)*vzfact*vzfact &
     &         + 0.25d0*fzbath_cent(1,nnhc-i,k)*vzfact*dt_ys

         end do

!-----------------------------------------------------------------------
!        /*   update the particle velocities                          */
!-----------------------------------------------------------------------

         pvxfact = exp(-0.5d0*vxbath_cent(1,1,k)*dt_ys)
         pvyfact = exp(-0.5d0*vybath_cent(1,1,k)*dt_ys)
         pvzfact = exp(-0.5d0*vzbath_cent(1,1,k)*dt_ys)

         scale_x = scale_x*pvxfact
         scale_y = scale_y*pvyfact
         scale_z = scale_z*pvzfact

!-----------------------------------------------------------------------
!        /*   calculate kinetic energy                                */
!-----------------------------------------------------------------------

         dkinx = dkinx*pvxfact*pvxfact
         dkiny = dkiny*pvyfact*pvyfact
         dkinz = dkinz*pvzfact*pvzfact

!-----------------------------------------------------------------------
!        /*   update the force                                        */
!-----------------------------------------------------------------------

         fxbath_cent(1,1,k) = (dkinx - natom*gkt)/qmass_cent(1,k)
         fybath_cent(1,1,k) = (dkiny - natom*gkt)/qmass_cent(1,k)
         fzbath_cent(1,1,k) = (dkinz - natom*gkt)/qmass_cent(1,k)

!-----------------------------------------------------------------------
!        /*   update the thermostat position                          */
!-----------------------------------------------------------------------

         do i = 1, nnhc

            xbath_cent(1,i,k) &
     &         = xbath_cent(1,i,k) + 0.5d0*vxbath_cent(1,i,k)*dt_ys
            ybath_cent(1,i,k) &
     &         = ybath_cent(1,i,k) + 0.5d0*vybath_cent(1,i,k)*dt_ys
            zbath_cent(1,i,k) &
     &         = zbath_cent(1,i,k) + 0.5d0*vzbath_cent(1,i,k)*dt_ys

         end do

!-----------------------------------------------------------------------
!        /*   update the thermostat velocities                        */
!-----------------------------------------------------------------------

         do i = 1, nnhc-1

            vxfact = exp(-0.125d0*vxbath_cent(1,i+1,k)*dt_ys)
            vyfact = exp(-0.125d0*vybath_cent(1,i+1,k)*dt_ys)
            vzfact = exp(-0.125d0*vzbath_cent(1,i+1,k)*dt_ys)

            vxbath_cent(1,i,k) = vxbath_cent(1,i,k)*vxfact*vxfact &
     &                  + 0.25d0*fxbath_cent(1,i,k)*vxfact*dt_ys
            vybath_cent(1,i,k) = vybath_cent(1,i,k)*vyfact*vyfact &
     &                  + 0.25d0*fybath_cent(1,i,k)*vyfact*dt_ys
            vzbath_cent(1,i,k) = vzbath_cent(1,i,k)*vzfact*vzfact &
     &                  + 0.25d0*fzbath_cent(1,i,k)*vzfact*dt_ys

            fxbath_cent(1,i+1,k) = (qmass_cent(i,k)*vxbath_cent(1,i,k) &
     &         *vxbath_cent(1,i,k) - gkt)/qmass_cent(i+1,k)
            fybath_cent(1,i+1,k) = (qmass_cent(i,k)*vybath_cent(1,i,k) &
     &         *vybath_cent(1,i,k) - gkt)/qmass_cent(i+1,k)
            fzbath_cent(1,i+1,k) = (qmass_cent(i,k)*vzbath_cent(1,i,k) &
     &         *vzbath_cent(1,i,k) - gkt)/qmass_cent(i+1,k)

         end do

         vxbath_cent(1,nnhc,k) = vxbath_cent(1,nnhc,k) &
     &      + 0.25d0*fxbath_cent(1,nnhc,k)*dt_ys
         vybath_cent(1,nnhc,k) = vybath_cent(1,nnhc,k) &
     &      + 0.25d0*fybath_cent(1,nnhc,k)*dt_ys
         vzbath_cent(1,nnhc,k) = vzbath_cent(1,nnhc,k) &
     &      + 0.25d0*fzbath_cent(1,nnhc,k)*dt_ys

!-----------------------------------------------------------------------
!     /*   massive nhc end                                            */
!-----------------------------------------------------------------------

      end do

!-----------------------------------------------------------------------
!     /*   main loop end                                              */
!-----------------------------------------------------------------------

      end do

!-----------------------------------------------------------------------
!     /*   update the paricle velocities                              */
!-----------------------------------------------------------------------

      do j = 1, natom

         vux(j,1) = vux(j,1)*scale_x
         vuy(j,1) = vuy(j,1)*scale_y
         vuz(j,1) = vuz(j,1)*scale_z

      end do

      return
      end





!***********************************************************************
      subroutine update_mnhc_cent
!***********************************************************************
!=======================================================================
!
!     Update Nose-Hoover chain thermostat attached to centroid.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   fictmass, vux, vuy, vuz, ysweight, qmass_cent, dt, &
     &   fxbath_cent, fybath_cent, fzbath_cent, gkt, vxbath_cent, &
     &   vybath_cent, vzbath_cent, xbath_cent, ybath_cent, zbath_cent, &
     &   nys, nnhc, ncolor, natom

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, iys, inhc, j, l

      real(8) :: dt_ys, dkinx, dkiny, dkinz, scalex, scaley, scalez, &
     &           vxfact, vyfact, vzfact, pvxfact, pvyfact, pvzfact

!-----------------------------------------------------------------------
!     /*   main loop start                                            */
!-----------------------------------------------------------------------

      do iys = 1, nys

!-----------------------------------------------------------------------
!     /*   step size                                                  */
!-----------------------------------------------------------------------

      dt_ys = dt*ysweight(iys) /2.d0

!-----------------------------------------------------------------------
!        /*   massive nhc start                                       */
!-----------------------------------------------------------------------

         do l = 1, ncolor
         do j = 1, natom

!           /*   calculate total kinetic energy of the system   */

            dkinx = fictmass(j,1)*vux(j,1)*vux(j,1)
            dkiny = fictmass(j,1)*vuy(j,1)*vuy(j,1)
            dkinz = fictmass(j,1)*vuz(j,1)*vuz(j,1)

            scalex = 1.d0
            scaley = 1.d0
            scalez = 1.d0

            fxbath_cent(j,1,l) = (dkinx - gkt)/qmass_cent(1,l)
            fybath_cent(j,1,l) = (dkiny - gkt)/qmass_cent(1,l)
            fzbath_cent(j,1,l) = (dkinz - gkt)/qmass_cent(1,l)

            do i = 2, nnhc
              fxbath_cent(j,i,l) = &
     &           (qmass_cent(i-1,l)*vxbath_cent(j,i-1,l) &
     &           *vxbath_cent(j,i-1,l) - gkt)/qmass_cent(i,l)
              fybath_cent(j,i,l) = &
     &           (qmass_cent(i-1,l)*vybath_cent(j,i-1,l) &
     &           *vybath_cent(j,i-1,l) - gkt)/qmass_cent(i,l)
              fzbath_cent(j,i,l) = &
     &           (qmass_cent(i-1,l)*vzbath_cent(j,i-1,l) &
     &           *vzbath_cent(j,i-1,l) - gkt)/qmass_cent(i,l)
            end do

!           /*   update the thermostat velocities   */

            vxbath_cent(j,nnhc,l) = vxbath_cent(j,nnhc,l) &
     &         + 0.25d0*fxbath_cent(j,nnhc,l)*dt_ys
            vybath_cent(j,nnhc,l) = vybath_cent(j,nnhc,l) &
     &         + 0.25d0*fybath_cent(j,nnhc,l)*dt_ys
            vzbath_cent(j,nnhc,l) = vzbath_cent(j,nnhc,l) &
     &         + 0.25d0*fzbath_cent(j,nnhc,l)*dt_ys

            do inhc = 1, nnhc-1

               vxfact &
     &            = exp(-0.125d0*vxbath_cent(j,nnhc-inhc+1,l)*dt_ys)
               vyfact &
     &            = exp(-0.125d0*vybath_cent(j,nnhc-inhc+1,l)*dt_ys)
               vzfact &
     &            = exp(-0.125d0*vzbath_cent(j,nnhc-inhc+1,l)*dt_ys)

               vxbath_cent(j,nnhc-inhc,l) = &
     &            vxbath_cent(j,nnhc-inhc,l)*vxfact*vxfact &
     &            + 0.25d0*fxbath_cent(j,nnhc-inhc,l)*vxfact*dt_ys
               vybath_cent(j,nnhc-inhc,l) = &
     &            vybath_cent(j,nnhc-inhc,l)*vyfact*vyfact &
     &            + 0.25d0*fybath_cent(j,nnhc-inhc,l)*vyfact*dt_ys
               vzbath_cent(j,nnhc-inhc,l) = &
     &            vzbath_cent(j,nnhc-inhc,l)*vzfact*vzfact &
     &            + 0.25d0*fzbath_cent(j,nnhc-inhc,l)*vzfact*dt_ys

            end do

!           /*   update the particle velocities   */

            pvxfact = exp(-0.5d0*vxbath_cent(j,1,l)*dt_ys)
            pvyfact = exp(-0.5d0*vybath_cent(j,1,l)*dt_ys)
            pvzfact = exp(-0.5d0*vzbath_cent(j,1,l)*dt_ys)

            scalex = scalex*pvxfact
            scaley = scaley*pvyfact
            scalez = scalez*pvzfact

!           /*   update the force   */

            fxbath_cent(j,1,l) &
     &         = (scalex*scalex*dkinx - gkt)/qmass_cent(1,l)
            fybath_cent(j,1,l) &
     &         = (scaley*scaley*dkiny - gkt)/qmass_cent(1,l)
            fzbath_cent(j,1,l) &
     &         = (scalez*scalez*dkinz - gkt)/qmass_cent(1,l)

!           /*   update the thermostat positions   */
            do inhc = 1, nnhc
               xbath_cent(j,inhc,l) = xbath_cent(j,inhc,l) &
     &            + 0.5d0*vxbath_cent(j,inhc,l)*dt_ys
               ybath_cent(j,inhc,l) = ybath_cent(j,inhc,l) &
     &            + 0.5d0*vybath_cent(j,inhc,l)*dt_ys
               zbath_cent(j,inhc,l) = zbath_cent(j,inhc,l) &
     &            + 0.5d0*vzbath_cent(j,inhc,l)*dt_ys
            end do

!          /*   update the thermostat velocities   */

            do inhc = 1, nnhc-1

               vxfact = exp(-0.125d0*vxbath_cent(j,inhc+1,l)*dt_ys)
               vyfact = exp(-0.125d0*vybath_cent(j,inhc+1,l)*dt_ys)
               vzfact = exp(-0.125d0*vzbath_cent(j,inhc+1,l)*dt_ys)

               vxbath_cent(j,inhc,l) = &
     &            vxbath_cent(j,inhc,l)*vxfact*vxfact &
     &            + 0.25d0*fxbath_cent(j,inhc,l)*vxfact*dt_ys
               vybath_cent(j,inhc,l) = &
     &            vybath_cent(j,inhc,l)*vyfact*vyfact &
     &            + 0.25d0*fybath_cent(j,inhc,l)*vyfact*dt_ys
               vzbath_cent(j,inhc,l) = &
     &            vzbath_cent(j,inhc,l)*vzfact*vzfact &
     &            + 0.25d0*fzbath_cent(j,inhc,l)*vzfact*dt_ys

               fxbath_cent(j,inhc+1,l) = &
     &            (qmass_cent(inhc,l)*vxbath_cent(j,inhc,l) &
     &            *vxbath_cent(j,inhc,l) - gkt)/qmass_cent(inhc+1,l)
               fybath_cent(j,inhc+1,l) = &
     &            (qmass_cent(inhc,l)*vybath_cent(j,inhc,l) &
     &            *vybath_cent(j,inhc,l) - gkt)/qmass_cent(inhc+1,l)
               fzbath_cent(j,inhc+1,l) = &
     &            (qmass_cent(inhc,l)*vzbath_cent(j,inhc,l) &
     &            *vzbath_cent(j,inhc,l) - gkt)/qmass_cent(inhc+1,l)

            end do

            vxbath_cent(j,nnhc,l) = vxbath_cent(j,nnhc,l) &
     &         + 0.25d0*fxbath_cent(j,nnhc,l)*dt_ys
            vybath_cent(j,nnhc,l) = vybath_cent(j,nnhc,l) &
     &         + 0.25d0*fybath_cent(j,nnhc,l)*dt_ys
            vzbath_cent(j,nnhc,l) = vzbath_cent(j,nnhc,l) &
     &         + 0.25d0*fzbath_cent(j,nnhc,l)*dt_ys

!           /*   update the paricle velocities   */

            vux(j,1) = vux(j,1)*scalex
            vuy(j,1) = vuy(j,1)*scaley
            vuz(j,1) = vuz(j,1)*scalez

!-----------------------------------------------------------------------
!     /*   massive nhc end                                            */
!-----------------------------------------------------------------------

         end do
         end do

!-----------------------------------------------------------------------
!        /*   massive nhc start                                       */
!-----------------------------------------------------------------------

         do l = ncolor, 1, -1
         do j = natom, 1, -1

!           /*   calculate total kinetic energy of the system   */

            dkinx = fictmass(j,1)*vux(j,1)*vux(j,1)
            dkiny = fictmass(j,1)*vuy(j,1)*vuy(j,1)
            dkinz = fictmass(j,1)*vuz(j,1)*vuz(j,1)

            scalex = 1.d0
            scaley = 1.d0
            scalez = 1.d0

            fxbath_cent(j,1,l) = (dkinx - gkt)/qmass_cent(1,l)
            fybath_cent(j,1,l) = (dkiny - gkt)/qmass_cent(1,l)
            fzbath_cent(j,1,l) = (dkinz - gkt)/qmass_cent(1,l)

            do i = 2, nnhc
              fxbath_cent(j,i,l) = &
     &           (qmass_cent(i-1,l)*vxbath_cent(j,i-1,l) &
     &           *vxbath_cent(j,i-1,l) - gkt)/qmass_cent(i,l)
              fybath_cent(j,i,l) = &
     &           (qmass_cent(i-1,l)*vybath_cent(j,i-1,l) &
     &           *vybath_cent(j,i-1,l) - gkt)/qmass_cent(i,l)
              fzbath_cent(j,i,l) = &
     &           (qmass_cent(i-1,l)*vzbath_cent(j,i-1,l) &
     &           *vzbath_cent(j,i-1,l) - gkt)/qmass_cent(i,l)
            end do

!           /*   update the thermostat velocities   */

            vxbath_cent(j,nnhc,l) = vxbath_cent(j,nnhc,l) &
     &         + 0.25d0*fxbath_cent(j,nnhc,l)*dt_ys
            vybath_cent(j,nnhc,l) = vybath_cent(j,nnhc,l) &
     &         + 0.25d0*fybath_cent(j,nnhc,l)*dt_ys
            vzbath_cent(j,nnhc,l) = vzbath_cent(j,nnhc,l) &
     &         + 0.25d0*fzbath_cent(j,nnhc,l)*dt_ys

            do inhc = 1, nnhc-1

               vxfact &
     &            = exp(-0.125d0*vxbath_cent(j,nnhc-inhc+1,l)*dt_ys)
               vyfact &
     &            = exp(-0.125d0*vybath_cent(j,nnhc-inhc+1,l)*dt_ys)
               vzfact &
     &            = exp(-0.125d0*vzbath_cent(j,nnhc-inhc+1,l)*dt_ys)

               vxbath_cent(j,nnhc-inhc,l) = &
     &            vxbath_cent(j,nnhc-inhc,l)*vxfact*vxfact &
     &            + 0.25d0*fxbath_cent(j,nnhc-inhc,l)*vxfact*dt_ys
               vybath_cent(j,nnhc-inhc,l) = &
     &            vybath_cent(j,nnhc-inhc,l)*vyfact*vyfact &
     &            + 0.25d0*fybath_cent(j,nnhc-inhc,l)*vyfact*dt_ys
               vzbath_cent(j,nnhc-inhc,l) = &
     &            vzbath_cent(j,nnhc-inhc,l)*vzfact*vzfact &
     &            + 0.25d0*fzbath_cent(j,nnhc-inhc,l)*vzfact*dt_ys

            end do

!           /*   update the particle velocities   */

            pvxfact = exp(-0.5d0*vxbath_cent(j,1,l)*dt_ys)
            pvyfact = exp(-0.5d0*vybath_cent(j,1,l)*dt_ys)
            pvzfact = exp(-0.5d0*vzbath_cent(j,1,l)*dt_ys)

            scalex = scalex*pvxfact
            scaley = scaley*pvyfact
            scalez = scalez*pvzfact

!           /*   update the force   */

            fxbath_cent(j,1,l) &
     &         = (scalex*scalex*dkinx - gkt)/qmass_cent(1,l)
            fybath_cent(j,1,l) &
     &         = (scaley*scaley*dkiny - gkt)/qmass_cent(1,l)
            fzbath_cent(j,1,l) &
     &         = (scalez*scalez*dkinz - gkt)/qmass_cent(1,l)

!           /*   update the thermostat positions   */
            do inhc = 1, nnhc
               xbath_cent(j,inhc,l) = xbath_cent(j,inhc,l) &
     &            + 0.5d0*vxbath_cent(j,inhc,l)*dt_ys
               ybath_cent(j,inhc,l) = ybath_cent(j,inhc,l) &
     &            + 0.5d0*vybath_cent(j,inhc,l)*dt_ys
               zbath_cent(j,inhc,l) = zbath_cent(j,inhc,l) &
     &            + 0.5d0*vzbath_cent(j,inhc,l)*dt_ys
            end do

!          /*   update the thermostat velocities   */

            do inhc = 1, nnhc-1

               vxfact = exp(-0.125d0*vxbath_cent(j,inhc+1,l)*dt_ys)
               vyfact = exp(-0.125d0*vybath_cent(j,inhc+1,l)*dt_ys)
               vzfact = exp(-0.125d0*vzbath_cent(j,inhc+1,l)*dt_ys)

               vxbath_cent(j,inhc,l) = &
     &            vxbath_cent(j,inhc,l)*vxfact*vxfact &
     &            + 0.25d0*fxbath_cent(j,inhc,l)*vxfact*dt_ys
               vybath_cent(j,inhc,l) = &
     &            vybath_cent(j,inhc,l)*vyfact*vyfact &
     &            + 0.25d0*fybath_cent(j,inhc,l)*vyfact*dt_ys
               vzbath_cent(j,inhc,l) = &
     &            vzbath_cent(j,inhc,l)*vzfact*vzfact &
     &            + 0.25d0*fzbath_cent(j,inhc,l)*vzfact*dt_ys

               fxbath_cent(j,inhc+1,l) = &
     &            (qmass_cent(inhc,l)*vxbath_cent(j,inhc,l) &
     &            *vxbath_cent(j,inhc,l) - gkt)/qmass_cent(inhc+1,l)
               fybath_cent(j,inhc+1,l) = &
     &            (qmass_cent(inhc,l)*vybath_cent(j,inhc,l) &
     &            *vybath_cent(j,inhc,l) - gkt)/qmass_cent(inhc+1,l)
               fzbath_cent(j,inhc+1,l) = &
     &            (qmass_cent(inhc,l)*vzbath_cent(j,inhc,l) &
     &            *vzbath_cent(j,inhc,l) - gkt)/qmass_cent(inhc+1,l)

            end do

            vxbath_cent(j,nnhc,l) = vxbath_cent(j,nnhc,l) &
     &         + 0.25d0*fxbath_cent(j,nnhc,l)*dt_ys
            vybath_cent(j,nnhc,l) = vybath_cent(j,nnhc,l) &
     &         + 0.25d0*fybath_cent(j,nnhc,l)*dt_ys
            vzbath_cent(j,nnhc,l) = vzbath_cent(j,nnhc,l) &
     &         + 0.25d0*fzbath_cent(j,nnhc,l)*dt_ys

!           /*   update the paricle velocities   */

            vux(j,1) = vux(j,1)*scalex
            vuy(j,1) = vuy(j,1)*scaley
            vuz(j,1) = vuz(j,1)*scalez

!-----------------------------------------------------------------------
!     /*   massive nhc end                                            */
!-----------------------------------------------------------------------

         end do
         end do

!-----------------------------------------------------------------------
!     /*   main loop end                                              */
!-----------------------------------------------------------------------

      end do

      return
      end

