!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     update thermostats attached to box
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine update_mnhc_box_c2
!***********************************************************************
!=======================================================================
!
!     Update Nose-Hoover chain thermostat attached to box.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   boxmass, vlog, ysweight, fbath_box, qmass_box, dt, volume, &
     &   boxdot, vbath_box, rbath_box, gkt, nys, nnhc, ncolor

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, iys, k

      real(8) :: dt_ys, dkin, scale, vfact, pvfact

!-----------------------------------------------------------------------
!     /*   calculate total kinetic energy of the system               */
!-----------------------------------------------------------------------

      dkin = boxmass(1,1)*vlog*vlog

!-----------------------------------------------------------------------
!     /*   velocity scaling factor                                    */
!-----------------------------------------------------------------------

      scale = 1.d0

!-----------------------------------------------------------------------
!     /*   main loop start                                            */
!-----------------------------------------------------------------------

      do iys = 1, nys

!-----------------------------------------------------------------------
!     /*   step size                                                  */
!-----------------------------------------------------------------------

      dt_ys = dt*ysweight(iys) /2.d0

!-----------------------------------------------------------------------
!     /*   massive nhc start                                          */
!-----------------------------------------------------------------------

      do k = 1, ncolor

!-----------------------------------------------------------------------
!        /*   update the force                                        */
!-----------------------------------------------------------------------

         fbath_box(1,k) = (dkin - gkt)/qmass_box(1,k)

         do i = 2, nnhc

            fbath_box(i,k) = &
     &         (qmass_box(i-1,k)*vbath_box(i-1,k) &
     &         *vbath_box(i-1,k) - gkt)/qmass_box(i,k)

         end do

!-----------------------------------------------------------------------
!        /*   update the thermostat velocities                        */
!-----------------------------------------------------------------------

         vbath_box(nnhc,k) = vbath_box(nnhc,k) &
     &      + 0.25d0*fbath_box(nnhc,k)*dt_ys

         do i = 1, nnhc-1

            vfact = exp(-0.125d0*vbath_box(nnhc-i+1,k)*dt_ys)

            vbath_box(nnhc-i,k) = vbath_box(nnhc-i,k)*vfact*vfact &
     &         + 0.25d0*fbath_box(nnhc-i,k)*vfact*dt_ys

         end do

!-----------------------------------------------------------------------
!        /*   update the particle velocities                          */
!-----------------------------------------------------------------------

         pvfact = exp(-0.5d0*vbath_box(1,k)*dt_ys)

         scale = scale*pvfact

!-----------------------------------------------------------------------
!        /*   calculate kinetic energy                                */
!-----------------------------------------------------------------------

         dkin = dkin*pvfact*pvfact

!-----------------------------------------------------------------------
!        /*   update the force                                        */
!-----------------------------------------------------------------------

         fbath_box(1,k) = (dkin - gkt)/qmass_box(1,k)

!-----------------------------------------------------------------------
!        /*   update the thermostat position                          */
!-----------------------------------------------------------------------

         do i = 1, nnhc

            rbath_box(i,k) = rbath_box(i,k) &
     &                      + 0.5d0*vbath_box(i,k)*dt_ys

         end do

!-----------------------------------------------------------------------
!        /*   update the thermostat velocities                        */
!-----------------------------------------------------------------------

         do i = 1, nnhc-1

            vfact = exp(-0.125d0*vbath_box(i+1,k)*dt_ys)

            vbath_box(i,k) = vbath_box(i,k)*vfact*vfact &
     &         + 0.25d0*fbath_box(i,k)*vfact*dt_ys

            fbath_box(i+1,k) = (qmass_box(i,k)*vbath_box(i,k) &
     &         *vbath_box(i,k) - gkt)/qmass_box(i+1,k)

         end do

         vbath_box(nnhc,k) &
     &      = vbath_box(nnhc,k) + 0.25d0*fbath_box(nnhc,k)*dt_ys

!-----------------------------------------------------------------------
!     /*   massive nhc end                                            */
!-----------------------------------------------------------------------

      end do

!-----------------------------------------------------------------------
!     /*   massive nhc start                                          */
!-----------------------------------------------------------------------

      do k = ncolor, 1, -1

!-----------------------------------------------------------------------
!        /*   update the force                                        */
!-----------------------------------------------------------------------

         fbath_box(1,k) = (dkin - gkt)/qmass_box(1,k)

         do i = 2, nnhc

            fbath_box(i,k) = &
     &         (qmass_box(i-1,k)*vbath_box(i-1,k) &
     &         *vbath_box(i-1,k) - gkt)/qmass_box(i,k)

         end do

!-----------------------------------------------------------------------
!        /*   update the thermostat velocities                        */
!-----------------------------------------------------------------------

         vbath_box(nnhc,k) = vbath_box(nnhc,k) &
     &      + 0.25d0*fbath_box(nnhc,k)*dt_ys

         do i = 1, nnhc-1

            vfact = exp(-0.125d0*vbath_box(nnhc-i+1,k)*dt_ys)

            vbath_box(nnhc-i,k) = vbath_box(nnhc-i,k)*vfact*vfact &
     &         + 0.25d0*fbath_box(nnhc-i,k)*vfact*dt_ys

         end do

!-----------------------------------------------------------------------
!        /*   update the particle velocities                          */
!-----------------------------------------------------------------------

         pvfact = exp(-0.5d0*vbath_box(1,k)*dt_ys)

         scale = scale*pvfact

!-----------------------------------------------------------------------
!        /*   calculate kinetic energy                                */
!-----------------------------------------------------------------------

         dkin = dkin*pvfact*pvfact

!-----------------------------------------------------------------------
!        /*   update the force                                        */
!-----------------------------------------------------------------------

         fbath_box(1,k) = (dkin - gkt)/qmass_box(1,k)

!-----------------------------------------------------------------------
!        /*   update the thermostat position                          */
!-----------------------------------------------------------------------

         do i = 1, nnhc

            rbath_box(i,k) = rbath_box(i,k) &
     &                      + 0.5d0*vbath_box(i,k)*dt_ys

         end do

!-----------------------------------------------------------------------
!        /*   update the thermostat velocities                        */
!-----------------------------------------------------------------------

         do i = 1, nnhc-1

            vfact = exp(-0.125d0*vbath_box(i+1,k)*dt_ys)

            vbath_box(i,k) = vbath_box(i,k)*vfact*vfact &
     &         + 0.25d0*fbath_box(i,k)*vfact*dt_ys

            fbath_box(i+1,k) = (qmass_box(i,k)*vbath_box(i,k) &
     &         *vbath_box(i,k) - gkt)/qmass_box(i+1,k)

         end do

         vbath_box(nnhc,k) &
     &      = vbath_box(nnhc,k) + 0.25d0*fbath_box(nnhc,k)*dt_ys

!-----------------------------------------------------------------------
!     /*   massive nhc end                                            */
!-----------------------------------------------------------------------

      end do

!-----------------------------------------------------------------------
!     /*   main loop end                                              */
!-----------------------------------------------------------------------

      end do

!-----------------------------------------------------------------------
!     /*   update the paricle velocities                              */
!-----------------------------------------------------------------------

      vlog = vlog*scale

      boxdot(1,1) = vlog * volume**(1.d0/3.d0)
      boxdot(2,2) = vlog * volume**(1.d0/3.d0)
      boxdot(3,3) = vlog * volume**(1.d0/3.d0)

      return
      end





!***********************************************************************
      subroutine update_mnhc_box_pp
!***********************************************************************
!=======================================================================
!
!     Update Nose-Hoover chain thermostat attached to box.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   boxmass, vbox, ysweight, qmass_box, dt, boxdot, box, &
     &   fxbath_box, fybath_box, fzbath_box, gkt, vxbath_box, &
     &   vybath_box, vzbath_box, xbath_box, ybath_box, zbath_box, &
     &   nys, nnhc, ncolor

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, iys, inhc, j, k, l

      real(8) :: dt_ys, dkinx, dkiny, dkinz, scalex, scaley, scalez, &
     &           vxfact, vyfact, vzfact, pvxfact, pvyfact, pvzfact

!-----------------------------------------------------------------------
!     /*   main loop start                                            */
!-----------------------------------------------------------------------

      do iys = 1, nys

!-----------------------------------------------------------------------
!     /*   step size                                                  */
!-----------------------------------------------------------------------

      dt_ys = dt*ysweight(iys) /2.d0

!-----------------------------------------------------------------------
!        /*   massive nhc start                                       */
!-----------------------------------------------------------------------

         do l = 1, ncolor
         do j = 1, 3

!           /*   calculate total kinetic energy of the system   */

            dkinx = boxmass(j,1)*vbox(j,1)*vbox(j,1)
            dkiny = boxmass(j,2)*vbox(j,2)*vbox(j,2)
            dkinz = boxmass(j,3)*vbox(j,3)*vbox(j,3)

            scalex = 1.d0
            scaley = 1.d0
            scalez = 1.d0

            fxbath_box(j,1,l) = (dkinx - gkt)/qmass_box(1,l)
            fybath_box(j,1,l) = (dkiny - gkt)/qmass_box(1,l)
            fzbath_box(j,1,l) = (dkinz - gkt)/qmass_box(1,l)

            do i = 2, nnhc
              fxbath_box(j,i,l) = &
     &           (qmass_box(i-1,l)*vxbath_box(j,i-1,l) &
     &           *vxbath_box(j,i-1,l) - gkt)/qmass_box(i,l)
              fybath_box(j,i,l) = &
     &           (qmass_box(i-1,l)*vybath_box(j,i-1,l) &
     &           *vybath_box(j,i-1,l) - gkt)/qmass_box(i,l)
              fzbath_box(j,i,l) = &
     &           (qmass_box(i-1,l)*vzbath_box(j,i-1,l) &
     &           *vzbath_box(j,i-1,l) - gkt)/qmass_box(i,l)
            end do

!           /*   update the thermostat velocities   */

            vxbath_box(j,nnhc,l) = vxbath_box(j,nnhc,l) &
     &         + 0.25d0*fxbath_box(j,nnhc,l)*dt_ys
            vybath_box(j,nnhc,l) = vybath_box(j,nnhc,l) &
     &         + 0.25d0*fybath_box(j,nnhc,l)*dt_ys
            vzbath_box(j,nnhc,l) = vzbath_box(j,nnhc,l) &
     &         + 0.25d0*fzbath_box(j,nnhc,l)*dt_ys

            do inhc = 1, nnhc-1

               vxfact &
     &            = exp(-0.125d0*vxbath_box(j,nnhc-inhc+1,l)*dt_ys)
               vyfact &
     &            = exp(-0.125d0*vybath_box(j,nnhc-inhc+1,l)*dt_ys)
               vzfact &
     &            = exp(-0.125d0*vzbath_box(j,nnhc-inhc+1,l)*dt_ys)

               vxbath_box(j,nnhc-inhc,l) = &
     &            vxbath_box(j,nnhc-inhc,l)*vxfact*vxfact &
     &            + 0.25d0*fxbath_box(j,nnhc-inhc,l)*vxfact*dt_ys
               vybath_box(j,nnhc-inhc,l) = &
     &            vybath_box(j,nnhc-inhc,l)*vyfact*vyfact &
     &            + 0.25d0*fybath_box(j,nnhc-inhc,l)*vyfact*dt_ys
               vzbath_box(j,nnhc-inhc,l) = &
     &            vzbath_box(j,nnhc-inhc,l)*vzfact*vzfact &
     &            + 0.25d0*fzbath_box(j,nnhc-inhc,l)*vzfact*dt_ys

            end do

!           /*   update the particle velocities   */

            pvxfact = exp(-0.5d0*vxbath_box(j,1,l)*dt_ys)
            pvyfact = exp(-0.5d0*vybath_box(j,1,l)*dt_ys)
            pvzfact = exp(-0.5d0*vzbath_box(j,1,l)*dt_ys)

            scalex = scalex*pvxfact
            scaley = scaley*pvyfact
            scalez = scalez*pvzfact

!           /*   update the force   */

            fxbath_box(j,1,l) &
     &         = (scalex*scalex*dkinx - gkt)/qmass_box(1,l)
            fybath_box(j,1,l) &
     &         = (scaley*scaley*dkiny - gkt)/qmass_box(1,l)
            fzbath_box(j,1,l) &
     &         = (scalez*scalez*dkinz - gkt)/qmass_box(1,l)

!           /*   update the thermostat positions   */
            do inhc = 1, nnhc
               xbath_box(j,inhc,l) = xbath_box(j,inhc,l) &
     &            + 0.5d0*vxbath_box(j,inhc,l)*dt_ys
               ybath_box(j,inhc,l) = ybath_box(j,inhc,l) &
     &            + 0.5d0*vybath_box(j,inhc,l)*dt_ys
               zbath_box(j,inhc,l) = zbath_box(j,inhc,l) &
     &            + 0.5d0*vzbath_box(j,inhc,l)*dt_ys
            end do

!          /*   update the thermostat velocities   */

            do inhc = 1, nnhc-1

               vxfact = exp(-0.125d0*vxbath_box(j,inhc+1,l)*dt_ys)
               vyfact = exp(-0.125d0*vybath_box(j,inhc+1,l)*dt_ys)
               vzfact = exp(-0.125d0*vzbath_box(j,inhc+1,l)*dt_ys)

               vxbath_box(j,inhc,l) = &
     &            vxbath_box(j,inhc,l)*vxfact*vxfact &
     &            + 0.25d0*fxbath_box(j,inhc,l)*vxfact*dt_ys
               vybath_box(j,inhc,l) = &
     &            vybath_box(j,inhc,l)*vyfact*vyfact &
     &            + 0.25d0*fybath_box(j,inhc,l)*vyfact*dt_ys
               vzbath_box(j,inhc,l) = &
     &            vzbath_box(j,inhc,l)*vzfact*vzfact &
     &            + 0.25d0*fzbath_box(j,inhc,l)*vzfact*dt_ys

               fxbath_box(j,inhc+1,l) = &
     &            (qmass_box(inhc,l)*vxbath_box(j,inhc,l) &
     &            *vxbath_box(j,inhc,l) - gkt)/qmass_box(inhc+1,l)
               fybath_box(j,inhc+1,l) = &
     &            (qmass_box(inhc,l)*vybath_box(j,inhc,l) &
     &            *vybath_box(j,inhc,l) - gkt)/qmass_box(inhc+1,l)
               fzbath_box(j,inhc+1,l) = &
     &            (qmass_box(inhc,l)*vzbath_box(j,inhc,l) &
     &            *vzbath_box(j,inhc,l) - gkt)/qmass_box(inhc+1,l)

            end do

            vxbath_box(j,nnhc,l) = vxbath_box(j,nnhc,l) &
     &         + 0.25d0*fxbath_box(j,nnhc,l)*dt_ys
            vybath_box(j,nnhc,l) = vybath_box(j,nnhc,l) &
     &         + 0.25d0*fybath_box(j,nnhc,l)*dt_ys
            vzbath_box(j,nnhc,l) = vzbath_box(j,nnhc,l) &
     &         + 0.25d0*fzbath_box(j,nnhc,l)*dt_ys

!           /*   update the paricle velocities   */

            vbox(j,1) = vbox(j,1)*scalex
            vbox(j,2) = vbox(j,2)*scaley
            vbox(j,3) = vbox(j,3)*scalez

!-----------------------------------------------------------------------
!     /*   massive nhc end                                            */
!-----------------------------------------------------------------------

         end do
         end do

!-----------------------------------------------------------------------
!        /*   massive nhc start                                       */
!-----------------------------------------------------------------------

         do l = ncolor, 1, -1
         do j = 3, 1, -1

!           /*   calculate total kinetic energy of the system   */

            dkinx = boxmass(j,1)*vbox(j,1)*vbox(j,1)
            dkiny = boxmass(j,2)*vbox(j,2)*vbox(j,2)
            dkinz = boxmass(j,3)*vbox(j,3)*vbox(j,3)

            scalex = 1.d0
            scaley = 1.d0
            scalez = 1.d0

            fxbath_box(j,1,l) = (dkinx - gkt)/qmass_box(1,l)
            fybath_box(j,1,l) = (dkiny - gkt)/qmass_box(1,l)
            fzbath_box(j,1,l) = (dkinz - gkt)/qmass_box(1,l)

            do i = 2, nnhc
              fxbath_box(j,i,l) = &
     &           (qmass_box(i-1,l)*vxbath_box(j,i-1,l) &
     &           *vxbath_box(j,i-1,l) - gkt)/qmass_box(i,l)
              fybath_box(j,i,l) = &
     &           (qmass_box(i-1,l)*vybath_box(j,i-1,l) &
     &           *vybath_box(j,i-1,l) - gkt)/qmass_box(i,l)
              fzbath_box(j,i,l) = &
     &           (qmass_box(i-1,l)*vzbath_box(j,i-1,l) &
     &           *vzbath_box(j,i-1,l) - gkt)/qmass_box(i,l)
            end do

!           /*   update the thermostat velocities   */

            vxbath_box(j,nnhc,l) = vxbath_box(j,nnhc,l) &
     &         + 0.25d0*fxbath_box(j,nnhc,l)*dt_ys
            vybath_box(j,nnhc,l) = vybath_box(j,nnhc,l) &
     &         + 0.25d0*fybath_box(j,nnhc,l)*dt_ys
            vzbath_box(j,nnhc,l) = vzbath_box(j,nnhc,l) &
     &         + 0.25d0*fzbath_box(j,nnhc,l)*dt_ys

            do inhc = 1, nnhc-1

               vxfact &
     &            = exp(-0.125d0*vxbath_box(j,nnhc-inhc+1,l)*dt_ys)
               vyfact &
     &            = exp(-0.125d0*vybath_box(j,nnhc-inhc+1,l)*dt_ys)
               vzfact &
     &            = exp(-0.125d0*vzbath_box(j,nnhc-inhc+1,l)*dt_ys)

               vxbath_box(j,nnhc-inhc,l) = &
     &            vxbath_box(j,nnhc-inhc,l)*vxfact*vxfact &
     &            + 0.25d0*fxbath_box(j,nnhc-inhc,l)*vxfact*dt_ys
               vybath_box(j,nnhc-inhc,l) = &
     &            vybath_box(j,nnhc-inhc,l)*vyfact*vyfact &
     &            + 0.25d0*fybath_box(j,nnhc-inhc,l)*vyfact*dt_ys
               vzbath_box(j,nnhc-inhc,l) = &
     &            vzbath_box(j,nnhc-inhc,l)*vzfact*vzfact &
     &            + 0.25d0*fzbath_box(j,nnhc-inhc,l)*vzfact*dt_ys

            end do

!           /*   update the particle velocities   */

            pvxfact = exp(-0.5d0*vxbath_box(j,1,l)*dt_ys)
            pvyfact = exp(-0.5d0*vybath_box(j,1,l)*dt_ys)
            pvzfact = exp(-0.5d0*vzbath_box(j,1,l)*dt_ys)

            scalex = scalex*pvxfact
            scaley = scaley*pvyfact
            scalez = scalez*pvzfact

!           /*   update the force   */

            fxbath_box(j,1,l) &
     &         = (scalex*scalex*dkinx - gkt)/qmass_box(1,l)
            fybath_box(j,1,l) &
     &         = (scaley*scaley*dkiny - gkt)/qmass_box(1,l)
            fzbath_box(j,1,l) &
     &         = (scalez*scalez*dkinz - gkt)/qmass_box(1,l)

!           /*   update the thermostat positions   */
            do inhc = 1, nnhc
               xbath_box(j,inhc,l) = xbath_box(j,inhc,l) &
     &            + 0.5d0*vxbath_box(j,inhc,l)*dt_ys
               ybath_box(j,inhc,l) = ybath_box(j,inhc,l) &
     &            + 0.5d0*vybath_box(j,inhc,l)*dt_ys
               zbath_box(j,inhc,l) = zbath_box(j,inhc,l) &
     &            + 0.5d0*vzbath_box(j,inhc,l)*dt_ys
            end do

!          /*   update the thermostat velocities   */

            do inhc = 1, nnhc-1

               vxfact = exp(-0.125d0*vxbath_box(j,inhc+1,l)*dt_ys)
               vyfact = exp(-0.125d0*vybath_box(j,inhc+1,l)*dt_ys)
               vzfact = exp(-0.125d0*vzbath_box(j,inhc+1,l)*dt_ys)

               vxbath_box(j,inhc,l) = &
     &            vxbath_box(j,inhc,l)*vxfact*vxfact &
     &            + 0.25d0*fxbath_box(j,inhc,l)*vxfact*dt_ys
               vybath_box(j,inhc,l) = &
     &            vybath_box(j,inhc,l)*vyfact*vyfact &
     &            + 0.25d0*fybath_box(j,inhc,l)*vyfact*dt_ys
               vzbath_box(j,inhc,l) = &
     &            vzbath_box(j,inhc,l)*vzfact*vzfact &
     &            + 0.25d0*fzbath_box(j,inhc,l)*vzfact*dt_ys

               fxbath_box(j,inhc+1,l) = &
     &            (qmass_box(inhc,l)*vxbath_box(j,inhc,l) &
     &            *vxbath_box(j,inhc,l) - gkt)/qmass_box(inhc+1,l)
               fybath_box(j,inhc+1,l) = &
     &            (qmass_box(inhc,l)*vybath_box(j,inhc,l) &
     &            *vybath_box(j,inhc,l) - gkt)/qmass_box(inhc+1,l)
               fzbath_box(j,inhc+1,l) = &
     &            (qmass_box(inhc,l)*vzbath_box(j,inhc,l) &
     &            *vzbath_box(j,inhc,l) - gkt)/qmass_box(inhc+1,l)

            end do

            vxbath_box(j,nnhc,l) = vxbath_box(j,nnhc,l) &
     &         + 0.25d0*fxbath_box(j,nnhc,l)*dt_ys
            vybath_box(j,nnhc,l) = vybath_box(j,nnhc,l) &
     &         + 0.25d0*fybath_box(j,nnhc,l)*dt_ys
            vzbath_box(j,nnhc,l) = vzbath_box(j,nnhc,l) &
     &         + 0.25d0*fzbath_box(j,nnhc,l)*dt_ys

!           /*   update the paricle velocities   */

            vbox(j,1) = vbox(j,1)*scalex
            vbox(j,2) = vbox(j,2)*scaley
            vbox(j,3) = vbox(j,3)*scalez

!-----------------------------------------------------------------------
!     /*   massive nhc end                                            */
!-----------------------------------------------------------------------

         end do
         end do

!-----------------------------------------------------------------------
!     /*   main loop end                                              */
!-----------------------------------------------------------------------

      end do

!-----------------------------------------------------------------------
!     /*   symmetrize box velocity                                    */
!-----------------------------------------------------------------------

      do j = 1, 3-1
      do i = j+1, 3
         vbox(i,j) = 0.5d0 * ( vbox(i,j) + vbox(j,i) )
         vbox(j,i) = vbox(i,j)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   symmetrize thermostat velocity                             */
!-----------------------------------------------------------------------

      do l = 1, ncolor
      do i = 1, nnhc
         vxbath_box(2,i,l) &
     &      = 0.5d0 * ( vxbath_box(2,i,l) + vybath_box(1,i,l) )
         vybath_box(1,i,l) = vxbath_box(2,i,l)
         vybath_box(3,i,l) &
     &      = 0.5d0 * ( vybath_box(3,i,l) + vzbath_box(2,i,l) )
         vzbath_box(2,i,l) = vybath_box(3,i,l)
         vzbath_box(1,i,l) &
     &      = 0.5d0 * ( vzbath_box(1,i,l) + vxbath_box(3,i,l) )
         vxbath_box(3,i,l) = vzbath_box(1,i,l)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   box velocity                                               */
!-----------------------------------------------------------------------

      boxdot(:,:) = 0.d0

      do k = 1, 3
      do j = 1, 3
      do i = 1, 3
         boxdot(i,j) = boxdot(i,j) + vbox(i,k)*box(k,j)
      end do
      end do
      end do

      return
      end
