!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Ded 30, 2022 by M. Shiga
!      Description:     thermostatted ring polymer molecular dynamics
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine trpmdcycle_MPI
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use cons_variables, only : ncons

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   without constraints                                        */
!-----------------------------------------------------------------------

      if ( ncons .eq. 0 ) then

         call trpmdcycle_0_MPI

      else

         call trpmdcycle_cons_MPI

      end if

      return
      end





!***********************************************************************
      subroutine trpmdcycle_0_MPI
!***********************************************************************
!=======================================================================
!
!     The RPMD molecular dynamics cycle.
!
!=======================================================================

!     /*   shared variables   */
      use common_variables, only : &
     &   istep, istep_start, istep_end, nstep, iexit

!     /*   local variables   */
      implicit none

!     /*   initialize step   */
      istep = istep_start
      istep_end = istep

!     /*   normal mode position -> Cartesian position   */
      call nm_trans_MPI(0)

!     /*   get interatomic forces   */
      call getforce_MPI

!     /*   Cartesian force -> normal mode force   */
      call nm_trans_force_MPI( 1 )

!     /*   get harmonic force  */
      call getforce_ref

!     /*   calculate the hamiltonian and temperature   */
      call standard_trpmd_0_MPI

!     /*   do some analysis   */
      call analysis_MPI ( 1 )

      do istep = istep_start+1, nstep

!        /*   current step   */
         istep_end = istep

!        /*   update the velocities by interatomic forces   */
         call update_posvel_trpmd_MPI( 1 )

!        /*   normal mode position -> Cartesian position   */
         call nm_trans_MPI( 0 )

!        /*   get interatomic forces   */
         call getforce_MPI

!        /*   Cartesian force -> normal mode force   */
         call nm_trans_force_MPI( 1 )

!        /*   get harmonic force  */
         call getforce_ref

!        /*   update the velocities by interatomic forces   */
         call update_posvel_trpmd_MPI( 2 )

!        /*   normal mode velocity -> Cartesian velocity   */
         call nm_trans_velocity_MPI( 0 )

!        /*   calculate the hamiltonian and temperature   */
         call standard_trpmd_0_MPI

!        /*   do some analysis   */
         call analysis_MPI ( 2 )

!        /*   output restart   */
         call backup_trpmd_MPI

!        /*   exit if `exit.dat' exists   */
         call softexit_MPI
         if ( iexit .eq. 1 ) exit

      end do

!     /*   current step   */
      istep = istep_end

      return
      end





!***********************************************************************
      subroutine trpmdcycle_cons_MPI
!***********************************************************************
!=======================================================================
!
!     The RPMD molecular dynamics cycle.
!
!=======================================================================

!     /*   shared variables   */
      use common_variables, only : &
     &   istep, istep_start, istep_end, nstep, iexit

!     /*   local variables   */
      implicit none

!     /*   initialize step   */
      istep = istep_start
      istep_end = istep

!     /*   normal mode position -> Cartesian position   */
      call nm_trans_MPI(0)

!     /*   get interatomic forces   */
      call getforce_MPI

!     /*   Cartesian force -> normal mode force   */
      call nm_trans_force_MPI( 1 )

!     /*   get harmonic force  */
      call getforce_ref

!     /*   get constrained force  */
      call getforce_ref_cons_cent_MPI

!     /*   calculate the hamiltonian and temperature   */
      call standard_trpmd_cons_MPI

!     /*   do some analysis   */
      call analysis_MPI ( 1 )

      do istep = istep_start+1, nstep

!        /*   current step   */
         istep_end = istep

!        /*   update the velocities by interatomic forces   */
         call update_posvel_trpmd_MPI( 1 )

!        /*   normal mode position -> Cartesian position   */
         call nm_trans_MPI( 0 )

!        /*   get interatomic forces   */
         call getforce_MPI

!        /*   Cartesian force -> normal mode force   */
         call nm_trans_force_MPI( 1 )

!        /*   get harmonic force  */
         call getforce_ref

!        /*   get constrained force  */
         call getforce_ref_cons_cent_MPI

!        /*   update the velocities by interatomic forces   */
         call update_posvel_trpmd_MPI( 2 )

!        /*   normal mode velocity -> Cartesian velocity   */
         call nm_trans_velocity_MPI( 0 )

!        /*   calculate the hamiltonian and temperature   */
         call standard_trpmd_cons_MPI

!        /*   do some analysis   */
         call analysis_MPI ( 2 )

!        /*   output restart   */
         call backup_trpmd_MPI

!        /*   exit if `exit.dat' exists   */
         call softexit_MPI
         if ( iexit .eq. 1 ) exit

      end do

!     /*   current step   */
      istep = istep_end

      return
      end





!***********************************************************************
      subroutine update_posvel_trpmd_MPI( ioption )
!***********************************************************************

      use common_variables, only :  integrator_trpmd
      implicit none
      integer :: ioption

      if      ( integrator_trpmd(1:11) .eq. 'LEIMKUHLER ' ) then
         call update_posvel_trpmd_leimkuhler_MPI( ioption )
      else if ( integrator_trpmd(1:15) .eq. 'VANDEN-EIJNDEN ' ) then
         call update_posvel_trpmd_vandeneijnden_MPI( ioption )
      end if

      return
      end





!***********************************************************************
      subroutine update_posvel_trpmd_vandeneijnden_MPI( ioption )
!***********************************************************************
!-----------------------------------------------------------------------
!     //    shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   ux, uy, uz, vux, vuy, vuz, fux, fuy, fuz, dt, fictmass, &
     &   fux_ref, fuy_ref, fuz_ref, dnmmass, physmass, beta, omega_p, &
     &   scale_trpmd, natom, nbead, myrank

!-----------------------------------------------------------------------
!     //    local variables
!-----------------------------------------------------------------------

      implicit none

      real(8) :: gx1, gy1, gz1, gx2, gy2, gz2, sigma, sqrt3inv, sqrtdt, &
     &           dt2h, dtdtsqrth, omega_mode, gamma, ax, ay, az, gasdev, &
     &           bx, by, bz

      integer :: i, j, ioption

!-----------------------------------------------------------------------
!     //    first half
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

         if ( myrank .eq. 0 ) then

            sqrt3inv = 1.d0/sqrt(3.d0)
            sqrtdt   = sqrt(dt)

            dt2h      = 0.5d0 * dt * dt
            dtdtsqrth = 0.5d0 * dt * sqrt(dt)

!           /*   centroid velocities and positions   */

            do i = 1, natom
               vux(i,1) = vux(i,1) + 0.5d0*dt*fux(i,1)/fictmass(i,1)
               vuy(i,1) = vuy(i,1) + 0.5d0*dt*fuy(i,1)/fictmass(i,1)
               vuz(i,1) = vuz(i,1) + 0.5d0*dt*fuz(i,1)/fictmass(i,1)
            end do

            do i = 1, natom
               ux(i,1) = ux(i,1) + vux(i,1)*dt
               uy(i,1) = uy(i,1) + vuy(i,1)*dt
               uz(i,1) = uz(i,1) + vuz(i,1)*dt
            end do

!           /*   non-centroid velocities and positions   */

            do j = 2, nbead
            do i = 1, natom

               omega_mode = sqrt(dnmmass(i,j)/physmass(i)) * omega_p

               gamma = scale_trpmd * ( 2.d0 * omega_mode )

               sigma = sqrt((2.d0*gamma)/(beta*fictmass(i,j)))

               gx1 = gasdev()
               gy1 = gasdev()
               gz1 = gasdev()

               gx2 = gasdev()
               gy2 = gasdev()
               gz2 = gasdev()

               bx = (fux(i,j)+fux_ref(i,j)) / fictmass(i,j)
               by = (fuy(i,j)+fuy_ref(i,j)) / fictmass(i,j)
               bz = (fuz(i,j)+fuz_ref(i,j)) / fictmass(i,j)

               ax = dt2h * ( bx - gamma*vux(i,j) ) &
     &            + dtdtsqrth * sigma * ( gx1 + gx2*sqrt3inv )
               ay = dt2h * ( by - gamma*vuy(i,j) ) &
     &            + dtdtsqrth * sigma * ( gy1 + gy2*sqrt3inv )
               az = dt2h * ( bz - gamma*vuz(i,j) ) &
     &            + dtdtsqrth * sigma * ( gz1 + gz2*sqrt3inv )

               ux(i,j) = ux(i,j) + vux(i,j)*dt + ax
               uy(i,j) = uy(i,j) + vuy(i,j)*dt + ay
               uz(i,j) = uz(i,j) + vuz(i,j)*dt + az

               vux(i,j) = ( 1.d0 - dt*gamma ) * vux(i,j) &
     &                  + sigma * sqrtdt * gx1 &
     &                  - gamma * ax &
     &                  + 0.5d0 * dt * bx
               vuy(i,j) = ( 1.d0 - dt*gamma ) * vuy(i,j) &
     &                  + sigma * sqrtdt * gy1 &
     &                  - gamma * ay &
     &                  + 0.5d0 * dt * by
               vuz(i,j) = ( 1.d0 - dt*gamma ) * vuz(i,j) &
     &                  + sigma * sqrtdt * gz1 &
     &                  - gamma * az &
     &                  + 0.5d0 * dt * bz

            end do
            end do

         end if

!        //    communication
         call my_mpi_bcast_real_2 ( vux, natom, nbead )
         call my_mpi_bcast_real_2 ( vuy, natom, nbead )
         call my_mpi_bcast_real_2 ( vuz, natom, nbead )
         call my_mpi_bcast_real_2 ( ux, natom, nbead )
         call my_mpi_bcast_real_2 ( uy, natom, nbead )
         call my_mpi_bcast_real_2 ( uz, natom, nbead )

!-----------------------------------------------------------------------
!     //    second half
!-----------------------------------------------------------------------

      else if ( ioption .eq. 2 ) then

         if ( myrank .eq. 0 ) then

!           /*   centroid and non-centroid velocities  */

            do i = 1, natom
               vux(i,1) = vux(i,1) + 0.5d0*dt*fux(i,1)/fictmass(i,1)
               vuy(i,1) = vuy(i,1) + 0.5d0*dt*fuy(i,1)/fictmass(i,1)
               vuz(i,1) = vuz(i,1) + 0.5d0*dt*fuz(i,1)/fictmass(i,1)
            end do

            do j = 2, nbead
            do i = 1, natom
               bx = (fux(i,j)+fux_ref(i,j)) / fictmass(i,j)
               by = (fuy(i,j)+fuy_ref(i,j)) / fictmass(i,j)
               bz = (fuz(i,j)+fuz_ref(i,j)) / fictmass(i,j)
               vux(i,j) = vux(i,j) + 0.5d0*dt*bx
               vuy(i,j) = vuy(i,j) + 0.5d0*dt*by
               vuz(i,j) = vuz(i,j) + 0.5d0*dt*bz
            end do
            end do

         end if

!        //    communication
         call my_mpi_bcast_real_2 ( vux, natom, nbead )
         call my_mpi_bcast_real_2 ( vuy, natom, nbead )
         call my_mpi_bcast_real_2 ( vuz, natom, nbead )

      end if

      return
      end





!***********************************************************************
      subroutine update_posvel_trpmd_leimkuhler_MPI( ioption )
!***********************************************************************
!-----------------------------------------------------------------------
!     //    shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   ux, uy, uz, vux, vuy, vuz, fux, fuy, fuz, dt, fictmass, &
     &   fux_ref, fuy_ref, fuz_ref, dnmmass, physmass, beta, omega_p, &
     &   hamiltonian_cor, scale_trpmd, natom, nbead, myrank

!-----------------------------------------------------------------------
!     //    local variables
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, ioption

      real(8) :: sigma, omega_mode, gamma, ax, ay, az, gasdev
      real(8) :: dlambda, dth, e1, e2, ekin_cor, factor, gx, gy, gz

!-----------------------------------------------------------------------
!     //    first half
!-----------------------------------------------------------------------

      dth = 0.5d0 * dt

!-----------------------------------------------------------------------
!     //    first half
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

         ekin_cor = 0.d0

         if ( myrank .eq. 0 ) then

!           /*   centroid velocities and positions   */

            do i = 1, natom
               ax = fux(i,1) / fictmass(i,1)
               ay = fuy(i,1) / fictmass(i,1)
               az = fuz(i,1) / fictmass(i,1)
               vux(i,1) = vux(i,1) + ax * dth
               vuy(i,1) = vuy(i,1) + ay * dth
               vuz(i,1) = vuz(i,1) + az * dth
            end do

            do i = 1, natom
               ux(i,1) = ux(i,1) + vux(i,1) * dt
               uy(i,1) = uy(i,1) + vuy(i,1) * dt
               uz(i,1) = uz(i,1) + vuz(i,1) * dt
            end do

!           /*   non-centroid velocities and positions   */

            do j = 2, nbead
            do i = 1, natom

!              //   acceleration by force
               ax = ( fux(i,j) + fux_ref(i,j) ) / fictmass(i,j)
               ay = ( fuy(i,j) + fuy_ref(i,j) ) / fictmass(i,j)
               az = ( fuz(i,j) + fuz_ref(i,j) ) / fictmass(i,j)

!              //   update velocity by half step 
               vux(i,j) = vux(i,j) + ax * dth
               vuy(i,j) = vuy(i,j) + ay * dth
               vuz(i,j) = vuz(i,j) + az * dth

!              //   update position by half step
               ux(i,j) = ux(i,j) + vux(i,j) * dth
               uy(i,j) = uy(i,j) + vuy(i,j) * dth
               uz(i,j) = uz(i,j) + vuz(i,j) * dth

!              //   lambda value
               dlambda = dnmmass(i,j) / physmass(i)

!              //   omega value
               omega_mode = sqrt(dlambda) * omega_p

!              //   gamma value
               gamma = scale_trpmd * ( 2.d0 * omega_mode )

!              //   exponent
               e1 = exp( - gamma * dt )

!              //   exponent squared
               e2 = e1 * e1

!              //   mean square deviation of velocities
               sigma = sqrt( (2.d0*gamma) / (beta*fictmass(i,j)) )

!              //   constant
               factor = sigma * sqrt( (1.d0-e2) / (2.d0*gamma) )

!              //   normal distibution
               gx = gasdev()
               gy = gasdev()
               gz = gasdev()

!              //   kinetic energy correction
               ekin_cor = ekin_cor - 0.5d0*fictmass(i,j)*vux(i,j)*vux(i,j)
               ekin_cor = ekin_cor - 0.5d0*fictmass(i,j)*vuy(i,j)*vuy(i,j)
               ekin_cor = ekin_cor - 0.5d0*fictmass(i,j)*vuz(i,j)*vuz(i,j)

!              //   update velocity by friction and random force
               vux(i,j) = vux(i,j)*e1 + factor*gx
               vuy(i,j) = vuy(i,j)*e1 + factor*gy
               vuz(i,j) = vuz(i,j)*e1 + factor*gz

!              //   kinetic energy correction
               ekin_cor = ekin_cor + 0.5d0*fictmass(i,j)*vux(i,j)*vux(i,j)
               ekin_cor = ekin_cor + 0.5d0*fictmass(i,j)*vuy(i,j)*vuy(i,j)
               ekin_cor = ekin_cor + 0.5d0*fictmass(i,j)*vuz(i,j)*vuz(i,j)

!              //   update postion by half step
               ux(i,j) = ux(i,j) + vux(i,j) * dth
               uy(i,j) = uy(i,j) + vuy(i,j) * dth
               uz(i,j) = uz(i,j) + vuz(i,j) * dth

            end do
            end do

         end if

!        //    communication
         call my_mpi_bcast_real_2 ( vux, natom, nbead )
         call my_mpi_bcast_real_2 ( vuy, natom, nbead )
         call my_mpi_bcast_real_2 ( vuz, natom, nbead )
         call my_mpi_bcast_real_2 ( ux, natom, nbead )
         call my_mpi_bcast_real_2 ( uy, natom, nbead )
         call my_mpi_bcast_real_2 ( uz, natom, nbead )
         call my_mpi_bcast_real_0 ( ekin_cor )

         hamiltonian_cor = hamiltonian_cor - ekin_cor

!-----------------------------------------------------------------------
!     //    second half
!-----------------------------------------------------------------------

      else if ( ioption .eq. 2 ) then

         if ( myrank .eq. 0 ) then

!-----------------------------------------------------------------------
!           /*   centroid velocities   */
!-----------------------------------------------------------------------

            do i = 1, natom
               ax = fux(i,1) / fictmass(i,1)
               ay = fuy(i,1) / fictmass(i,1)
               az = fuz(i,1) / fictmass(i,1)
               vux(i,1) = vux(i,1) + ax * dth
               vuy(i,1) = vuy(i,1) + ay * dth
               vuz(i,1) = vuz(i,1) + az * dth
            end do

!-----------------------------------------------------------------------
!           /*   non-centroid velocities   */
!-----------------------------------------------------------------------

!           //   loop of modes and atoms
            do j = 2, nbead
            do i = 1, natom

!              //   acceleration by force
               ax = ( fux(i,j) + fux_ref(i,j) ) / fictmass(i,j)
               ay = ( fuy(i,j) + fuy_ref(i,j) ) / fictmass(i,j)
               az = ( fuz(i,j) + fuz_ref(i,j) ) / fictmass(i,j)

!              //   update velocity by half step 
               vux(i,j) = vux(i,j) + ax * dth
               vuy(i,j) = vuy(i,j) + ay * dth
               vuz(i,j) = vuz(i,j) + az * dth

!           //   loop of modes and atoms
            end do
            end do

         end if

!        //    communication
         call my_mpi_bcast_real_2 ( vux, natom, nbead )
         call my_mpi_bcast_real_2 ( vuy, natom, nbead )
         call my_mpi_bcast_real_2 ( vuz, natom, nbead )

      end if

      return
      end





!***********************************************************************
      subroutine standard_trpmd_0_MPI
!***********************************************************************
!=======================================================================
!
!     calculate ``Hamiltonian'' and ``temperature''
!     no thermostat.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   temp, ekin, boltz, qkin, ekin, potential, hamiltonian, dnmmass, &
     &   hamiltonian_sys, omega_p2, ux, uy, uz, ebath_cent, ebath_mode, &
     &   hamiltonian_cor, ndof, iprint_std, nbead, natom

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: imode, iatom

      real(8) :: factqk

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      call standard_init_MPI( iset )

      if ( iprint_std .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   temp  =  instantaneous temperature                         */
!-----------------------------------------------------------------------

!     /*   calculate ekin =  fictitious kinetic energy   */
      call kinetic_energy

      temp = 2.d0*ekin/dble(ndof)/boltz

!-----------------------------------------------------------------------
!     /*   qkin  =  harmonic potential                                */
!-----------------------------------------------------------------------

      qkin = 0.d0

      do imode = 2, nbead
      do iatom = 1, natom
         factqk = 0.5d0*dnmmass(iatom,imode)*omega_p2
         qkin = qkin &
     &        + factqk*ux(iatom,imode)*ux(iatom,imode) &
     &        + factqk*uy(iatom,imode)*uy(iatom,imode) &
     &        + factqk*uz(iatom,imode)*uz(iatom,imode)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian_sys  =  system Hamiltonian                     */
!-----------------------------------------------------------------------

      hamiltonian_sys = ekin + qkin + potential

!-----------------------------------------------------------------------
!     /*   ebath_cent  =  thermostats attached to centroid            */
!-----------------------------------------------------------------------

      ebath_cent = 0.d0

!-----------------------------------------------------------------------
!     /*   ebath_mode  =  thermostats attached to non-centroid        */
!-----------------------------------------------------------------------

      ebath_mode = 0.d0

!-----------------------------------------------------------------------
!     /*   hamiltonian =  total Hamiltonian                           */
!-----------------------------------------------------------------------

      hamiltonian = hamiltonian_sys + ebath_mode + ebath_cent

!-----------------------------------------------------------------------
!     /*   energy correction                                          */
!-----------------------------------------------------------------------

      hamiltonian = hamiltonian + hamiltonian_cor

!-----------------------------------------------------------------------
!     /*   output                                                     */
!-----------------------------------------------------------------------

      call standard_output_MPI

      return
      end





!***********************************************************************
      subroutine standard_trpmd_cons_MPI
!***********************************************************************
!=======================================================================
!
!     calculate ``Hamiltonian'' and ``temperature''
!     no thermostat.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   temp, ekin, boltz, qkin, ekin, potential, hamiltonian, dnmmass, &
     &   hamiltonian_sys, omega_p2, ux, uy, uz, ebath_cent, ebath_mode, &
     &   hamiltonian_cor, ndof, iprint_std, nbead, natom

      use cons_variables, only : &
     &   pot_ref_cons, potential_ref_cons

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: imode, iatom, i

      real(8) :: factqk

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      call standard_init_MPI( iset )

      if ( iprint_std .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   temp  =  instantaneous temperature                         */
!-----------------------------------------------------------------------

!     /*   calculate ekin =  fictitious kinetic energy   */
      call kinetic_energy

      temp = 2.d0*ekin/dble(ndof)/boltz

!-----------------------------------------------------------------------
!     /*   qkin  =  harmonic potential                                */
!-----------------------------------------------------------------------

      qkin = 0.d0

      do imode = 2, nbead
      do iatom = 1, natom
         factqk = 0.5d0*dnmmass(iatom,imode)*omega_p2
         qkin = qkin &
     &        + factqk*ux(iatom,imode)*ux(iatom,imode) &
     &        + factqk*uy(iatom,imode)*uy(iatom,imode) &
     &        + factqk*uz(iatom,imode)*uz(iatom,imode)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian_sys  =  system Hamiltonian                     */
!-----------------------------------------------------------------------

      hamiltonian_sys = ekin + qkin + potential

!-----------------------------------------------------------------------
!     /*   ebath_cent  =  thermostats attached to centroid            */
!-----------------------------------------------------------------------

      ebath_cent = 0.d0

!-----------------------------------------------------------------------
!     /*   ebath_mode  =  thermostats attached to non-centroid        */
!-----------------------------------------------------------------------

      ebath_mode = 0.d0

!-----------------------------------------------------------------------
!     /*   energy of constraints                                      */
!-----------------------------------------------------------------------

      potential_ref_cons = 0.d0

      do i = 1, nbead
         potential_ref_cons = potential_ref_cons + pot_ref_cons(i)
      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian =  total Hamiltonian                           */
!-----------------------------------------------------------------------

      hamiltonian = hamiltonian_sys + potential_ref_cons &
     &   + ebath_mode + ebath_cent

!-----------------------------------------------------------------------
!     /*   energy correction                                          */
!-----------------------------------------------------------------------

      hamiltonian = hamiltonian + hamiltonian_cor

!-----------------------------------------------------------------------
!     /*   output                                                     */
!-----------------------------------------------------------------------

      call standard_output_MPI

      return
      end





!***********************************************************************
      subroutine backup_trpmd_MPI
!***********************************************************************
!=======================================================================
!
!     finalize the calculation.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   hamiltonian_cor, istep_end, nstep, iexit, iprint_rest, iounit, &
     &   myrank

      implicit none

!-----------------------------------------------------------------------
!     /*   conditions                                                 */
!-----------------------------------------------------------------------

      if ( istep_end .eq. nstep ) then
         continue
      else if ( iexit .eq. 1 ) then
         continue
      else
         if ( iprint_rest .le. 0 ) then
            return
         else
            if ( mod(istep_end,iprint_rest) .eq. 0 ) then
               continue
            else
               return
            end if
         end if
      end if

!-----------------------------------------------------------------------
!     /*   write out restart file                                     */
!-----------------------------------------------------------------------

      call restart_position_MPI( 3 )
      call restart_velocity_MPI( 3 )

!-----------------------------------------------------------------------
!     /*   in `step.ini', print the step number for restart           */
!-----------------------------------------------------------------------

      if ( myrank .eq. 0 ) then
         open ( iounit, file = 'step.ini' )
         write ( iounit, '(i8)' ) istep_end
         write ( iounit, '(d24.16)' ) hamiltonian_cor
         close( iounit )
      end if

!-----------------------------------------------------------------------
!     /*   save averages                                              */
!-----------------------------------------------------------------------

      call analysis_MPI ( 3 )

!-----------------------------------------------------------------------
!     /*   print final geometry                                       */
!-----------------------------------------------------------------------

      call print_final_xyz_MPI

      return
      end
