!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Dec 2, 2019 by M. Shiga
!      Description:     transition path sampling
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      module tps_variables
!***********************************************************************

!     /*   tps counter   */
      integer :: itrial_tps

!     /*   starting tps counter   */
      integer :: itrial_start_tps

!     /*   current tps counter   */
      integer :: itrial_end_tps

!     /*   number of trial moves   */
      integer :: ntrial_tps

!     /*   kicking temperature   */
      real(8) :: temp_kick_tps

!     /*   number of acceptance   */
      integer :: naccept_tps = 0

!     /*   number of rejection  */
      integer :: nreject_tps = 0

!     /*   acceptance ratio  */
      real(8) :: ratio_tps = 0.d0

!     /*   status   */
      character(len=1), dimension(:), allocatable :: status_tps

!     /*   collective variables   */
      character(len=2), dimension(:), allocatable :: params_tps_meta

!     /*   file number   */
      integer :: iounit_tps = 81

!     /*   number of steps printed in trj.out   */
      integer :: nstep_trj

!***********************************************************************
      end module tps_variables
!***********************************************************************





!***********************************************************************
      subroutine setup_tps
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   physmass, fictmass, tnm, tnminv, ux, uy, uz, vx, vy, vz, &
     &   natom, nbead, ivel_start, ipos_start, iounit

      use tps_variables, only : &
     &   temp_kick_tps, itrial_start_tps, naccept_tps, &
     &   nreject_tps, ntrial_tps, status_tps

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, itest, ierr

!-----------------------------------------------------------------------
!     /*   check error                                                */
!-----------------------------------------------------------------------

      ierr = 0

      if ( mod(nbead,2) .ne. 0 ) then
         write( 6, '(a)' ) 'Error - nbead must be even for TPS.'
         write( 6, '(a)' )
         ierr = 1
      end if

      call error_handling ( ierr, 'subroutine setup_tps', 20 )

!-----------------------------------------------------------------------
!     /*   parameters for Suzuki-Yoshida propagator ( order = nys )   */
!-----------------------------------------------------------------------

      call suzuki_yoshida

!-----------------------------------------------------------------------
!     /*   atomic mass                                                */
!-----------------------------------------------------------------------

      do j = 1, nbead
      do i = 1, natom
         fictmass(i,j) = physmass(i)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   unit matrix (just for convenience)                         */
!-----------------------------------------------------------------------

      do j = 1, nbead
      do i = 1, nbead
         tnm(i,j)    = 0.d0
         tnminv(i,j) = 0.d0
      end do
      end do

      do i = 1, nbead
         tnm(i,i)    = 1.d0
         tnminv(i,i) = 1.d0
      end do

!-----------------------------------------------------------------------
!     /*   atom position                                              */
!-----------------------------------------------------------------------

      if      ( ipos_start .eq. 0 ) then

!        /*   initialize position   */
         call init_position_cart

!        /*   duplicate position   */
         do j = 1, nbead/2
            ux(:,j+nbead/2) = ux(:,j)
            uy(:,j+nbead/2) = uy(:,j)
            uz(:,j+nbead/2) = uz(:,j)
         end do

!        /*   ux --> x   */
         call nm_trans_cart( 2 )

      else if ( ipos_start .eq. 1 ) then

!        /*   restart position   */
         call restart_position( 2 )

      else

!        /*   error handling   */
         call error_handling( 1, 'subroutine setup_tps', 20 )

      end if

!-----------------------------------------------------------------------
!     /*   atom velocity                                              */
!-----------------------------------------------------------------------

      if      ( ivel_start .eq. 0 ) then

!        /*   initialize velocity   */
         call init_velocity_cart

!        /*   duplicate velocity   */
         do j = 1, nbead/2
            i = j + nbead/2
            vx(:,i) = vx(:,j)
            vy(:,i) = vy(:,j)
            vz(:,i) = vz(:,j)
         end do

!        /*   vx --> vux   */
         call nm_trans_velocity_cart( 3 )

      else if ( ivel_start .eq. 1 ) then

!        /*   restart velocity   */
         call restart_velocity( 2 )

      else

!        /*   error handling   */
         call error_handling( 1, 'subroutine setup_tps', 20 )

      end if

!-----------------------------------------------------------------------
!     /*   memory allocations                                         */
!-----------------------------------------------------------------------

      if ( .not. allocated(status_tps) ) then
         allocate( status_tps(nbead) )
      end if

!-----------------------------------------------------------------------
!     /*   temperature for random velocity kick                       */
!-----------------------------------------------------------------------

      call read_real1 ( temp_kick_tps, '<temp_kick_tps>', 15, iounit )

!-----------------------------------------------------------------------
!     /*   number of tps trajectories                                 */
!-----------------------------------------------------------------------

      call read_int1 ( ntrial_tps, '<ntrial_tps>', 12, iounit )

!-----------------------------------------------------------------------
!     /*   counter values for transition path sampling                */
!-----------------------------------------------------------------------

!     /*   error flag   */
      ierr = 0

!     /*   check existence of restart file   */
      call testfile ( 'averages.ini', 12, itest )

!     /*   if exists   */
      if ( itest .eq. 0 ) then

!        /*   read tps counter from restart file   */
         open ( iounit, file = 'averages.ini' )
         read ( iounit, *, iostat=ierr ) itrial_start_tps
         read ( iounit, *, iostat=ierr ) naccept_tps
         read ( iounit, *, iostat=ierr ) nreject_tps
         do j = 1, nbead
            read( iounit, *, iostat=ierr ) status_tps(j)
         end do
         close( iounit )

!     /*   if not   */
      else

!        /*   reset tps counter   */
         itrial_start_tps = 0
         naccept_tps = 0
         nreject_tps = 0
         do j = 1, nbead
            status_tps(j) = 'I'
         end do

!     /*   end of if statement   */
      end if

!     /*   check error   */
      call error_handling ( ierr, 'subroutine setup_tps', 20 )

!-----------------------------------------------------------------------
!     /*   collective variables for transition path sampling          */
!-----------------------------------------------------------------------

      call setup_tps_param

      return
      end





!***********************************************************************
      subroutine tpscycle
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   istep, istep_start, istep_end, nstep, iexit

      use tps_variables, only : &
     &   itrial_tps, itrial_end_tps, itrial_start_tps, ntrial_tps

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   initial setup                                              */
!-----------------------------------------------------------------------

!     //   initialize trial
      itrial_tps = itrial_start_tps
      itrial_end_tps = itrial_tps

!     //   initialize step
      istep = istep_start
      istep_end = istep

!     //   ux --> x
      call nm_trans_cart( 2 )

!     //   tps loop start
      do itrial_tps = itrial_start_tps+1, ntrial_tps

!        /*   reinitialize step   */
         if ( ( ( itrial_tps .eq. itrial_start_tps+1 ) .and. &
     &          ( istep_start .eq. nstep ) ) .or. &
     &          ( itrial_tps .gt. itrial_start_tps+1 ) ) &
     &      istep_start = 0

!        //   initialize step
         istep = istep_start
         istep_end = istep

!        /*   initial momentum kick   */
         if ( (itrial_tps.ne.1) .and. (istep.eq.0) ) call kick_tps

!        //   update force
         call getforce

!        //   collective variables
         call get_cv_meta

!        //   standard output
         call standard_tps

!        /*   do some analysis   */
         call analysis_tps ( 1 )

!        //   md loop start
         do istep = istep_start+1, nstep

!           //   current step
            istep_end = istep

!           //   update velocity
            call update_vel_tps

!           //   update position
            call update_pos_tps

!           //   update force
            call getforce

!           //   update velocity
            call update_vel_tps

!           //   collective variables
            call get_cv_meta

!           //   standard output
            call standard_tps

!           //   output restart
            call backup_tps

!           /*   do some analysis   */
            call analysis_tps ( 2 )

!           //   exit if `exit.dat' exists
            call softexit
            if ( iexit .eq. 1 ) exit

!        //   md loop end
         end do

!        /*   current step   */
         istep = istep_end

!        //   exit if `exit.dat' exists
         if ( iexit .eq. 1 ) exit

!        //   final step
         if ( istep .eq. nstep ) then

!           //   update trial
            itrial_end_tps = itrial_tps

!           /*   judge acceptance   */
            call judge_tps

!           /*   do some analysis   */
            call analysis_tps ( 3 )

!           /*   save final geometry   */
            call save_tps

!           /*   restart files   */
            call backup_tps

!        //   final step
         end if

!     //   tps loop end
      end do

      return
      end





!***********************************************************************
      subroutine update_vel_tps
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   vx, vy, vz, fx, fy, fz, dt, physmass, natom, nbead

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j

!-----------------------------------------------------------------------
!     //   update velocity: forward paths
!-----------------------------------------------------------------------

      do j = 1, nbead/2
      do i = 1, natom
         vx(i,j) = vx(i,j) + 0.5d0 * fx(i,j) * nbead / physmass(i) * dt
         vy(i,j) = vy(i,j) + 0.5d0 * fy(i,j) * nbead / physmass(i) * dt
         vz(i,j) = vz(i,j) + 0.5d0 * fz(i,j) * nbead / physmass(i) * dt
      end do
      end do

!-----------------------------------------------------------------------
!     //   update velocity: backward paths
!-----------------------------------------------------------------------

      do j = nbead/2+1, nbead
      do i = 1, natom
         vx(i,j) = vx(i,j) - 0.5d0 * fx(i,j) * nbead / physmass(i) * dt
         vy(i,j) = vy(i,j) - 0.5d0 * fy(i,j) * nbead / physmass(i) * dt
         vz(i,j) = vz(i,j) - 0.5d0 * fz(i,j) * nbead / physmass(i) * dt
      end do
      end do

!-----------------------------------------------------------------------
!     /*   save velocity                                              */
!-----------------------------------------------------------------------

!      vux(:,:) = vx(:,:)
!      vuy(:,:) = vy(:,:)
!      vuz(:,:) = vz(:,:)

      return
      end





!***********************************************************************
      subroutine update_pos_tps
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, vx, vy, vz, dt, natom, nbead

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j

!-----------------------------------------------------------------------
!     //   update position: forward paths
!-----------------------------------------------------------------------

      do j = 1, nbead/2
      do i = 1, natom
         x(i,j) = x(i,j) + vx(i,j) * dt
         y(i,j) = y(i,j) + vy(i,j) * dt
         z(i,j) = z(i,j) + vz(i,j) * dt
      end do
      end do

!-----------------------------------------------------------------------
!     //   update position: backward paths
!-----------------------------------------------------------------------

      do j = nbead/2+1, nbead
      do i = 1, natom
         x(i,j) = x(i,j) - vx(i,j) * dt
         y(i,j) = y(i,j) - vy(i,j) * dt
         z(i,j) = z(i,j) - vz(i,j) * dt
      end do
      end do

!-----------------------------------------------------------------------
!     /*   apply boundary condition                                   */
!-----------------------------------------------------------------------

      call pbc_xyz

!-----------------------------------------------------------------------
!     /*   save position                                              */
!-----------------------------------------------------------------------

!      ux(:,:) = x(:,:)
!      uy(:,:) = y(:,:)
!      uz(:,:) = z(:,:)

      return
      end





!***********************************************************************
      subroutine standard_tps
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   physmass, vx, vy, vz, temp, boltz, ekin, &
     &   hamiltonian, potential, char_date, istep, &
     &   iprint_std, iounit, iounit_std, natom, nbead

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, itest

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( iset .eq. 0 ) then

         call read_int1 ( iprint_std, '<iprint_std>', 12, iounit )

         call testfile ( 'standard.out', 12, itest )

         if ( itest .eq. 1 ) then

            open ( iounit_std, file = 'standard.out')

            write(iounit_std,'(a)') &
     &      '====================================================' // &
     &      '=========================='
            write(iounit_std,'(a)') &
     &      '    step     energy [au]  potential [au]  temp [K]  ' // &
     &      'wall clock time           '
            write(iounit_std,'(a)') &
     &      '----------------------------------------------------' // &
     &      '--------------------------'

            write(         6,'(a)') &
     &      '====================================================' // &
     &      '=========================='
            write(         6,'(a)') &
     &      '    step     energy [au]  potential [au]  temp [K]  ' // &
     &      'wall clock time           '
            write(         6,'(a)') &
     &      '----------------------------------------------------' // &
     &      '--------------------------'

         else

            open( iounit_std, file = 'standard.out', access = 'append' )

            write(         6,'(a)') &
     &      '====================================================' // &
     &      '=========================='
            write(         6,'(a)') &
     &      '    step     energy [au]  potential [au]  temp [K]  ' // &
     &      'wall clock time           '
            write(         6,'(a)') &
     &      '----------------------------------------------------' // &
     &      '--------------------------'

         end if

         iset = 1

      end if

!-----------------------------------------------------------------------
!     /*   kinetic energy                                             */
!-----------------------------------------------------------------------

      ekin = 0.d0

      do j = 1, nbead
      do i = 1, natom
         ekin = ekin + 0.5d0*physmass(i)*vx(i,j)*vx(i,j)
         ekin = ekin + 0.5d0*physmass(i)*vy(i,j)*vy(i,j)
         ekin = ekin + 0.5d0*physmass(i)*vz(i,j)*vz(i,j)
      end do
      end do

      temp = 2.d0*ekin/dble(3*natom*nbead)/boltz

!-----------------------------------------------------------------------
!     /*   total energy                                               */
!-----------------------------------------------------------------------

      hamiltonian = ekin/dble(nbead) + potential

!-----------------------------------------------------------------------
!     /*   output                                                     */
!-----------------------------------------------------------------------

      if ( mod(istep,iprint_std) .eq. 0 ) then

!        /*   wall clock time   */
         call getdate

!        /*   output   */
         write(iounit_std,'(i8,2f16.8,f10.2,2x,a28)') &
     &      istep, hamiltonian, potential, temp, char_date

         write(         6,'(i8,2f16.8,f10.2,2x,a28)') &
     &      istep, hamiltonian, potential, temp, char_date

      end if

      return
      end





!***********************************************************************
      subroutine backup_tps
!***********************************************************************
!=======================================================================
!
!     finalize the calculation.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, vx, vy, vz, ux, uy, uz, vux, vuy, vuz, nbead, &
     &   istep_end, iprint_rest, iounit, nstep

      use tps_variables, only : &
     &   itrial_end_tps, naccept_tps, nreject_tps, status_tps

      implicit none

      integer :: j

!-----------------------------------------------------------------------
!     /*   conditions                                                 */
!-----------------------------------------------------------------------

      if ( mod(istep_end,iprint_rest) .eq. 0 ) then
         if ( iprint_rest .le. 0 ) then
            return
         else
            continue
         end if
      else if ( istep_end .eq. nstep ) then
         continue
      else
         return
      end if

!-----------------------------------------------------------------------
!     /*   copy x, y, z, vx, vy, vz -> ux, uy, uz, vux, vuy, vuz      */
!-----------------------------------------------------------------------

      ux(:,:) = x(:,:)
      uy(:,:) = y(:,:)
      uz(:,:) = z(:,:)

      vux(:,:) = vx(:,:)
      vuy(:,:) = vy(:,:)
      vuz(:,:) = vz(:,:)

!-----------------------------------------------------------------------
!     /*   write out restart file                                     */
!-----------------------------------------------------------------------

!     /*   system position   */
      call restart_position( 4 )

!     /*   system velocity   */
      call restart_velocity( 4 )

!-----------------------------------------------------------------------
!     /*   in `step.ini', print the step number for restart           */
!-----------------------------------------------------------------------

      open ( iounit, file = 'step.ini' )
         write ( iounit, '(i8)' ) istep_end
      close( iounit )

!-----------------------------------------------------------------------
!     /*   write tps counter to restart file                          */
!-----------------------------------------------------------------------

      open ( iounit, file = 'averages.ini' )
         write( iounit, '(i8)' ) itrial_end_tps
         write( iounit, '(i8)' ) naccept_tps
         write( iounit, '(i8)' ) nreject_tps
         do j = 1, nbead
            write( iounit, '(a)' ) status_tps(j)
         end do
      close( iounit )

      return
      end





!***********************************************************************
      subroutine analysis_tps ( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   fx, fy, fz, x, y, z, vx, vy, vz, pot, au_length, nstep, &
     &   natom, nbead, iounit, istep, iounit_xyz, iounit_trj

      use analysis_variables, only : iprint_trj

      use meta_variables, only : params_meta, smeta, nmeta

      use tps_variables, only : params_tps_meta, status_tps, nstep_trj

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: ioption, i, j, k, jflag, kflag

      real(8) :: fxn, fyn, fzn, smin, smax

!-----------------------------------------------------------------------
!     /*   ioption = 1:  initialize/restart                           */
!-----------------------------------------------------------------------

      if ( ( ioption .eq. 0 ) .or. ( ioption .eq. 1 ) )  then

         call read_int1 ( iprint_trj, '<iprint_trj>', 12, iounit )

         nstep_trj = nstep / iprint_trj

         if ( istep .le. 1 ) status_tps(:) = 'I'

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 2:  start analysis                               */
!-----------------------------------------------------------------------

      if ( ( ioption .eq. 1 ) .or. ( ioption .eq. 2 ) ) then

         if ( mod(istep,iprint_trj) .eq. 0 ) then

            open ( iounit, file = 'trj.out', access = 'append' )

            do k = 1, nbead
            do i = 1, natom

               fxn = fx(i,k) * nbead
               fyn = fy(i,k) * nbead
               fzn = fz(i,k) * nbead

               write( iounit, '(i8,10e24.16)' ) &
     &            istep, x(i,k),  y(i,k),  z(i,k), &
     &                   vx(i,k), vy(i,k), vz(i,k), &
     &                   fxn, fyn, fzn, pot(k)

            end do
            end do

            close( iounit )

         end if

!-----------------------------------------------------------------------
!        /*   collective variables                                    */
!-----------------------------------------------------------------------

!        /*   loop of forward trajectory   */
         do j = 1, nbead/2

            k = j + nbead/2

            jflag = 0
            kflag = 0

            do i = 1, nmeta

               smin = params_meta(1,i)
               smax = params_meta(2,i)

               if ( ( params_tps_meta(i)(1:2) .eq. 'RP' ) .and. &
     &              ( smeta(i,j) .lt. smin ) ) jflag = jflag + 1 

               if ( ( params_tps_meta(i)(1:2) .eq. 'RP' ) .and. &
     &              ( smeta(i,k) .lt. smin ) ) kflag = kflag + 1

               if ( ( params_tps_meta(i)(1:2) .eq. 'RP' ) .and. &
     &              ( smeta(i,j) .gt. smax ) ) jflag = jflag - 1

               if ( ( params_tps_meta(i)(1:2) .eq. 'RP' ) .and. &
     &              ( smeta(i,k) .gt. smax ) ) kflag = kflag - 1

               if ( ( params_tps_meta(i)(1:2) .eq. 'PR' ) .and. &
     &              ( smeta(i,j) .lt. smin ) ) jflag = jflag - 1

               if ( ( params_tps_meta(i)(1:2) .eq. 'PR' ) .and. &
     &              ( smeta(i,k) .lt. smin ) ) kflag = kflag - 1

               if ( ( params_tps_meta(i)(1:2) .eq. 'PR' ) .and. &
     &              ( smeta(i,j) .gt. smax ) ) jflag = jflag + 1

               if ( ( params_tps_meta(i)(1:2) .eq. 'PR' ) .and. &
     &              ( smeta(i,k) .gt. smax ) ) kflag = kflag + 1

            end do

            if ( jflag .eq. +nmeta ) status_tps(j)(1:1) = 'R'
            if ( jflag .eq. -nmeta ) status_tps(j)(1:1) = 'P'
            if ( kflag .eq. +nmeta ) status_tps(k)(1:1) = 'R'
            if ( kflag .eq. -nmeta ) status_tps(k)(1:1) = 'P'

!        /*   loop of forward trajectory   */
         end do

         if ( mod(istep,iprint_trj) .eq. 0 ) then

            open ( iounit, file = 'tps.out', access = 'append' )

            if ( nmeta .eq. 1 ) then
               do i = 1, nbead/2
                  write(iounit,'(i8,2f10.4,1x,a,1x,a)') &
     &               istep, smeta(1,i), smeta(1,i+nbead/2), &
     &               status_tps(i), status_tps(i+nbead/2)
               end do
            else if ( nmeta .eq. 2 ) then
               do i = 1, nbead/2
                  write(iounit,'(i8,4f10.4,1x,a,1x,a)') &
     &               istep, (smeta(j,i), j=1,2), &
     &               (smeta(j,i+nbead/2),j=1,2), &
     &               status_tps(i), status_tps(i+nbead/2)
               end do
            else if ( nmeta .ge. 3 ) then
               do i = 1, nbead/2
                  write(iounit,'(i8,6f10.4,1x,a,1x,a)') &
     &               istep, (smeta(j,i), j=1,3), &
     &               (smeta(j,i+nbead/2),j=1,3), &
     &               status_tps(i), status_tps(i+nbead/2)
               end do
            end if

            close( iounit )

         end if
 
      end if

!-----------------------------------------------------------------------
!     /*   ioption = 3:  finalize                                     */
!-----------------------------------------------------------------------

      if ( ioption .eq. 3 ) then

         call system( 'rm -f trj.out' )

         open ( iounit, file = 'tps.out', access = 'append' )

         if ( nmeta .eq. 1 ) then
            do i = 1, nbead/2
               write(iounit,'(i8,2f10.4,1x,a,1x,a,a)') &
     &            istep, smeta(1,i), smeta(1,i+nbead/2), &
     &            status_tps(i), status_tps(i+nbead/2), &
     &            ' END'
            end do
         else if ( nmeta .eq. 2 ) then
            do i = 1, nbead/2
               write(iounit,'(i8,4f10.4,1x,a,1x,a,a)') &
     &            istep, (smeta(j,i), j=1,2), &
     &            (smeta(j,i+nbead/2),j=1,2), &
     &            status_tps(i), status_tps(i+nbead/2), &
     &            ' END'
            end do
         else if ( nmeta .ge. 3 ) then
            do i = 1, nbead/2
               write(iounit,'(i8,6f10.4,1x,a,1x,a)') &
     &            istep, (smeta(j,i), j=1,3), &
     &            (smeta(j,i+nbead/2),j=1,3), &
     &            status_tps(i), status_tps(i+nbead/2), &
     &            ' END'
            end do
         end if

         close( iounit )

      end if

      return
      end





!***********************************************************************
      subroutine judge_tps
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   natom, nbead, iounit

      use tps_variables, only : &
     &   ratio_tps, status_tps, iounit_tps, naccept_tps, nreject_tps, &
     &   nstep_trj

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     //   initialization
      implicit none

!     //   characters
      character(len=3)  :: char_3
      character(len=8)  :: char_8
      character(len=20) :: char_20

!     //   integers
      integer :: i, j, n, itest, istep_trj, ierr

!     //   real vectors
      real(8), dimension(:), allocatable :: &
     &   d1, d2, d3, d4, d5, d6, d7, d8, d9, d10

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

      if ( .not. allocated(d1)  ) allocate( d1(natom)  )
      if ( .not. allocated(d2)  ) allocate( d2(natom)  )
      if ( .not. allocated(d3)  ) allocate( d3(natom)  )
      if ( .not. allocated(d4)  ) allocate( d4(natom)  )
      if ( .not. allocated(d5)  ) allocate( d5(natom)  )
      if ( .not. allocated(d6)  ) allocate( d6(natom)  )
      if ( .not. allocated(d7)  ) allocate( d7(natom)  )
      if ( .not. allocated(d8)  ) allocate( d8(natom)  )
      if ( .not. allocated(d9)  ) allocate( d9(natom)  )
      if ( .not. allocated(d10) ) allocate( d10(natom) )

!-----------------------------------------------------------------------
!     /*   update acceptance and rejection                            */
!-----------------------------------------------------------------------

!     //   loop of forward trajectories
      do j = 1, nbead/2

!        //   if trajectories are good
         if ( ( ( status_tps(j)(1:1) .eq. 'R' ) .and. &
     &          ( status_tps(j+nbead/2)(1:1) .eq. 'P' ) ) .or. &
     &        ( ( status_tps(j)(1:1) .eq. 'P' ) .and. &
     &          ( status_tps(j+nbead/2)(1:1) .eq. 'R' ) ) ) then

!           //   number of trajectories accepted
            naccept_tps = naccept_tps + 1

!        //   if trajectories are bad
         else

!           //   number of trajectories rejected
            nreject_tps = nreject_tps + 1

!        //   end of if statement
         end if

!     //   loop of forward trajectories
      end do

!     //   acceptance ratio
      ratio_tps = dble(naccept_tps) / dble( naccept_tps + nreject_tps )

!-----------------------------------------------------------------------
!     /*   update acceptance and rejection                            */
!-----------------------------------------------------------------------

      ierr = 0

!-----------------------------------------------------------------------
!     /*   make scratch directory                                     */
!-----------------------------------------------------------------------

!     /*   find scratch directory   */
      call testfile( 'scr', 3, itest )

!     /*   if not found creat scratch directory   */
      if ( itest .ne. 0 ) call system( 'mkdir -p scr' )

!-----------------------------------------------------------------------
!     /*   split data for each bead and for each step                 */
!-----------------------------------------------------------------------

!     //   open the master output file
      open ( iounit, file = 'trj.out' )

!     //   loop of steps in master output file
      do istep_trj = 0, nstep_trj

!        //   char_8: step number
         call int8_to_char( istep_trj, char_8 )

!        //   loop of beads
         do j = 1, nbead

!           //   char_3: bead number
            call int3_to_char( j, char_3 )

!           //   loop of atoms
            do i = 1, natom

!              //   read step number and 10 data
               read ( iounit, *, iostat=ierr ) &
     &            n,  d1(i), d2(i), d3(i), &
     &                d4(i), d5(i), d6(i), &
     &                d7(i), d8(i), d9(i), d10(i)

!           //   loop of atoms
            end do

!           //   find error
            if ( ierr .ne. 0 ) exit

!           //   file name
            char_20 = 'scr/' // char_3(1:3) // char_8(1:8) // '.scr'

!           //   open file
            open ( iounit_tps, file = char_20, position ='append' )

!           //   loop of atoms
            do i = 1, natom

!              //   write step number and 10 data
               write( iounit_tps, '(i8,10e24.16)' ) &
     &            n,  d1(i), d2(i), d3(i), &
     &                d4(i), d5(i), d6(i), &
     &                d7(i), d8(i), d9(i), d10(i)

!           //   loop of atoms
            end do

!        //   loop of beads
         end do

!        //   find error
         if ( ierr .ne. 0 ) exit

!     //   loop of steps in master output file
      end do

!     //   close the master output file
      close( iounit )

!     /*   check error   */
      call error_handling ( ierr, 'subroutine judge_tps', 20 )

!-----------------------------------------------------------------------
!     /*   gather them for each bead                                  */
!-----------------------------------------------------------------------

!     //   loop of beads of forward trajectory
      do j = 1, nbead/2

!        //   char_3: bead number
         call int3_to_char( j, char_3 )

!        //   find file
         call testfile( 'tps.'//char_3//'.ini', 11, itest )

!        //   a trajectory from reactant to product
         if ( ( status_tps(j)(1:1) .eq. 'R' ) .and. &
     &        ( status_tps(j+nbead/2)(1:1) .eq. 'P' ) ) then

!           //   char_3: bead number
            call int3_to_char( j, char_3 )

!           //   open ini file
            open ( iounit, file = 'tps.' // char_3 // '.ini' )

!           //   char_3: bead number
            call int3_to_char( j, char_3 )

!           //   loop of steps
            do istep_trj = nstep_trj, 0, -1

!              //   step number
               call int8_to_char( istep_trj, char_8 )

!              //   file name
               char_20 = 'scr/' // char_3(1:3) // char_8(1:8) // '.scr'

!              //   open file
               open ( iounit_tps, file = char_20 )

!              //   loop of atoms
               do i = 1, natom

!                 //   read step number and 10 data
                  read ( iounit_tps, *, iostat=ierr ) &
     &               n,  d1(i), d2(i), d3(i), &
     &                   d4(i), d5(i), d6(i), &
     &                   d7(i), d8(i), d9(i), d10(i)

!              //   loop of atoms
               end do

!              //   close file
               close( iounit_tps )

!              //   loop of atoms
               do i = 1, natom

!                 //   write step number and 10 data
                  write( iounit, '(i8,10e24.16)' ) &
     &              -n,  d1(i), d2(i), d3(i), &
     &                   d4(i), d5(i), d6(i), &
     &                   d7(i), d8(i), d9(i), d10(i)

!              //   loop of atoms
               end do

!           //   loop of steps
            end do

!           //   char_3: bead number
            call int3_to_char( j+nbead/2, char_3 )

!           //   loop of steps
            do istep_trj = 1, nstep_trj, 1

!              //   step number
               call int8_to_char( istep_trj, char_8 )

!              //   file name
               char_20 = 'scr/' // char_3(1:3) // char_8(1:8) // '.scr'

!              //   open file
               open ( iounit_tps, file = char_20 )

!              //   loop of atoms
               do i = 1, natom

!                 //   read step number and 10 data
                  read ( iounit_tps, *, iostat=ierr ) &
     &               n,  d1(i), d2(i), d3(i), &
     &                   d4(i), d5(i), d6(i), &
     &                   d7(i), d8(i), d9(i), d10(i)

!              //   loop of atoms
               end do

!               //   close file   */
               close( iounit_tps )

!              //   loop of atoms
               do i = 1, natom

!                 //   write step number and 10 data
                  write( iounit, '(i8,10e24.16)' ) &
     &               n,  d1(i), d2(i), d3(i), &
     &                   d4(i), d5(i), d6(i), &
     &                   d7(i), d8(i), d9(i), d10(i)

!              //   loop of atoms
               end do

!           //   loop of steps
            end do

!           //   close ini file
            close( iounit )

!        //   a trajectory from reactant to product
         else if ( ( status_tps(j)(1:1) .eq. 'P' ) .and. &
     &             ( status_tps(j+nbead/2)(1:1) .eq. 'R' ) ) then

!           //   char_3: bead number
            call int3_to_char( j, char_3 )

!           //   open ini file
            open ( iounit, file = 'tps.' // char_3 // '.ini' )

!           //   char_3: bead number
            call int3_to_char( j+nbead/2, char_3 )

!           //   loop of steps
            do istep_trj = nstep_trj, 0, -1

!              //   step number
               call int8_to_char( istep_trj, char_8 )

!              //   file name
               char_20 = 'scr/' // char_3(1:3) // char_8(1:8) // '.scr'

!              //   open file
               open ( iounit_tps, file = char_20 )

!              //   loop of atoms
               do i = 1, natom

!                 //   read step number and 10 data
                  read ( iounit_tps, *, iostat=ierr ) &
     &               n,  d1(i), d2(i), d3(i), &
     &                   d4(i), d5(i), d6(i), &
     &                   d7(i), d8(i), d9(i), d10(i)

!              //   loop of atoms
               end do

!              //   close file
               close( iounit_tps )

!              //   loop of atoms
               do i = 1, natom

!                 //   write step number and 10 data
                  write( iounit, '(i8,10e24.16)' ) &
     &              -n,  d1(i), d2(i), d3(i), &
     &                   d4(i), d5(i), d6(i), &
     &                   d7(i), d8(i), d9(i), d10(i)

!              //   loop of atoms
               end do

!           //   loop of steps
            end do

!           //   char_3: bead number
            call int3_to_char( j, char_3 )

!           //   loop of steps
            do istep_trj = 1, nstep_trj, 1

!              //   step number
               call int8_to_char( istep_trj, char_8 )

!              //   file name
               char_20 = 'scr/' // char_3(1:3) // char_8(1:8) // '.scr'

!              //   open file
               open ( iounit_tps, file = char_20 )

!              //   loop of atoms
               do i = 1, natom

!                 //   read step number and 10 data
                  read ( iounit_tps, *, iostat=ierr ) &
     &               n,  d1(i), d2(i), d3(i), &
     &                   d4(i), d5(i), d6(i), &
     &                   d7(i), d8(i), d9(i), d10(i)

!              //   loop of atoms
               end do

!               //   close file   */
               close( iounit_tps )

!              //   loop of atoms
               do i = 1, natom

!                 //   write step number and 10 data
                  write( iounit, '(i8,10e24.16)' ) &
     &               n,  d1(i), d2(i), d3(i), &
     &                   d4(i), d5(i), d6(i), &
     &                   d7(i), d8(i), d9(i), d10(i)

!              //   loop of atoms
               end do

!           //   loop of steps
            end do

!           //   close ini file
            close( iounit )

!        //   for a rejected but initial trajectory, save it
         else if ( itest .ne. 0 ) then

!           //   char_3: bead number
            call int3_to_char( j, char_3 )

!           //   open ini file
            open ( iounit, file = 'tps.' // char_3 // '.ini' )

!           //   char_3: bead number
            call int3_to_char( j, char_3 )

!           //   loop of steps
            do istep_trj = nstep_trj, 0, -1

!              //   step number
               call int8_to_char( istep_trj, char_8 )

!              //   file name
               char_20 = 'scr/' // char_3(1:3) // char_8(1:8) // '.scr'

!              //   open file
               open ( iounit_tps, file = char_20 )

!              //   loop of atoms
               do i = 1, natom

!                 //   read step number and 10 data
                  read ( iounit_tps, *, iostat=ierr ) &
     &               n,  d1(i), d2(i), d3(i), &
     &                   d4(i), d5(i), d6(i), &
     &                   d7(i), d8(i), d9(i), d10(i)

!              //   loop of atoms
               end do

!              //   close file
               close( iounit_tps )

!              //   loop of atoms
               do i = 1, natom

!                 //   write step number and 10 data
                  write( iounit, '(i8,10e24.16)' ) &
     &              -n,  d1(i), d2(i), d3(i), &
     &                   d4(i), d5(i), d6(i), &
     &                   d7(i), d8(i), d9(i), d10(i)

!              //   loop of atoms
               end do

!           //   loop of steps
            end do

!           //   char_3: bead number
            call int3_to_char( j+nbead/2, char_3 )

!           //   loop of steps
            do istep_trj = 1, nstep_trj, 1

!              //   step number
               call int8_to_char( istep_trj, char_8 )

!              //   file name
               char_20 = 'scr/' // char_3(1:3) // char_8(1:8) // '.scr'

!              //   open file
               open ( iounit_tps, file = char_20 )

!              //   loop of atoms
               do i = 1, natom

!                 //   read step number and 10 data
                  read ( iounit_tps, *, iostat=ierr ) &
     &               n,  d1(i), d2(i), d3(i), &
     &                   d4(i), d5(i), d6(i), &
     &                   d7(i), d8(i), d9(i), d10(i)

!              //   loop of atoms
               end do

!               //   close file   */
               close( iounit_tps )

!              //   loop of atoms
               do i = 1, natom

!                 //   write step number and 10 data
                  write( iounit, '(i8,10e24.16)' ) &
     &               n,  d1(i), d2(i), d3(i), &
     &                   d4(i), d5(i), d6(i), &
     &                   d7(i), d8(i), d9(i), d10(i)

!              //   loop of atoms
               end do

!           //   loop of steps
            end do

!           //   close ini file
            close( iounit )

!        //   end of if statement
         end if

!     //   loop of beads of forward trajectory
      end do

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

      if ( allocated(d1)  ) deallocate( d1  )
      if ( allocated(d2)  ) deallocate( d2  )
      if ( allocated(d3)  ) deallocate( d3  )
      if ( allocated(d4)  ) deallocate( d4  )
      if ( allocated(d5)  ) deallocate( d5  )
      if ( allocated(d6)  ) deallocate( d6  )
      if ( allocated(d7)  ) deallocate( d7  )
      if ( allocated(d8)  ) deallocate( d8  )
      if ( allocated(d9)  ) deallocate( d9  )
      if ( allocated(d10) ) deallocate( d10 )

!-----------------------------------------------------------------------
!     /*   remove scratch directory                                   */
!-----------------------------------------------------------------------

!     /*   find scratch directory   */
      call testfile( 'scr', 3, itest )

!     /*   if not found creat scratch directory   */
      if ( itest .eq. 0 ) call system( 'rm -rf scr' )

      return
      end





!***********************************************************************
      subroutine judge_old_tps
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   au_length, natom, nbead, iounit

      use tps_variables, only : &
     &   ratio_tps, status_tps, iounit_tps, naccept_tps, nreject_tps, &
     &   nstep_trj

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: kstep, jbead, i, l, n, itest, ierr

      real(8) :: d1, d2, d3, d4, d5, d6, d7, d8, d9, d10

      character(len=3) :: c3

!-----------------------------------------------------------------------
!     /*   update acceptance and rejection                            */
!-----------------------------------------------------------------------

      do jbead = 1, nbead/2

         if ( ( ( status_tps(jbead)(1:1) .eq. 'R' ) .and. &
     &          ( status_tps(jbead+nbead/2)(1:1) .eq. 'P' ) ) .or. &
     &        ( ( status_tps(jbead)(1:1) .eq. 'P' ) .and. &
     &          ( status_tps(jbead+nbead/2)(1:1) .eq. 'R' ) ) ) then

            naccept_tps = naccept_tps + 1

         else

            nreject_tps = nreject_tps + 1

         end if

      end do

      ratio_tps = dble(naccept_tps) / dble( naccept_tps + nreject_tps )

!-----------------------------------------------------------------------
!     /*   update acceptance and rejection                            */
!-----------------------------------------------------------------------

      ierr = 0

      do jbead = 1, nbead/2

         call int3_to_char( jbead, c3 )

         call testfile( 'tps.'//c3//'.ini', 11, itest )

         if ( itest .ne. 0 ) then

            open ( iounit_tps, file = 'tps.' // c3 // '.ini' )

            do kstep = nstep_trj, 1, -1

               l = kstep*natom*nbead + natom*(nbead/2+jbead-1)

               open ( iounit, file = 'trj.out' )

               do i = 1, l
                  read ( iounit, *, iostat=ierr )
               end do

               do i = 1, natom
                  read ( iounit, *, iostat=ierr ) &
     &               n,  d1, d2, d3, &
     &                   d4, d5, d6, &
     &                   d7, d8, d9, d10
                  write( iounit_tps, '(i8,10e24.16)' ) &
     &               n,  d1, d2, d3, &
     &                   d4, d5, d6, &
     &                   d7, d8, d9, d10
               end do

               close( iounit )

            end do

            do kstep = 0, nstep_trj, 1

               l = kstep*natom*nbead + natom*(jbead-1)

               open ( iounit, file = 'trj.out' )

               do i = 1, l
                  read ( iounit, *, iostat=ierr )
               end do

               do i = 1, natom
                  read ( iounit, *, iostat=ierr ) &
     &                n,  d1, d2, d3, &
     &                    d4, d5, d6, &
     &                    d7, d8, d9, d10
                  write( iounit_tps, '(i8,10e24.16)' ) &
     &                n,  d1, d2, d3, &
     &                    d4, d5, d6, &
     &                    d7, d8, d9, d10
               end do

               close( iounit )

            end do

            close ( iounit_tps )

         else if ( ( status_tps(jbead)(1:1) .eq. 'R' ) .and. &
     &             ( status_tps(jbead+nbead/2)(1:1) .eq. 'P' ) ) then 

            open ( iounit_tps, file = 'tps.' // c3 // '.ini' )

            do kstep = nstep_trj, 1, -1

               l = kstep*natom*nbead + natom*(nbead/2+jbead-1)

               open ( iounit, file = 'trj.out' )

               do i = 1, l
                  read ( iounit, *, iostat=ierr )
               end do

               do i = 1, natom
                  read ( iounit, *, iostat=ierr ) &
     &               n,  d1, d2, d3, &
     &                   d4, d5, d6, &
     &                   d7, d8, d9, d10
                  write( iounit_tps, '(i8,10e24.16)' ) &
     &               n,  d1, d2, d3, &
     &                   d4, d5, d6, &
     &                   d7, d8, d9, d10
               end do

               close( iounit )

            end do

            do kstep = 0, nstep_trj, 1

               l = kstep*natom*nbead + natom*(jbead-1)

               open ( iounit, file = 'trj.out' )

               do i = 1, l
                  read ( iounit, *, iostat=ierr )
               end do

               do i = 1, natom
                  read ( iounit, *, iostat=ierr ) &
     &                n,  d1, d2, d3, &
     &                    d4, d5, d6, &
     &                    d7, d8, d9, d10
                  write( iounit_tps, '(i8,10e24.16)' ) &
     &                n,  d1, d2, d3, &
     &                    d4, d5, d6, &
     &                    d7, d8, d9, d10
               end do

               close( iounit )

            end do

            close ( iounit_tps )

         else if ( ( status_tps(jbead)(1:1) .eq. 'P' ) .and. &
     &             ( status_tps(jbead+nbead/2)(1:1) .eq. 'R' ) ) then 

            open ( iounit_tps, file = 'tps.' // c3 // '.ini' )

            do kstep = nstep_trj, 0, -1

               l = kstep*natom*nbead + natom*(jbead-1)

               open ( iounit, file = 'trj.out' )

               do i = 1, l
                  read ( iounit, *, iostat=ierr )
               end do

               do i = 1, natom
                  read ( iounit, *, iostat=ierr ) &
     &                n,  d1, d2, d3, &
     &                    d4, d5, d6, &
     &                    d7, d8, d9, d10
                  write( iounit_tps, '(i8,10e24.16)' ) &
     &                n,  d1, d2, d3, &
     &                    d4, d5, d6, &
     &                    d7, d8, d9, d10
               end do

               close( iounit )

            end do

            do kstep = 1, nstep_trj, 1

               l = kstep*natom*nbead + natom*(nbead/2+jbead-1)

               open ( iounit, file = 'trj.out' )

               do i = 1, l
                  read ( iounit, *, iostat=ierr )
               end do

               do i = 1, natom
                  read ( iounit, *, iostat=ierr ) &
     &               n,  d1, d2, d3, &
     &                   d4, d5, d6, &
     &                   d7, d8, d9, d10
                  write( iounit_tps, '(i8,10e24.16)' ) &
     &               n,  d1, d2, d3, &
     &                   d4, d5, d6, &
     &                   d7, d8, d9, d10
               end do

               close( iounit )

            end do

            close ( iounit_tps )

         end if

         if ( ierr .ne. 0 ) exit

      end do

!     /*   check error   */
      call error_handling ( ierr, 'subroutine judge_tps', 20 )

      return
      end





!***********************************************************************
      subroutine save_tps
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   au_length, species, natom, nbead, iounit

      use tps_variables, only : iounit_tps, itrial_tps, nstep_trj

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: kstep, jbead, i, n, ierr

      real(8) :: d1, d2, d3, d4, d5, d6, d7, d8, d9, d10

      character(len=3) :: c3
      character(len=8) :: c8

      real(8) :: bohr2ang = au_length * 1.d+10

!-----------------------------------------------------------------------
!     /*   update acceptance and rejection                            */
!-----------------------------------------------------------------------

      ierr = 0

      do jbead = 1, nbead/2

         i = nbead/2 * ( itrial_tps -1 ) + jbead

         call int3_to_char( jbead, c3 )
         call int8_to_char( i, c8 )

         open ( iounit_tps, file = 'tps.' // c3 // '.ini' )
         open ( iounit, file = 'tps.' // c8 // '.xyz' )

         do kstep = 1, 2*nstep_trj+1

            write( iounit, '(i8)' ) natom
            write( iounit, '(i8)' ) kstep

            do i = 1, natom
               read ( iounit_tps, *, iostat=ierr ) &
     &            n,  d1, d2, d3, &
     &                d4, d5, d6, &
     &                d7, d8, d9, d10
               write( iounit, '(a4,3f12.5)' ) &
     &            species(i)(1:4), &
     &            d1*bohr2ang, d2*bohr2ang, d3*bohr2ang
            end do

         end do

         close ( iounit_tps )
         close ( iounit )

         if ( ierr .ne. 0 ) exit

      end do

      call error_handling( ierr, 'subroutine save_tps', 19 )

      return
      end





!***********************************************************************
      subroutine recover_tps( jstep, j, ierr )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, vx, vy, vz, natom, iounit, nbead

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, n, jstep, ierr

      character(len=3) :: c3

!-----------------------------------------------------------------------
!     /*   file name                                                  */
!-----------------------------------------------------------------------

      call int3_to_char( j, c3 )

!-----------------------------------------------------------------------
!     /*   read position and velocity of jstep                        */
!-----------------------------------------------------------------------

      open ( iounit, file = 'tps.' // c3 // '.ini' )

      do i = 1, natom*(jstep-1)
         read ( iounit, *, iostat=ierr )
      end do

      do i = 1, natom
         read ( iounit, *, iostat=ierr) &
     &      n, x(i,j), y(i,j), z(i,j), vx(i,j), vy(i,j), vz(i,j)
      end do

      close( iounit )

!-----------------------------------------------------------------------
!     /*   copy forward trajectory to backward trajectory             */
!-----------------------------------------------------------------------

      x(:,j+nbead/2) = x(:,j)
      y(:,j+nbead/2) = y(:,j)
      z(:,j+nbead/2) = z(:,j)

      vx(:,j+nbead/2) = vx(:,j)
      vy(:,j+nbead/2) = vy(:,j)
      vz(:,j+nbead/2) = vz(:,j)

      return
      end





!***********************************************************************
      subroutine setup_tps_param
!***********************************************************************
!=======================================================================
!
!     read parameters for transition path sampling
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iounit, nbead

      use meta_variables, only : &
     &   smeta, params_meta, req_meta, &
     &   itype_meta, i_meta, j_meta, k_meta, l_meta, &
     &   mu_meta, nu_meta, ipbc_meta, nmeta

      use tps_variables, only : &
     &   params_tps_meta

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: k, ierr

      character(len=8) :: char_meta

      character(len=120) :: char_line

!-----------------------------------------------------------------------
!     /*   read number of collective variables                        */
!-----------------------------------------------------------------------

!     /*   read integer   */
      call read_int1 ( nmeta, '<nmeta>', 7, iounit )

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

!     /*   actual cv position   */
      if ( .not. allocated( smeta ) ) &
     &   allocate( smeta(nmeta,nbead) )

!     /*   type of cv   */
      if ( .not. allocated( itype_meta ) ) &
     &   allocate( itype_meta(nmeta) )

!     /*   boundary condition of cv   */
      if ( .not. allocated( ipbc_meta ) ) &
     &   allocate( ipbc_meta(nmeta) )

!     /*   atoms i, j, k, l of cv   */
      if ( .not. allocated( i_meta ) ) &
     &   allocate( i_meta(nmeta) )
      if ( .not. allocated( j_meta ) ) &
     &   allocate( j_meta(nmeta) )
      if ( .not. allocated( k_meta ) ) &
     &   allocate( k_meta(nmeta) )
      if ( .not. allocated( l_meta ) ) &
     &   allocate( l_meta(nmeta) )

!     /*   rational function parameters for coordination number   */
      if ( .not. allocated( nu_meta ) ) &
     &   allocate( nu_meta(nmeta,2) )
      if ( .not. allocated( mu_meta ) ) &
     &   allocate( mu_meta(nmeta,2) )
      if ( .not. allocated( req_meta ) ) &
     &   allocate( req_meta(nmeta,2) )

!     /*   minimum and maximum   */
      if ( .not. allocated( params_meta ) ) &
     &   allocate( params_meta(2,nmeta) )

!     /*   RP or PR   */
      if ( .not. allocated( params_tps_meta ) ) &
     &   allocate( params_tps_meta(nmeta) )

!-----------------------------------------------------------------------
!     /*   read type of collective variables                          */
!-----------------------------------------------------------------------

!     /*   file open   */
      open ( iounit, file = 'input.dat' )

!     /*   search for tag    */
      call search_tag ( '<nmeta>', 7, iounit, ierr )

!     /*   read integer   */
      read ( iounit, *, iostat=ierr )

      do k = 1, nmeta

         read ( iounit, *, iostat=ierr ) char_meta

         backspace( iounit )

         if      ( ( char_meta(1:6) .eq. '1     ' ) .or. &
     &             ( char_meta(1:6) .eq. 'DIST  ' ) ) then
            itype_meta(k) = 1
         else if ( ( char_meta(1:6) .eq. '2     ' ) .or. &
     &             ( char_meta(1:6) .eq. 'ANGL  ' ) ) then
            itype_meta(k) = 2
         else if ( ( char_meta(1:6) .eq. '3     ' ) .or. &
     &             ( char_meta(1:6) .eq. 'DIH   ' ) ) then
            itype_meta(k) = 3
         else if ( ( char_meta(1:6) .eq. '4     ' ) .or. &
     &             ( char_meta(1:6) .eq. 'DIFF  ' ) ) then
            itype_meta(k) = 4
         else if ( ( char_meta(1:6) .eq. '5     ' ) .or. &
     &             ( char_meta(1:6) .eq. 'CN    ' ) ) then
            itype_meta(k) = 5
         else if ( ( char_meta(1:6) .eq. '6     ' ) .or. &
     &             ( char_meta(1:6) .eq. 'DCN   ' ) ) then
            itype_meta(k) = 6
         else if ( ( char_meta(1:6) .eq. '7     ' ) .or. &
     &             ( char_meta(1:6) .eq. 'XYZ   ' ) ) then
            itype_meta(k) = 7
         else if ( ( char_meta(1:6) .eq. '8     ' ) .or. &
     &             ( char_meta(1:6) .eq. 'DXYZ  ' ) ) then
            itype_meta(k) = 8
         else
            ierr = 1
            exit
         end if

         read ( iounit, '(a)', iostat=ierr ) char_line

         if      ( itype_meta(k) .eq. 1 ) then
            read ( char_line, *, iostat=ierr ) &
     &         char_meta, i_meta(k), j_meta(k), &
     &         params_meta(1,k), params_meta(2,k), &
     &         params_tps_meta(k)
            if ( ierr .ne. 0 ) exit
         else if ( itype_meta(k) .eq. 2 ) then
            read ( char_line, *, iostat=ierr ) &
     &         char_meta, i_meta(k), j_meta(k), k_meta(k), &
     &         params_meta(1,k), params_meta(2,k), &
     &         params_tps_meta(k)
            if ( ierr .ne. 0 ) exit
         else if ( itype_meta(k) .eq. 3 ) then
            read ( char_line, *, iostat=ierr ) &
     &         char_meta, i_meta(k), j_meta(k), k_meta(k), l_meta(k), &
     &         params_meta(1,k), params_meta(2,k), &
     &         params_tps_meta(k)
            if ( ierr .ne. 0 ) exit
         else if ( itype_meta(k) .eq. 4 ) then
            read ( char_line, *, iostat=ierr ) &
     &         char_meta, i_meta(k), j_meta(k), k_meta(k), &
     &         params_meta(1,k), params_meta(2,k), &
     &         params_tps_meta(k)
            if ( ierr .ne. 0 ) exit
         else if ( itype_meta(k) .eq. 5 ) then
            read ( char_line, *, iostat=ierr ) &
     &         char_meta, i_meta(k), j_meta(k), &
     &         nu_meta(k,1), mu_meta(k,1), req_meta(k,1), &
     &         params_meta(1,k), params_meta(2,k), &
     &         params_tps_meta(k)
            if ( ierr .ne. 0 ) exit
         else if ( itype_meta(k) .eq. 6 ) then
            read ( char_line, *, iostat=ierr ) &
     &         char_meta, i_meta(k), j_meta(k), &
     &         nu_meta(k,1), mu_meta(k,1), req_meta(k,1), &
     &                    k_meta(k), l_meta(k), &
     &         nu_meta(k,2), mu_meta(k,2), req_meta(k,2), &
     &         params_meta(1,k), params_meta(2,k), &
     &         params_tps_meta(k)
            if ( ierr .ne. 0 ) exit
         else if ( itype_meta(k) .eq. 7 ) then
            read ( char_line, *, iostat=ierr ) &
     &         char_meta, i_meta(k), j_meta(k), &
     &         params_meta(1,k), params_meta(2,k), &
     &         params_tps_meta(k)
            if ( ierr .ne. 0 ) exit
         else if ( itype_meta(k) .eq. 8 ) then
            read ( char_line, *, iostat=ierr ) &
     &         char_meta, i_meta(k), j_meta(k), k_meta(k), &
     &         params_meta(1,k), params_meta(2,k), &
     &         params_tps_meta(k)
            if ( ierr .ne. 0 ) exit
         else
            ierr = 1
            exit
         end if

      end do

!     /*   file close   */
      close( iounit )

!     /*   error message   */
      if ( ierr .ne. 0 ) then
         write( 6, '(a)' ) &
     &      'Error - keyword <nmeta> is incorrect.'
      end if

!     /*   check error   */
      call error_handling ( ierr, 'subroutine setup_tps_param', 26 )

      return
      end





!***********************************************************************
      subroutine kick_tps
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   boltz, physmass, vx, vy, vz, temperature, natom, nbead

      use tps_variables, only : &
     &   temp_kick_tps, nstep_trj

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   initialize   */
      implicit none

!     /*   random number and its gaussian deviation   */
      real(8) :: gasdev, randomno, ranf1

!     /*   kinetic energies   */
      real(8) :: ekin_old, ekin_new

!     /*   internal temperatures   */
      real(8) :: temp_new, temp_old

!     /*   real numbers   */
      real(8) :: f1, f2, vsigma

!     /*   starting step   */
      integer :: istep_tps(nbead)

!     /*   integers   */
      integer :: i, j, ierr

!     /*   vector of old velocity   */
      real(8) :: sx(natom), sy(natom), sz(natom)

!     /*   vector of kick/new velocity   */
      real(8) :: tx(natom), ty(natom), tz(natom)

!     /*   shift of initial step   */
      real(8) :: scale

!-----------------------------------------------------------------------
!     /*   scaling factor                                             */
!-----------------------------------------------------------------------

      scale = dble(nstep_trj) / 8.d0

!-----------------------------------------------------------------------
!     /*   initial step of forward and backward trajectories          */
!-----------------------------------------------------------------------

!     /*   initial step   */
      istep_tps(:) = nstep_trj+1

!     /*   loop of forward trajectories   */
      do i = 1, nbead/2

!        /*   uniform random number from 0 to 1   */
         randomno = ranf1()

!        /*   initial step is shifted   */
         istep_tps(i) = istep_tps(i) + nint( scale*randomno-scale/2.d0 )

!        /*   initial step is shifted   */
         istep_tps(i) = min( max( istep_tps(i), 1 ), 2*nstep_trj+1 )

!     /*   loop of forward trajectories   */
      end do

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

!     /*   loop of forward trajectories   */
      do i = 1, nbead/2

!        /*   read geometry at origin   */
         call recover_tps( istep_tps(i), i, ierr )

!-----------------------------------------------------------------------
!        /*   f1: old velocity norm                                   */
!-----------------------------------------------------------------------

!        /*   old velocity norm   */
         f1 = 0.d0

!        /*   loop of atoms   */
         do j = 1, natom

!           /*   old velocity   */
            sx(j) = sqrt(physmass(j)) * vx(j,i)
            sy(j) = sqrt(physmass(j)) * vy(j,i)
            sz(j) = sqrt(physmass(j)) * vz(j,i)

!           /*   old velocity norm   */
            f1 = f1 + sx(j) * sx(j)
            f1 = f1 + sy(j) * sy(j)
            f1 = f1 + sz(j) * sz(j)

!        /*   loop of atoms   */
         end do

!        /*   old kinetic energy   */
         ekin_old = 0.5d0 * f1

!        /*   old temperature   */
         temp_old = 2.d0 * ekin_old / dble(3*natom) / boltz

!-----------------------------------------------------------------------
!        /*   s: normalized old velocity vector                       */
!-----------------------------------------------------------------------

!        /*   normalization factor   */
         f1 = 1.d0 / sqrt( f1 )

!        /*   loop of atoms   */
         do j = 1, natom

!           /*   normalized old velocity   */
            sx(j) = sx(j) * f1
            sy(j) = sy(j) * f1
            sz(j) = sz(j) * f1

!        /*   loop of atoms   */
         end do

!-----------------------------------------------------------------------
!        /*   t: kick vector from gaussian distribution               */
!-----------------------------------------------------------------------

!        /*   norm   */
         f1 = 0.d0

!        /*   loop of atoms   */
         do j = 1, natom

!           /*   scaled kick velocity   */
            tx(j) = gasdev()
            ty(j) = gasdev()
            tz(j) = gasdev()

!           /*   kick velocity norm   */
            f1 = f1 + tx(j) * tx(j)
            f1 = f1 + ty(j) * ty(j)
            f1 = f1 + tz(j) * tz(j)

!        /*   loop of atoms   */
         end do

!-----------------------------------------------------------------------
!        /*   t: normalized kick vector                               */
!-----------------------------------------------------------------------

!        /*   normalization factor   */
         f1 = 1.d0 / sqrt( f1 )

!        /*   loop of atoms   */
         do j = 1, natom

!           /*   normalized kick vector   */
            tx(j) = tx(j) * f1
            ty(j) = ty(j) * f1
            tz(j) = tz(j) * f1

!        /*   loop of atoms   */
         end do

!-----------------------------------------------------------------------
!        /*   t: weighted average of old and kick vectors             */
!-----------------------------------------------------------------------

!        /*   tentative velocity norm   */
         f1 = 0.d0

!        /*   weights   */
         f2 = sqrt( temp_kick_tps / temperature )

!        /*   loop of atoms   */
         do j = 1, natom

!           /*   tentative velocity   */
            tx(j) = tx(j) * f2 + sx(j)
            ty(j) = ty(j) * f2 + sy(j)
            tz(j) = tz(j) * f2 + sz(j)

!           /*   tentative velocity norm   */
            f1 = f1 + tx(j) * tx(j)
            f1 = f1 + ty(j) * ty(j)
            f1 = f1 + tz(j) * tz(j)

!        /*   loop of atoms   */
         end do

!-----------------------------------------------------------------------
!        /*   s: maxwell distribution                                 */
!-----------------------------------------------------------------------

!        /*   new velocity norm  */
         f2 = 0.d0

!        /*   loop of atoms   */
         do j = 1, natom

            vsigma = sqrt( boltz*temperature )

!           /*   new velocity of forward trajectory   */
            sx(j) = gasdev() * vsigma
            sy(j) = gasdev() * vsigma
            sz(j) = gasdev() * vsigma

!           /*   new velocity norm   */
            f2 = f2 + sx(j) * sx(j)
            f2 = f2 + sy(j) * sy(j)
            f2 = f2 + sz(j) * sz(j)

         end do

!-----------------------------------------------------------------------
!        /*   new kinetic energy                                      */
!-----------------------------------------------------------------------

!        /*   scaling factor   */
         f2 = sqrt( f2 / f1 )

!        /*   new velocity norm   */
         f1 = 0.d0

!        /*   loop of atoms   */
         do j = 1, natom

!           /*   new velocity   */
            vx(j,i) = tx(j) * f2 / sqrt(physmass(j))
            vy(j,i) = ty(j) * f2 / sqrt(physmass(j))
            vz(j,i) = tz(j) * f2 / sqrt(physmass(j))

!           /*   new velocity of backward trajectory   */
            vx(j,nbead/2+i) = vx(j,i)
            vy(j,nbead/2+i) = vy(j,i)
            vz(j,nbead/2+i) = vz(j,i)

!           /*   kick velocity norm   */
            f1 = f1 + physmass(j) * vx(j,i) * vx(j,i)
            f1 = f1 + physmass(j) * vy(j,i) * vy(j,i)
            f1 = f1 + physmass(j) * vz(j,i) * vz(j,i)

!        /*   loop of atoms   */
         end do

!        /*   new kinetic energy   */
         ekin_new = 0.5d0 * f1

!        /*   new temperature   */
         temp_new = 2.d0 * ekin_new / dble(3*natom) / boltz

!     /*   loop of forward trajectories   */
      end do

!-----------------------------------------------------------------------
!     /*   error termination                                          */
!-----------------------------------------------------------------------

      call error_handling( ierr, 'subroutine kick_tps', 19 )

      return
      end
