!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     free energy of toy model
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      module toymodel_variables
!***********************************************************************

      integer :: ng = 100
      real(8) :: boxl = 6.d0
      real(8), dimension(:,:), allocatable :: req
      real(8), dimension(:,:), allocatable :: alpha

!***********************************************************************
      end module toymodel_variables
!***********************************************************************





!***********************************************************************
      subroutine toymodel
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : nbead, iounit

      use afed_variables, only : fenergy_afed, f_afed, hess_afed

      use cons_variables, only : rcons, ncons

      use toymodel_variables, only : req, alpha, ng, boxl

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, l

      real(8) :: delta(ncons,ncons), dsi, dsj, v

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      call toymodel_init

!-----------------------------------------------------------------------
!        //   starting position
!-----------------------------------------------------------------------

      if ( iset .eq. 0 ) then

         call toymodel_pes

         rcons(1,:) = -1.0d0
         rcons(2,:) =  0.5d0

         iset = 1

      end if

!-----------------------------------------------------------------------
!     /*   initialization                                             */
!-----------------------------------------------------------------------

      delta(:,:) = 0.d0

      do i = 1, ncons
         delta(i,i) = 1.d0
      end do

      fenergy_afed = 0.d0

      f_afed(:) = 0.d0

      hess_afed(:,:) = 0.d0

!-----------------------------------------------------------------------
!     /*   potential, force, hessian                                  */
!-----------------------------------------------------------------------

      do k = 1, nbead

         do l = 1, ng

            v = 1.d0 / dble(ng)

            do i = 1, ncons
               dsi = rcons(i,k) - req(i,l)
               dsi = dsi - boxl*nint(dsi/boxl)
               v = v * exp( - alpha(i,l) * dsi * dsi )
            end do

            fenergy_afed = fenergy_afed + v

            do i = 1, ncons
               dsi = rcons(i,k) - req(i,l)
               dsi = dsi - boxl*nint(dsi/boxl)
               f_afed(i) = f_afed(i) + 2.d0 * alpha(i,l) * dsi * v
            end do

            do i = 1, ncons
            do j = 1, ncons
               dsi = rcons(i,k) - req(i,l)
               dsi = dsi - boxl*nint(dsi/boxl)
               dsj = rcons(j,k) - req(j,l)
               dsj = dsj - boxl*nint(dsj/boxl)
               hess_afed(i,j) = hess_afed(i,j) &
     &             - 2.d0 * alpha(i,l) * delta(i,j) * v &
     &             + 4.d0 * alpha(i,l) * alpha(j,l) * dsi * dsj * v
            end do
            end do

         end do

      end do

      fenergy_afed = fenergy_afed / dble(nbead)

      f_afed(:) = f_afed(:) / dble(nbead)

      hess_afed(:,:) = hess_afed(:,:) / dble(nbead)

      return
      end





!***********************************************************************
      subroutine toymodel_init
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use cons_variables, only : ncons

      use toymodel_variables, only : req, alpha, ng, boxl

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, l

      real(8) :: ranf1

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialization                                             */
!-----------------------------------------------------------------------

      if ( iset .eq. 0 ) then

         if ( .not. allocated( req ) ) &
     &      allocate( req(ncons,ng) )
         if ( .not. allocated( alpha ) ) &
     &      allocate( alpha(ncons,ng) )

         do l = 1, ng
         do i = 1, ncons
            req(i,l) = boxl*ranf1()
            alpha(i,l) = boxl*ranf1() + 1.d0
         end do
         end do

         iset = 1

      end if

      return
      end





!***********************************************************************
      subroutine toymodel_pes
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : iounit, iounit_tmp, nbead

      use afed_variables, only : fenergy_afed

      use cons_variables, only : rcons, ncons

      use toymodel_variables, only : req, alpha, ng, boxl

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, l, m

      real(8) :: dsi, v

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      call toymodel_init

!-----------------------------------------------------------------------
!     /*   free energy surface                                        */
!-----------------------------------------------------------------------

      open ( iounit_tmp, file = 'toymodel.out' )

      do i = -80, 80

         do j = -80, 80

            rcons(1,:) = dble(i)*0.075d0
            rcons(2,:) = dble(j)*0.075d0

            do k = 1, nbead

               do l = 1, ng

                  v = 1.d0 / dble(ng)

                  do m = 1, ncons
                     dsi = rcons(m,k) - req(m,l)
                     dsi = dsi - boxl*nint(dsi/boxl)
                     v = v * exp( - alpha(m,l) * dsi * dsi )
                  end do

                  fenergy_afed = fenergy_afed + v

               end do

            end do

            fenergy_afed = fenergy_afed / dble(nbead)

            write( iounit_tmp, '(3f10.5)' ) &
     &         rcons(1,1), rcons(2,1), fenergy_afed

         end do

         write( iounit_tmp, '(3f10.5)' )

      end do

      close( iounit_tmp )

      return
      end
