!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     compare analytical and numerical virial
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine testvirial
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   box, boxinv, potential, volume, ux, uy, uz, fdiff, fbox, &
     &   pressure, sigma_ref, tension, iounit, natom, iboundary, iexit

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k

      real(8) :: r1, r2, r3, s1, s2, s3, det3, pot_m, pot_p

      real(8), dimension(3,3) :: box0
      real(8), dimension(3,3) :: boxinv0
      real(8) :: volume0

      real(8), dimension(natom) :: ux0
      real(8), dimension(natom) :: uy0
      real(8), dimension(natom) :: uz0

      real(8), dimension(3,3) :: fbox_ref
      real(8), dimension(3,3) :: fbox_fd

!-----------------------------------------------------------------------
!     /*   error termination for free boundary                        */
!-----------------------------------------------------------------------

      if ( iboundary .eq. 0 ) then

         write( 6, '(a)' ) &
     &     'Error - Virial is valid only for periodic boundary.'

         write( 6, '(a)' )

         call error_handling ( 1, 'subroutine testvirial', 21 )

      end if

!-----------------------------------------------------------------------
!     /*   normal mode position -> cartesian position                 */
!-----------------------------------------------------------------------

      call nm_trans( 0 )

!-----------------------------------------------------------------------
!     /*   get fbox as analytical derivatives                         */
!-----------------------------------------------------------------------

!     /*   calculate potential   */
      call getforce

!     /*   virial to fbox   */
      call getfbox

!     /*   store box matrix   */
      fbox_ref(:,:) = fbox(:,:)

!     /*   add pressure and tension contributions   */
      call addfbox &
     &   ( fbox_ref, volume, pressure, boxinv, box, sigma_ref )

!-----------------------------------------------------------------------
!     /*   get fbox as numerical derivatives                          */
!-----------------------------------------------------------------------

!     /*   store box matrix   */
      box0(:,:) = box(:,:)

!     /*   store coordinates   */
      ux0(:)    = ux(:,1)
      uy0(:)    = uy(:,1)
      uz0(:)    = uz(:,1)

!     /*   inverse box matrix   */
      call inv3 ( box0, boxinv0 )

!     /*   volume   */
      volume0 = det3( box0 )

!-----------------------------------------------------------------------

      do i = 1, 3
      do j = 1, 3

!        /*   back to original box matrix   */
         box(:,:)  = box0(:,:)

!        /*   positive shift box matrix   */
         box(i,j)  = box0(i,j) + fdiff

!        /*   shift box volume   */
         volume    = det3( box )

!        /*   inverse matrix of box   */
         call inv3 ( box, boxinv )

!        /*   shift centroid positions   */

         do k = 1, natom

            r1 = ux0(k)
            r2 = uy0(k)
            r3 = uz0(k)

            s1 = boxinv0(1,1)*r1 + boxinv0(1,2)*r2 + boxinv0(1,3)*r3
            s2 = boxinv0(2,1)*r1 + boxinv0(2,2)*r2 + boxinv0(2,3)*r3
            s3 = boxinv0(3,1)*r1 + boxinv0(3,2)*r2 + boxinv0(3,3)*r3

            ux(k,1) = box(1,1)*s1 + box(1,2)*s2 + box(1,3)*s3
            uy(k,1) = box(2,1)*s1 + box(2,2)*s2 + box(2,3)*s3
            uz(k,1) = box(3,1)*s1 + box(3,2)*s2 + box(3,3)*s3

         end do

!        /*   normal mode position -> cartesian position   */
         call nm_trans( 0 )

!        /*   calculate potential   */
         call getforce

!        /*   potential with positive shift   */
         pot_p = potential

!        /*   add pressure and tension contribution   */
         call addebox &
     &      ( pot_p, pressure, volume, boxinv0, volume0, box, tension )

!-----------------------------------------------------------------------

!        /*   back to original box matrix   */
         box(:,:)  = box0(:,:)

!        /*   negative shift box matrix   */
         box(i,j)  = box0(i,j) - fdiff

!        /*   box volume   */
         volume    = det3( box )

!        /*   inverse matrix of box   */
         call inv3 ( box, boxinv )

!        /*   shift centroid positions   */

         do k = 1, natom

            r1 = ux0(k)
            r2 = uy0(k)
            r3 = uz0(k)

            s1 = boxinv0(1,1)*r1 + boxinv0(1,2)*r2 + boxinv0(1,3)*r3
            s2 = boxinv0(2,1)*r1 + boxinv0(2,2)*r2 + boxinv0(2,3)*r3
            s3 = boxinv0(3,1)*r1 + boxinv0(3,2)*r2 + boxinv0(3,3)*r3

            ux(k,1) = box(1,1)*s1 + box(1,2)*s2 + box(1,3)*s3
            uy(k,1) = box(2,1)*s1 + box(2,2)*s2 + box(2,3)*s3
            uz(k,1) = box(3,1)*s1 + box(3,2)*s2 + box(3,3)*s3

         end do

!        /*   normal mode position -> cartesian position   */
         call nm_trans( 0 )

!        /*   calculate potential   */
         call getforce

!        /*   potential with negative shift   */
         pot_m = potential

!        /*   add pressure and tension contribution   */
         call addebox &
     &      ( pot_m, pressure, volume, boxinv0, volume0, box, tension )

!        /*   finite-difference value of potential   */
         fbox_fd(i,j) = - (pot_p - pot_m)/(2.d0*fdiff)

!-----------------------------------------------------------------------

         call standard_testvirial( i, j, fbox_ref, fbox_fd )

!-----------------------------------------------------------------------

!        /*   exit if `exit.dat' exists   */
         call softexit
         if ( iexit .eq. 1 ) go to 100

!-----------------------------------------------------------------------

      end do
      end do

  100 continue

      return
      end
