!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Jan 23, 2025 by M. Shiga
!      Description:     compare analytical and numerical forces
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine testforce_XMPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   fdiff, iounit_std, x, y, z, pot, fx, fy, fz, pux, puy, puz, &
     &   fux_ref, fuy_ref, fuz_ref, iexit, natom, nbead

      use XMPI_variables, only : &
     &   jstart_bead, jend_bead, jstart_atom, jend_atom

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, m

      real(8), dimension(nbead) :: pot_m, pot_p

!-----------------------------------------------------------------------
!     /*   preparation of parallel MPI                                */
!-----------------------------------------------------------------------

      call prep_XMPI

!-----------------------------------------------------------------------
!     /*   normal mode position -> Cartesian position                 */
!-----------------------------------------------------------------------

      call nm_trans_MPI( 0 )

!-----------------------------------------------------------------------
!     /*   force calculation                                          */
!-----------------------------------------------------------------------

      call getforce_XMPI

!-----------------------------------------------------------------------
!     /*   save force                                                 */
!-----------------------------------------------------------------------

      do m = jstart_bead, jend_bead
      do i = jstart_atom, jend_atom
         fux_ref(i,m)  = fx(i,m) * dble(nbead)
         fuy_ref(i,m)  = fy(i,m) * dble(nbead)
         fuz_ref(i,m)  = fz(i,m) * dble(nbead)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   start finite difference                                    */
!-----------------------------------------------------------------------

      do i = 1, natom

!-----------------------------------------------------------------------
!        /*   shift x                                                 */
!-----------------------------------------------------------------------

         do m = jstart_bead, jend_bead
            x(i,m) = x(i,m) - fdiff
         end do 

         call getforce_XMPI

         do m = 1, nbead
            pot_m(m)  = pot(m)
         end do

         do m = jstart_bead, jend_bead
            x(i,m) = x(i,m) + 2.d0*fdiff
         end do

         call getforce_XMPI

         do m = 1, nbead
            pot_p(m)  = pot(m)
         end do

         do m = jstart_bead, jend_bead
            x(i,m) = x(i,m) - fdiff
         end do

         do m = 1, nbead
            pux(i,m)  = - ( pot_p(m) - pot_m(m) ) /(2.d0*fdiff)
         end do

!-----------------------------------------------------------------------
!        /*   shift y                                                 */
!-----------------------------------------------------------------------

         do m = jstart_bead, jend_bead
            y(i,m) = y(i,m) - fdiff
         end do

         call getforce_XMPI

         do m = 1, nbead
            pot_m(m)  = pot(m)
         end do

         do m = jstart_bead, jend_bead
            y(i,m) = y(i,m) + 2.d0*fdiff
         end do

         call getforce_XMPI

         do m = 1, nbead
            pot_p(m)  = pot(m)
         end do

         do m = jstart_bead, jend_bead
            y(i,m) = y(i,m) - fdiff
         end do

         do m = 1, nbead
            puy(i,m)  = - ( pot_p(m) - pot_m(m) ) /(2.d0*fdiff)
         end do

!-----------------------------------------------------------------------
!        /*   shift z                                                 */
!-----------------------------------------------------------------------

         do m = jstart_bead, jend_bead
            z(i,m) = z(i,m) - fdiff
         end do

         call getforce_XMPI

         do m = 1, nbead
            pot_m(m)  = pot(m)
         end do

         do m = jstart_bead, jend_bead
            z(i,m) = z(i,m) + 2.d0*fdiff
         end do

         call getforce_XMPI

         do m = 1, nbead
            pot_p(m)  = pot(m)
         end do

         do m = jstart_bead, jend_bead
            z(i,m) = z(i,m) - fdiff
         end do

         do m = 1, nbead
            puz(i,m)  = - ( pot_p(m) - pot_m(m) ) /(2.d0*fdiff)
         end do

!-----------------------------------------------------------------------
!        /*   output                                                  */
!-----------------------------------------------------------------------

         call standard_testforce_XMPI( i )

!-----------------------------------------------------------------------
!        /*   exit if `exit.dat' exists                               */
!-----------------------------------------------------------------------

         call softexit_MPI
         if ( iexit .eq. 1 ) exit

!-----------------------------------------------------------------------
!     /*   finite difference loop                                     */
!-----------------------------------------------------------------------

      end do

      call my_mpi_barrier

!-----------------------------------------------------------------------
!     /*   end of routine                                             */
!-----------------------------------------------------------------------

      return
      end
