!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga, B. Thomsen
!      Last updated:    Jul 30, 2019 by M. Shiga
!      Description:     static calculation
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine static_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   au_energy, avogadro, x, y, z, vx, vy, vz, pot, fx, fy, fz, &
     &   vux, vuy, vuz, ux, uy, uz, dipx, dipy, dipz, vir, iprint_minfo, &
     &   natom, nbead, iounit, istep, mbox, iounit_trj, myrank

      implicit none

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      integer :: i, j

      real(8) :: fxan, fyan, fzan

      character(len=80) :: char_line

      real(8) :: har2kcal, har2kj

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      istep   = 0

      vx(:,:) = 0.d0
      vy(:,:) = 0.d0
      vz(:,:) = 0.d0
      
!     // Check if we are printing a .minfo file
      call read_int1_MPI ( iprint_minfo, '<iprint_minfo>', 14, iounit )
!-----------------------------------------------------------------------
!     /*   normal mode position -> Cartesian position                 */
!-----------------------------------------------------------------------

      call nm_trans_MPI(0)

!-----------------------------------------------------------------------
!     /*   calculate potential energy and forces                      */
!-----------------------------------------------------------------------

      call getforce_MPI

!-----------------------------------------------------------------------
!     /*   master process only                                        */
!-----------------------------------------------------------------------

      if ( myrank .ne. 0 ) return

!-----------------------------------------------------------------------
!     /*   print                                                      */
!-----------------------------------------------------------------------

!     //   from hartree to kcal/mol
      har2kcal = au_energy / 1000.d0 * avogadro / 4.184d0

!     //   from hartree to kcal/mol
      har2kj   = au_energy / 1000.d0 * avogadro

      write( 6, '(a)' ) &
     &   '==========================================================' &
     &    // '===================='
      write( 6, '(a)' ) &
     &   '                           potential energy values'
      write( 6, '(a)' ) &
     &   '  bead                 hartree' &
     &    // '                kcal/mol                  kJ/mol'
      write( 6, '(a)' ) &
     &   '----------------------------------------------------------' &
     &    // '--------------------'

      do i = 1, nbead
         write( 6, '(i6,f24.8,2f24.6)' ) i, &
     &      pot(i), pot(i)*har2kcal, pot(i)*har2kj
      end do

      write( 6, '(a)' ) &
     &   '----------------------------------------------------------' &
     &    // '--------------------'


!-----------------------------------------------------------------------
!     /*   output forces                                              */
!-----------------------------------------------------------------------

      char_line = 'forces.out'

      call ioforce( 1, char_line, 10, iounit, &
     &               pot, fx, fy, fz, natom, nbead )

!-----------------------------------------------------------------------
!     /*   output dipoles                                             */
!-----------------------------------------------------------------------

      char_line = 'dipole.out'

      call iodipole( 1, char_line, 10, iounit, &
     &                dipx, dipy, dipz, nbead )

!-----------------------------------------------------------------------
!     /*   output virials                                             */
!-----------------------------------------------------------------------

      char_line = 'virial.out'

      call iovirial( 1, char_line, 10, iounit, vir )

!-----------------------------------------------------------------------
!     /*    output minfo file if requested                            */
!-----------------------------------------------------------------------

      if ( myrank .eq. 0 ) then
         if ( iprint_minfo .ne. -1 ) then
            call print_minfo ( 1, 0 )
         end if
      end if

!-----------------------------------------------------------------------
!     /*   output restart files                                       */
!-----------------------------------------------------------------------

      vux(:,:) = 0.d0
      vuy(:,:) = 0.d0
      vuz(:,:) = 0.d0

      open ( iounit,  file = 'geometry.ini' )

         do j = 1, nbead
         do i = 1, natom

            write( iounit, '(i8,6e24.16,3i4)' ) &
     &         istep,  ux(i,j),  uy(i,j),  uz(i,j), &
     &                vux(i,j), vuy(i,j), vuz(i,j), mbox(1:3,i,j)

         end do
         end do

      close( iounit )

!-----------------------------------------------------------------------
!     /*   output restart files                                       */
!-----------------------------------------------------------------------

      open ( iounit,  file = 'step.ini' )

          write( iounit, '(i8)' ) istep

      close( iounit )

!-----------------------------------------------------------------------
!     /*   output trajectory file                                     */
!-----------------------------------------------------------------------

      open ( iounit_trj,  file = 'trj.out' )

         do j = 1, nbead
         do i = 1, natom

            fxan = fx(i,j)*nbead
            fyan = fy(i,j)*nbead
            fzan = fz(i,j)*nbead

            write( iounit_trj, '(i8,10e24.16)' ) &
     &         istep,  x(i,j),  y(i,j),  z(i,j), &
     &                vx(i,j), vy(i,j), vz(i,j), &
     &                fxan, fyan, fzan, pot(j)

         end do
         end do

      close( iounit_trj )

!-----------------------------------------------------------------------
!     /*   end of the routine                                         */
!-----------------------------------------------------------------------

      return
      end
