!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Apr 10, 2022 by M. Shiga
!      Description:     standard output of Tully fewest switches
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine standard_tfs_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   temp, boltz, ekin, vx, vy, vz, physmass, potential, &
     &   hamiltonian, istep, iounit, natom, iounit_std, &
     &   iounit_tfs, iounit_nac, myrank, ndof, nbead

      use multistate_variables, only : &
     &   pop_state, pop_sum, branch_ratio, cstate, vstate, &
     &   gxstate, gystate, gzstate, dipxstate, dipystate, dipzstate, &
     &   dxstate, dystate, dzstate, iprint_tfs, iprint_nac, &
     &   nstate, istate_tfs

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: itest, i, j, k, l

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      call standard_init_MPI( iset )

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( iset .eq. 1 ) then

         call read_int1_MPI ( iprint_tfs, '<iprint_tfs>', 12, iounit )

         if ( ( myrank .eq. 0 ) .and. ( iprint_tfs .gt. 0 ) ) then

         call testfile ( 'tfs.out', 7, itest )

         if ( itest .eq. 1 ) then

            open ( iounit_tfs, file = 'tfs.out' )

            write( iounit_tfs, '(a)' ) &
     &      '==========================================' // &
     &      '================================'
            write( iounit_tfs, '(a)' ) &
     &      '    step  bead   s occ     pop/s   ratio/s' // &
     &      '     potential/s   potential/occ'
            write( iounit_tfs, '(a)' ) &
     &      '------------------------------------------' // &
     &      '--------------------------------'

            close( iounit_tfs )

         end if

         end if

         call read_int1_MPI ( iprint_nac, '<iprint_nac>', 12, iounit )

         iset = 2

      end if

!-----------------------------------------------------------------------
!     /*   kinetic energy                                             */
!-----------------------------------------------------------------------

      ekin = 0.d0

      do l = 1, nbead
      do k = 1, natom
         ekin = ekin + 0.5d0*physmass(k)*vx(k,l)*vx(k,l) &
     &               + 0.5d0*physmass(k)*vy(k,l)*vy(k,l) &
     &               + 0.5d0*physmass(k)*vz(k,l)*vz(k,l)
      end do
      end do

      ekin = ekin / dble(nbead)

      temp = 2.d0*ekin * dble(nbead) / dble(ndof) / boltz

!-----------------------------------------------------------------------
!     /*   total energy                                               */
!-----------------------------------------------------------------------

      hamiltonian = ekin + potential

!-----------------------------------------------------------------------
!     /*   population of states                                       */
!-----------------------------------------------------------------------

      do l = 1, nbead
      do i = 1, nstate
         pop_state(i,l) = dreal(dconjg(cstate(i,l))*cstate(i,l))
      end do
      end do

!-----------------------------------------------------------------------
!     /*   sum of population ( = 1.0 )                                */
!-----------------------------------------------------------------------

      pop_sum(:) = 0.d0

      do l = 1, nbead
      do i = 1, nstate
         pop_sum(l) = pop_sum(l) + pop_state(i,l)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   branching ratio                                            */
!-----------------------------------------------------------------------

      branch_ratio(:) = 0.d0

      do l = 1, nbead
         branch_ratio(istate_tfs(l)) &
     &      = branch_ratio(istate_tfs(l)) + 1.d0
      end do

      branch_ratio(:) = branch_ratio(:) / nbead

!-----------------------------------------------------------------------
!     /*   print energy                                               */
!-----------------------------------------------------------------------

      call standard_output_MPI

!-----------------------------------------------------------------------
!     /*   print population                                           */
!-----------------------------------------------------------------------

      if ( ( myrank .eq. 0 ) .and. ( iprint_tfs .gt. 0 ) ) then
      if ( mod(istep,iprint_tfs) .eq. 0 ) then

         open( iounit_tfs, file = 'tfs.out', access = 'append' )

         do l = 1, nbead
         do i = 1, nstate
            write(iounit_tfs,'(i8,i6,2i4,2f10.6,2f16.8)') &
     &         istep, l, i, istate_tfs(l), &
     &         pop_state(i,l), branch_ratio(i), &
     &         vstate(i,i,l), vstate(istate_tfs(l),istate_tfs(l),l)
         end do
         end do

         close( iounit_tfs )

      end if
      end if

!-----------------------------------------------------------------------
!     /*   print all including nonadiabatic coupling elements         */
!-----------------------------------------------------------------------

      if ( ( myrank .eq. 0 ) .and. ( iprint_nac .gt. 0 ) ) then
      if ( mod(istep,iprint_nac) .eq. 0 ) then

         open( iounit_nac, file = 'nac.out', access = 'append' )

         do l = 1, nbead

!           /*   state coefficient   */

            do j = 1, nstate
               write( iounit_nac, '(2e24.16)' ) cstate(j,l)
            end do

!           /*   potential   */

            do k = 1, nstate
            do j = 1, nstate
               write( iounit_nac, '(e24.16)' ) vstate(j,k,l)
            end do
            end do

!           /*   gradient   */

            do k = 1, nstate
            do j = 1, nstate
            do i = 1, natom
               write( iounit_nac, '(3e24.16)' ) &
     &            gxstate(j,k,i,l), gystate(j,k,i,l), gzstate(j,k,i,l)
            end do
            end do
            end do

!           /*   dipole moment   */

            do k = 1, nstate
              write( iounit_nac, '(3e24.16)' ) &
     &           dipxstate(k,l), dipystate(k,l), dipzstate(k,l)
            end do

!           /*   nonadiabatic coupling matrix   */

            do k = 1, nstate
            do j = 1, nstate
            do i = 1, natom
               write( iounit_nac, '(3e24.16)' ) &
     &            dxstate(j,k,i,l), dystate(j,k,i,l), dzstate(j,k,i,l)
            end do
            end do
            end do

         end do

         close( iounit_nac )

      end if
      end if

      return
      end
