!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Apr 10, 2022 by M. Shiga
!      Description:     polymers standard output
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine standard_polymers_atom_MPI
!***********************************************************************
!=======================================================================
!
!     calculate ``Hamiltonian'' and ``temperature''
!     thermostat type III.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   temp, ekin, boltz, qkin, ekin, potential, hamiltonian, gkt, &
     &   hamiltonian_sys, omega_p2, dnmmass, ux, uy, uz, qmass_cent, &
     &   xbath_cent, ybath_cent, zbath_cent, qmass, &
     &   vxbath_cent, vybath_cent, vzbath_cent, xbath, ybath, zbath, &
     &   vxbath, vybath, vzbath, ebath_cent, ebath_mode, char_date, &
     &   ndof, iprint_std, iounit, iounit_std, nbead, natom, istep, &
     &   ncolor, nnhc, myrank, myrank_world

      use polymers_variables, only : jpoly, npoly

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: imode, iatom, i, j, k, m

      real(8) :: factqk

      integer, save :: iset = 0, jset = 0

      character(len=3) :: char_num

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( iset .le. jpoly ) then

         call read_int1_MPI ( iprint_std, '<iprint_std>', 12, iounit )

         if ( iprint_std .le. 0 ) return

         if ( myrank .eq. 0 ) then

!           /*   polymer number  */
            call int3_to_char( jpoly, char_num )

!           /*   open file   */
            open ( iounit_std, &
     &             file = 'poly.' // char_num // '/standard.out', &
     &             access = 'append' )

            write( iounit_std, '(a)' ) &
     &      '====================================================' // &
     &      '=========================='
            write( iounit_std, '(a)' ) &
     &      '    step     energy [au]  potential [au]  temp [K]  ' // &
     &      'wall clock time           '
            write( iounit_std, '(a)' ) &
     &      '----------------------------------------------------' // &
     &      '--------------------------'

!           /*   close file   */
            close( iounit_std )

         end if

         iset = jpoly + 1

      end if

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( jset .eq. 0 ) then

         call read_int1_MPI ( iprint_std, '<iprint_std>', 12, iounit )

         if ( iprint_std .le. 0 ) return

         if ( myrank_world .eq. 0 ) then

            write( 6, '(a)' ) &
     &      '====================================================' // &
     &      '=========================='
            write( 6, '(a)' ) &
     &      'num   step    energy [au] potential [au]  temp [K]  ' // &
     &      'wall clock time           '
            write( 6, '(a)' ) &
     &      '----------------------------------------------------' // &
     &      '--------------------------'

            flush( 6 )

         end if

         jset = 1

      end if

      if ( iprint_std .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   temp  =  instantaneous temperature                         */
!-----------------------------------------------------------------------

!     /*   calculate ekin =  fictitious kinetic energy   */
      call kinetic_energy

      temp = 2.d0*ekin/dble(ndof)/boltz

!-----------------------------------------------------------------------
!     /*   qkin  =  harmonic potential                                */
!-----------------------------------------------------------------------

      qkin = 0.d0

      do imode = 2, nbead
      do iatom = 1, natom
         factqk = 0.5d0*dnmmass(iatom,imode)*omega_p2
         qkin = qkin &
     &        + factqk*ux(iatom,imode)*ux(iatom,imode) &
     &        + factqk*uy(iatom,imode)*uy(iatom,imode) &
     &        + factqk*uz(iatom,imode)*uz(iatom,imode)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian_sys  =  system Hamiltonian                     */
!-----------------------------------------------------------------------

      hamiltonian_sys = ekin + qkin + potential

!-----------------------------------------------------------------------
!     /*   ebath_cent  =  thermostats attached to centroid            */
!-----------------------------------------------------------------------

      ebath_cent = 0.d0

      do m = 1, ncolor
      do i = 1, nnhc
      do j = 1, natom
         ebath_cent = ebath_cent &
     &   + 0.5d0*qmass_cent(i,m)*vxbath_cent(j,i,m)*vxbath_cent(j,i,m) &
     &   + 0.5d0*qmass_cent(i,m)*vybath_cent(j,i,m)*vybath_cent(j,i,m) &
     &   + 0.5d0*qmass_cent(i,m)*vzbath_cent(j,i,m)*vzbath_cent(j,i,m) &
     &   + gkt*xbath_cent(j,i,m) &
     &   + gkt*ybath_cent(j,i,m) &
     &   + gkt*zbath_cent(j,i,m)
      end do
      end do
      end do

!-----------------------------------------------------------------------
!     /*   ebath_mode  =  thermostats attached to non-centroid        */
!-----------------------------------------------------------------------

      ebath_mode = 0.d0

      do i = 2, nbead
         do j = 1, nnhc
         do k = 1, natom
            ebath_mode = ebath_mode &
     &           + 0.5d0*qmass(i)*vxbath(k,j,i)*vxbath(k,j,i) &
     &           + 0.5d0*qmass(i)*vybath(k,j,i)*vybath(k,j,i) &
     &           + 0.5d0*qmass(i)*vzbath(k,j,i)*vzbath(k,j,i) &
     &           + gkt*xbath(k,j,i) &
     &           + gkt*ybath(k,j,i) &
     &           + gkt*zbath(k,j,i)
         end do
         end do
      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian =  total Hamiltonian                           */
!-----------------------------------------------------------------------

      hamiltonian = hamiltonian_sys + ebath_mode + ebath_cent

!-----------------------------------------------------------------------
!     /*   output                                                     */
!-----------------------------------------------------------------------

!     /*   periodic output   */
      if ( mod(istep,iprint_std) .eq. 0 ) then

!        /*   master rank   */
         if ( myrank .eq. 0 ) then

!           /*   polymer number  */
            call int3_to_char( jpoly, char_num )

!           /*   open file   */
            open ( iounit_std, &
     &             file = 'poly.' // char_num // '/standard.out', &
     &             access = 'append' )

!           /*   wall clock time   */
            call getdate

!           /*   output   */
            write( iounit_std, '(i8,2f16.8,f10.2,2x,a26)' ) &
     &         istep, hamiltonian, potential, temp, char_date

!           /*   close file   */
            close( iounit_std )

!        /*   master rank   */
         end if

!        /*   loop of polymers   */
         do j = 1, npoly

!           /*   polymer by polymer   */
            if ( j .eq. jpoly ) then

!              /*   master rank   */
               if ( myrank .eq. 0 ) then

!                 /*   polymer number  */
                  call int3_to_char( jpoly, char_num )

!                 /*   wall clock time   */
                  call getdate

!                 /*   output   */
                  write( 6, '(i3,i7,2f15.7,f10.2,2x,a26)' ) &
     &            jpoly, istep, hamiltonian, potential, temp, char_date

!                 /*   make sure output   */
                  flush( 6 )

!              /*   master rank   */
               end if

!           /*   polymer by polymer  */
            end if

!           /*   wait   */
            call my_mpi_barrier_world

!        /*   loop of polymers   */
         end do

         flush( 6 )

!     /*   periodic output   */
      end if

      return
      end





!***********************************************************************
      subroutine standard_polymers_cons_MPI
!***********************************************************************
!=======================================================================
!
!     calculate ``Hamiltonian'' and ``temperature''
!     thermostat type III.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   temp, ekin, boltz, qkin, ekin, potential, hamiltonian, gkt, &
     &   hamiltonian_sys, omega_p2, dnmmass, ux, uy, uz, qmass_cent, &
     &   xbath_cent, ybath_cent, zbath_cent, qmass, &
     &   vxbath_cent, vybath_cent, vzbath_cent, xbath, ybath, zbath, &
     &   vxbath, vybath, vzbath, ebath_cent, ebath_mode, char_date, &
     &   ndof, iprint_std, iounit, iounit_std, nbead, natom, istep, &
     &   ncolor, nnhc, myrank, myrank_world

      use polymers_variables, only : jpoly, npoly

      use cons_variables, only : pot_ref_cons, potential_ref_cons

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: imode, iatom, i, j, k, m

      real(8) :: factqk

      integer, save :: iset = 0, jset = 0

      character(len=3) :: char_num

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( iset .le. jpoly ) then

         call read_int1_MPI ( iprint_std, '<iprint_std>', 12, iounit )

         if ( iprint_std .le. 0 ) return

         if ( myrank .eq. 0 ) then

!           /*   polymer number  */
            call int3_to_char( jpoly, char_num )

!           /*   open file   */
            open ( iounit_std, &
     &             file = 'poly.' // char_num // '/standard.out', &
     &             access = 'append' )

            write( iounit_std, '(a)' ) &
     &      '====================================================' // &
     &      '=========================='
            write( iounit_std, '(a)' ) &
     &      '    step     energy [au]  potential [au]  temp [K]  ' // &
     &      'wall clock time           '
            write( iounit_std, '(a)' ) &
     &      '----------------------------------------------------' // &
     &      '--------------------------'

!           /*   close file   */
            close( iounit_std )

         end if

         iset = jpoly + 1

      end if

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( jset .eq. 0 ) then

         call read_int1_MPI ( iprint_std, '<iprint_std>', 12, iounit )

         if ( iprint_std .le. 0 ) return

         if ( myrank_world .eq. 0 ) then

            write(         6,'(a)') &
     &      '====================================================' // &
     &      '=========================='
            write(         6,'(a)') &
     &      'num   step    energy [au] potential [au]  temp [K]  ' // &
     &      'wall clock time           '
            write(         6,'(a)') &
     &      '----------------------------------------------------' // &
     &      '--------------------------'

            flush( 6 )

         end if

         jset = 1

      end if

      if ( iprint_std .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   temp  =  instantaneous temperature                         */
!-----------------------------------------------------------------------

!     /*   calculate ekin =  fictitious kinetic energy   */
      call kinetic_energy

      temp = 2.d0*ekin/dble(ndof)/boltz

!-----------------------------------------------------------------------
!     /*   qkin  =  harmonic potential                                */
!-----------------------------------------------------------------------

      qkin = 0.d0

      do imode = 2, nbead
      do iatom = 1, natom
         factqk = 0.5d0*dnmmass(iatom,imode)*omega_p2
         qkin = qkin &
     &        + factqk*ux(iatom,imode)*ux(iatom,imode) &
     &        + factqk*uy(iatom,imode)*uy(iatom,imode) &
     &        + factqk*uz(iatom,imode)*uz(iatom,imode)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian_sys  =  system Hamiltonian                     */
!-----------------------------------------------------------------------

      hamiltonian_sys = ekin + qkin + potential

!-----------------------------------------------------------------------
!     /*   ebath_cent  =  thermostats attached to centroid            */
!-----------------------------------------------------------------------

      ebath_cent = 0.d0

      do m = 1, ncolor
      do i = 1, nnhc
      do j = 1, natom
         ebath_cent = ebath_cent &
     &   + 0.5d0*qmass_cent(i,m)*vxbath_cent(j,i,m)*vxbath_cent(j,i,m) &
     &   + 0.5d0*qmass_cent(i,m)*vybath_cent(j,i,m)*vybath_cent(j,i,m) &
     &   + 0.5d0*qmass_cent(i,m)*vzbath_cent(j,i,m)*vzbath_cent(j,i,m) &
     &   + gkt*xbath_cent(j,i,m) &
     &   + gkt*ybath_cent(j,i,m) &
     &   + gkt*zbath_cent(j,i,m)
      end do
      end do
      end do

!-----------------------------------------------------------------------
!     /*   ebath_mode  =  thermostats attached to non-centroid        */
!-----------------------------------------------------------------------

      ebath_mode = 0.d0

      do i = 2, nbead
         do j = 1, nnhc
         do k = 1, natom
            ebath_mode = ebath_mode &
     &           + 0.5d0*qmass(i)*vxbath(k,j,i)*vxbath(k,j,i) &
     &           + 0.5d0*qmass(i)*vybath(k,j,i)*vybath(k,j,i) &
     &           + 0.5d0*qmass(i)*vzbath(k,j,i)*vzbath(k,j,i) &
     &           + gkt*xbath(k,j,i) &
     &           + gkt*ybath(k,j,i) &
     &           + gkt*zbath(k,j,i)
         end do
         end do
      end do

!-----------------------------------------------------------------------
!     /*   energy of constraints                                      */
!-----------------------------------------------------------------------

      potential_ref_cons = 0.d0

      do i = 1, nbead
         potential_ref_cons = potential_ref_cons + pot_ref_cons(i)
      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian =  total Hamiltonian                           */
!-----------------------------------------------------------------------

      hamiltonian = hamiltonian_sys + potential_ref_cons &
     &            + ebath_mode + ebath_cent

!-----------------------------------------------------------------------
!     /*   output                                                     */
!-----------------------------------------------------------------------

!     /*   periodic output   */
      if ( mod(istep,iprint_std) .eq. 0 ) then

!        /*   master rank   */
         if ( myrank .eq. 0 ) then

!           /*   polymer number  */
            call int3_to_char( jpoly, char_num )

!           /*   open file   */
            open ( iounit_std, &
     &             file = 'poly.' // char_num // '/standard.out', &
     &             access = 'append' )

!           /*   wall clock time   */
            call getdate

!           /*   output   */
            write( iounit_std,'(i8,2f16.8,f10.2,2x,a26)' ) &
     &         istep, hamiltonian, potential, temp, char_date

!           /*   close file   */
            close( iounit_std )

!        /*   master rank   */
         end if

!        /*   loop of polymers   */
         do j = 1, npoly

!           /*   polymer by polymer   */
            if ( j .eq. jpoly ) then

!              /*   master rank   */
               if ( myrank .eq. 0 ) then

!                 /*   polymer number  */
                  call int3_to_char( jpoly, char_num )

!                 /*   wall clock time   */
                  call getdate

!                 /*   output   */
                  write( 6, '(i3,i7,2f15.7,f10.2,2x,a26)' ) &
     &            jpoly, istep, hamiltonian, potential, temp, char_date

!              /*   master rank   */
               end if

!           /*   polymer by polymer  */
            end if

!           /*   wait   */
            call my_mpi_barrier_world

!        /*   loop of polymers   */
         end do

         flush( 6 )

!     /*   periodic output   */
      end if

      return
      end
