!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Apr 10, 2022 by M. Shiga
!      Description:     standard output of NtH ensemble
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine standard_nth_pp
!***********************************************************************
!=======================================================================
!
!     calculate ``Hamiltonian'' and ``temperature''
!     no thermostat.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   temp, ekin, boltz, qkin, dnmmass, ux, uy, uz, vir_iso, box, &
     &   potential, hamiltonian, hamiltonian_sys, fictmass, pres_iso, &
     &   ekin, vux, vuy, vuz, omega_p2, vir, ebaro, boxmass, vbox, &
     &   volume, gbox, strain, boxinv_ref, tension, volume_ref, &
     &   nbead, natom, ndof, iprint_std

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: imode, iatom, i, j, k, l

      real(8) :: factqk, dkin

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      call standard_init( iset )

      if ( iprint_std .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   temp  =  instantaneous temperature                         */
!-----------------------------------------------------------------------

!     /*   calculate ekin =  fictitious kinetic energy   */
      call kinetic_energy

      temp = 2.d0*ekin/dble(ndof)/boltz

!-----------------------------------------------------------------------
!     /*   qkin  =  harmonic potential                                */
!-----------------------------------------------------------------------

      qkin = 0.d0

      do imode = 2, nbead
      do iatom = 1, natom
         factqk = 0.5d0*dnmmass(iatom,imode)*omega_p2
         qkin = qkin &
     &        + factqk*ux(iatom,imode)*ux(iatom,imode) &
     &        + factqk*uy(iatom,imode)*uy(iatom,imode) &
     &        + factqk*uz(iatom,imode)*uz(iatom,imode)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian_sys  =  system Hamiltonian                     */
!-----------------------------------------------------------------------

      hamiltonian_sys = ekin + qkin + potential

!-----------------------------------------------------------------------
!     /*   metric matrix                                              */
!-----------------------------------------------------------------------

      gbox(:,:) = 0.d0

      do k = 1, 3
      do j = 1, 3
      do i = 1, 3
         gbox(i,j) = gbox(i,j) + box(k,i)*box(k,j)
      end do
      end do
      end do

!-----------------------------------------------------------------------
!     /*   strain matrix                                              */
!-----------------------------------------------------------------------

      strain(:,:) = 0.d0

      do l = 1, 3
      do k = 1, 3
      do j = 1, 3
      do i = 1, 3

         strain(i,j) = strain(i,j) &
     &               + boxinv_ref(k,i)*gbox(k,l)*boxinv_ref(l,j)

      end do
      end do
      end do
      end do

      do i = 1, 3
         strain(i,i) = strain(i,i) - 1.d0
      end do

      strain(:,:) = 0.5d0*strain(:,:)

!-----------------------------------------------------------------------
!     /*   ebaro =  barostats                                         */
!-----------------------------------------------------------------------

      ebaro = 0.d0

      do j = 1, 3
      do i = 1, 3
         ebaro = ebaro - volume_ref*tension(i,j)*strain(j,i)
      end do
      end do

      do j = 1, 3
      do i = 1, 3
         ebaro = ebaro + 0.5d0*boxmass(i,j)*vbox(i,j)*vbox(i,j)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian =  total Hamiltonian                           */
!-----------------------------------------------------------------------

      hamiltonian = hamiltonian_sys + ebaro

!-----------------------------------------------------------------------
!     /*   kinetic energy of centroids                                */
!-----------------------------------------------------------------------

      dkin = 0.d0

      do i = 1, natom

         dkin = dkin + fictmass(i,1)*vux(i,1)*vux(i,1) &
     &               + fictmass(i,1)*vuy(i,1)*vuy(i,1) &
     &               + fictmass(i,1)*vuz(i,1)*vuz(i,1)

      end do

      dkin = 0.5d0*dkin

!-----------------------------------------------------------------------
!     /*   isotropic virial                                           */
!-----------------------------------------------------------------------

      vir_iso = vir(1,1) + vir(2,2) + vir(3,3)

!-----------------------------------------------------------------------
!     /*   isotropic pressure                                         */
!-----------------------------------------------------------------------

      pres_iso = ( 2.d0*dkin + vir_iso ) / ( 3.d0 * volume )

!-----------------------------------------------------------------------
!     /*   output                                                     */
!-----------------------------------------------------------------------

      call standard_output

      return
      end

