!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Apr 10, 2022 by M. Shiga
!      Description:     standard output of NPT ensemble
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine standard_npt_c2_MPI
!***********************************************************************
!=======================================================================
!
!     calculate ``Hamiltonian'' and ``temperature''
!     no thermostat.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   temp, ekin, boltz, qkin, dnmmass, ux, uy, uz, pres_iso, &
     &   ekin, potential, hamiltonian, hamiltonian_sys, fictmass, &
     &   vux, vuy, vuz, omega_p2, vir, pres, ebaro, boxmass, vlog, &
     &   pressure, volume, ebath_mode, ebath_cent, qmass, qmass_cent, &
     &   xbath, ybath, zbath, xbath_cent, ybath_cent,qmass_box, &
     &   zbath_cent, vxbath, vybath, vzbath, vxbath_cent, vybath_cent, &
     &   vzbath_cent, ebath_box, vbath_box, rbath_box, gkt, ncolor, &
     &   nnhc, nbead, natom, ndof, iprint_std

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: imode, iatom, i, j, k, m

      real(8) :: factqk

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      call standard_init_MPI( iset )

      if ( iprint_std .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   temp  =  instantaneous temperature                         */
!-----------------------------------------------------------------------

!     /*   calculate ekin =  fictitious kinetic energy   */
      call kinetic_energy

      temp = 2.d0*ekin/dble(ndof)/boltz

!-----------------------------------------------------------------------
!     /*   qkin  =  harmonic potential                                */
!-----------------------------------------------------------------------

      qkin = 0.d0

      do imode = 2, nbead
      do iatom = 1, natom
         factqk = 0.5d0*dnmmass(iatom,imode)*omega_p2
         qkin = qkin &
     &        + factqk*ux(iatom,imode)*ux(iatom,imode) &
     &        + factqk*uy(iatom,imode)*uy(iatom,imode) &
     &        + factqk*uz(iatom,imode)*uz(iatom,imode)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian_sys  =  system Hamiltonian                     */
!-----------------------------------------------------------------------

      hamiltonian_sys = ekin + qkin + potential

!-----------------------------------------------------------------------
!     /*   ebath_cent  =  thermostats attached to centroid            */
!-----------------------------------------------------------------------

      ebath_cent = 0.d0

      do m = 1, ncolor
      do i = 1, nnhc
      do j = 1, natom
         ebath_cent = ebath_cent &
     &   + 0.5d0*qmass_cent(i,m)*vxbath_cent(j,i,m)*vxbath_cent(j,i,m) &
     &   + 0.5d0*qmass_cent(i,m)*vybath_cent(j,i,m)*vybath_cent(j,i,m) &
     &   + 0.5d0*qmass_cent(i,m)*vzbath_cent(j,i,m)*vzbath_cent(j,i,m) &
     &   + gkt*xbath_cent(j,i,m) &
     &   + gkt*ybath_cent(j,i,m) &
     &   + gkt*zbath_cent(j,i,m)
      end do
      end do
      end do

!-----------------------------------------------------------------------
!     /*   ebath_mode  =  thermostats attached to non-centroid        */
!-----------------------------------------------------------------------

      ebath_mode = 0.d0

      do i = 2, nbead
         do j = 1, nnhc
         do k = 1, natom
            ebath_mode = ebath_mode &
     &           + 0.5d0*qmass(i)*vxbath(k,j,i)*vxbath(k,j,i) &
     &           + 0.5d0*qmass(i)*vybath(k,j,i)*vybath(k,j,i) &
     &           + 0.5d0*qmass(i)*vzbath(k,j,i)*vzbath(k,j,i) &
     &           + gkt*xbath(k,j,i) &
     &           + gkt*ybath(k,j,i) &
     &           + gkt*zbath(k,j,i)
         end do
         end do
      end do

!-----------------------------------------------------------------------
!     /*   ebath_box  =  thermostats attached to box                  */
!-----------------------------------------------------------------------

      ebath_box = 0.d0

      do m = 1, ncolor
      do i = 1, nnhc
         ebath_box = ebath_box &
     &             + 0.5d0*qmass_box(i,m)*vbath_box(i,m)*vbath_box(i,m) &
     &             + gkt*rbath_box(i,m)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   ebaro =  barostats                                         */
!-----------------------------------------------------------------------

      ebaro = 0.5d0*boxmass(1,1)*vlog*vlog + pressure*volume

!-----------------------------------------------------------------------
!     /*   hamiltonian =  total Hamiltonian                           */
!-----------------------------------------------------------------------

      hamiltonian = hamiltonian_sys &
     &            + ebath_mode + ebath_cent + ebath_box + ebaro

!-----------------------------------------------------------------------
!     /*   pressure                                                   */
!-----------------------------------------------------------------------

      pres(:,:) =  vir(:,:)

      do i = 1, natom

         pres(1,1) = pres(1,1) + fictmass(i,1)*vux(i,1)*vux(i,1)
         pres(1,2) = pres(1,2) + fictmass(i,1)*vux(i,1)*vuy(i,1)
         pres(1,3) = pres(1,3) + fictmass(i,1)*vux(i,1)*vuz(i,1)
         pres(2,1) = pres(2,1) + fictmass(i,1)*vuy(i,1)*vux(i,1)
         pres(2,2) = pres(2,2) + fictmass(i,1)*vuy(i,1)*vuy(i,1)
         pres(2,3) = pres(2,3) + fictmass(i,1)*vuy(i,1)*vuz(i,1)
         pres(3,1) = pres(3,1) + fictmass(i,1)*vuz(i,1)*vux(i,1)
         pres(3,2) = pres(3,2) + fictmass(i,1)*vuz(i,1)*vuy(i,1)
         pres(3,3) = pres(3,3) + fictmass(i,1)*vuz(i,1)*vuz(i,1)

      end do

      pres(:,:) = pres(:,:) / volume

!-----------------------------------------------------------------------
!     /*   isotropic pressure                                         */
!-----------------------------------------------------------------------

      pres_iso = ( pres(1,1) + pres(2,2) + pres(3,3) ) / 3.d0

!-----------------------------------------------------------------------
!     /*   output                                                     */
!-----------------------------------------------------------------------

      call standard_output_MPI

      return
      end





!***********************************************************************
      subroutine standard_npt_pp_MPI
!***********************************************************************
!=======================================================================
!
!     calculate ``Hamiltonian'' and ``temperature''
!     no thermostat.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   temp, ekin, boltz, qkin, dnmmass, ux, uy, uz, vir_iso, &
     &   potential, hamiltonian, hamiltonian_sys, fictmass, pres_iso, &
     &   ekin, vux, vuy, vuz, omega_p2, vir, ebaro, boxmass, vbox, &
     &   pressure, volume, ebath_mode, ebath_cent, gkt, qmass, &
     &   qmass_cent, xbath, ybath, zbath, xbath_cent, ybath_cent, &
     &   zbath_cent, vxbath, vybath, vzbath, vxbath_cent, vybath_cent, &
     &   vzbath_cent, ebath_box, xbath_box, ybath_box, zbath_box, &
     &   qmass_box, vxbath_box, vybath_box, vzbath_box, ncolor, nnhc, &
     &   nbead, natom, ndof, iprint_std

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: imode, iatom, i, j, k, m

      real(8) :: factqk, dkin

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      call standard_init_MPI( iset )

      if ( iprint_std .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   temp  =  instantaneous temperature                         */
!-----------------------------------------------------------------------

!     /*   calculate ekin =  fictitious kinetic energy   */
      call kinetic_energy

      temp = 2.d0*ekin/dble(ndof)/boltz

!-----------------------------------------------------------------------
!     /*   qkin  =  harmonic potential                                */
!-----------------------------------------------------------------------

      qkin = 0.d0

      do imode = 2, nbead
      do iatom = 1, natom
         factqk = 0.5d0*dnmmass(iatom,imode)*omega_p2
         qkin = qkin &
     &        + factqk*ux(iatom,imode)*ux(iatom,imode) &
     &        + factqk*uy(iatom,imode)*uy(iatom,imode) &
     &        + factqk*uz(iatom,imode)*uz(iatom,imode)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian_sys  =  system Hamiltonian                     */
!-----------------------------------------------------------------------

      hamiltonian_sys = ekin + qkin + potential

!-----------------------------------------------------------------------
!     /*   ebath_cent  =  thermostats attached to centroid            */
!-----------------------------------------------------------------------

      ebath_cent = 0.d0

      do m = 1, ncolor
      do i = 1, nnhc
      do j = 1, natom
         ebath_cent = ebath_cent &
     &   + 0.5d0*qmass_cent(i,m)*vxbath_cent(j,i,m)*vxbath_cent(j,i,m) &
     &   + 0.5d0*qmass_cent(i,m)*vybath_cent(j,i,m)*vybath_cent(j,i,m) &
     &   + 0.5d0*qmass_cent(i,m)*vzbath_cent(j,i,m)*vzbath_cent(j,i,m) &
     &   + gkt*xbath_cent(j,i,m) &
     &   + gkt*ybath_cent(j,i,m) &
     &   + gkt*zbath_cent(j,i,m)
      end do
      end do
      end do

!-----------------------------------------------------------------------
!     /*   ebath_mode  =  thermostats attached to non-centroid        */
!-----------------------------------------------------------------------

      ebath_mode = 0.d0

      do i = 2, nbead
         do j = 1, nnhc
         do k = 1, natom
            ebath_mode = ebath_mode &
     &           + 0.5d0*qmass(i)*vxbath(k,j,i)*vxbath(k,j,i) &
     &           + 0.5d0*qmass(i)*vybath(k,j,i)*vybath(k,j,i) &
     &           + 0.5d0*qmass(i)*vzbath(k,j,i)*vzbath(k,j,i) &
     &           + gkt*xbath(k,j,i) &
     &           + gkt*ybath(k,j,i) &
     &           + gkt*zbath(k,j,i)
         end do
         end do
      end do

!-----------------------------------------------------------------------
!     /*   ebath_box  =  thermostats attached to box                  */
!-----------------------------------------------------------------------

      ebath_box = 0.d0

      do m = 1, ncolor
      do i = 1, nnhc
      do j = 1, 3
         ebath_box = ebath_box &
     &      + 0.5d0*qmass_box(i,m)*vxbath_box(j,i,m)*vxbath_box(j,i,m) &
     &      + 0.5d0*qmass_box(i,m)*vybath_box(j,i,m)*vybath_box(j,i,m) &
     &      + 0.5d0*qmass_box(i,m)*vzbath_box(j,i,m)*vzbath_box(j,i,m) &
     &      + gkt*xbath_box(j,i,m) &
     &      + gkt*ybath_box(j,i,m) &
     &      + gkt*zbath_box(j,i,m)
      end do
      end do
      end do

!-----------------------------------------------------------------------
!     /*   ebaro =  barostats                                         */
!-----------------------------------------------------------------------

      ebaro = pressure*volume

      do j = 1, 3
      do i = 1, 3
         ebaro = ebaro + 0.5d0*boxmass(i,j)*vbox(i,j)*vbox(i,j)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian =  total Hamiltonian                           */
!-----------------------------------------------------------------------

      hamiltonian = hamiltonian_sys &
     &            + ebath_mode + ebath_cent + ebath_box + ebaro

!-----------------------------------------------------------------------
!     /*   kinetic energy of centroids                                */
!-----------------------------------------------------------------------

      dkin = 0.d0

      do i = 1, natom

         dkin = dkin + fictmass(i,1)*vux(i,1)*vux(i,1) &
     &               + fictmass(i,1)*vuy(i,1)*vuy(i,1) &
     &               + fictmass(i,1)*vuz(i,1)*vuz(i,1)

      end do

      dkin = 0.5d0*dkin

!-----------------------------------------------------------------------
!     /*   isotropic virial                                           */
!-----------------------------------------------------------------------

      vir_iso = vir(1,1) + vir(2,2) + vir(3,3)

!-----------------------------------------------------------------------
!     /*   isotropic pressure                                         */
!-----------------------------------------------------------------------

      pres_iso = ( 2.d0*dkin + vir_iso ) / ( 3.d0 * volume )

!-----------------------------------------------------------------------
!     /*   output                                                     */
!-----------------------------------------------------------------------

      call standard_output_MPI

      return
      end

