!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Apr 10, 2022 by M. Shiga
!      Description:     standard output of NPH ensemble
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine standard_nph_c1
!***********************************************************************
!=======================================================================
!
!     calculate ``Hamiltonian'' and ``temperature''
!     no thermostat.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   temp, ekin, boltz, dnmmass, ux, uy, uz, pres_iso, pressure, &
     &   ekin, potential, hamiltonian, hamiltonian_sys, fictmass, qkin, &
     &   vux, vuy, vuz, omega_p2, vir, pres, ebaro, volmass, vvol, &
     &   volume, nbead, natom, ndof, iprint_std

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: imode, iatom, i

      real(8) :: factqk

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      call standard_init( iset )

      if ( iprint_std .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   temp  =  instantaneous temperature                         */
!-----------------------------------------------------------------------

!     /*   calculate ekin =  fictitious kinetic energy   */
      call kinetic_energy

      temp = 2.d0*ekin/dble(ndof)/boltz

!-----------------------------------------------------------------------
!     /*   qkin  =  harmonic potential                                */
!-----------------------------------------------------------------------

      qkin = 0.d0

      do imode = 2, nbead
      do iatom = 1, natom
         factqk = 0.5d0*dnmmass(iatom,imode)*omega_p2
         qkin = qkin &
     &        + factqk*ux(iatom,imode)*ux(iatom,imode) &
     &        + factqk*uy(iatom,imode)*uy(iatom,imode) &
     &        + factqk*uz(iatom,imode)*uz(iatom,imode)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian_sys  =  system Hamiltonian                     */
!-----------------------------------------------------------------------

      hamiltonian_sys = ekin + qkin + potential

!-----------------------------------------------------------------------
!     /*   ebaro =  barostats                                         */
!-----------------------------------------------------------------------

      ebaro = 0.5d0*volmass*vvol*vvol + pressure*volume

!-----------------------------------------------------------------------
!     /*   hamiltonian =  total Hamiltonian                           */
!-----------------------------------------------------------------------

      hamiltonian = hamiltonian_sys + ebaro

!-----------------------------------------------------------------------
!     /*   pressure                                                   */
!-----------------------------------------------------------------------

      pres(:,:) =  vir(:,:)

      do i = 1, natom

         pres(1,1) = pres(1,1) + fictmass(i,1)*vux(i,1)*vux(i,1)
         pres(1,2) = pres(1,2) + fictmass(i,1)*vux(i,1)*vuy(i,1)
         pres(1,3) = pres(1,3) + fictmass(i,1)*vux(i,1)*vuz(i,1)
         pres(2,1) = pres(2,1) + fictmass(i,1)*vuy(i,1)*vux(i,1)
         pres(2,2) = pres(2,2) + fictmass(i,1)*vuy(i,1)*vuy(i,1)
         pres(2,3) = pres(2,3) + fictmass(i,1)*vuy(i,1)*vuz(i,1)
         pres(3,1) = pres(3,1) + fictmass(i,1)*vuz(i,1)*vux(i,1)
         pres(3,2) = pres(3,2) + fictmass(i,1)*vuz(i,1)*vuy(i,1)
         pres(3,3) = pres(3,3) + fictmass(i,1)*vuz(i,1)*vuz(i,1)

      end do

      pres(:,:) = pres(:,:) / volume

!-----------------------------------------------------------------------
!     /*   isotropic pressure                                         */
!-----------------------------------------------------------------------

      pres_iso = ( pres(1,1) + pres(2,2) + pres(3,3) ) / 3.d0

!-----------------------------------------------------------------------
!     /*   output                                                     */
!-----------------------------------------------------------------------

      call standard_output

      return
      end





!***********************************************************************
      subroutine standard_nph_c2
!***********************************************************************
!=======================================================================
!
!     calculate ``Hamiltonian'' and ``temperature''
!     no thermostat.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   temp, ekin, boltz, dnmmass, ux, uy, uz, pres_iso, pressure, &
     &   ekin, potential, hamiltonian, hamiltonian_sys, fictmass, qkin, &
     &   vux, vuy, vuz, omega_p2, vir, pres, ebaro, boxmass, vlog, &
     &   volume, nbead, natom, ndof, iprint_std

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: imode, iatom, i

      real(8) :: factqk

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      call standard_init( iset )

      if ( iprint_std .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   temp  =  instantaneous temperature                         */
!-----------------------------------------------------------------------

!     /*   calculate ekin =  fictitious kinetic energy   */
      call kinetic_energy

      temp = 2.d0*ekin/dble(ndof)/boltz

!-----------------------------------------------------------------------
!     /*   qkin  =  harmonic potential                                */
!-----------------------------------------------------------------------

      qkin = 0.d0

      do imode = 2, nbead
      do iatom = 1, natom
         factqk = 0.5d0*dnmmass(iatom,imode)*omega_p2
         qkin = qkin &
     &        + factqk*ux(iatom,imode)*ux(iatom,imode) &
     &        + factqk*uy(iatom,imode)*uy(iatom,imode) &
     &        + factqk*uz(iatom,imode)*uz(iatom,imode)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian_sys  =  system Hamiltonian                     */
!-----------------------------------------------------------------------

      hamiltonian_sys = ekin + qkin + potential

!-----------------------------------------------------------------------
!     /*   ebaro =  barostats                                         */
!-----------------------------------------------------------------------

      ebaro = 0.5d0*boxmass(1,1)*vlog*vlog + pressure*volume

!-----------------------------------------------------------------------
!     /*   hamiltonian =  total Hamiltonian                           */
!-----------------------------------------------------------------------

      hamiltonian = hamiltonian_sys + ebaro

!-----------------------------------------------------------------------
!     /*   pressure                                                   */
!-----------------------------------------------------------------------

      pres(:,:) =  vir(:,:)

      do i = 1, natom

         pres(1,1) = pres(1,1) + fictmass(i,1)*vux(i,1)*vux(i,1)
         pres(1,2) = pres(1,2) + fictmass(i,1)*vux(i,1)*vuy(i,1)
         pres(1,3) = pres(1,3) + fictmass(i,1)*vux(i,1)*vuz(i,1)
         pres(2,1) = pres(2,1) + fictmass(i,1)*vuy(i,1)*vux(i,1)
         pres(2,2) = pres(2,2) + fictmass(i,1)*vuy(i,1)*vuy(i,1)
         pres(2,3) = pres(2,3) + fictmass(i,1)*vuy(i,1)*vuz(i,1)
         pres(3,1) = pres(3,1) + fictmass(i,1)*vuz(i,1)*vux(i,1)
         pres(3,2) = pres(3,2) + fictmass(i,1)*vuz(i,1)*vuy(i,1)
         pres(3,3) = pres(3,3) + fictmass(i,1)*vuz(i,1)*vuz(i,1)

      end do

      pres(:,:) = pres(:,:) / volume

!-----------------------------------------------------------------------
!     /*   isotropic pressure                                         */
!-----------------------------------------------------------------------

      pres_iso = ( pres(1,1) + pres(2,2) + pres(3,3) ) / 3.d0

!-----------------------------------------------------------------------
!     /*   output                                                     */
!-----------------------------------------------------------------------

      call standard_output

      return
      end





!***********************************************************************
      subroutine standard_nph_pp
!***********************************************************************
!=======================================================================
!
!     calculate ``Hamiltonian'' and ``temperature''
!     no thermostat.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   temp, ekin, boltz, qkin, dnmmass, ux, uy, uz, vir_iso, ekin, &
     &   potential, hamiltonian, hamiltonian_sys, fictmass, pres_iso, &
     &   vux, vuy, vuz, omega_p2, vir, ebaro, boxmass, vbox, volume, &
     &   pressure, nbead, natom, ndof, iprint_std

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: imode, iatom, i, j

      real(8) :: factqk, dkin

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      call standard_init( iset )

      if ( iprint_std .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   temp  =  instantaneous temperature                         */
!-----------------------------------------------------------------------

!     /*   calculate ekin =  fictitious kinetic energy   */
      call kinetic_energy

      temp = 2.d0*ekin/dble(ndof)/boltz

!-----------------------------------------------------------------------
!     /*   qkin  =  harmonic potential                                */
!-----------------------------------------------------------------------

      qkin = 0.d0

      do imode = 2, nbead
      do iatom = 1, natom
         factqk = 0.5d0*dnmmass(iatom,imode)*omega_p2
         qkin = qkin &
     &        + factqk*ux(iatom,imode)*ux(iatom,imode) &
     &        + factqk*uy(iatom,imode)*uy(iatom,imode) &
     &        + factqk*uz(iatom,imode)*uz(iatom,imode)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian_sys  =  system Hamiltonian                     */
!-----------------------------------------------------------------------

      hamiltonian_sys = ekin + qkin + potential

!-----------------------------------------------------------------------
!     /*   ebaro =  barostats                                         */
!-----------------------------------------------------------------------

      ebaro = pressure*volume

      do j = 1, 3
      do i = 1, 3
         ebaro = ebaro + 0.5d0*boxmass(i,j)*vbox(i,j)*vbox(i,j)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian =  total Hamiltonian                           */
!-----------------------------------------------------------------------

      hamiltonian = hamiltonian_sys + ebaro

!-----------------------------------------------------------------------
!     /*   kinetic energy of centroids                                */
!-----------------------------------------------------------------------

      dkin = 0.d0

      do i = 1, natom

         dkin = dkin + fictmass(i,1)*vux(i,1)*vux(i,1) &
     &               + fictmass(i,1)*vuy(i,1)*vuy(i,1) &
     &               + fictmass(i,1)*vuz(i,1)*vuz(i,1)

      end do

      dkin = 0.5d0*dkin

!-----------------------------------------------------------------------
!     /*   isotropic virial                                           */
!-----------------------------------------------------------------------

      vir_iso = vir(1,1) + vir(2,2) + vir(3,3)

!-----------------------------------------------------------------------
!     /*   isotropic pressure                                         */
!-----------------------------------------------------------------------

      pres_iso = ( 2.d0*dkin + vir_iso ) / ( 3.d0 * volume )

!-----------------------------------------------------------------------
!     /*   output                                                     */
!-----------------------------------------------------------------------

      call standard_output

      return
      end

