!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Apr 10, 2022 by M. Shiga
!      Description:     standard output for multiscale simulations
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine standard_multi_me_MPI
!***********************************************************************
!=======================================================================
!
!     calculate ``Hamiltonian'' and ``temperature''
!     thermostat type III.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   temp, ekin, boltz, qkin, ekin, potential, hamiltonian, gkt, &
     &   hamiltonian_sys, omega_p2, dnmmass, ux, uy, uz, &
     &   xbath_cent, ybath_cent, zbath_cent, ebath_cent, ebath_mode, &
     &   vxbath_cent, vybath_cent, vzbath_cent, xbath, ybath, zbath, &
     &   vxbath, vybath, vzbath, ncolor, nnhc, ndof, nbead, natom, &
     &   iprint_std

      use qmmm_variables, only : &
     &   qmass_cent_multi_a, qmass_cent_multi_b, qmass_multi_a, &
     &   qmass_multi_b, layer

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: imode, iatom, i, j, k, m

      real(8) :: factqk

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      call standard_init_MPI( iset )

      if ( iprint_std .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   temp  =  instantaneous temperature                         */
!-----------------------------------------------------------------------

!     /*   calculate ekin =  fictitious kinetic energy   */
      call kinetic_energy

      temp = 2.d0*ekin/dble(ndof)/boltz

!-----------------------------------------------------------------------
!     /*   qkin  =  harmonic potential                                */
!-----------------------------------------------------------------------

      qkin = 0.d0

      do imode = 2, nbead
      do iatom = 1, natom
         factqk = 0.5d0*dnmmass(iatom,imode)*omega_p2
         qkin = qkin &
     &        + factqk*ux(iatom,imode)*ux(iatom,imode) &
     &        + factqk*uy(iatom,imode)*uy(iatom,imode) &
     &        + factqk*uz(iatom,imode)*uz(iatom,imode)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian_sys  =  system Hamiltonian                     */
!-----------------------------------------------------------------------

      hamiltonian_sys = ekin + qkin + potential

!-----------------------------------------------------------------------
!     /*   ebath_cent  =  thermostats attached to centroid            */
!-----------------------------------------------------------------------

      ebath_cent = 0.d0

      do j = 1, natom
         if ( layer(j)(1:1) .ne. 'A' ) cycle
         do m = 1, ncolor
         do i = 1, nnhc
            ebath_cent = ebath_cent &
     &         + 0.5d0*qmass_cent_multi_a(i,m) &
     &                *vxbath_cent(j,i,m)*vxbath_cent(j,i,m) &
     &         + 0.5d0*qmass_cent_multi_a(i,m) &
     &                *vybath_cent(j,i,m)*vybath_cent(j,i,m) &
     &         + 0.5d0*qmass_cent_multi_a(i,m) &
     &                *vzbath_cent(j,i,m)*vzbath_cent(j,i,m) &
     &         + gkt*xbath_cent(j,i,m) &
     &         + gkt*ybath_cent(j,i,m) &
     &         + gkt*zbath_cent(j,i,m)
         end do
         end do
      end do

      do j = 1, natom
         if ( layer(j)(1:1) .ne. 'B' ) cycle
         do m = 1, ncolor
         do i = 1, nnhc
            ebath_cent = ebath_cent &
     &         + 0.5d0*qmass_cent_multi_b(i,m) &
     &                *vxbath_cent(j,i,m)*vxbath_cent(j,i,m) &
     &         + 0.5d0*qmass_cent_multi_b(i,m) &
     &                *vybath_cent(j,i,m)*vybath_cent(j,i,m) &
     &         + 0.5d0*qmass_cent_multi_b(i,m) &
     &                *vzbath_cent(j,i,m)*vzbath_cent(j,i,m) &
     &         + gkt*xbath_cent(j,i,m) &
     &         + gkt*ybath_cent(j,i,m) &
     &         + gkt*zbath_cent(j,i,m)
         end do
         end do
      end do

!-----------------------------------------------------------------------
!     /*   ebath_mode  =  thermostats attached to non-centroid        */
!-----------------------------------------------------------------------

      ebath_mode = 0.d0

      do k = 1, natom
         if ( layer(k)(1:1) .ne. 'A' ) cycle
         do i = 2, nbead
         do j = 1, nnhc
            ebath_mode = ebath_mode &
     &         + 0.5d0*qmass_multi_a(i)*vxbath(k,j,i)*vxbath(k,j,i) &
     &         + 0.5d0*qmass_multi_a(i)*vybath(k,j,i)*vybath(k,j,i) &
     &         + 0.5d0*qmass_multi_a(i)*vzbath(k,j,i)*vzbath(k,j,i) &
     &         + gkt*xbath(k,j,i) &
     &         + gkt*ybath(k,j,i) &
     &         + gkt*zbath(k,j,i)
         end do
         end do
      end do

      do k = 1, natom
         if ( layer(k)(1:1) .ne. 'B' ) cycle
         do i = 2, nbead
         do j = 1, nnhc
            ebath_mode = ebath_mode &
     &        + 0.5d0*qmass_multi_b(i)*vxbath(k,j,i)*vxbath(k,j,i) &
     &        + 0.5d0*qmass_multi_b(i)*vybath(k,j,i)*vybath(k,j,i) &
     &        + 0.5d0*qmass_multi_b(i)*vzbath(k,j,i)*vzbath(k,j,i) &
     &        + gkt*xbath(k,j,i) &
     &        + gkt*ybath(k,j,i) &
     &        + gkt*zbath(k,j,i)
         end do
         end do
      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian =  total Hamiltonian                           */
!-----------------------------------------------------------------------

      hamiltonian = hamiltonian_sys + ebath_mode + ebath_cent

!-----------------------------------------------------------------------
!     /*   output                                                     */
!-----------------------------------------------------------------------

      call standard_output_MPI

      return
      end

